import requests
from bs4 import BeautifulSoup
import json
import datetime

# Funkce pro extrakci titulů z jedné konkrétní tabulky
def extract_titles_from_table(table_soup):
    titles = []
    if not table_soup:
        return []
    
    # Hledáme všechny řádky v tabulce
    rows = table_soup.find_all('tr')
    
    for row in rows:
        # Hledáme odkaz s názvem
        link = row.find('a')
        if link:
            title = link.get_text(strip=True)
            # Ochrana: někdy tam jsou prázdné linky nebo "View more", chceme jen smysluplné názvy
            if title and len(title) > 1:
                titles.append(title)
        
        # Jakmile máme 10 titulů, končíme (pro tuto tabulku)
        if len(titles) >= 10:
            break
            
    return titles

# Funkce pro načtení dat platformy (Shows i Movies)
def fetch_platform_data(platform_name, platform_url):
    print(f"Zpracovávám {platform_name}...")
    data = {
        "TV Shows": [],
        "Movies": []
    }
    try:
        html = requests.get(platform_url, headers={'User-Agent': 'Mozilla/5.0'}).text
        soup = BeautifulSoup(html, 'html.parser')
        
        # Najdeme všechny tabulky na stránce
        tables = soup.find_all('table')
        
        # FlixPatrol má obvykle: 1. tabulka = TV Shows, 2. tabulka = Movies
        if len(tables) >= 1:
            data["TV Shows"] = extract_titles_from_table(tables[0])
        
        if len(tables) >= 2:
            data["Movies"] = extract_titles_from_table(tables[1])
            
    except Exception as e:
        print(f"Chyba při zpracování {platform_url}: {e}")
    
    return data

# Funkce pro načtení kalendáře premiér (Původní funkční kód)
def fetch_calendar():
    url = "https://flixpatrol.com/calendar/"
    print("Načítám kalendář premiér...")
    try:
        html = requests.get(url, headers={'User-Agent': 'Mozilla/5.0'}).text
        soup = BeautifulSoup(html, 'html.parser')
        events = []
        rows = soup.select('table tr')
        for row in rows[1:]:
            cols = [c.get_text(strip=True) for c in row.find_all('td')]
            if len(cols) >= 5:
                event = {
                    "release_date": cols[0],
                    "title": cols[1],
                    "type": cols[2],
                    "country": cols[3],
                    "platform": cols[4],
                    "genres": cols[5:]
                }
                events.append(event)
        return events
    except Exception as e:
        print(f"Chyba při načítání kalendáře: {e}")
        return []

# === HLAVNÍ ČÁST ===

platforms = {
    'Netflix': 'https://flixpatrol.com/top10/netflix/',
    'HBO Max': 'https://flixpatrol.com/top10/hbo-max/',
    'Disney+': 'https://flixpatrol.com/top10/disney/',
    'Prime Video': 'https://flixpatrol.com/top10/amazon-prime/',
    'Apple TV+': 'https://flixpatrol.com/top10/apple-tv/',
    'Paramount+': 'https://flixpatrol.com/top10/paramount-plus/'
}

# 1. Načtení TOP 10 pro všechny platformy
top10_data = {}
for name, url in platforms.items():
    top10_data[name] = fetch_platform_data(name, url)

# 2. Načtení kalendáře
calendar_data = fetch_calendar()

# 3. Sestavení reportu
report = {
    "date": datetime.datetime.now().strftime("%Y-%m-%d"),
    "top10": top10_data,
    "calendar": calendar_data
}

# 4. Uložení do JSON
with open("streaming_report.json", "w", encoding="utf-8") as f:
    json.dump(report, f, indent=4, ensure_ascii=False)

# 5. Vytvoření přehledného TXT reportu
with open("streaming_report.txt", "w", encoding="utf-8") as f:
    f.write(f"=== STREAMING REPORT ({datetime.datetime.now().strftime('%d.%m.%Y')}) ===\n\n")
    
    for name, content in top10_data.items():
        f.write(f"--- {name.upper()} ---\n")
        
        f.write("📺 TV Shows:\n")
        if content["TV Shows"]:
            for i, t in enumerate(content["TV Shows"], start=1):
                f.write(f"  {i}. {t}\n")
        else:
            f.write("  (Data nenalezena)\n")
            
        f.write("\n🎬 Movies:\n")
        if content["Movies"]:
            for i, t in enumerate(content["Movies"], start=1):
                f.write(f"  {i}. {t}\n")
        else:
            f.write("  (Data nenalezena)\n")
        
        f.write("\n" + "="*30 + "\n\n")

    f.write("=== KALENDÁŘ PREMIÉR ===\n\n")
    for event in calendar_data:
        genres = ', '.join(event['genres']) if event['genres'] else ''
        f.write(f"📅 {event['release_date']} | {event['platform']} | {event['type']} | {event['title']} ({genres})\n")

print("\n✅ Hotovo! Soubory byly vytvořeny:")
print(" - streaming_report.json (Strukturovaná data)")
print(" - streaming_report.txt (Čitelný přehled)")