
# -*- coding: utf-8 -*-
"""
Series actions helpers: lokální akce pro označování pokroku v seriálech.
Oddělené od routeru pro lepší údržbu a přehled.
"""
import re
import os
from urllib.parse import unquote, urlparse
import json
import xbmc
import xbmcgui
import xbmcaddon
import series_manager  # top-level modul vedle yawsp.py



ADDON_ID = 'plugin.video.mmirousek'
ADDON = xbmcaddon.Addon(ADDON_ID)
ADDON_NAME = ADDON.getAddonInfo('name')


def series_mark_episode(params, _addon=ADDON, _profile=None):
    """
    Označení jedné epizody (zhlédnuté / rozkoukané / reset).
    Očekává: series_name, season (int/str), episode (int/str), progress (float 0..100)
    """
    try:
        if _profile is None:
            _profile = xbmc.translatePath(_addon.getAddonInfo('profile'))
        sm = series_manager.SeriesManager(_addon, _profile)

        series_name = params.get('series_name')
        season = int(params.get('season'))
        episode = int(params.get('episode'))
        progress = float(params.get('progress'))

        sm.mark_episode_progress(series_name, season, episode, progress, set_by='manual')
        xbmcgui.Dialog().notification(ADDON_NAME, "Epizoda označena.", xbmcgui.NOTIFICATION_INFO, 2000)
        xbmc.executebuiltin('Container.Refresh()')
    except Exception as e:
        xbmc.log(f'[SeriesActions] CHYBA series_mark_episode – {e}', xbmc.LOGERROR)
        xbmcgui.Dialog().notification(ADDON_NAME, "Chyba při označení epizody", xbmcgui.NOTIFICATION_ERROR, 3000)


def series_mark_season(params, _addon=ADDON, _profile=None):
    """
    Označení celé sezóny (bulk) – nastaví progress všem epizodám dané sezóny,
    přepočítá stav sezóny a uloží.
    Očekává: series_name (str), tmdb_id (str/int – volitelné), season (str/int), progress (float 0..100)
    """
    try:
        if _profile is None:
            _profile = xbmc.translatePath(_addon.getAddonInfo('profile'))
        sm = series_manager.SeriesManager(_addon, _profile)

        series_name = params.get('series_name')
        tmdb_id = params.get('tmdb_id')  # může být None
        season = str(params.get('season')).strip()
        progress = float(params.get('progress'))

        # Načti data podle TMDb ID (pokud existuje), jinak podle názvu
        data = None
        if tmdb_id:
            file_path = os.path.join(sm.series_db_path, f"tmdb_{tmdb_id}.json")
            if os.path.exists(file_path):
                with open(file_path, 'r', encoding='utf-8') as f:
                    data = json.load(f)
        if not data:
            data = sm.load_series_data(series_name)
        if not data:
            xbmcgui.Dialog().notification(ADDON_NAME, "Data seriálu nenalezena", xbmcgui.NOTIFICATION_ERROR, 3000)
            return

        if season not in data.get('seasons', {}):
            xbmcgui.Dialog().notification(ADDON_NAME, f"Sezóna {season} nenalezena", xbmcgui.NOTIFICATION_ERROR, 3000)
            return

        # Označ všechny epizody dané sezóny
        for ep_num in list(data['seasons'][season].keys()):
            ep = data['seasons'][season][ep_num]
            ep['progress'] = round(progress, 1)
            ep['watched'] = progress >= 90.0
            if ep['watched']:
                ep['play_count'] = ep.get('play_count', 0) + 1
                ep['last_watched_at'] = xbmc.getInfoLabel('System.Date') + 'T' + xbmc.getInfoLabel('System.Time')
            ep['set_by'] = 'manual'

        # Přepočítej stav sezóny a ulož
        sm.recalc_and_save_season_state(data, season)

        xbmcgui.Dialog().notification(
            ADDON_NAME,
            f"Sezóna {season} označena jako {'zhlédnutá' if progress >= 90 else 'resetována'}",
            xbmcgui.NOTIFICATION_INFO, 2500
        )

        # Návrat na detail seriálu (seznam sezón)
        xbmc.executebuiltin(f'Container.Update(plugin://{ADDON_ID}/?action=series_detail&series_name={series_name})')

    except Exception as e:
        xbmc.log(f'[SeriesActions] CHYBA series_mark_season – {e}', xbmc.LOGERROR)
        xbmcgui.Dialog().notification(ADDON_NAME, "Chyba při označení sezóny", xbmcgui.NOTIFICATION_ERROR, 3000)



def csfd_lookup_enhanced(series_name: str,
                         max_comments: int = 5,
                         top_n: int = 12):
    """
    ČSFD lookup 2025 – 100% bez „více / méně“ (funguje 23.11.2025)
    """
    import os, re, json, time, hashlib
    import xbmcaddon, xbmcgui, xbmcvfs
    import requests

    ADDON = xbmcaddon.Addon('plugin.video.mmirousek')
    BASE_URL = "https://www.csfd.cz"
    PROFILE = xbmcvfs.translatePath(ADDON.getAddonInfo('profile'))
    CACHE_DIR = os.path.join(PROFILE, 'csfd_cache')
    if not xbmcvfs.exists(CACHE_DIR):
        xbmcvfs.mkdirs(CACHE_DIR)

    headers = {
        "User-Agent": "Mozilla/5.0 (Kodi Plugin)",
        "Referer": BASE_URL,
        "Accept-Language": "cs,en;q=0.8",
    }

    # ====================== CACHE ======================
    def _ck(url): return hashlib.sha1(url.encode('utf-8')).hexdigest()
    def _cache_read(url):
        fp = os.path.join(CACHE_DIR, _ck(url) + '.json')
        if xbmcvfs.exists(fp):
            with xbmcvfs.File(fp) as f: raw = f.read()
            try:
                obj = json.loads(raw)
                if time.time() - obj.get('_time', 0) < 86400: return obj.get('payload')
            except: pass
        return None

    def _cache_write(url, payload):
        fp = os.path.join(CACHE_DIR, _ck(url) + '.json')
        with xbmcvfs.File(fp, 'w') as f:
            f.write(json.dumps({'_time': time.time(), 'payload': payload}, ensure_ascii=False))

    def _get(url):
        # TOTO JE KLÍČOVÉ: ?section=info rovnou rozbalí celý blok
        if '/prehled/' in url and not any(x in url for x in ['?', '#']):
            url = url.rstrip('/') + '?section=info'
        elif '/prehled/' in url and '?' in url:
            url = url.split('?')[0] + '?section=info'

        cached = _cache_read(url)
        if cached: return cached

        wait = 1.5
        for _ in range(6):
            try:
                r = requests.get(url, headers=headers, timeout=12)
                if r.status_code == 200:
                    html = r.text
                    _cache_write(url, html)
                    return html
                elif r.status_code == 429:
                    xbmcgui.Dialog().notification("ČSFD", f"Rate limit – čekám {int(wait)}s", xbmcgui.NOTIFICATION_INFO, 2000)
                    time.sleep(wait)
                    wait = min(wait * 2, 20)
            except:
                time.sleep(1)
        return None

    # ====================== VYHLEDÁVÁNÍ (tvoje původní – funguje) ======================
    def get_candidates(html):
        results = []
        if not html: return results

        for m in re.finditer(r'href="(/serial/[^"]+)"[^>]*>([^<]+)</a>', html, re.IGNORECASE):
            href, title = m.group(1), m.group(2).strip()
            if title and all(title.lower() != t.lower() for t, _ in results):
                results.append((title, BASE_URL + href))
            if len(results) >= top_n: break

        if not results:
            for m in re.finditer(r'href="(/film/[^"]+)"[^>]*>([^<]+)</a>', html, re.IGNORECASE):
                href, title = m.group(1), m.group(2).strip()
                if title and all(title.lower() != t.lower() for t, _ in results):
                    results.append((title, BASE_URL + href))
                if len(results) >= top_n: break
        return results

# ====================== DETAIL PARSER ======================

    def parse_detail(html: str) -> dict:
        """
        Čistě regex parser: vrací dict vždy; nikdy None.
        """
        import re
        d = {"rating": "N/A", "year": "", "countries": [], "genres": [],
            "original_title": "", "poster": "", "crew": {}, "plot": "", "comments": []}
        if not html:
            return d

        flags = re.DOTALL | re.IGNORECASE

        # --- 1) Vyčištění rušivých bloků do html2 (musí být před rating blokem!) ---
        html2 = re.sub(r'<a[^>]*class="button[^"]*"[^>]*>.*?</a>', '', html, flags=flags)
        html2 = re.sub(r'<span[^>]*class="[^"]*more[^"]*"[^>]*>.*?</span>', '', html2, flags=flags)
        html2 = re.sub(r'<div[^>]*class="[^"]*more[^"]*"[^>]*>.*?</div>', '', html2, flags=flags)

        # --- 2) RATING: podpora více variant (data atribut / číslo v elementu / fallback) ---
        rating_val = None

        # Varianta A: atribut data-rating="72" (někdy i data-percent="72")
        m = re.search(r'data-(?:rating|percent)\s*=\s*"(\d{1,3})"', html2, flags)
        if m:
            rating_val = m.group(1)

        # Varianta B: číslo uvnitř elementu s class="rating-average" (např. ...> 72 %)
        if rating_val is None:
            m = re.search(r'rating-average[^>]*>\s*([0-9]{1,3})\s*%', html2, flags)
            if m:
                rating_val = m.group(1)

        # Varianta C: obecný fallback (původní volnější tvar)
        if rating_val is None:
            m = re.search(r'rating-average[^0-9]*([0-9]{1,3})\s*%', html2, flags)
            if m:
                rating_val = m.group(1)

        if rating_val is not None:
            try:
                v = int(rating_val)
                if 0 <= v <= 100:
                    d["rating"] = f"{v}%"
            except Exception:
                pass

        # --- 3) ROK ---
        m = re.search(r'<div[^>]*class="origin"[^>]*>.*?\((\d{4})\)', html2, flags)
        if m:
            d["year"] = m.group(1)

        # --- 4) ZEMĚ ---
        for m in re.finditer(r'class="flag"[^>]*title="([^"]+)"', html2, flags):
            c = (m.group(1) or '').strip()
            if c and c not in d["countries"]:
                d["countries"].append(c)

        # --- 5) ŽÁNRY ---
        m = re.search(r'<div[^>]*class="genres"[^>]*>(.*?)</div', html2, flags)
        if m:
            raw = re.sub(r'<[^>]+>', '', m.group(1))
            d["genres"] = [g.strip() for g in raw.split('/') if g.strip()]

        # --- 6) ORIGINÁLNÍ NÁZEV (z <ul class="film-names"> -> druhá položka) ---
        m = re.search(r'<ul[^>]*class="film-names"[^>]*>([\s\S]*?)</ul>', html2, flags)
        if m:
            names_block = re.sub(r'\s+', ' ', re.sub(r'<[^>]+>', '', m.group(1))).strip()
            parts = [p.strip() for p in names_block.split('/') if p.strip()]
            if len(parts) >= 2:
                d["original_title"] = parts[1]

        # --- 7) POSTER (image|img).pmgstatic.com/... (jpg|jpeg|png) ---
        m = re.search(r'(?:image|img)\.pmgstatic\.com/[^"]+\.(?:jpg|jpeg|png)', html2, flags)
        if m:
            pic = m.group(0)
            d["poster"] = pic if pic.startswith('http') else ("https://" + pic)

        # --- 8) CREW (Režie / Scénář / Kamera / Hudba) ---
        for role in ["Režie", "Scénář", "Kamera", "Hudba"]:
            block = re.search(fr'<h4>{role}:</h4>(.*?)</div>', html2, flags)
            if block:
                names = re.findall(r'<a [^>]*>([^<]+)</a>', block.group(1))
                if names:
                    d["crew"][role] = ", ".join([n.strip() for n in names if n.strip()])

        # --- 9) OBSAH ---
        m = re.search(r'<div[^>]*class="plot-text"[^>]*>\s*<p>\s*(.*?)\s*</p>', html2, flags)
        if m:
            raw_plot = re.sub(r'<br\s*/?>', '\n', m.group(1))
            d["plot"] = re.sub(r'<[^>]+>', '', raw_plot).strip()
        else:
            m2 = re.search(r'class="(?:plot-full|plot-full hidden)"[^>]*>\s*([^<]+?)\s*<', html2, flags)
            if m2:
                raw_plot = re.sub(r'<br\s*/?>', '\n', m2.group(1))
                d["plot"] = re.sub(r'<[^>]+>', '', raw_plot).strip()

        # --- 10) KOMENTÁŘE ---
        comments_block_m = re.search(r'<h4>Komentáře:</h4>(.*?)<aside', html2, flags)
        if comments_block_m:
            comments_block = comments_block_m.group(1)
            for mm in re.finditer(r'class="review-content"[^>]*>(.*?)</div>', comments_block, flags):
                txt = mm.group(1)
                txt = re.sub(r'<a[^>]*class="button[^"]*"[^>]*>.*?</a>', '', txt, flags=flags)
                txt = re.sub(r'<br\s*/?>', '\n', txt)
                txt = re.sub(r'<[^>]+>', '', txt).strip()
                if txt and len(txt) > 15:
                    d["comments"].append(txt)
                if len(d["comments"]) >= max_comments:
                    break

        return d
    
            # ====================== WORKFLOW ======================
    search_url = f"{BASE_URL}/hledat/?q={series_name.replace(' ', '+')}"
    html = _get(search_url)
    if not html:
        xbmcgui.Dialog().notification("ČSFD", "Chyba připojení", xbmcgui.NOTIFICATION_ERROR, 3000)
        return

    candidates = get_candidates(html)
    if not candidates:
        xbmcgui.Dialog().notification("ČSFD", "Nic nenalezeno", xbmcgui.NOTIFICATION_WARNING, 4000)
        return

    idx = xbmcgui.Dialog().select(f"ČSFD • {series_name}", [t for t, _ in candidates])
    if idx < 0: return

    chosen_title, chosen_url = candidates[idx]
    detail_html = _get(chosen_url)          # ← tady se přidá ?section=info
    info = parse_detail(detail_html)

    # ====================== ČISTÝ VÝSTUP ======================
    lines = [
        f"Název: {chosen_title}",
    ]
    if info["original_title"]: lines.append(f"Originál: {info['original_title']}")
    if info["rating"] != "N/A": lines.append(f"Hodnocení: {info['rating']}")
    if info["year"]: lines.append(f"Rok: {info['year']}")
    if info["countries"]: lines.append(f"Země: {', '.join(info['countries'])}")
    if info["genres"]: lines.append(f"Žánry: {', '.join(info['genres'])}")
    if "Režie" in info["crew"]: lines.append(f"Režie: {info['crew']['Režie']}")
    if "Scénář" in info["crew"]: lines.append(f"Scénář: {info['crew']['Scénář']}")
    if info["plot"]: lines += ["", "Obsah:", info["plot"]]
    lines.append(f"\nURL: {chosen_url}")
    if info["comments"]:
        lines += ["", "Komentáře:"] + info["comments"]
    else:
        lines.append("\n(žádné komentáře)")

    xbmcgui.Dialog().textviewer(f"ČSFD • {chosen_title}", "\n".join(lines))