# -*- coding: utf-8 -*-
# resources/lib/scrobble.py
# Scrobbling do Trakt + lokální historie
# - jednotný log prefix [SCROBBLE]
# - TMDb parsování epizod přes tmdb_utils
# - ukládání stream_url do lokální historie
# - žádné přímé čtení Trakt client_id/token (jen get_trakt_headers())

import json
import requests
import xbmc
import xbmcgui
import xbmcaddon
import xbmcvfs
from datetime import datetime


from resources.lib import local_history
from resources.lib import tmdb_utils
from resources.lib.trakt.trakt_service import get_trakt_headers  # používáme JEN tohle

PREFIX = "[SCROBBLE]"
ADDON_ID = 'plugin.video.mmirousek'
TRAKT_API_SCROBBLE_URL = 'https://api.trakt.tv/scrobble'

# stav (info/debug)
_last_known_progress = 0.0
_progress_updated = False
_ADDON_INSTANCE = None


# ============================== UTIL / COMMON ===============================

def _get_addon_instance():
    """Lazy načtení Addon instance s ošetřením chyb."""
    global _ADDON_INSTANCE
    if _ADDON_INSTANCE is None:
        try:
            _ADDON_INSTANCE = xbmcaddon.Addon(ADDON_ID)
        except Exception as e:
            xbmc.log(f'{PREFIX}[UTIL] CHYBA inicializace ADDON: {e}', xbmc.LOGERROR)
            _ADDON_INSTANCE = False
    return _ADDON_INSTANCE if _ADDON_INSTANCE is not False else None


def _get_log_tag():
    """Sestaví čitelný tag s názvem addonu."""
    ADDON = _get_addon_instance()
    name = ADDON.getAddonInfo("name") if ADDON else "StreamManager"
    return f'{PREFIX}[{name}]'


def _clean_settings():
    """Vyčistí dočasné položky scrobble stavu v settings."""
    try:
        addon = _get_addon_instance()
        if addon:
            addon.setSetting('trakt_last_payload', '')
            addon.setSetting('trakt_last_progress', '')
            addon.setSetting('trakt_progress_updated', '')
    except Exception as e:
        xbmc.log(f'{PREFIX}[UTIL] Chyba při čištění nastavení: {e}', xbmc.LOGERROR)


# ============================= DATA O PŘEHRÁVANÉM ===========================

def get_item_data(fallback_title=None):
    """
    Vrátí payload pro Trakt scrobble:
      - pro epizodu: { "name", "tmdb_id", "show": {"ids":{"tvdb":...}}, "episode": {"season":..,"number":..} }
      - pro film:    { "name", "tmdb_id", "movie": {"ids":{"tmdb":...}} }
      - fallback:    { "name", "tmdb_id": "0", "media_type": "movie"|"episode" }
    """
    LOG = _get_log_tag()

    title = xbmc.getInfoLabel('ListItem.Title')
    if not title or title == 'ListItem.Title':
        title = fallback_title
    if not title:
        xbmc.log(f'{LOG} [UTIL] Žádný název – selhání.', xbmc.LOGWARNING)
        return None

    xbmc.log(f'{LOG} [UTIL] Zpracovávám: "{title}"', xbmc.LOGDEBUG)

    # 1) Pokus o epizodu (SxxEyy / 1x02)
    try:
        parsed = tmdb_utils.parse_episode_title(title)
    except Exception as e:
        parsed = None
        xbmc.log(f'{LOG} [UTIL] parse_episode_title chyba: {e}', xbmc.LOGWARNING)

    if parsed:
        show_name, season, episode = parsed
        xbmc.log(f'{LOG} [UTIL] Seriál detekován: "{show_name}" S{season}E{episode}', xbmc.LOGINFO)
        tv_data = tmdb_utils.search_tmdb_tv(show_name)
        if tv_data and tv_data.get('tvdb_id'):
            payload = {
                "name": tv_data["title"],
                "tmdb_id": str(tv_data["tmdb_id"]),
                "show": {"ids": {"tvdb": int(tv_data['tvdb_id'])}},
                "episode": {"season": season, "number": episode}
            }
            xbmc.log(f'{LOG} [UTIL] → TVDB:{tv_data["tvdb_id"]} ({tv_data["title"]})', xbmc.LOGINFO)
            return payload
        xbmc.log(f'{LOG} [UTIL] TMDb nenašel show podle "{show_name}" – zkusím film/fallback.', xbmc.LOGWARNING)

    # 2) Film
    movie_data = tmdb_utils.search_tmdb_movie(title)
    if movie_data and movie_data.get('tmdb_id'):
        payload = {
            "name": movie_data["title"],
            "tmdb_id": str(movie_data["tmdb_id"]),
            "movie": {"ids": {"tmdb": movie_data['tmdb_id']}}
        }
        xbmc.log(f'{LOG} [UTIL] → TMDB:{movie_data["tmdb_id"]} {movie_data["title"]}', xbmc.LOGINFO)
        return payload

    # 3) Fallback – vyčištěný název, fiktivní tmdb_id="0"
    final_clean_name = tmdb_utils.clean_title(title)
    is_episode = bool(tmdb_utils.parse_episode_title(title))
    fb_payload = {
        "name": final_clean_name,
        "tmdb_id": "0",
        "media_type": "episode" if is_episode else "movie"
    }
    xbmc.log(f'{LOG} [UTIL] TMDb: nic nenašel, fallback payload: {fb_payload}', xbmc.LOGWARNING)
    return fb_payload


# ================================== TIK =====================================

def _update_progress():
    """
    Každou sekundu (volá service.py):
      - pokud není lock, jednorázově připraví a zamkne payload,
      - uloží progress (last_progress) a flag progress_updated=true.
    """
    addon = xbmcaddon.Addon(ADDON_ID)

    # Respect lock
    if addon.getSetting('trakt_payload_locked') == 'true':
        xbmc.log(f'{PREFIX}[TIK] Payload zamknutý – nepřepisuji', xbmc.LOGDEBUG)
    else:
        # Jednorázový fallback – pokus o payload v tiku
        title = xbmc.getInfoLabel('Player.Title')
        if title and title not in ('Player.Title', ''):
            try:
                payload = get_item_data(title)
                if payload:
                    addon.setSetting('trakt_last_payload', json.dumps(payload))
                    addon.setSetting('trakt_payload_locked', 'true')
                    xbmc.executebuiltin('UpdateAddonRepos')  # lehký nudge pro UI cache
                    xbmc.log(f'{PREFIX}[TIK] Payload uložen a zamknut', xbmc.LOGINFO)
            except Exception as e:
                xbmc.log(f'{PREFIX}[TIK] Fallback chyba při přípravě payloadu: {e}', xbmc.LOGWARNING)

    # Ulož progres
    player = xbmc.Player()
    if player.isPlaying():
        try:
            current = player.getTime()
            total = player.getTotalTime()
            if total > 0:
                progress = (current / total) * 100.0
                addon.setSetting('trakt_last_progress', str(progress))
                addon.setSetting('trakt_progress_updated', 'true')
                global _last_known_progress, _progress_updated
                _last_known_progress = progress
                _progress_updated = True
                xbmc.log(f'{PREFIX}[TIK] Progress uložen: {progress:.1f}%', xbmc.LOGDEBUG)
        except Exception as e:
            xbmc.log(f'{PREFIX}[TIK] Chyba při zjišťování času: {e}', xbmc.LOGWARNING)
    else:
        # Nic nehraje → nic neukládáme
        pass


# ================================= START ====================================

def scrobble_start(fallback_title=None):
    """
    Připraví payload (z historie nebo z ListItemu), uloží ho do settings a pošle Trakt /start.
    """
    LOG = _get_log_tag()
    ADDON = xbmcaddon.Addon(ADDON_ID)
    item = None

    # 1) Start z lokální historie (pokud přehráno z menu „Naposledy sledované“)
    history_item_json = ADDON.getSetting('trakt_launch_history_item')
    if history_item_json:
        ADDON.setSetting('trakt_launch_history_item', '')  # vyčistit po použití
        history_item = json.loads(history_item_json)
        trakt_payload = history_item.get('trakt_payload')
        if trakt_payload:
            item = history_item.copy()
            item.update(trakt_payload)
            xbmc.log(f'{LOG} [START] Rekonstruován Trakt payload z lokální historie.', xbmc.LOGINFO)
        else:
            xbmc.log(f'{LOG} [START] Chybí trakt_payload v lokální historii, pokračuji z ListItem.', xbmc.LOGWARNING)

    # 2) Fallback: z ListItemu
    if not item:
        item = get_item_data(fallback_title)
        if not item:
            xbmc.log(f'{LOG} [START] Žádná data – přeskočeno.', xbmc.LOGWARNING)
            return

    # 2a) Získat stream URL pro lokální historii
    try:
        player = xbmc.Player()
        if player.isPlaying():
            stream_url = player.getPlayingFile()
            if stream_url:
                item['stream_url'] = stream_url
                xbmc.log(f'{LOG} [START] Získán stream URL pro historii.', xbmc.LOGINFO)
    except Exception as e:
        xbmc.log(f'{LOG} [START] Chyba při získávání stream URL: {e}', xbmc.LOGWARNING)

    # 3) Uložit payload + init progress
    addon_inst = _get_addon_instance()
    if not addon_inst:
        xbmc.log(f'{LOG} [START] Nelze získat instanci addonu – konec.', xbmc.LOGERROR)
        return
    try:
        addon_inst.setSetting('trakt_last_payload', json.dumps(item))
        addon_inst.setSetting('trakt_last_progress', '0.0')
        addon_inst.setSetting('trakt_progress_updated', 'true')
        xbmc.log(f'{LOG} [START] Payload a progress init uložen.', xbmc.LOGINFO)
        xbmc.sleep(300)  # malá prodleva
    except Exception as e:
        xbmc.log(f'{LOG} [START] Chyba uložení payloadu/init: {e}', xbmc.LOGERROR)

    # 4) Odeslat START na Trakt (pokud povoleno a autorizováno)
    if not ADDON.getSettingBool('trakt_enabled'):
        xbmc.log(f'{LOG} [START] Trakt je zakázán – ignoruji.', xbmc.LOGINFO)
        return

    headers = get_trakt_headers()
    if not headers:
        xbmc.log(f'{LOG} [START] Trakt není autorizován (get_trakt_headers() vrátilo None).', xbmc.LOGINFO)
        return

    # Očistit payload pro Trakt (bez lokálních klíčů)
    trakt_payload = item.copy()
    for k in ('stream_url', 'name', 'tmdb_id'):
        trakt_payload.pop(k, None)

    data = {"progress": 0.1, **trakt_payload}

    try:
        r = requests.post(f"{TRAKT_API_SCROBBLE_URL}/start", json=data, headers=headers, timeout=10)
        if r.status_code in (201, 409):
            xbmc.log(f'{LOG} [START] ÚSPĚCH', xbmc.LOGINFO)
        else:
            xbmc.log(f'{LOG} [START] CHYBA HTTP {r.status_code}', xbmc.LOGERROR)
    except Exception as e:
        xbmc.log(f'{LOG} [START] Síťová chyba: {e}', xbmc.LOGERROR)


# ================================= PAUSE ====================================

def scrobble_pause():
    """Odešle Trakt /pause (pokud povoleno a autorizováno)."""
    LOG = _get_log_tag()

    # Načíst uložený payload
    item = None
    try:
        addon = xbmcaddon.Addon(ADDON_ID)
        payload_json = addon.getSetting('trakt_last_payload')
        if payload_json:
            item = json.loads(payload_json)
            xbmc.log(f'{LOG} [PAUSE] Payload načten.', xbmc.LOGINFO)
    except Exception as e:
        xbmc.log(f'{LOG} [PAUSE] Chyba načtení payloadu: {e}', xbmc.LOGWARNING)

    if not item:
        xbmc.log(f'{LOG} [PAUSE] Žádný payload – přeskočeno.', xbmc.LOGWARNING)
        return

    addon = _get_addon_instance()
    if addon is None or not addon.getSettingBool('trakt_enabled'):
        xbmc.log(f'{LOG} [PAUSE] Trakt je zakázán – ignoruji.', xbmc.LOGINFO)
        return

    headers = get_trakt_headers()
    if not headers:
        xbmc.log(f'{LOG} [PAUSE] Trakt není autorizován – ignoruji.', xbmc.LOGINFO)
        return

    trakt_payload = item.copy()
    for k in ('stream_url', 'name', 'tmdb_id'):
        trakt_payload.pop(k, None)
    data = {"progress": 0.0, **trakt_payload}

    try:
        r = requests.post(f"{TRAKT_API_SCROBBLE_URL}/pause", json=data, headers=headers, timeout=10)
        if r.status_code in (201, 409):
            xbmc.log(f'{LOG} [PAUSE] ÚSPĚCH', xbmc.LOGINFO)
        else:
            xbmc.log(f'{LOG} [PAUSE] CHYBA HTTP {r.status_code}', xbmc.LOGERROR)
    except Exception as e:
        xbmc.log(f'{LOG} [PAUSE] Chyba: {e}', xbmc.LOGERROR)


# ================================= STOP =====================================

def scrobble_stop():
    """
    Při stopu:
    - Načte payload + progress,
    - progress < 90 % → uloží jako rozkoukáno (pokud >5 %), VŽDY uloží do lokální historie (i bez ID),
    - progress >= 90 % → pošle Trakt /stop (pokud povoleno) a uloží zhlédnuto do lokální historie (i bez ID),
    - epizody: uloží progress do SeriesManager + pošle Trakt STOP,
    - vyčistí dočasné hodnoty a odemkne lock.
    """
    LOG = _get_log_tag()
    xbmc.log(f'{LOG} [STOP] Spouštím...', xbmc.LOGINFO)

    addon = _get_addon_instance()
    if addon is None:
        xbmc.log(f'{LOG} [STOP] Nelze získat instanci Addonu – konec.', xbmc.LOGERROR)
        return

    # --- Payload z settings ---
    item = None
    try:
        fresh_addon = xbmcaddon.Addon(ADDON_ID)
        payload_json = fresh_addon.getSetting('trakt_last_payload')
        xbmc.log(f'{PREFIX}[STOP] Načteno z disku: {payload_json}', xbmc.LOGDEBUG)
        if payload_json and payload_json.strip():
            item = json.loads(payload_json)
            xbmc.log(f'{PREFIX}[STOP] Payload načten.', xbmc.LOGINFO)
        else:
            xbmc.log(f'{PREFIX}[STOP] Payload prázdný – zkusím fallback.', xbmc.LOGWARNING)
    except Exception as e:
        xbmc.log(f'{PREFIX}[STOP] Chyba načítání payloadu: {e}', xbmc.LOGERROR)

    # Fallback: poskládat z ListItem.Title
    if not item:
        fb_title = xbmc.getInfoLabel('ListItem.Title')
        if fb_title and fb_title != 'ListItem.Title':
            try:
                item = get_item_data(fallback_title=fb_title)
                if item:
                    xbmc.log(f'{PREFIX}[STOP] Fallback OK: {item}', xbmc.LOGINFO)
                else:
                    xbmc.log(f'{PREFIX}[STOP] Fallback selhal (get_item_data).', xbmc.LOGWARNING)
            except Exception as e:
                xbmc.log(f'{PREFIX}[STOP] Fallback chyba: {e}', xbmc.LOGERROR)
        else:
            xbmc.log(f'{PREFIX}[STOP] ListItem.Title nedostupný – konec.', xbmc.LOGERROR)
            return

    if not item:
        xbmc.log(f'{PREFIX}[STOP] Žádná data – přeskočeno.', xbmc.LOGERROR)
        return

    # --- Progress ---
    progress = 0.0
    updated = False
    try:
        fresh_addon_for_progress = xbmcaddon.Addon(ADDON_ID)
        xbmc.sleep(200)  # malá prodleva, aby se stihl uložit poslední tik
        prog_str = fresh_addon_for_progress.getSetting('trakt_last_progress')
        upd_str = fresh_addon_for_progress.getSetting('trakt_progress_updated')
        if prog_str and upd_str == 'true':
            progress = float(prog_str)
            updated = True
            xbmc.log(f'{PREFIX}[STOP] Progress načten (settings): {progress:.1f}%', xbmc.LOGINFO)
    except Exception as e:
        xbmc.log(f'{PREFIX}[STOP] Chyba čtení progressu: {e}', xbmc.LOGWARNING)

    global _last_known_progress, _progress_updated
    if not updated and _progress_updated:
        progress = _last_known_progress
        updated = True
        xbmc.log(f'{PREFIX}[STOP] Progress převzat z in-memory: {progress:.1f}%', xbmc.LOGINFO)

    if not updated:
        try:
            p = xbmc.Player()
            if p.isPlaying():
                current = p.getTime()
                total = p.getTotalTime()
                if total > 0:
                    progress = (current / total) * 100.0
                    updated = True
                    xbmc.log(f'{PREFIX}[STOP] Progress dopočítán z Playeru: {progress:.1f}%', xbmc.LOGINFO)
        except Exception as e:
            xbmc.log(f'{PREFIX}[STOP] Chyba fallbacku z Playeru: {e}', xbmc.LOGWARNING)

    if not updated:
        xbmc.log(f'{PREFIX}[STOP] Progress nebyl aktualizován – STOP přeskočen.', xbmc.LOGWARNING)
        _clean_settings()
        return

    # --- Připrav payload pro lokální DB / Trakt ---
    item_for_local_db = item.copy()
    item_for_local_db['progress'] = progress

    trakt_payload = item.copy()
    for k in ('stream_url', 'name', 'tmdb_id', 'trakt_payload', 'media_type', 'watched_at'):
        trakt_payload.pop(k, None)
    item_for_local_db['trakt_payload'] = trakt_payload

    mt = (item_for_local_db.get('media_type')
          or ('episode' if ('show' in item_for_local_db and 'episode' in item_for_local_db) else None)
          or ('movie' if 'movie' in item_for_local_db else 'movie'))

    # >>> ID EXTRACT >>> doplň ID z URL a názvu (IMDb/TMDb/TVDB/Webshare)
    try:
        from resources.lib import tmdb_utils as TMU
        src_blob = ' '.join([
            str(item_for_local_db.get('stream_url', '')),
            str(item_for_local_db.get('name', '')),
            str(item.get('stream_url', '')),
            str(item.get('name', '')),
        ])
        ext_ids = TMU.extract_external_ids(src_blob)

        # Doplnit do lokální položky (preferovat typ dle media_type)
        item_for_local_db = TMU.merge_ids(item_for_local_db, ext_ids, media_type=mt)

        # U EPIZOD doplnit TVDB do Trakt payloadu (show.ids.tvdb), pokud chybí
        if mt == 'episode' and item_for_local_db.get('trakt_payload'):
            show = item_for_local_db['trakt_payload'].setdefault('show', {})
            ids  = show.setdefault('ids', {})
            if ext_ids.get('tvdb') and not ids.get('tvdb'):
                ids['tvdb'] = ext_ids['tvdb']
                show['ids'] = ids
                item_for_local_db['trakt_payload']['show'] = show

    except Exception as e:
        xbmc.log(f'{PREFIX}[STOP] ID extract chyba: {e}', xbmc.LOGWARNING)
    # <<< ID EXTRACT <<<

    # ✅ EPIZODY: SeriesManager + Trakt STOP (beze změny)
    if mt == 'episode':
        try:
            series_name = item.get('name') or 'Neznámý seriál'
            season = item.get('episode', {}).get('season') or item.get('season_num')
            episode = item.get('episode', {}).get('number') or item.get('episode_num')
            if season and episode:
                import series_manager
                profile_path = xbmcvfs.translatePath(xbmcaddon.Addon(ADDON_ID).getAddonInfo('profile'))
                sm = series_manager.SeriesManager(xbmcaddon.Addon(ADDON_ID), profile_path)
                sm.mark_episode_progress(series_name, int(season), int(episode), progress, set_by='auto')
                xbmc.log(f"[SCROBBLE][STOP] Progress uložen do SeriesManager: {series_name} S{season}E{episode} -> {progress}%", xbmc.LOGINFO)

                # Trakt STOP pro epizodu (pokud povoleno)
                if addon.getSettingBool('trakt_enabled'):
                    headers = get_trakt_headers()
                    if headers:
                        ep_payload = item.copy()
                        for k in ('stream_url', 'name', 'tmdb_id', 'trakt_payload', 'media_type', 'watched_at'):
                            ep_payload.pop(k, None)
                        data = {"progress": min(progress, 99.9), **ep_payload}
                        try:
                            resp = requests.post(f"{TRAKT_API_SCROBBLE_URL}/stop", json=data, headers=headers, timeout=10)
                            if resp.status_code in (201, 409):
                                xbmc.log("[SCROBBLE][STOP] Trakt STOP úspěšně odeslán pro epizodu.", xbmc.LOGINFO)
                            else:
                                xbmc.log(f"[SCROBBLE][STOP] Trakt STOP chyba HTTP {resp.status_code}", xbmc.LOGERROR)
                        except Exception as e:
                            xbmc.log(f"[SCROBBLE][STOP] Síťová chyba při odeslání Trakt STOP: {e}", xbmc.LOGERROR)
                    else:
                        xbmc.log("[SCROBBLE][STOP] Trakt není autorizován – STOP přeskočen.", xbmc.LOGWARNING)
            else:
                xbmc.log("[SCROBBLE][STOP] Chybí S/E pro epizodu – progress nezapsán.", xbmc.LOGWARNING)
        except Exception as e:
            xbmc.log(f"[SCROBBLE][STOP] CHYBA zápisu progressu do SeriesManager: {e}", xbmc.LOGERROR)

    # ✅ FILMY: ukládej VŽDY (i bez TMDb ID)
    if mt == 'movie':
        # Rozkoukáno (<90 %)
        if progress < 90.0:
            xbmc.log(f'{PREFIX}[STOP] PROGRESS < 90% ({progress:.1f}%) – Trakt STOP přeskočen.', xbmc.LOGWARNING)
            if progress > 5.0:
                try:
                    from resources.lib import local_history
                    # pokud tmdb_id chybí, nastavíme '0' (store_item si s tím poradí)
                    if not item_for_local_db.get('tmdb_id'):
                        item_for_local_db['tmdb_id'] = '0'
                    local_history.store_item(item_for_local_db)
                    xbmc.log(f'{PREFIX}[STOP] Rozkoukáno uloženo (lokální historie).', xbmc.LOGINFO)
                except Exception as e:
                    xbmc.log(f'{PREFIX}[STOP] Chyba ukládání rozkoukána: {e}', xbmc.LOGERROR)
        else:
            # Zhlédnuto (≥90 %) → Trakt STOP (pokud povoleno) + lokálně
            trakt_successful = False
            headers = get_trakt_headers()
            if headers and addon.getSettingBool('trakt_enabled'):
                xbmc.log(f'{PREFIX}[STOP] Trakt povolen – posílám /stop.', xbmc.LOGINFO)
                mv_payload = item.copy()
                for k in ('stream_url', 'name', 'tmdb_id', 'trakt_payload', 'media_type', 'watched_at'):
                    mv_payload.pop(k, None)
                data = {
                    "progress": min(progress, 99.9),
                    "watched_at": datetime.utcnow().strftime('%Y-%m-%dT%H:%M:%S.000Z'),
                    **mv_payload
                }
                xbmc.log(f"[SCROBBLE][STOP] Trakt STOP payload: {json.dumps(data)}", xbmc.LOGDEBUG)
                try:
                    resp = requests.post(f"{TRAKT_API_SCROBBLE_URL}/stop", json=data, headers=headers, timeout=10)
                    if resp.status_code in (201, 409):
                        trakt_successful = True
                        xbmc.log(f'{PREFIX}[STOP] ÚSPĚCH – Trakt zhlédnuto.', xbmc.LOGINFO)
                    else:
                        xbmc.log(f'{PREFIX}[STOP] CHYBA HTTP {resp.status_code}', xbmc.LOGERROR)
                except Exception as e:
                    xbmc.log(f'{PREFIX}[STOP] Síťová chyba Traktu: {e}', xbmc.LOGERROR)
            else:
                xbmc.log(f'{PREFIX}[STOP] Trakt není autorizován/povolen – odeslání přeskočeno.', xbmc.LOGINFO)

            xbmcgui.Dialog().notification("Trakt" if trakt_successful else "Lokálně", "Zhlédnuto", xbmcgui.NOTIFICATION_INFO)
            # Uložit zhlédnuto VŽDY i bez tmdb_id
            item_for_local_db['progress'] = 100.0
            try:
                from resources.lib import local_history
                if not item_for_local_db.get('tmdb_id'):
                    item_for_local_db['tmdb_id'] = '0'
                local_history.store_item(item_for_local_db)
                xbmc.log(f'{PREFIX}[STOP] Zhlédnuto uloženo (lokální historie).', xbmc.LOGINFO)
            except Exception as e:
                xbmc.log(f'{PREFIX}[STOP] Chyba ukládání zhlédnuto: {e}', xbmc.LOGERROR)

    # --- Vyčistit nastavení + odemknout ---
    _clean_settings()
    try:
        addon.setSetting('trakt_payload_locked', 'false')
        addon.setSetting('trakt_last_payload', '')
        addon.setSetting('trakt_last_progress', '0.0')
        addon.setSetting('trakt_progress_updated', 'false')
        xbmc.log(f'{PREFIX}[STOP] Zámek a dočasná data vyčištěna.', xbmc.LOGINFO)
    except Exception as e:
        xbmc.log(f'{PREFIX}[STOP] Chyba při čištění dočasných dat: {e}', xbmc.LOGWARNING)
