# -*- coding: utf-8 -*-
# resources/lib/flix_report.py
import os
import re
import io
import time
import tempfile
import shutil
import xbmc
import xbmcaddon
import xbmcvfs
import urllib.request
import urllib.error
import urllib.robotparser

ADDON_ID = 'plugin.video.mmirousek'
ADDON    = xbmcaddon.Addon(ADDON_ID)

USER_AGENT = ('Mozilla/5.0 (Windows NT 10.0; Win64; x64) '
              'AppleWebKit/537.36 (KHTML, like Gecko) '
              'Chrome/121.0 Safari/537.36 YaWSP/1.0')

FLIX_URL      = 'https://flixpatrol.com/'
PROFILE_PATH  = xbmcvfs.translatePath(ADDON.getAddonInfo('profile'))
CACHE_HTML    = os.path.join(PROFILE_PATH, 'flixpatrol.html')
CACHE_REPORT  = os.path.join(PROFILE_PATH, 'streaming_report.txt')
DUMP_HTML     = os.path.join(PROFILE_PATH, 'flixpatrol_dump.html')  # pro ladění
TTL_SECONDS   = 24 * 3600  # stáhnout max 1× denně

PLATFORMS = [
    ("Netflix",     "netflix"),
    ("HBO Max",     "hbo max"),
    ("Disney+",     "disney"),
    ("Prime Video", "prime"),
    ("Paramount+",  "paramount"),
    ("Apple TV+",   "apple tv"),
]

ALIAS_GROUPS = {
    "Netflix":     ["Netflix"],
    "HBO Max":     ["HBO Max", "Max"],
    "Disney+":     ["Disney+"],
    "Prime Video": ["Prime", "Amazon Prime", "Amazon"],
    "Paramount+":  ["Paramount+"],
    "Apple TV+":   ["Apple TV+"],
}

# --- Per-platform URL map (lazy fetch na klik) --------------------------------
PLATFORM_URLS = {
    "Netflix":      "https://flixpatrol.com/top10/netflix/world/",
    "HBO Max":      "https://flixpatrol.com/top10/hbo-max/world/",
    "Disney+":      "https://flixpatrol.com/top10/disney/world/",
    "Prime Video":  "https://flixpatrol.com/top10/amazon-prime/world/",
    "Paramount+":   "https://flixpatrol.com/top10/paramount-plus/world/",
    "Apple TV+":    "https://flixpatrol.com/top10/apple-tv/world/",
}

def _platform_cache_path(nice_name: str) -> str:
    slug = (nice_name or '').lower().replace(' ', '_').replace('+','plus')
    return os.path.join(PROFILE_PATH, f'flix_{slug}.html')


# -----------------------------------------------------------------------------
# IO utils
# -----------------------------------------------------------------------------
def _ensure_profile():
    if not os.path.exists(PROFILE_PATH):
        os.makedirs(PROFILE_PATH)

def _atomic_write(path, text):
    d = os.path.dirname(path)
    fd, tmp = tempfile.mkstemp(dir=d, prefix='tmp_', suffix='.txt')
    with os.fdopen(fd, 'w', encoding='utf-8') as f:
        f.write(text)
    shutil.move(tmp, path)


# -----------------------------------------------------------------------------
# robots.txt compliance
# -----------------------------------------------------------------------------
def _robots_permit(url):
    try:
        rp = urllib.robotparser.RobotFileParser()
        rp.set_url('https://flixpatrol.com/robots.txt')
        rp.read()
        ok = (rp.can_fetch(USER_AGENT, url) and rp.can_fetch('*', url))
        xbmc.log(f'[FlixReport] robots can_fetch={ok}', xbmc.LOGDEBUG)
        return ok
    except Exception as e:
        xbmc.log(f'[FlixReport] robots.txt check failed: {e}', xbmc.LOGWARNING)
        return False  # raději nepokračovat bez robots ⇒ zůstaň u cache


# -----------------------------------------------------------------------------
# Homepage download (pro textový report a kalendář)
# -----------------------------------------------------------------------------
def download_flix_html(force=False):
    """Stáhne flixpatrol.com do profilu atomicky. Respektuje TTL + robots.txt."""
    _ensure_profile()
    now = time.time()
    if (not force) and os.path.exists(CACHE_HTML):
        try:
            age = now - os.path.getmtime(CACHE_HTML)
            xbmc.log(f'[FlixReport] cache age={int(age)}s; TTL={TTL_SECONDS}s', xbmc.LOGDEBUG)
            if age < TTL_SECONDS:
                xbmc.log('[FlixReport] Using cached flixpatrol.html', xbmc.LOGINFO)
                return CACHE_HTML
        except Exception as e:
            xbmc.log(f'[FlixReport] cache mtime read failed: {e}', xbmc.LOGDEBUG)

    if not _robots_permit(FLIX_URL):
        xbmc.log('[FlixReport] robots disallow or fetch failed – using cache', xbmc.LOGWARNING)
        return CACHE_HTML if os.path.exists(CACHE_HTML) else None

    req = urllib.request.Request(FLIX_URL, headers={
        'User-Agent': USER_AGENT,
        'Accept': 'text/html,application/xhtml+xml',
        'Accept-Language': 'cs-CZ,cs;q=0.9,en-US;q=0.8,en;q=0.7',
        'Connection': 'close'
    })
    try:
        xbmc.log('[FlixReport] downloading flixpatrol.com ...', xbmc.LOGINFO)
        with urllib.request.urlopen(req, timeout=12) as resp:
            status = getattr(resp, 'status', 200)
            if status != 200:
                xbmc.log(f'[FlixReport] HTTP {status}', xbmc.LOGWARNING)
                return CACHE_HTML if os.path.exists(CACHE_HTML) else None
            html_text = resp.read().decode('utf-8', errors='replace')
            _atomic_write(CACHE_HTML, html_text)
            xbmc.log(f'[FlixReport] Downloaded {len(html_text)} bytes → {CACHE_HTML}', xbmc.LOGINFO)
            # dump pro ladění (část)
            try:
                _atomic_write(DUMP_HTML, html_text[-80000:])
            except Exception:
                pass
            return CACHE_HTML
    except Exception as e:
        xbmc.log(f'[FlixReport] Download error: {e}', xbmc.LOGERROR)
        return CACHE_HTML if os.path.exists(CACHE_HTML) else None


def _norm(s): return (s or '').strip()


# -----------------------------------------------------------------------------
# HTML-aware parser (homepage) – Top Movies / Top TV Shows
# -----------------------------------------------------------------------------
def _alias_to_nice(alias):
    alias = (alias or '').strip()
    for nice, aliases in ALIAS_GROUPS.items():
        if alias in aliases:
            return nice
    # speciál: "Max" mapuj na "HBO Max"
    if alias.lower() == 'max':
        return 'HBO Max'
    return None

def _parse_html_sections(html):
    """
    Robustní HTML parser bez bs4:
    1) Najde pozice H2/H3 se slovy 'Top Movies' a 'Top TV Shows' (sekce).
    2) V každé sekci najde platformové H3 (<h3>Netflix</h3> atd.).
    3) Z každé platformové podsekce vyextrahuje první 10 /title/...NÁZEV</a>.
    Vrací: (movies_map, tv_map), kde klíče jsou normalizované názvy platforem.
    """
    html_min = re.sub(r'\s+', ' ', html)

    sec_movies = []
    sec_tv     = []

    for m in re.finditer(r'(?i)<h[23][^>]*>\s*Top\s+Movies\s*</h[23]>', html_min):
        start = m.start()
        nxt = re.search(r'(?i)<h[23][^>]*>\s*Top\s+(Movies|TV\s*Shows)\s*</h[23]>', html_min[m.end():])
        end = m.end() + (nxt.start() if nxt else 0)
        if end <= start: end = len(html_min)
        sec_movies.append(html_min[start:end])

    for m in re.finditer(r'(?i)<h[23][^>]*>\s*Top\s+TV\s*Shows\s*</h[23]>', html_min):
        start = m.start()
        nxt = re.search(r'(?i)<h[23][^>]*>\s*Top\s+(Movies|TV\s*Shows)\s*</h[23]>', html_min[m.end():])
        end = m.end() + (nxt.start() if nxt else 0)
        if end <= start: end = len(html_min)
        sec_tv.append(html_min[start:end])

    def collect(sec_list, label):
        out = {}
        for sec in sec_list:
            for plat_alias in ["Netflix", "HBO Max", "Max", "Disney+", "Prime", "Amazon Prime", "Amazon", "Paramount+", "Apple TV+"]:
                for h in re.finditer(rf'(?i)<h[3][^>]*>[^<]*{re.escape(plat_alias)}[^<]*</h[3]>', sec):
                    start = h.end()
                    nxt_h3 = re.search(r'(?i)<h[3][^>]*>.*?</h[3]>', sec[start:])
                    end = start + (nxt_h3.start() if nxt_h3 else len(sec)-start)
                    block = sec[start:end]
                    titles = []
                    for a in re.finditer(r'(?i)<a[^>]+href="/title/[^"]+"[^>]*>(.*?)</a>', block):
                        name = _norm(re.sub(r'<[^>]+>', '', a.group(1)))
                        if name and name not in titles:
                            titles.append(name)
                        if len(titles) >= 10:
                            break
                    if titles:
                        nice = _alias_to_nice(plat_alias)
                        if nice:
                            out.setdefault(nice, titles[:10])
                        xbmc.log(f'[FlixReport] HTML {label}: {plat_alias} -> {len(titles)}', xbmc.LOGDEBUG)
        return out

    movies_map = collect(sec_movies, 'Movies')
    tv_map     = collect(sec_tv, 'TV')
    return movies_map, tv_map


# -----------------------------------------------------------------------------
# Markdown-like parser (fallback #1)
# -----------------------------------------------------------------------------
def _parse_primary(text):
    xbmc.log('[FlixReport] _parse_primary() start', xbmc.LOGDEBUG)
    parts_movies, parts_tv = [], []
    for m in re.finditer(r'(?i)Top Movies', text):
        start = m.start()
        next_m = re.search(r'(?i)Top (Movies|TV Shows)', text[start+1:])
        end = start + 1 + (next_m.start() if next_m else len(text))
        parts_movies.append(text[start:end])
    for m in re.finditer(r'(?i)Top TV Shows', text):
        start = m.start()
        next_m = re.search(r'(?i)Top (Movies|TV Shows)', text[start+1:])
        end = start + 1 + (next_m.start() if next_m else len(text))
        parts_tv.append(text[start:end])

    re_h3_platform = re.compile(r'^\s*#{2,3}\s*\[?(Netflix|HBO Max|Disney\+|Prime|Paramount\+|Apple TV\+)\]?\s*$', re.MULTILINE)
    re_item_line   = re.compile(r'^\s*-\s*\[?([^\]]+)\]?\s*$', re.MULTILINE)
    re_title_line  = re.compile(r'^\s*(?:- )?([A-Za-z0-9].+)$', re.MULTILINE)

    def collect_from_parts(parts):
        plat_map = {}
        for part in parts:
            labels = list(re_h3_platform.finditer(part))
            if not labels:
                continue
            for i, lab in enumerate(labels):
                plat  = lab.group(1)
                start = lab.start()
                end   = (labels[i+1].start() if i+1 < len(labels) else len(part))
                sub   = part[start:end]
                items = []
                for mm in re_item_line.finditer(sub):
                    t = _norm(mm.group(1))
                    if t and t not in items:
                        items.append(t)
                    if len(items) >= 10: break
                if not items:
                    for ln in sub.splitlines():
                        if re.search(r'^\s*#{1,6}\s*', ln):  # nadpis
                            continue
                        mtt = re_title_line.match(ln)
                        if mtt:
                            t = _norm(mtt.group(1))
                            low = t.lower()
                            if not t or 'top movies' in low or 'top tv shows' in low:
                                continue
                            if t not in items:
                                items.append(t)
                            if len(items) >= 10: break
                plat_map[_alias_to_nice(plat) or plat] = items[:10]
        return plat_map

    movies_map = collect_from_parts(parts_movies)
    tv_map     = collect_from_parts(parts_tv)
    xbmc.log(f'[FlixReport] _parse_primary: movies_map={list(movies_map.keys())}, tv_map={list(tv_map.keys())}', xbmc.LOGDEBUG)
    return movies_map, tv_map


# -----------------------------------------------------------------------------
# Volný fallback (fallback #2)
# -----------------------------------------------------------------------------
def _fallback_collect_by_platform(text):
    xbmc.log('[FlixReport] fallback_collect_by_platform() start', xbmc.LOGWARNING)
    plat_sections = {}
    re_h3 = re.compile(r'(?m)^\s*#{2,3}\s*\[(Netflix|HBO Max|Disney\+|Prime|Paramount\+|Apple TV\+)\]\s*$')
    for m in re_h3.finditer(text):
        plat = m.group(1)
        start = m.end()
        nxt = re_h3.search(text, start)
        end = nxt.start() if nxt else len(text)
        block = text[start:end]
        titles = []
        for am in re.finditer(r'\[([^\]]+)\]\(/title/[^)]+\)', block):
            t = _norm(am.group(1))
            if t and t not in titles:
                titles.append(t)
        if not titles:
            for ln in block.splitlines():
                ln = _norm(ln)
                if not ln:
                    continue
                if ln.startswith('- '):
                    ln = ln[2:].strip()
                if ln and ln not in titles:
                    titles.append(ln)
        plat_sections[plat] = titles[:20]
        xbmc.log(f'[FlixReport] fallback: {plat} found {len(titles)} titles', xbmc.LOGDEBUG)

    movies_map, tv_map = {}, {}
    for nice, _slug in PLATFORMS:
        for a in ALIAS_GROUPS.get(nice, [nice]):
            if a in plat_sections and a not in movies_map:
                titles = plat_sections[a]
                movies_map[nice] = titles[:10]
                tv_map[nice]     = titles[10:20]
    return movies_map, tv_map


# -----------------------------------------------------------------------------
# Per-platform DOWNLOAD + PARSE (lazy fetch na klik v menu)
# -----------------------------------------------------------------------------
def download_platform_html(nice_name: str, force=False):
    """
    Stáhne HTML pro danou platformu (např. Netflix world TOP10) do lokální cache.
    Respektuje TTL (24 h) + robots.txt. Vrací cestu k souboru nebo None.
    """
    url = PLATFORM_URLS.get(nice_name)
    if not url:
        xbmc.log(f'[FlixReport] Unknown platform: {nice_name}', xbmc.LOGERROR)
        return None
    path = _platform_cache_path(nice_name)
    _ensure_profile()
    now = time.time()

    if (not force) and os.path.exists(path):
        try:
            age = now - os.path.getmtime(path)
            xbmc.log(f'[FlixReport] PLAT[{nice_name}] cache age={int(age)}s; TTL={TTL_SECONDS}s', xbmc.LOGDEBUG)
            if age < TTL_SECONDS:
                xbmc.log(f'[FlixReport] Using cached platform HTML: {path}', xbmc.LOGINFO)
                return path
        except Exception:
            pass

    # robots
    try:
        rp = urllib.robotparser.RobotFileParser()
        rp.set_url('https://flixpatrol.com/robots.txt')
        rp.read()
        if not (rp.can_fetch(USER_AGENT, url) and rp.can_fetch('*', url)):
            xbmc.log(f'[FlixReport] robots disallow for {url} – using cache', xbmc.LOGWARNING)
            return path if os.path.exists(path) else None
    except Exception as e:
        xbmc.log(f'[FlixReport] robots check failed ({url}): {e}', xbmc.LOGWARNING)
        return path if os.path.exists(path) else None

    req = urllib.request.Request(url, headers={
        'User-Agent': USER_AGENT,
        'Accept': 'text/html,application/xhtml+xml',
        'Accept-Language': 'cs-CZ,cs;q=0.9,en-US;q=0.8,en;q=0.7',
        'Connection': 'close'
    })
    try:
        xbmc.log(f'[FlixReport] downloading platform page: {url}', xbmc.LOGINFO)
        with urllib.request.urlopen(req, timeout=12) as resp:
            status = getattr(resp, 'status', 200)
            if status != 200:
                xbmc.log(f'[FlixReport] PLATFORM HTTP {status} for {nice_name}', xbmc.LOGWARNING)
                return path if os.path.exists(path) else None
            html_text = resp.read().decode('utf-8', errors='replace')
            _atomic_write(path, html_text)
            xbmc.log(f'[FlixReport] PLATFORM {nice_name} downloaded {len(html_text)} bytes → {path}', xbmc.LOGINFO)
            return path
    except Exception as e:
        xbmc.log(f'[FlixReport] PLATFORM {nice_name} download error: {e}', xbmc.LOGERROR)
        return path if os.path.exists(path) else None


def parse_platform_html(nice_name: str, path: str):
    """
    Z platformové stránky vytáhni 10 titulů pro filmy + 10 pro seriály.
    Vrací {"Filmy":[...], "Seriály":[...]}.
    """
    if not path or not os.path.exists(path):
        xbmc.log(f'[FlixReport] parse_platform_html: missing {path}', xbmc.LOGERROR)
        return {"Filmy": [], "Seriály": []}
    with io.open(path, 'r', encoding='utf-8') as f:
        html = f.read()
    html_min = re.sub(r'\s+', ' ', html)

    def collect_for(label: str):
        # najdi sekci H2/H3 s Top Movies / Top TV Shows
        m = re.search(rf'(?i)<h[23][^>]*>\s*Top\s+{label}\s*</h[23]>', html_min)
        if not m:
            return []
        start = m.end()
        nxt = re.search(r'(?i)<h[23][^>]*>\s*Top\s+(Movies|TV\s*Shows)\s*</h[23]>', html_min[start:])
        end = start + (nxt.start() if nxt else len(html_min)-start)
        block = html_min[start:end]
        titles = []
        # obecně: odkazy /title/... do 10 ks
        for a in re.finditer(r'(?i)<a[^>]+href="/title/[^"]+"[^>]*>(.*?)</a>', block):
            name = _norm(re.sub(r'<[^>]+>', '', a.group(1)))
            if name and name not in titles:
                titles.append(name)
            if len(titles) >= 10: break
        xbmc.log(f'[FlixReport] PLATPARSE {nice_name} {label}: {len(titles)}', xbmc.LOGINFO)
        return titles

    return {"Filmy": collect_for("Movies"), "Seriály": collect_for("TV\\s*Shows")}


# -----------------------------------------------------------------------------
# Orchestrátor parserů (homepage)
# -----------------------------------------------------------------------------
def parse_flix_html(path):
    """Vrátí {'top': {...}, 'calendar': [...]}, s detailním logem a HTML snapshotem pro debug."""
    if not path or not os.path.exists(path):
        xbmc.log('[FlixReport] parse_flix_html: file missing', xbmc.LOGERROR)
        return {"top": {}, "calendar": []}
    with io.open(path, 'r', encoding='utf-8') as f:
        html = f.read()

    xbmc.log('[FlixReport] parse_flix_html() begin', xbmc.LOGDEBUG)

    # 1) Zkus HTML-aware parser
    movies_map, tv_map = _parse_html_sections(html)

    # 2) Pokud nic, zkus primary (markdown-like)
    if not any((movies_map, tv_map)):
        xbmc.log('[FlixReport] HTML parse empty -> try primary', xbmc.LOGWARNING)
        movies_map, tv_map = _parse_primary(html)

    # 3) Pokud pořád nic, zkus volný fallback
    if not any((movies_map, tv_map)):
        xbmc.log('[FlixReport] primary parse empty -> try fallback', xbmc.LOGWARNING)
        movies_map, tv_map = _fallback_collect_by_platform(html)

    # Slož finální 'top'
    top = {}
    for nice, _slug in PLATFORMS:
        m, s = [], []
        for a in ALIAS_GROUPS.get(nice, [nice]):
            if not m and (a in movies_map): m = movies_map.get(a, [])
            if not s and (a in tv_map):     s = tv_map.get(a, [])
        if not m and (nice in movies_map): m = movies_map.get(nice, [])
        if not s and (nice in tv_map):     s = tv_map.get(nice, [])
        top[nice] = {"Filmy": m, "Seriály": s}
        xbmc.log(f'[FlixReport] TOP {nice}: films={len(m)}, shows={len(s)}', xbmc.LOGINFO)

    # --- Kalendář (lehké vytěžení) ---
    calendar = []
    html_min = re.sub(r'\s+', ' ', html)
    cal_anchor = re.search(r'(?i)VOD\s+calendar\s*-\s*New\s+Movies\s+and\s+TV\s+Shows\s+Streaming\s+Right\s+Now', html_min)
    if cal_anchor:
        tail = html_min[cal_anchor.end():]
        for row in re.finditer(r'(?i)>(Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec)\s+\d{1,2}<.*?</tr>', tail):
            snippet = row.group(0)
            date_m  = re.search(r'(?i)>(Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec)\s+\d{1,2}<', snippet)
            date_s  = date_m.group(0).strip('><') if date_m else ''
            title_m = re.search(r'(?i)<a[^>]*>([^<]+)</a>', snippet)
            title_s = _norm(title_m.group(1)) if title_m else ''
            plat_s  = ''
            for p in ['Netflix','Amazon','Prime','Disney+','Paramount+','Apple TV+','HBO','Hulu','Peacock']:
                if p in snippet: plat_s = p; break
            ctype_s = 'Movie' if re.search(r'(?i)\bMovie\b', snippet) else ('TV Show' if re.search(r'(?i)\bTV\s*Show\b', snippet) else '')
            if date_s or title_s:
                calendar.append({"date": date_s, "title": title_s or snippet[:120], "platform": plat_s, "type": ctype_s})
    xbmc.log(f'[FlixReport] Calendar items: {len(calendar)}', xbmc.LOGINFO)

    # Pokud je vše prázdné, ulož dump (už uložen v downloadu) a upozorni do logu
    all_empty = all((not top[p]['Filmy'] and not top[p]['Seriály']) for p,_ in PLATFORMS)
    if all_empty:
        xbmc.log(f'[FlixReport] WARNING: parsed empty TOP. Check HTML dump at: {DUMP_HTML}', xbmc.LOGWARNING)

    return {"top": top, "calendar": calendar}


# -----------------------------------------------------------------------------
# Report (text) a API pro router/menu
# -----------------------------------------------------------------------------
def build_text_report(model):
    lines = []
    lines.append("STREAMING REPORT\n")
    lines.append("TOP 10 podle platforem\n")
    for nice, _ in PLATFORMS:
        block = model["top"].get(nice) or {}
        movies = block.get("Filmy") or []
        shows  = block.get("Seriály") or []
        lines.append(f"{nice}")
        lines.append(" Filmy:")
        lines += [f"  {i}. {t}" for i, t in enumerate(movies[:10], 1)] or ["  n/a"]
        lines.append(" Seriály:")
        lines += [f"  {i}. {t}" for i, t in enumerate(shows[:10], 1)] or ["  n/a"]
    lines.append("Kalendář novinek")
    if model["calendar"]:
        for i, it in enumerate(model["calendar"], 1):
            lines.append(f"{i}. {it['date']}  {it.get('platform') or '—'}  {it.get('title') or '—'}  ({it.get('type') or '—'})")
    else:
        lines.append("n/a")
    return "\n".join(lines)


def refresh_daily_on_start():
    """Volat při startu doplňku: stáhne HTML jen pokud cache starší než TTL."""
    try:
        _ = download_flix_html(force=False)
    except Exception as e:
        xbmc.log(f'[FlixReport] refresh on start failed: {e}', xbmc.LOGWARNING)


def get_model(force=False):
    """Vrátí model (top+calendar) – s případným forcerefresh a detailními logy."""
    path = download_flix_html(force=force)
    if not path:
        xbmc.log('[FlixReport] get_model: no HTML path available', xbmc.LOGERROR)
        return {"top": {}, "calendar": []}
    model = parse_flix_html(path)
    for nice, _ in PLATFORMS:
        block = model["top"].get(nice) or {}
        xbmc.log(f'[FlixReport] SUM {nice}: F={len(block.get("Filmy") or [])}, S={len(block.get("Seriály") or [])}', xbmc.LOGINFO)
    return model


def list_platforms(handle, build_url_fn):
    """Zobrazí seznam platforem pro 'Novinky na VOD' jako složky + ruční refresh."""
    import xbmcgui, xbmcplugin
    for nice, slug in PLATFORMS:
        li = xbmcgui.ListItem(label=nice)
        li.setArt({'icon': 'DefaultFolder.png'})
        url = build_url_fn(action='vod_news_platform', p=nice)
        xbmcplugin.addDirectoryItem(handle, url, li, isFolder=True)
    # ruční refresh celé cache homepage (volitelný)
    li = xbmcgui.ListItem(label='[Obnovit data (vynutit stažení homepage)]')
    li.setArt({'icon': 'DefaultRefresh.png'})
    url = build_url_fn(action='vod_news_refresh')
    xbmcplugin.addDirectoryItem(handle, url, li, isFolder=False)


def list_titles_for_platform(handle, build_url_fn, platform_name, force=False):
    """
    Pro zvolenou platformu zobrazí TOP 10 Filmy + TOP 10 Seriály.
    Položky jsou klikatelné -> otevře se Webshare vyhledávání s q=title (tvůj 'search').
    """
    import xbmcgui, xbmcplugin
    # 1) Zkus nejdřív globální model (homepage)
    model = get_model(force=False)
    block = model["top"].get(platform_name) or {}
    movies = block.get("Filmy") or []
    shows  = block.get("Seriály") or []

    # 2) Pokud je prázdno, stáhni a rozparsuj přímo platformu (lazy, TTL 24h)
    if (not movies) and (not shows):
        p_path = download_platform_html(platform_name, force=force)
        p_data = parse_platform_html(platform_name, p_path)
        movies = p_data.get("Filmy") or []
        shows  = p_data.get("Seriály") or []

    xbmc.log(f'[FlixReport] list_titles_for_platform({platform_name}): movies={len(movies)} shows={len(shows)}', xbmc.LOGINFO)

    # Sekce + položky (klik → action=search, q=název)
    if movies:
        li_head = xbmcgui.ListItem(label=f'— {platform_name}: Filmy —')
        li_head.setArt({'icon': 'DefaultMovies.png'})
        xbmcplugin.addDirectoryItem(handle, build_url_fn(action='noop'), li_head, isFolder=False)
        for t in movies[:10]:
            li = xbmcgui.ListItem(label=t)
            li.setInfo('video', {'title': t, 'mediatype': 'video'})
            url = build_url_fn(action='search', q=t)
            xbmcplugin.addDirectoryItem(handle, url, li, isFolder=True)
    else:
        li = xbmcgui.ListItem(label=f'{platform_name}: Filmy — n/a')
        xbmcplugin.addDirectoryItem(handle, build_url_fn(action='noop'), li, isFolder=False)

    if shows:
        li_head = xbmcgui.ListItem(label=f'— {platform_name}: Seriály —')
        li_head.setArt({'icon': 'DefaultTvShows.png'})
        xbmcplugin.addDirectoryItem(handle, build_url_fn(action='noop'), li_head, isFolder=False)
        for t in shows[:10]:
            li = xbmcgui.ListItem(label=t)
            li.setInfo('video', {'title': t, 'mediatype': 'video'})
            url = build_url_fn(action='search', q=t)
            xbmcplugin.addDirectoryItem(handle, url, li, isFolder=True)
    else:
        li = xbmcgui.ListItem(label=f'{platform_name}: Seriály — n/a')
        xbmcplugin.addDirectoryItem(handle, build_url_fn(action='noop'), li, isFolder=False)


def generate_and_show_report(force=False):
    """Textový viewer s celým reportem z homepage (pokud jsou data k dispozici)."""
    html_path = download_flix_html(force=force)
    if not html_path:
        import xbmcgui
        xbmcgui.Dialog().ok('FlixPatrol', 'Nepodařilo se stáhnout flixpatrol.com. Zkus znovu později.')
        return
    model = parse_flix_html(html_path)
    text  = build_text_report(model)
    _atomic_write(CACHE_REPORT, text)
    import xbmcgui
    xbmcgui.Dialog().textviewer('Novinky na VOD (FlixPatrol)', text)
