# -*- coding: utf-8 -*-
# resources/lib/ext/csfd_toplist/router.py
"""
Adaptér, který umožní volat původní ČSFD Toplist uvnitř hlavního pluginu:
- Nastaví správný handle/BASE_URL,
- přepíše build_url() tak, aby generoval URL s prefixem akcí 'csfd_*'
  (ty pak odchytí hlavní router v yawsp.py).
- Exponuje funkce, které volá yawsp.Router.
"""
import sys
from urllib.parse import urlencode
import xbmc
import xbmcgui
import xbmcplugin
import xbmcaddon
import xbmcvfs


# Načti původní modul UI/logiky (obsahuje root_menu(), prepare_menu() atd.)
from resources.lib.ext.csfd_toplist import default as csfd  # ← tvůj default.py

ADDON = xbmcaddon.Addon('plugin.video.mmirousek')
TMDB_IMG_BASE = "https://image.tmdb.org/t/p/w500"

# --- napojení na aktuální kontext pluginu ---
_HANDLE = int(sys.argv[1])
_BASE_URL = sys.argv[0]

# Přepíšeme kontext uvnitř csfd.default tak, aby používal tento handle/BASE_URL
csfd.HANDLE = _HANDLE
csfd.BASE_URL = _BASE_URL
def get_playlist_path():
    addon_path = ADDON.getAddonInfo("path")
    return xbmcvfs.translatePath(addon_path + "/resources/media/playlist.m3u")


# Původní default.py používá csfd.build_url({... 'action': 'xyz' ...})
# Zde přemapujeme 'action' -> 'csfd_xyz' a vrátíme URL pro hlavní router.
def _build_url_patched(query: dict) -> str:
    # necháme původní parametry, jen přepneme prefix akce
    q = dict(query) if query else {}
    act = q.pop('action', None)
    if act:
        q['action'] = f'csfd_{act}'  # → csfd_prepare_menu, csfd_browse, ...
    return '{}?{}'.format(_BASE_URL, urlencode(q, encoding='utf-8'))

# Monkey-patch: UI modul bude používat „náš“ build_url
csfd.build_url = _build_url_patched  # type: ignore

# --- Veřejné wrappery pro yawsp.Router ---

def show_root():
    """Hlavní menu ČSFD Toplist v rámci pluginu."""
    csfd.root_menu()  # vytvoří položky a volá xbmcplugin.endOfDirectory()

def prepare_menu():
    csfd.prepare_menu()

def clean_cache_menu():
    csfd.clean_cache_menu()

def clean_cache(what: str = 'all'):
    csfd.do_clean_cache(what or 'all')

def prepare_all(category: str = 'films'):
    csfd.do_prepare_all((category or 'films').lower())

def update_all():
    csfd.do_update_all()

def browse_menu(category: str = 'series'):
    csfd.browse_menu((category or 'series').lower())

def list_countries(category: str = 'series'):
    csfd.list_countries((category or 'series').lower())

def browse(category: str = 'series'):
    # přímé zobrazení: načti JSON a vyrenderuj
    try:
        items = csfd.load_json((category or 'series').lower())
    except Exception:
        xbmcgui.Dialog().notification("ČSFD Toplist",
                                      "Chyba čtení JSON (viz log).",
                                      xbmcgui.NOTIFICATION_ERROR, 5000)
        xbmcplugin.endOfDirectory(_HANDLE, succeeded=False)
        return
    csfd.render_items(items, is_tv=(category == 'series'))

def prepare_m3u_index():
    """
    Wrapper pro indexaci M3U -> JSON (TMDb) s progress dialogem.
    Cestu k M3U bere z nastavení addonu.
    """
    import os
    import xbmcaddon
    from resources.lib import playlist_index

    m3u_path = get_playlist_path()

    if not os.path.isfile(m3u_path):
        xbmcgui.Dialog().notification("M3U index", f"Soubor playlistu nenalezen:\n{m3u_path}",
                                      xbmcgui.NOTIFICATION_WARNING, 4000)
        xbmcplugin.endOfDirectory(_HANDLE, succeeded=False)
        return

    # Seriály v playlistu nejsou – indexujeme jako FILMY (is_tv=False)
    idx = playlist_index.rebuild_if_needed(m3u_path, is_tv=False, force=True, show_progress=True)
    if not idx:
        xbmcgui.Dialog().notification("M3U index", "Indexace selhala.", xbmcgui.NOTIFICATION_ERROR, 4000)
        xbmcplugin.endOfDirectory(HANDLE, succeeded=False)
        return

    cnt = (idx.get('meta') or {}).get('count', 0)
    xbmcgui.Dialog().ok("M3U index", f"Hotovo.\nPočet položek v indexu: {cnt}")
    # návrat do menu Příprava dat
    prepare_menu()

def browse_filtered(category: str = 'series', countries: str = None, country: str = None):
    try:
        items = csfd.load_json((category or 'series').lower())
    except Exception:
        xbmcgui.Dialog().notification("ČSFD Toplist",
                                      "Chyba čtení JSON (viz log).",
                                      xbmcgui.NOTIFICATION_ERROR, 5000)
        xbmcplugin.endOfDirectory(_HANDLE, succeeded=False)
        return

    wanted = set()
    if countries:
        for c in countries.split('\n'):
            c = (c or '').strip().lower()
            if c:
                wanted.add(c)
    elif country:
        c = (country or '').strip().lower()
        if c:
            wanted.add(c)

    if not wanted:
        csfd.render_items(items, is_tv=(category == 'series'))
        return

    # OR filtr podle zemí – shodné s původním routerem
    filtered = []
    for it in items:
        clist = [ (c or '').strip().lower() for c in (it.get('countries') or []) ]
        if any(c in wanted for c in clist):
            filtered.append(it)
    if not filtered:
        xbmcgui.Dialog().notification("ČSFD Toplist",
                                      "Žádné položky pro zvolený filtr zemí.",
                                      xbmcgui.NOTIFICATION_INFO, 3000)
        xbmcplugin.endOfDirectory(_HANDLE, succeeded=False)
        return

    csfd.render_items(filtered, is_tv=(category == 'series'))

# resources/lib/ext/csfd_toplist/router.py

# nahoře v souboru (pokud ještě není), definuj TMDb obrázkovou základnu:


def show_extra_playlist_movies(params=None):
    """
    Zobrazí filmy z playlistu (playlist_index.json), které NEJSOU v Toplistu (csfd_filmy.json).
    Klik na položku přehraje přes play_m3u_stream.
    Stránkování: 50 položek na stránku.
    """
    import json
    import xbmcgui
    import xbmcplugin
    import xbmcaddon
    import xbmcvfs

    PAGE_SIZE = 50

    # 1) Zjisti číslo stránky z params
    page = 1
    if params and 'page' in params:
        try:
            page = int(params['page'])
        except Exception:
            page = 1
    page = max(1, int(page))  # pojistka

    # 2) Načti playlist index
    try:
        from resources.lib import playlist_index
        idx = playlist_index.load_index_json()
    except Exception as e:
        xbmcgui.Dialog().notification("Playlist", f"Nelze načíst index: {e}", xbmcgui.NOTIFICATION_ERROR, 4000)
        xbmcplugin.endOfDirectory(_HANDLE, succeeded=False)
        return

    items = (idx or {}).get('items') or []
    if not items:
        xbmcgui.Dialog().notification("Playlist", "Index playlistu nenalezen. Nejprve jej vytvořte v Přípravě dat.", xbmcgui.NOTIFICATION_WARNING, 4000)
        xbmcplugin.endOfDirectory(_HANDLE, succeeded=False)
        return

    # 3) Načti Toplist (filmy)
    addon = xbmcaddon.Addon('plugin.video.mmirousek')
    prof = xbmcvfs.translatePath(addon.getAddonInfo('profile'))
    if not prof.endswith(('/', '\\')):
        prof += '/'
    csfd_filmy = prof + 'csfd_filmy.json'

    toplist_ids = set()
    if xbmcvfs.exists(csfd_filmy):
        try:
            f = xbmcvfs.File(csfd_filmy, 'r')
            raw = f.read()
            f.close()
            if isinstance(raw, bytes):
                raw = raw.decode('utf-8', 'ignore')
            data = json.loads(raw or '[]')
            for it in data:
                tid = int(it.get('tmdb_id') or 0)
                if tid > 0:
                    toplist_ids.add(tid)
        except Exception:
            pass

    # 4) Filtrovat „mimo Top 300“
    extra = []
    for it in items:
        if (it.get('media_type') or '').lower() == 'movie':
            tid = int(it.get('tmdb_id') or 0)
            if tid == 0 or tid not in toplist_ids:
                extra.append(it)

    if not extra:
        xbmcgui.Dialog().notification("Playlist", "Žádné další filmy mimo Toplist.", xbmcgui.NOTIFICATION_INFO, 3000)
        xbmcplugin.endOfDirectory(_HANDLE, succeeded=False)
        return

    # 5) Stránkování
    total = len(extra)
    start = (page - 1) * PAGE_SIZE
    end = start + PAGE_SIZE
    page_items = extra[start:end]

    # 6) Navigační položka „Předchozí stránka“ na začátku
    if start > 0:
        prev_url = _build_url_patched({'action': 'extra_playlist', 'page': page - 1})
        li = xbmcgui.ListItem(label="← Předchozí stránka")
        xbmcplugin.addDirectoryItem(_HANDLE, prev_url, li, isFolder=True)

    # 7) Vykreslit seznam položek
    for it in page_items:
        base_title = it.get('display_title') or it.get('title_raw') or '???'
        year = it.get('year')
        # Přidáme rok do názvu, pokud existuje
        title = f"{base_title} ({year})" if year else base_title

        li = xbmcgui.ListItem(label=title)
        li.setProperty('IsPlayable', 'true')

        # Artwork – TMDb poster, pokud ho máme v indexu, jinak logo z M3U
        art = {}
        if it.get('poster_path'):
            poster_url = TMDB_IMG_BASE + it['poster_path']
            art.update({'thumb': poster_url, 'icon': poster_url, 'poster': poster_url})
        elif it.get('logo'):
            art.update({'thumb': it['logo'], 'icon': it['logo']})
        if art:
            li.setArt(art)

        # Krátké info (plot/rok)
        info_labels = {
            'title': base_title,  # čistý název bez roku
            'plot': it.get('plot') or 'Položka z M3U playlistu.',
            'year': year or 0,
            'mediatype': 'movie'
        }
        li.setInfo('video', info_labels)

        play_url = _build_url_patched({'action': 'play_m3u_stream', 'm3u_url': it.get('m3u_url') or ''})
        xbmcplugin.addDirectoryItem(_HANDLE, play_url, li, isFolder=False)

    # 8) Navigační položka „Další stránka“ na konci
    if end < total:
        next_url = _build_url_patched({'action': 'extra_playlist', 'page': page + 1})
        li = xbmcgui.ListItem(label="Další stránka →")
        xbmcplugin.addDirectoryItem(_HANDLE, next_url, li, isFolder=True)

    # 9) Nastavit content type na 'movies' → Estuary nabídne více view typů
    xbmcplugin.setContent(_HANDLE, 'movies')

    # 10) Pokud není nastavený view, nastavíme List view (50 = seznam v Estuary)
    xbmc.executebuiltin('Container.SetViewMode(50)')

    xbmcplugin.endOfDirectory(_HANDLE)