# -*- coding: utf-8 -*-
"""
trakt_service.py — Core logika, API obálka a cache managery pro Trakt.tv.
Data-layer pro Kodi addon.
"""
import sys
import os
import json
import time
import gzip
import hashlib
from typing import Any, Dict, Optional, List
from datetime import datetime

# Externí knihovny
import requests

# Kodi/XBMC moduly (Nutné pro cestu k souborům, dialogy, atd.)
import xbmc
import xbmcvfs
import xbmcaddon
import xbmcgui
from xbmcvfs import translatePath
from urllib.parse import urlencode

from resources.lib.tmdb_module import IMAGE_BASE_URL_POSTER # Příklad (pokud ji používáte)
ADDON = xbmcaddon.Addon('plugin.video.mmirousek')

# === KONFIGURACE TRAKT ===
TRAKT_CLIENT_ID = "4c2c45518d4accf713853b837c1251d787878434d76e71ca279db0aae40ae343"
TRAKT_API_URL = "https://api.trakt.tv"

# === TTL (sekundy) ===
TTL_MEDIUM = 10 * 60
TTL_LONG   = 24 * 60 * 60
TTL_FAST   = 60 * 60
TTL_TMDB_DETAILS = 14 * 24 * 60 * 60
TMDB_PREFETCH_WORKERS = 4

# === TRAKT HEADERS / TOKEN HELPERY ===
def _trakt_headers_base() -> Dict[str, str]:
    h = {
        'trakt-api-version': '2',
        'trakt-api-key': TRAKT_CLIENT_ID,
        'User-Agent': 'plugin.video.mmirousek/1.0 (+Kodi)'
    }
    try:
        lang = ADDON.getSetting('trakt_lang') or ''
        if lang:
            h['Accept-Language'] = lang
    except Exception:
        pass
    return h

def _auth_headers(for_post: bool = False) -> Optional[Dict[str, str]]:
    token = ADDON.getSetting('trakt_access_token')
    if not token:
        return None
    h = _trakt_headers_base()
    h['Authorization'] = f'Bearer {token}'
    if for_post:
        h['Content-Type'] = 'application/json'
    return h

def get_trakt_headers(for_post: bool = False):
    """
    Vrátí hlavičky pro volání Trakt API.
    - for_post=True → vyžaduje platný access token, jinak None
    - for_post=False → pokud není token, vrátí veřejné hlavičky (api-key, UA)
    """
    h_auth = _auth_headers(for_post=for_post)
    if h_auth:
        return h_auth
    if for_post:
        return None
    return _trakt_headers_base()


# === CACHE UTILS ===
def _build_cache_key(endpoint: str, params: Optional[Dict[str, Any]], auth: bool) -> str:
    p = params or {}
    try:
        p_ser = json.dumps(p, sort_keys=True, ensure_ascii=False)
    except Exception:
        p_ser = str(p)
    return f"{endpoint}\n{p_ser}\n{'A' if auth else 'N'}"

def _pick_pagination_headers(h: Dict[str, Any]) -> Dict[str, Any]:
    keep = {}
    for k, v in (h.items() if isinstance(h, dict) else []):
        if isinstance(k, str) and k.lower().startswith('x-pagination-'):
            keep[k] = v
    return keep

# === TMDb CACHE LOGIKA (Musí být zde, protože ji používá inkrementální update) ===
_TMDBCACHE = None

def _ensure_tmdb_cache():
    global _TMDBCACHE
    if _TMDBCACHE is None:
        # FileCache může být definována až níže → inicializujeme lhostejně až při prvním použití
        try:
            _TMDBCACHE = FileCache(namespace='tmdb')
        except NameError:
            # Pokud FileCache ještě není definována, necháme None (modul se dokončí a pak bude možné vytvořit)
            _TMDBCACHE = None

def _tmdb_cache_key(media_type: str, tmdb_id: int) -> str:
    return f"tmdb:{('movie' if media_type=='movie' else 'tv')}:{int(tmdb_id)}"

def get_tmdb_details_from_cache_only(tmdb_id: int, media_type: str):
    if not tmdb_id:
        return None
    _ensure_tmdb_cache()
    if _TMDBCACHE is None:
        return None
    key = _tmdb_cache_key(media_type, tmdb_id)
    obj = _TMDBCACHE.get(key)
    return (obj or {}).get('data') if obj else None

def store_tmdb_details(tmdb_id: int, media_type: str, details: dict, ttl_sec: int = TTL_TMDB_DETAILS):
    if not (tmdb_id and isinstance(details, dict) and details):
        return
    _ensure_tmdb_cache()
    if _TMDBCACHE is None:
        return
    key = _tmdb_cache_key(media_type, tmdb_id)
    _TMDBCACHE.set(key, details, hdr=None, ttl_sec=ttl_sec)

import threading
_TMDB_SEM = threading.Semaphore(TMDB_PREFETCH_WORKERS)

# Pozdější (lazy) import get_tmdb_details_fallback — vyhneme se ImportError při importu modulu,
# pokud tmdb_module neobsahuje přesně toto jméno.
def _import_tmdb_fallback():
    try:
        from resources.lib.tmdb_module import get_tmdb_details_fallback
        return get_tmdb_details_fallback
    except Exception as e:
        xbmc.log(f"[TMDb Prefetch] get_tmdb_details_fallback not available: {e}", xbmc.LOGWARNING)
        return None

def _prefetch_tmdb_detail_async(tmdb_id: int, media_type: str):
    """
    Na pozadí stáhne TMDb detaily (CZ) a uloží do cache, pokud tam nejsou.
    První návštěva listu tak bude rychlá; další už využije cached poster+plot.
    """
    if not tmdb_id:
        return

    # Pokud už je v cache, nic nedělej
    existing = get_tmdb_details_from_cache_only(tmdb_id, media_type)
    if existing:
        return

    def _task():
        with _TMDB_SEM:
            try:
                _get_fallback = _import_tmdb_fallback()
                if not _get_fallback:
                    return
                det = _get_fallback(tmdb_id, 'movie' if media_type == 'movie' else 'tv')
                if det:
                    store_tmdb_details(tmdb_id, media_type, det, ttl_sec=TTL_TMDB_DETAILS)
                    xbmc.log(f"[TMDb Prefetch] cached {media_type} tmdb_id={tmdb_id}", xbmc.LOGINFO)
            except Exception as e:
                xbmc.log(f"[TMDb Prefetch] error tmdb_id={tmdb_id}: {e}", xbmc.LOGWARNING)

    try:
        threading.Thread(target=_task, name=f"tmdb_prefetch_{media_type}_{tmdb_id}", daemon=True).start()
    except Exception:
        # Fallback bez threadingu (neblokovat listing!)
        xbmc.log("[TMDb Prefetch] failed to start thread; skipping", xbmc.LOGWARNING)


# ====================================================================
# --- TŘÍDY PRO PŘÍSTUP K DATŮM A CACHE ---
# ====================================================================
# === FILE CACHE (gzip) =======================================================
class FileCache:
    """Jednoduchý souborový cache layer (gzip JSON)."""
    def __init__(self, namespace: str = 'trakt'):
        profile = translatePath(ADDON.getAddonInfo('profile'))
        cache_root = os.path.join(profile, f'{namespace}_cache')
        os.makedirs(cache_root, exist_ok=True)
        self.root = cache_root

    @staticmethod
    def _hash_key(key: str) -> str:
        return hashlib.md5(key.encode('utf-8')).hexdigest()

    def _path(self, key: str) -> str:
        fname = self._hash_key(key) + '.json.gz'
        return os.path.join(self.root, fname)

    def get(self, key: str) -> Optional[Dict[str, Any]]:
        path = self._path(key)
        if not os.path.exists(path):
            return None
        try:
            with gzip.open(path, 'rt', encoding='utf-8') as f:
                obj = json.load(f)
            ts = obj.get('ts', 0)
            ttl = obj.get('ttl', 0)
            if ttl > 0 and (time.time() - ts) > ttl:
                try:
                    os.remove(path)
                except Exception:
                    pass
                return None
            return obj
        except Exception as e:
            xbmc.log(f"[TraktCache] Read fail: {e}", xbmc.LOGWARNING)
            return None

    def set(self, key: str, data: Any, hdr: Optional[Dict[str, Any]], ttl_sec: int):
        path = self._path(key)
        payload = {'ts': int(time.time()), 'ttl': int(ttl_sec), 'hdr': hdr or {}, 'data': data}
        try:
            with gzip.open(path, 'wt', encoding='utf-8') as f:
                json.dump(payload, f, ensure_ascii=False)
        except Exception as e:
            xbmc.log(f"[TraktCache] Write fail: {e}", xbmc.LOGWARNING)

    def clear_namespace(self):
        try:
            for fname in os.listdir(self.root):
                if fname.endswith('.gz'):
                    os.remove(os.path.join(self.root, fname))
        except Exception as e:
            xbmc.log(f"[TraktCache] Clear namespace error: {e}", xbmc.LOGERROR)


# === Trakt API obálka ========================================================
class TraktAPI:
    def __init__(self) -> None:
        self.base = TRAKT_API_URL
        self.last_headers: Dict[str, str] = {}
        self.cache = FileCache('trakt')

    def refresh_token_if_needed(self) -> bool:
        """
        Tichý refresh pomocí refresh_tokenu. Vrací True při úspěchu.
        Spustí se pouze pokud refresh_token a client_secret existují.
        """
        LOG = "mm.trakt_refresh"
        import time, requests, xbmc, xbmcaddon

        ADDON = xbmcaddon.Addon('plugin.video.mmirousek')
        client_secret = ADDON.getSetting('trakt_client_secret') or ""
        refresh_token = ADDON.getSetting('trakt_refresh_token') or ""
        if not (client_secret and refresh_token):
            xbmc.log(f"[{LOG}] missing refresh_token or client_secret → skip", xbmc.LOGINFO)
            return False

        url = f"{self.base}/oauth/token"
        payload = {
            "refresh_token": refresh_token,
            "client_id": "4c2c45518d4accf713853b837c1251d787878434d76e71ca279db0aae40ae343",
            "client_secret": client_secret,
            "grant_type": "refresh_token"
        }
        try:
            r = requests.post(url, json=payload, timeout=10)
            if r.status_code == 200:
                data = r.json() or {}
                ADDON.setSetting('trakt_access_token', data.get('access_token') or '')
                if data.get('refresh_token'):
                    ADDON.setSetting('trakt_refresh_token', data.get('refresh_token'))
                ADDON.setSetting('trakt_token_issued_at', str(int(time.time())))
                ADDON.setSetting('trakt_token_expires_in', str(data.get('expires_in', 0)))
                xbmc.log(f"[{LOG}] token refreshed OK", xbmc.LOGINFO)
                return True
            else:
                xbmc.log(f"[{LOG}] refresh failed status={r.status_code}", xbmc.LOGERROR)
                return False
        except Exception as e:
                xbmc.log(f"[{LOG}] refresh exception: {e}", xbmc.LOGERROR)
                return False
        

    # --- GET wrapper ---


    def get(self, endpoint: str, params=None, auth: bool = True, timeout: int = 12,
            cache_ttl: int = 0, cache_key: Optional[str] = None):
        import time, requests, xbmc, xbmcgui

        url = f"{self.base}/{endpoint.lstrip('/')}"
        headers = _auth_headers(False) if auth else _trakt_headers_base()
        if auth and not headers:
            return None  # Bez tokenu → žádná chyba, jen None

        # Cache-first
        key = cache_key or _build_cache_key(endpoint, params, auth)
        if cache_ttl > 0:
            cached = self.cache.get(key)
            if cached is not None:
                self.last_headers = cached.get('hdr', {}) or {}
                xbmc.log(f"[TraktCache] HIT {endpoint} {params}", xbmc.LOGINFO)
                return cached.get('data')

        def _request_once(hdrs):
            return requests.get(url, headers=hdrs, params=params, timeout=timeout)

        try:
            r = _request_once(headers)

            # Rate-limit retry
            if r.status_code == 429:
                time.sleep(int(r.headers.get('Retry-After', '3')))
                r = _request_once(headers)

            # 401 → pokus o refresh + retry
            if r.status_code == 401 and auth:
                if self.refresh_token_if_needed():
                    hdrs_retry = _auth_headers(False)
                    if hdrs_retry:
                        r = _request_once(hdrs_retry)

            r.raise_for_status()
            self.last_headers = dict(r.headers) if hasattr(r, 'headers') else {}
            data = r.json()
            if cache_ttl > 0:
                self.cache.set(key, data, _pick_pagination_headers(self.last_headers), cache_ttl)
            return data

        except requests.HTTPError as e:
            if getattr(e.response, "status_code", None) == 401:
                xbmcgui.Dialog().notification('Trakt', 'Neplatný token – přihlaste se znovu.', xbmcgui.NOTIFICATION_ERROR, 3000)
            xbmc.log(f"[TraktAPI] HTTP ERROR: {e} [{endpoint}]", xbmc.LOGERROR)
            return None
        except Exception as e:
            xbmc.log(f"[TraktAPI] ERROR: {e} [{endpoint}]", xbmc.LOGERROR)
            return None


    # --- Public endpoints / helpers ---
    def genres(self, media_type: str) -> List[Dict[str, Any]]:
        return self.get(f'genres/{media_type}', auth=False, cache_ttl=TTL_LONG) or []

    def movies(self, mode: str, page: int, limit: int,
               genres: Optional[str], years: Optional[str]):
        endpoint = f"movies/{mode}"
        params = {'extended': 'min', 'page': page, 'limit': limit}
        if genres: params['genres'] = genres
        if years:  params['years']  = years
        return self.get(endpoint, params=params, auth=False, cache_ttl=TTL_FAST)

    def shows(self, mode: str, page: int, limit: int,
              genres: Optional[str], years: Optional[str]):
        endpoint = f"shows/{mode}"
        params = {'extended': 'min', 'page': page, 'limit': limit}
        if genres: params['genres'] = genres
        if years:  params['years']  = years
        return self.get(endpoint, params=params, auth=False, cache_ttl=TTL_FAST)

    def list_popular_or_trending(self, mode: str, page: int, limit: int):
        endpoint = 'lists/popular' if mode == 'popular' else 'lists/trending'
        params = {'extended': 'min', 'page': page, 'limit': limit}
        return self.get(endpoint, params=params, auth=False, cache_ttl=TTL_FAST)

    def list_items(self, list_id: str, page: int, limit: int):
        endpoint = f'lists/{list_id}/items'
        params = {'extended': 'min', 'page': page, 'limit': limit}
        return self.get(endpoint, params=params, auth=True, cache_ttl=TTL_FAST)

    def people(self, media_type: str, trakt_id_or_slug: str, cache_ttl: int = TTL_LONG, auth: bool = False):
        endpoint = f'{media_type}/{trakt_id_or_slug}/people'
        xbmc.log(f"[TraktAPI.people] endpoint={endpoint} auth={'A' if auth else 'N'} ttl={cache_ttl}", xbmc.LOGINFO)
        return self.get(endpoint, params=None, auth=auth, cache_ttl=cache_ttl)

    def related(self, media_type: str, trakt_id_or_slug: str, page: int, limit: int):
        endpoint = f'{media_type}/{trakt_id_or_slug}/related'
        params = {'extended': 'full', 'page': page, 'limit': limit}
        return self.get(endpoint, params=params, auth=True, cache_ttl=TTL_MEDIUM)

    def users_watched_shows(self, page: int = 1, limit: int = 200,
                            extended: bool = False, extended_str: Optional[str] = 'min'):
        endpoint = 'users/me/watched/shows'
        params = {'page': page, 'limit': limit}
        params['extended'] = extended_str or ('full' if extended else 'min')
        return self.get(endpoint, params=params, auth=True, cache_ttl=TTL_MEDIUM) or []

    def users_watched_movies(self, page: int = 1, limit: int = 200, extended: bool = False, extended_str: str = 'min', updated_since: Optional[str] = None, cache_ttl: int = 0):
            """
            Historie filmů uživatele (watched movies) — vrací i plays/last_watched_at.
            """
            endpoint = 'users/me/watched/movies'
            
            # Sestavení základních parametrů
            params = {'page': page, 'limit': limit}
            params['extended'] = extended_str or ('full' if extended else 'min')
            
            # Podmíněné přidání 'updated_since' pro inkrementální update
            if updated_since:
                params['updated_since'] = updated_since
                
            # Nyní voláme API se správně sestavenými parametry a TTL
            return self.get(endpoint, params=params, auth=True, cache_ttl=cache_ttl) or []


    def users_watched_shows_history(self, page: int = 1, limit: int = 200,
                                    extended: bool = True, extended_str: Optional[str] = 'full'):
        """
        Historie seriálů uživatele (watched shows) — vrací i plays/last_watched_at.
        """
        endpoint = 'users/me/watched/shows'
        params = {'page': page, 'limit': limit}
        params['extended'] = extended_str or ('full' if extended else 'min')
        return self.get(endpoint, params=params, auth=True, cache_ttl=0) or []

    def users_history(self, type_: str = 'episodes',
                      start_at: Optional[str] = None, end_at: Optional[str] = None,
                      page: int = 1, limit: int = 200, extended: str = 'min',
                      cache_ttl: int = TTL_MEDIUM):
        endpoint = f'users/me/history/{type_}'
        params = {'page': page, 'limit': limit}
        if start_at: params['start_at'] = start_at
        if end_at:   params['end_at'] = end_at
        if extended: params['extended'] = extended
        return self.get(endpoint, params=params, auth=True, cache_ttl=cache_ttl) or []

    def shows_progress_watched(self, trakt_id_or_slug: str,
                               hidden: bool = False, specials: bool = False,
                               count_specials: bool = False, last_activity: bool = True,
                               cache_ttl: int = 0):
        if not trakt_id_or_slug:
            return None
        endpoint = f"shows/{trakt_id_or_slug}/progress/watched"
        params = {
            'hidden': 'true' if hidden else 'false',
            'specials': 'true' if specials else 'false',
            'count_specials': 'true' if count_specials else 'false',
            'last_activity': 'true' if last_activity else 'false'
        }
        return self.get(endpoint, params=params, auth=True, cache_ttl=cache_ttl)

    def last_activities(self) -> Optional[Dict[str, Any]]:
        return self.get('sync/last_activities', auth=True, cache_ttl=0)

    # NEW — show summary (public endpoint) — doplnění titulu/roku/ids/overview
    def show_summary(self, trakt_id_or_slug: str, extended: str = 'min') -> Dict[str, Any]:
        if not trakt_id_or_slug:
            return {}
        endpoint = f'shows/{trakt_id_or_slug}'
        params = {'extended': extended} if extended else None
        data = self.get(endpoint, params=params, auth=False, cache_ttl=TTL_LONG) or {}
        if isinstance(data, dict):
            return {
                'title': data.get('title') or '',
                'year': data.get('year'),
                'ids': data.get('ids') or {},
                'overview': data.get('overview') or ''
            }
        return {}

    # --- Collections (Movies/Shows) ---
    def collection_movies(self, sort_desc: bool = True):
        data = self.get('users/me/collection/movies', params={'extended': 'full'}, auth=True, cache_ttl=0) or []
        def _sort_key(row): return (row.get('listed_at') or row.get('collected_at') or '')
        data.sort(key=_sort_key, reverse=sort_desc)
        return data

    def collection_shows(self, sort_desc: bool = True):
        data = self.get('users/me/collection/shows', params={'extended': 'full'}, auth=True, cache_ttl=0) or []
        def _sort_key(row): return (row.get('listed_at') or row.get('collected_at') or '')
        data.sort(key=_sort_key, reverse=sort_desc)
        return data

# === Progress cache manager (cache-first) ====================================
class ProgressCacheManager:
    """
    Správa lokální cache pokroku seriálů:
    - progress_cache.json se strukturou:
    {
      "last_activity": ISO8601,
      "shows": {
        "<trakt_or_slug>": {
          "title": str,
          "year": int,
          "ids": {"trakt": int, "slug": str, "tmdb": int},
          "overview": str,
          "last_activity": ISO8601 "",
          "progress": {
            "aired": int,
            "completed": int,
            "next_episode": {"season": int, "number": int, "title": str} {}
          }
        }
      },
      "_last_full_fetch_ts": int (epoch) [informativně]
    }
    """
    def __init__(self, api: TraktAPI):
        self.api = api
        self.profile_root = translatePath(ADDON.getAddonInfo('profile'))
        os.makedirs(self.profile_root, exist_ok=True)
        self.cache_path = os.path.join(self.profile_root, 'progress_cache.json')

    def load_cache(self) -> Dict[str, Any]:
        if not os.path.exists(self.cache_path):
            return {"last_activity": "", "shows": {}}
        try:
            with open(self.cache_path, 'r', encoding='utf-8') as f:
                data = json.load(f)
            if not isinstance(data, dict):
                return {"last_activity": "", "shows": {}}
            data.setdefault("last_activity", "")
            data.setdefault("shows", {})
            return data
        except Exception as e:
            xbmc.log(f"[ProgressCache] read error: {e}", xbmc.LOGWARNING)
            return {"last_activity": "", "shows": {}}

    def save_cache(self, cache: Dict[str, Any]) -> None:
        try:
            with open(self.cache_path, 'w', encoding='utf-8') as f:
                json.dump(cache, f, ensure_ascii=False, indent=2)
            xbmc.log("[ProgressCache] cache saved", xbmc.LOGINFO)
        except Exception as e:
            xbmc.log(f"[ProgressCache] write error: {e}", xbmc.LOGERROR)

    def _compact_show_entry(self, show: Dict[str, Any], prog: Dict[str, Any]) -> Dict[str, Any]:
        ids = show.get('ids', {}) or {}
        title = show.get('title') or ''
        year = show.get('year')
        overview = show.get('overview') or ''
        p_air = (prog.get('aired') or 0) if isinstance(prog, dict) else 0
        p_comp = (prog.get('completed') or 0) if isinstance(prog, dict) else 0
        next_ep = (prog.get('next_episode') or {}) if isinstance(prog, dict) else {}
        last_act = (prog.get('last_watched_at') or prog.get('last_activity') or '') if isinstance(prog, dict) else ''
        return {
            "title": title,
            "year": year,
            "ids": {"trakt": ids.get('trakt'), "slug": ids.get('slug'), "tmdb": ids.get('tmdb')},
            "overview": overview,
            "last_activity": last_act,
            "progress": {
                "aired": int(p_air),
                "completed": int(p_comp),
                "next_episode": {
                    "season": next_ep.get('season') if isinstance(next_ep.get('season'), int) else None,
                    "number": next_ep.get('number') if isinstance(next_ep.get('number'), int) else None,
                    "title": next_ep.get('title') or ''
                } if isinstance(next_ep, dict) else {}
            }
        }

    def _full_fetch(self) -> Dict[str, Any]:
        xbmc.log("[ProgressCache] full fetch start", xbmc.LOGINFO)
        cache = {"last_activity": "", "shows": {}}
        watched = self.api.users_watched_shows(page=1, limit=200, extended=False, extended_str='min') or []
        show_items: List[Dict[str, Any]] = []
        for row in watched:
            if isinstance(row, dict):
                sh = row.get('show') or row or {}
            elif isinstance(row, str):
                sh = {'title': '', 'year': None, 'ids': {'slug': row}, 'overview': ''}
            else:
                continue
            ids = sh.get('ids') or {}
            ident = ids.get('trakt') or ids.get('slug')
            # doplnit summary pokud chybí title/rok/tmdb
            if ident and (not sh.get('title') or sh.get('year') in (None, 0) or not ids.get('tmdb')):
                summary = self.api.show_summary(ident, extended='min') or {}
                if summary:
                    sh['title'] = summary.get('title') or sh.get('title') or ''
                    sh['year'] = summary.get('year') if summary.get('year') is not None else sh.get('year')
                    sh['overview'] = summary.get('overview') or sh.get('overview') or ''
                    sids = summary.get('ids') or {}
                    ids.setdefault('tmdb', sids.get('tmdb'))
                    ids.setdefault('slug', sids.get('slug'))
                    ids.setdefault('trakt', sids.get('trakt'))
                    sh['ids'] = ids
            if ident:
                show_items.append(sh)
        for idx, sh in enumerate(show_items):
            ids = sh.get('ids') or {}
            ident = ids.get('trakt') or ids.get('slug')
            if not ident:
                continue
            prog = self.api.shows_progress_watched(
                ident, hidden=False, specials=False, count_specials=False,
                last_activity=True, cache_ttl=0
            ) or {}
            entry = self._compact_show_entry(sh, prog)
            cache["shows"][str(ident)] = entry
            if (idx + 1) % 20 == 0:
                xbmc.log(f"[ProgressCache] fetched {idx + 1}/{len(show_items)}", xbmc.LOGINFO)
        act = self.api.last_activities() or {}
        episodes_ts = ((act.get('episodes') or {}).get('watched_at')) or ''
        cache["last_activity"] = episodes_ts or ''
        xbmc.log("[ProgressCache] full fetch done", xbmc.LOGINFO)
        return cache

    def ensure_cache(self) -> Dict[str, Any]:
        """
        Cache-first:
        - Pokud lokální cache neexistuje / je prázdná → full fetch + uložení.
        - Jinak vrátí lokální cache BEZ jakéhokoli kontaktu se serverem.
        """
        cache = self.load_cache()
        if not cache.get("shows"):
            xbmc.log("[ProgressCache] init: empty cache → full fetch", xbmc.LOGINFO)
            cache = self._full_fetch()
            self.save_cache(cache)
            return cache
        xbmc.log("[ProgressCache] using local cache (no server checks)", xbmc.LOGINFO)
        return cache

    def full_fetch_with_progress(self, show_cancel_button: bool = True, chunk_log_step: int = 20) -> bool:
        """
        Full fetch s progress barem a možností zrušit.
        - Stáhne watched shows → postupně progress/watched pro každou show.
        - Průběžně informuje dialog (2-argumentové update kvůli kompatibilitě Kodi API).
        - Po dokončení uloží cache a vrátí True; při zrušení False.
        """
        dlg = None
        try:
            dlg = xbmcgui.DialogProgress()
            dlg.create("Trakt • Aktualizace cache", "Získávám seznam rozkoukaných seriálů...")

            watched = self.api.users_watched_shows(page=1, limit=200, extended=False, extended_str='min') or []
            show_items: List[Dict[str, Any]] = []
            for row in watched:
                if isinstance(row, dict):
                    sh = row.get('show') or row or {}
                elif isinstance(row, str):
                    sh = {'title': '', 'year': None, 'ids': {'slug': row}, 'overview': ''}
                else:
                    continue
                ids = sh.get('ids') or {}
                ident = ids.get('trakt') or ids.get('slug')
                if ident:
                    show_items.append(sh)

            total = len(show_items)
            if total == 0:
                dlg.update(100, "Žádné rozkoukané seriály.")
                dlg.close()
                xbmcgui.Dialog().notification("Trakt", "Žádné rozkoukané seriály.", xbmcgui.NOTIFICATION_INFO, 2500)
                return True

            cache = {"last_activity": "", "shows": {}}
            for idx, sh in enumerate(show_items, start=1):
                if dlg.iscanceled():
                    dlg.close()
                    xbmc.log("[ProgressCache] full fetch canceled by user", xbmc.LOGINFO)
                    return False

                ident = (sh.get('ids') or {}).get('trakt') or (sh.get('ids') or {}).get('slug')
                if ident and (not sh.get('title') or sh.get('year') in (None, 0) or not (sh.get('ids') or {}).get('tmdb')):
                    summary = self.api.show_summary(str(ident), extended='min') or {}
                    sids = summary.get('ids') or {}
                    ids = sh.get('ids') or {}
                    ids.setdefault('tmdb', sids.get('tmdb'))
                    ids.setdefault('slug', sids.get('slug'))
                    ids.setdefault('trakt', sids.get('trakt'))
                    sh['ids'] = ids
                    sh['title'] = summary.get('title') or sh.get('title') or ''
                    sh['year'] = summary.get('year') if summary.get('year') is not None else sh.get('year')
                    sh['overview'] = summary.get('overview') or sh.get('overview') or ''

                prog = self.api.shows_progress_watched(
                    str(ident), hidden=False, specials=False, count_specials=False,
                    last_activity=True, cache_ttl=0
                ) or {}
                entry = self._compact_show_entry(sh, prog)
                cache["shows"][str(ident)] = entry

                percent = int((idx / float(total)) * 100.0)
                msg = f"Stahuji progres {idx}/{total}\n{sh.get('title') or str(ident)}"
                dlg.update(percent, msg)
                if idx % chunk_log_step == 0:
                    xbmc.log(f"[ProgressCache] fetched {idx}/{total}", xbmc.LOGINFO)

            act = self.api.last_activities() or {}
            episodes_ts = ((act.get('episodes') or {}).get('watched_at')) or ''
            cache["last_activity"] = episodes_ts or ''
            try:
                import time as _t
                cache["_last_full_fetch_ts"] = int(_t.time())
            except Exception:
                pass

            self.save_cache(cache)
            dlg.update(100, "Hotovo.")
            dlg.close()
            xbmcgui.Dialog().notification("Trakt", "Cache pokroku aktualizována.", xbmcgui.NOTIFICATION_INFO, 2500)
            return True

        except Exception as e:
            if dlg:
                try:
                    dlg.close()
                except Exception:
                    pass
            xbmc.log(f"[ProgressCache] full_fetch_with_progress error: {e}", xbmc.LOGERROR)
            xbmcgui.Dialog().notification("Trakt", "Chyba při aktualizaci cache.", xbmcgui.NOTIFICATION_ERROR, 3000)
            return False

    def drop_show_from_cache(self, ident: str) -> bool:
        try:
            cache = self.load_cache()
            shows = cache.get("shows", {}) or {}
            if ident in shows:
                del shows[ident]
                cache["shows"] = shows
                self.save_cache(cache)
                xbmc.log(f"[ProgressCache] dropped show {ident} from cache", xbmc.LOGINFO)
                return True
            return False
        except Exception as e:
            xbmc.log(f"[ProgressCache] drop_show_from_cache error: {e}", xbmc.LOGERROR)
            return False

    def drop_show_by_trakt_id(self, trakt_id: int) -> bool:
        ident_str = str(trakt_id)
        try:
            cache = self.load_cache()
            shows = cache.get("shows", {}) or {}
            if ident_str in shows:
                del shows[ident_str]
                cache["shows"] = shows
                self.save_cache(cache)
                xbmc.log(f"[ProgressCache] dropped show key={ident_str}", xbmc.LOGINFO)
                return True
            removed = False
            for k, v in list(shows.items()):
                ids = (v or {}).get("ids", {}) or {}
                if str(ids.get("trakt") or "") == ident_str:
                    del shows[k]
                    removed = True
            if removed:
                cache["shows"] = shows
                self.save_cache(cache)
                xbmc.log(f"[ProgressCache] dropped show by ids.trakt={ident_str}", xbmc.LOGINFO)
                return True
            return False
        except Exception as e:
            xbmc.log(f"[ProgressCache] drop_show_by_trakt_id error: {e}", xbmc.LOGERROR)
            return False

class WatchedMoviesCacheManager:
    """Správa lokální cache pro historii filmů."""
    def __init__(self):
        self.profile_root = translatePath(ADDON.getAddonInfo('profile'))
        os.makedirs(self.profile_root, exist_ok=True)
        self.cache_path = os.path.join(self.profile_root, 'watched_movies_cache.json')

    def load_cache(self) -> list:
        if not os.path.exists(self.cache_path):
            return []
        try:
            with open(self.cache_path, 'r', encoding='utf-8') as f:
                data = json.load(f)
                return data if isinstance(data, list) else []
        except Exception as e:
            xbmc.log(f"[MoviesCache] read error: {e}", xbmc.LOGWARNING)
            return []

    def save_cache(self, data: list) -> None:
        try:
            with open(self.cache_path, 'w', encoding='utf-8') as f:
                json.dump(data, f, ensure_ascii=False, indent=2)
            xbmc.log("[MoviesCache] cache saved", xbmc.LOGINFO)
        except Exception as e:
            xbmc.log(f"[MoviesCache] write error: {e}", xbmc.LOGERROR)

class WatchedShowsCacheManager:
    """Správa lokální cache pro historii seriálů."""
    def __init__(self):
        self.profile_root = translatePath(ADDON.getAddonInfo('profile'))
        os.makedirs(self.profile_root, exist_ok=True)
        self.cache_path = os.path.join(self.profile_root, 'watched_shows_cache.json')

    def load_cache(self) -> list:
        if not os.path.exists(self.cache_path):
            return []
        try:
            with open(self.cache_path, 'r', encoding='utf-8') as f:
                data = json.load(f)
                return data if isinstance(data, list) else []
        except Exception as e:
            xbmc.log(f"[ShowsCache] read error: {e}", xbmc.LOGWARNING)
            return []

    def save_cache(self, data: list) -> None:
        try:
            with open(self.cache_path, 'w', encoding='utf-8') as f:
                json.dump(data, f, ensure_ascii=False, indent=2)
            xbmc.log("[ShowsCache] cache saved", xbmc.LOGINFO)
        except Exception as e:
            xbmc.log(f"[ShowsCache] write error: {e}", xbmc.LOGERROR)

# ====================================================================
# --- SAMOSTATNÉ FUNKCE (CACHE REFRESH) ---
# ====================================================================

# --- Progress cache FULL refresh helper --------------------------------------

def trakt_progress_cache_refresh_full_ui():
    """
    Aktualizuje všechny Trakt cache: pokrok seriálů + historie filmů + historie seriálů.
    Zobrazuje progress dialog.
    """
    LOG = "mm.trakt_cache_refresh"
    try:
        api = TraktAPI()
        dlg = xbmcgui.DialogProgress()
        dlg.create("Trakt • Aktualizace cache", "Stahuji pokrok seriálů...")
        # 1) Pokrok seriálů
        pcm = ProgressCacheManager(api)
        ok = pcm.full_fetch_with_progress(show_cancel_button=True, chunk_log_step=20)
        if not ok:
            dlg.close()
            return

        # 2) Historie filmů
        dlg.update(50, "Stahuji historii filmů...")
        movies_data = api.users_watched_movies(page=1, limit=500, extended=True, extended_str='full') or []
        WatchedMoviesCacheManager().save_cache(movies_data)

        # 3) Historie seriálů
        dlg.update(80, "Stahuji historii seriálů...")
        shows_data = api.users_watched_shows_history(page=1, limit=500, extended=True, extended_str='full') or []
        WatchedShowsCacheManager().save_cache(shows_data)

        dlg.update(100, "Hotovo.")
        dlg.close()
        xbmcgui.Dialog().notification('Trakt', 'Cache pokroku + historie aktualizována.', xbmcgui.NOTIFICATION_INFO, 3000)
    except Exception as e:
        xbmc.log(f"[{LOG}] error: {e}", xbmc.LOGERROR)
        try:
            dlg.close()
        except Exception:
            pass
        xbmcgui.Dialog().notification('Trakt', 'Chyba při aktualizaci cache.', xbmcgui.NOTIFICATION_ERROR, 3000)

# ====================================================================
# --- INKREMENTÁLNÍ AKTUALIZACE (Servisní funkce) ---
# ====================================================================

# V resources/lib/trakt/trakt_service.py

def trakt_cache_refresh_incremental_ui():
    """
    Inkrementální aktualizace cache Trakt s progress barem.
    - Aktualizuje progress_cache.json (seriály).
    - Aktualizuje watched_movies_cache.json (filmy).
    - NOVĚ: Odstraňuje zastaralé záznamy z watched_shows_cache.json (Historie seriálů).
    """
    import os, json
    from datetime import datetime
    
    # 1. Instanciace API a příprava cest
    api = TraktAPI()

    try:
        profile_root = translatePath(ADDON.getAddonInfo('profile'))
        last_update_file = os.path.join(profile_root, 'trakt_last_update.json')

        # Načtení last_update
        last_ts = None
        if xbmcvfs.exists(last_update_file):
            try:
                with xbmcvfs.File(last_update_file, 'r') as f:
                    content = f.read()
                    obj = json.loads(content) if content else {}
                    last_ts = obj.get('last_update')
            except Exception as e:
                xbmc.log(f"[TraktCacheUpdate] read last_update failed: {e}", xbmc.LOGWARNING)

        # Progress dialog
        dlg = xbmcgui.DialogProgress()
        dlg.create("Trakt • Rychlá aktualizace", "Zjišťuji změny...")

        params = {'extended': 'min'}
        if last_ts:
            params['updated_since'] = last_ts

        # Pomocné struktury
        changed_shows = {} 
        tmdb_prefetch_items = []
        movie_updates_map = {} 

        def _update_progress(pct: int, line1: str = "", line2: str = ""):
            try:
                dlg.update(max(0, min(100, int(pct))), line1 or "", line2 or "")
            except Exception:
                pass

        # --- A) Změny ve Watched Movies (jen pro prefetch) ---
        _update_progress(10, "Zjišťuji změny: filmy...")
        if dlg.iscanceled(): dlg.close(); return
        
        movies = api.get('sync/watched/movies', params=params, auth=True, cache_ttl=0) or []
        for m in movies:
            if not isinstance(m, dict): continue
            movie_obj = m.get('movie')
            if not isinstance(movie_obj, dict): continue
            ids = movie_obj.get('ids', {}) or {}
            tmdb_id = ids.get('tmdb')
            if isinstance(tmdb_id, int) and tmdb_id > 0:
                tmdb_prefetch_items.append(('movie', tmdb_id))

        # --- B) Změny ve Watched Shows (pro update cache + prefetch) ---
        _update_progress(25, "Zjišťuji změny: seriály...")
        if dlg.iscanceled(): dlg.close(); return
        
        shows = api.get('sync/watched/shows', params=params, auth=True, cache_ttl=0) or []
        for s in shows:
            if not isinstance(s, dict): continue
            show_obj = s.get('show')
            if not isinstance(show_obj, dict): continue
            
            ids = show_obj.get('ids', {}) or {}
            tmdb_id = ids.get('tmdb')
            if isinstance(tmdb_id, int) and tmdb_id > 0:
                tmdb_prefetch_items.append(('tv', tmdb_id))
            
            trakt_id = ids.get('trakt')
            if trakt_id:
                # Klíčové: Ukládáme Trakt ID všech seriálů, které se změnily.
                changed_shows[str(trakt_id)] = show_obj

        # --- C) Změny v Historii (History endpoint - zachytí jednotlivé epizody) ---
        _update_progress(40, "Zjišťuji změny: historie...")
        if dlg.iscanceled(): dlg.close(); return
        
        history = api.get('sync/history', params=params, auth=True, cache_ttl=0) or []
        for h in history:
            if not isinstance(h, dict): continue
            media_type = h.get('type')
            
            if media_type == 'movie':
                obj = h.get('movie')
                if obj and obj.get('ids', {}).get('tmdb'):
                    tmdb_prefetch_items.append(('movie', obj['ids']['tmdb']))
                    
            elif media_type in ('show', 'episode'):
                show_obj = h.get('show')
                if show_obj and isinstance(show_obj, dict):
                    ids = show_obj.get('ids', {}) or {}
                    tmdb_id = ids.get('tmdb')
                    trakt_id = ids.get('trakt')
                    
                    if tmdb_id: tmdb_prefetch_items.append(('tv', tmdb_id))
                    if trakt_id: 
                        # Ukládáme Trakt ID všech seriálů, které se změnily (pokryto krokem B, ale pro jistotu)
                        changed_shows[str(trakt_id)] = show_obj
        
        # Načteme poslední aktivity TEĎ, abychom měli data pro Kroky D, E, F a G
        act = api.last_activities() or {}


        # --- D) UPDATE PROGRESS CACHE (Seriály) ---
        if changed_shows:
            count_upd = len(changed_shows)
            _update_progress(50, f"Aktualizuji pokrok u {count_upd} seriálů...")
            
            pcm = ProgressCacheManager(api)
            current_cache = pcm.load_cache()
            if not current_cache.get("shows"):
                current_cache["shows"] = {}

            idx_s = 0
            for t_id, sh_obj in changed_shows.items():
                if dlg.iscanceled(): dlg.close(); return
                idx_s += 1
                
                # Získání čerstvého stavu seriálu
                prog = api.shows_progress_watched(
                    t_id, hidden=False, specials=False, count_specials=False,
                    last_activity=True, cache_ttl=0
                ) or {}
                
                entry = pcm._compact_show_entry(sh_obj, prog)
                current_cache["shows"][str(t_id)] = entry
                
                if idx_s % 5 == 0:
                    _update_progress(50 + int(10 * idx_s / count_upd), f"Aktualizuji pokrok {idx_s}/{count_upd}")

            # Uložení timestampu poslední aktivity epizod do cache (pro UI)
            episodes_ts = ((act.get('episodes') or {}).get('watched_at')) or ''
            current_cache["last_activity"] = episodes_ts
            
            pcm.save_cache(current_cache)
            xbmc.log(f"[TraktCacheUpdate] Updated progress for {count_upd} shows", xbmc.LOGINFO)
        else:
            xbmc.log("[TraktCacheUpdate] No shows changed.", xbmc.LOGINFO)
        
        
        # --- E) NOVÝ KROK: ODSTRANĚNÍ ZASTARALÝCH ZÁZNAMŮ Z WATCHED SHOWS CACHE ---
        # Tato cache se neaktualizuje inkrementálně, ale odstraníme poškozené/zastaralé záznamy.
        if changed_shows:
            _update_progress(65, f"Odstraňuji zastaralé záznamy pro Historii seriálů ({len(changed_shows)})...")
            if dlg.iscanceled(): dlg.close(); return
            
            wscm = WatchedShowsCacheManager() # Manager pro watched_shows_cache.json
            current_shows_cache = wscm.load_cache() # List seriálových objektů
            
            changed_trakt_ids = set(changed_shows.keys())
            updated_shows_cache = []
            removed_count = 0
            
            for show_entry in current_shows_cache:
                # watched_shows_cache.json vrací strukturu, kde objekt seriálu je v klíči 'show'
                trakt_id = (show_entry.get('show') or {}).get('ids', {}).get('trakt')
                
                if trakt_id and str(trakt_id) in changed_trakt_ids:
                    # Pokud se Trakt ID seriálu objevilo ve změnách, záznam odstraníme.
                    removed_count += 1
                else:
                    # Ponecháme nezměněný záznam.
                    updated_shows_cache.append(show_entry)

            if removed_count > 0:
                wscm.save_cache(updated_shows_cache)
                xbmc.log(f"[TraktCacheUpdate] Removed {removed_count} stale entries from WatchedShowsCache.", xbmc.LOGINFO)
            else:
                xbmc.log("[TraktCacheUpdate] No stale entries found in WatchedShowsCache.", xbmc.LOGDEBUG)


        # --- F) UPDATE MOVIES CACHE (Filmy) ---
        _update_progress(75, "Aktualizuji historii zhlédnutých filmů...") # Změna z 70 na 75
        if dlg.iscanceled(): dlg.close(); return
        
        wcm = WatchedMoviesCacheManager()
        current_movie_cache = wcm.load_cache()
        
        # Stáhneme jen změněné filmy
        movie_updates = api.users_watched_movies(
            updated_since=last_ts, 
            extended=True, 
            extended_str='full', 
            cache_ttl=0
        ) or []
        
        movie_updates_map = {
            (r.get('movie') or {}).get('ids', {}).get('trakt'): r 
            for r in movie_updates
        }
        
        if movie_updates_map:
            newly_watched_count = len(movie_updates_map)
            
            updated_cache = []
            updated_cache.extend(movie_updates)
            
            for movie_entry in current_movie_cache:
                trakt_id = (movie_entry.get('movie') or {}).get('ids', {}).get('trakt')
                if trakt_id and trakt_id not in movie_updates_map:
                    updated_cache.append(movie_entry)
            
            wcm.save_cache(updated_cache)
            xbmc.log(f"[TraktCacheUpdate] Merged {newly_watched_count} movie updates.", xbmc.LOGINFO)
        else:
            xbmc.log("[TraktCacheUpdate] No movies changed.", xbmc.LOGINFO)


        # --- G) TMDb Prefetch (obrázky) ---
        uniq = list(set(tmdb_prefetch_items))
        total_img = len(uniq)
        
        # Přepočítání progress bar pro zbylé kroky (z 75 na 95)
        step_base = 75.0
        step_span = 20.0 
        
        if total_img > 0:
            per_item = step_span / float(total_img)
            prefetched = 0
            
            for (mt, tid) in uniq:
                if dlg.iscanceled(): dlg.close(); return
                try:
                    _prefetch_tmdb_detail_async(tid, mt)
                    prefetched += 1
                except Exception: pass
                
                pct = int(step_base + per_item * prefetched)
                _update_progress(pct, "Doplňuji obrázky...", f"{prefetched}/{total_img}")


        # --- H) Uložení času aktualizace (Serverový čas) ---
        _update_progress(95, "Ukládám čas poslední aktivity...")
        
        episodes_ts = ((act.get('episodes') or {}).get('watched_at')) or ''
        movies_ts = ((act.get('movies') or {}).get('watched_at')) or ''
        
        ts_list = [t for t in [episodes_ts, movies_ts] if t]
        
        if ts_list:
            new_last_ts = max(ts_list)
        else:
            new_last_ts = datetime.utcnow().strftime('%Y-%m-%dT%H:%M:%S.000Z')
        
        try:
            with xbmcvfs.File(last_update_file, 'w') as f:
                f.write(json.dumps({'last_update': new_last_ts}))
        except Exception as e:
            xbmc.log(f"[TraktCacheUpdate] write last_update failed: {e}", xbmc.LOGERROR)

        _update_progress(100, "Hotovo.")
        dlg.close()
        
        # I) Notifikace a Refresh
        stats_msg = f"Seriály: {len(changed_shows)}, Filmy: {len(movie_updates_map)}"
        xbmcgui.Dialog().notification('Trakt', f'Aktualizováno ({stats_msg})', xbmcgui.NOTIFICATION_INFO, 3000)
        
        xbmc.executebuiltin('Container.Refresh')

    except Exception as e:
        try: dlg.close()
        except: pass
        xbmc.log(f"[TraktCacheUpdate] fatal: {e}", xbmc.LOGERROR)
        xbmcgui.Dialog().notification('Trakt', 'Chyba aktualizace (viz log).', xbmcgui.NOTIFICATION_ERROR, 3500)