# -*- coding: utf-8 -*-
import xbmc
import xbmcgui
import xbmcplugin
import xbmcaddon
import sys
import urllib.parse

from resources.lib.tmdb_module import TMDb, API_URL, IMAGE_BASE_URL_POSTER
from resources.lib.ui_utils import build_trakt_context

# Univerzální get_url z Trakt modulu (funguje všude v pluginu)
from resources.lib.trakt.trakt_module import get_url

addon_handle = int(sys.argv[1])
addon = xbmcaddon.Addon('plugin.video.mmirousek')
base_url = sys.argv[0]

def build_url(query):
    return base_url + '?' + urllib.parse.urlencode(query)


def show_seasons(tmdb_id, title, trakt_id=None, slug=None):
    """Zobrazí sezóny seriálu + Trakt kontextové menu"""
    if not tmdb_id:
        xbmcplugin.endOfDirectory(addon_handle, succeeded=False)
        return

    api_key = addon.getSetting('tmdb_api_key')
    if not api_key:
        xbmcgui.Dialog().ok('TMDb', 'Není nastaven TMDb API klíč v nastavení doplňku.')
        xbmcplugin.endOfDirectory(addon_handle, succeeded=False)
        return

    try:
        tmdb = TMDb(api_key)
        data = tmdb._call_api_raw(f"{API_URL}/tv/{tmdb_id}").json()
    except Exception as e:
        xbmc.log(f"[TMDb] Chyba při načítání seriálu: " + str(e), xbmc.LOGERROR)
        xbmcgui.Dialog().notification('TMDb', 'Chyba při načítání seriálu', xbmcgui.NOTIFICATION_ERROR)
        xbmcplugin.endOfDirectory(addon_handle, succeeded=False)
        return

    for season in data.get('seasons', []):
        sn = season.get('season_number')
        if sn == 0 or sn is None:
            continue

        label = f"S{sn:02d} – {season.get('name', 'Sezóna ' + str(sn))} ({season.get('episode_count', 0)} epizod)"

        url = build_url({
            'action': 'tmdb_episodes',
            'tmdb_id': tmdb_id,
            'season_num': sn,
            'title': title,
            'trakt_id': trakt_id or '',
            'slug': slug or ''
        })

        li = xbmcgui.ListItem(label)
        if season.get('poster_path'):
            poster = IMAGE_BASE_URL_POSTER + season['poster_path']
            li.setArt({'thumb': poster, 'icon': poster, 'poster': poster})

        li.setInfo('video', {
            'plot': season.get('overview') or '',
            'season': sn,
            'mediatype': 'season',
            'title': season.get('name')
        })

        # KONEXT MENU – pro celý seriál
        context = build_trakt_context(
            get_url_fn=get_url,
            title=title,
            tmdb_id=int(tmdb_id),
            trakt_id=int(trakt_id) if trakt_id else None,
            slug=slug,
            media_type='shows',
            context_type='progress'
        )
        if context:
            li.addContextMenuItems(context)

        li.setProperty('IsPlayable', 'false')
        xbmcplugin.addDirectoryItem(addon_handle, url, li, isFolder=True)

    xbmcplugin.endOfDirectory(addon_handle)


def show_episodes(tmdb_id, season_num, title, trakt_id=None, slug=None):
    """Zobrazí epizody sezóny + plné Trakt kontextové menu včetně označení epizody"""
    try:
        season_num = int(season_num)
    except:
        xbmcplugin.endOfDirectory(addon_handle, succeeded=False)
        return

    api_key = addon.getSetting('tmdb_api_key')
    try:
        tmdb = TMDb(api_key)
        data = tmdb._call_api_raw(f"{API_URL}/tv/{tmdb_id}/season/{season_num}").json()
    except Exception as e:
        xbmc.log(f"[TMDb] Chyba při načítání sezóny {season_num}: {e}", xbmc.LOGERROR)
        xbmcgui.Dialog().notification('TMDb', 'Chyba při načítání epizod', xbmcgui.NOTIFICATION_ERROR)
        xbmcplugin.endOfDirectory(addon_handle, succeeded=False)
        return

    for ep in data.get('episodes', []):
        en = ep.get('episode_number')
        name = ep.get('name') or f"Epizoda {en}"
        air_date = ep.get('air_date', '')

        label = f"S{season_num:02d}E{en:02d} – {name}"
        if air_date:
            label += f"  •  {air_date.split('-')[0] if '-' in air_date else air_date}"

        url = build_url({'action': 'series_search', 'what': title})

        li = xbmcgui.ListItem(label)

        if ep.get('still_path'):
            still = IMAGE_BASE_URL_POSTER + ep['still_path']
            li.setArt({'thumb': still, 'icon': still})

        li.setInfo('video', {
            'title': name,
            'plot': ep.get('overview') or '',
            'aired': air_date,
            'season': season_num,
            'episode': en,
            'mediatype': 'episode'
        })

        # RYCHLÉ OZNAČENÍ EPIZODY
        context_items = [
            ('Označit jako zhlédnutou (teď)',
             f"RunPlugin({get_url(action='series_trakt_mark_episode', tmdb_id=tmdb_id, season=season_num, episode=en)})"),
            ('Označit epizodu (vybrat datum)…',
             f"RunPlugin({get_url(action='series_trakt_mark_episode_prompt', tmdb_id=tmdb_id)})"),
        ]

        # Plný Trakt kontext (watchlist, related, mazání progressu…)
        full_context = build_trakt_context(
            get_url_fn=get_url,
            title=title,
            tmdb_id=int(tmdb_id),
            trakt_id=int(trakt_id) if trakt_id else None,
            slug=slug,
            media_type='shows',
            context_type='progress'
        )
        context_items.extend(full_context)

        if context_items:
            li.addContextMenuItems(context_items)

        li.setProperty('IsPlayable', 'false')
        xbmcplugin.addDirectoryItem(addon_handle, url, li, isFolder=True)

    xbmcplugin.endOfDirectory(addon_handle)