# resources/lib/stream_utils.py
import xbmc
import xbmcgui
import re
import os

LOG_TAG = '[StreamManager StreamPicker]'

# --- 1. FUNKCE PRO BODOVÁNÍ KVALITY (SCORING) ---
def get_quality_score(filename):
    """
    Analyzuje název souboru a přiřadí mu skóre kvality pro řazení.
    """
    filename = filename.lower()
    score = 0.0
    
    # Prioritní skóre (základní rozlišení)
    if '2160p' in filename or '4k' in filename:
        score += 5.0  # Nejvyšší skóre
    elif '1080p' in filename:
        score += 4.0
    elif '720p' in filename:
        score += 3.0
    else:
        score += 2.0 # Výchozí (neidentifikováno)

    # Modifikátory (upřesnění kvality)
    if 'hdr' in filename or 'dv' in filename or 'atmos' in filename:
        score += 0.5
    if 'web-dl' in filename or 'bluray' in filename:
        score += 0.4
    if 'webrip' in filename:
        score += 0.3
    if 'cam' in filename or 'ts' in filename or 'screener' in filename:
        score -= 2.0 # Velké snížení priority
        
    return score

# --- 2. ŘAZENÍ STREAMŮ ---
def smart_sort_streams(stream_list):
    """
    Seřadí seznam streamů na základě skóre kvality a velikosti souboru.
    
    Předpokládá, že každý prvek v stream_list je slovník s klíči:
    'file_name', 'stream_url', a 'size_bytes' (velikost v bytech).
    """
    
    # Přidáme skóre ke každému streamu
    for stream in stream_list:
        score = get_quality_score(stream.get('file_name', ''))
        stream['score'] = score
        
    # Složené řazení: 1. Skóre (sestupně), 2. Velikost souboru (sestupně)
    sorted_list = sorted(
        stream_list, 
        key=lambda item: (item.get('score', 0.0), item.get('size_bytes', 0)),
        reverse=True 
    )
    
    return sorted_list


# --- 3. HLAVNÍ FUNKCE PRO VÝBĚR STREAMU ---
def select_stream(available_streams):
    """
    Řadí dostupné streamy a zobrazí dialog pro výběr kvality streamu.
    
    :param available_streams: List streamovacích slovníků z Webshare.
    :return: Vybraná stream URL, nebo None při zrušení
    """
    if not available_streams:
        xbmc.log(f'{LOG_TAG} Žádné streamy k výběru.', xbmc.LOGERROR)
        return None
        
    # Seřadíme streamy pomocí Smart Sorting
    sorted_streams = smart_sort_streams(available_streams)

    # Vytvoříme možnosti pro dialog Kodi
    choices = []
    # Mapování indexu dialogu na URL (protože dialog.select vrací index)
    index_to_url_map = {} 
    
    for i, stream in enumerate(sorted_streams):
        file_name = stream.get('file_name', 'Neznámý soubor')
        
        # Převedeme velikost z byte na MB/GB pro lepší čitelnost
        size_bytes = stream.get('size_bytes', 0)
        if size_bytes > (1024 * 1024 * 1024): # Větší než 1 GB
            size_label = f"{round(size_bytes / (1024 * 1024 * 1024), 1)} GB"
        elif size_bytes > (1024 * 1024): # Větší než 1 MB
            size_label = f"{round(size_bytes / (1024 * 1024), 1)} MB"
        else:
            size_label = f"{size_bytes} B"
        
        # Vytvoříme přehledný popisek
        choice_label = f"{file_name} ({size_label})"
        choices.append(choice_label)
        index_to_url_map[i] = stream.get('stream_url')
        
    if len(available_streams) == 1:
        xbmc.log(f'{LOG_TAG} K dispozici je pouze jeden stream, výběr přeskočen.', xbmc.LOGINFO)
        return available_streams[0].get('stream_url')
        
    # Zobrazíme dialog pro výběr
    dialog = xbmcgui.Dialog()
    selection_index = dialog.select('Vyberte kvalitu streamu:', choices)

    if selection_index == -1:
        xbmc.log(f'{LOG_TAG} Výběr streamu zrušen uživatelem.', xbmc.LOGINFO)
        return None
    
    # Vrátíme vybranou URL
    selected_url = index_to_url_map.get(selection_index)
    
    xbmc.log(f'{LOG_TAG} Vybrán stream: {choices[selection_index]}', xbmc.LOGINFO)
    return selected_url

import os
import xbmcaddon
import xbmc

__addon__ = xbmcaddon.Addon()

    