
# -*- coding: utf-8 -*-
import io
import os
import json
import re
import xbmc
import xbmcgui
import xbmcplugin
import xbmcvfs
from xml.etree import ElementTree as ET
from resources.lib.ui_utils import clean_filename, ensure_folder, make_target_path

# ──────────────────────────────────────────────────────────────────────────────
# Historie vyhledávání (předáváme profile + addon místo globálů)
# ──────────────────────────────────────────────────────────────────────────────
def loadsearch(profile, history_file_name='search_history'):
    """Načte historii vyhledávání ze souboru a ošetří neexistující soubor."""
    try:
        with io.open(os.path.join(profile, history_file_name), 'r', encoding='utf8') as f:
            return json.loads(f.read())
    except FileNotFoundError:
        xbmc.log("YaWSP INFO: Soubor historie vyhledávání nenalezen, vracím prázdnou historii.", xbmc.LOGINFO)
        return []
    except Exception as e:
        xbmc.log(f"YaWSP ERROR: Chyba při načítání historie vyhledávání: {e}", xbmc.LOGERROR)
        return []

def storesearch(what, profile, addon, history_file_name='search_history'):
    """Uloží dotaz do historie (omezeno nastavením shistory)."""
    if not what:
        return
    size = int(addon.getSetting('shistory'))
    history = loadsearch(profile, history_file_name)
    if what in history:
        history.remove(what)
    history = [what] + history
    if len(history) > size:
        history = history[:size]
    try:
        with io.open(os.path.join(profile, history_file_name), 'w', encoding='utf8') as f:
            f.write(json.dumps(history))
    except Exception:
        xbmc.log("YaWSP ERROR: Chyba při ukládání historie vyhledávání.", xbmc.LOGERROR)

def removesearch(what, profile, history_file_name='search_history'):
    """Odstraní položku z historie vyhledávání."""
    if not what:
        return
    history = loadsearch(profile, history_file_name)
    if what in history:
        history.remove(what)
    try:
        with io.open(os.path.join(profile, history_file_name), 'w', encoding='utf8') as f:
            f.write(json.dumps(history))
    except Exception:
        xbmc.log("YaWSP ERROR: Chyba při mazání položky z historie.", xbmc.LOGERROR)

# ──────────────────────────────────────────────────────────────────────────────
# Hledání na Webshare (plná 1:1 logika, předáváme vše jako argumenty)
# ──────────────────────────────────────────────────────────────────────────────
def dosearch(token, what, category, sort, limit, offset, action,
             addon, handle, get_url, api, revalidate, popinfo,
             none_placeholder='%#NONE#%'):
    xbmc.log(f"[YaWSP] dosearch(): what={what}, category={category}, sort={sort}, limit={limit}, offset={offset}", xbmc.LOGINFO)


    # — 1) PRVNÍ POKUS: bez tokenu (guest) —
    data = {
        'what': '' if what == none_placeholder else what,
        'category': category,
        'sort': sort,
        'limit': limit,
        'offset': offset,
        'maybe_removed': 'true'
    }
    response = api('search', data)
    try:
        xml = ET.fromstring(response.content)
    except Exception as e:
        xbmc.log(f"[YaWSP] ERROR search parse (guest): {e}", xbmc.LOGERROR)
        xml = None

    ok = (xml is not None) and (xml.find('status') is not None) and (xml.find('status').text == 'OK')

    # — 2) TICHÝ RETRY: pokus o login/revalidate + druhý pokus s wst —
    if not ok:
        try:
            _token = revalidate()
        except Exception as e:
            xbmc.log(f"[YaWSP] revalidate() failed: {e}", xbmc.LOGERROR)
            _token = None
        if _token:
            data['wst'] = _token
            response = api('search', data)
            try:
                xml = ET.fromstring(response.content)
            except Exception as e:
                xbmc.log(f"[YaWSP] ERROR search parse (retry): {e}", xbmc.LOGERROR)
                xml = None
            ok = (xml is not None) and (xml.find('status') is not None) and (xml.find('status').text == 'OK')

    if not ok:
        popinfo(addon.getLocalizedString(30107), icon=xbmcgui.NOTIFICATION_WARNING)
        xbmcplugin.endOfDirectory(handle, succeeded=True)
        return

    # — zpracování výsledků —
    results = []
    for file in xml.iter('file'):
        item = {}
        for e in file:
            item[e.tag] = e.text
        results.append(item)

    def quality_score(item):
        score = 0
        q = (item.get('quality') or '').lower()
        info = (item.get('ainfo') or '').lower()
        if '2160' in q or '4k' in q: score += 100
        elif '1080' in q: score += 80
        elif '720' in q: score += 50
        else: score += 20
        for feature in ['hdr', 'dolby vision', 'dv', 'hevc', 'x265']:
            if feature in q or feature in info:
                score += 10
        lang = (item.get('lang') or '').lower()
        if 'cz' in lang or 'cs' in lang:
            score += 20
        try:
            size = float(item.get('size', 0))
            score += size / (1024 * 1024 * 100)
        except:
            pass
        return score

    results.sort(key=lambda x: quality_score(x), reverse=True)
    xbmc.log(f"[YaWSP] Počet nalezených souborů: {len(results)}", xbmc.LOGINFO)

    for item in results:
        ws_name = item.get('name', '') or ''
        target_path = make_target_path(addon, ws_name)

        # Kontextové menu: Přidat do fronty
        add_url = get_url(action='queue_add', ident=item['ident'], name=ws_name,
                          size=item.get('size', '0'), target_path=target_path)
        commands = [("Přidat do fronty", f"RunPlugin({add_url})")]

        label = f"{item.get('name')} ({item.get('size','?')})"
        li = xbmcgui.ListItem(label=label)
        li.setInfo('video', {'title': label})
        li.setProperty('IsPlayable', 'true')
        li.addContextMenuItems(commands, replaceItems=True)

        xbmcplugin.addDirectoryItem(
            handle,
            get_url(action='play', ident=item['ident'], name=item['name']),
            li, False
        )

    xbmcplugin.endOfDirectory(handle)

# ──────────────────────────────────────────────────────────────────────────────
# Kořen vyhledávání (UI) – plná verze, bez globálů
# ──────────────────────────────────────────────────────────────────────────────
def search(params, addon, handle, profile, get_url, api, revalidate, popinfo,
           categories=None, sorts=None, none_placeholder='%#NONE#%'):
    """
    Kořen vyhledávání (Webshare) + nové hledání + historie.
    Opravuje propisování placeholderu NONE_WHAT (%#NONE#%) do vstupního dialogu.
    """
    # Výchozí hodnoty, pokud nejsou dodány z yawsp
    categories = categories or ['', 'video', 'images', 'audio', 'archives', 'docs', 'adult']
    sorts = sorts or ['', 'recent', 'rating', 'largest', 'smallest']

    def _is_placeholder(v: str) -> bool:
        if not isinstance(v, str):
            return False
        v = v.strip()
        return (v == none_placeholder) or ("%#NONE#%" in v)

    def _sanitize_default(v: str) -> str:
        if not v:
            return ""
        return "" if _is_placeholder(v) else v.strip()

    xbmcplugin.setPluginCategory(handle, addon.getAddonInfo('name') + " \\ " + addon.getLocalizedString(30201))  # "Vyhledávání"
    updateListing = False
    token = None
    what = None

    # 1) Odstranění položky z historie
    if 'remove' in params:
        removesearch(params['remove'], profile)
        updateListing = True

    # 2) Přidání identu do fronty (původní kompatibilita)
    if 'toqueue' in params:
        # Import zde by opět vedl ke kruhové závislosti; použij zavedenou akci queue_add
        # toqueue(params['toqueue'], token)  # vynecháno – nahrazeno v dosearch kontextovým menu
        updateListing = True

    # 3) Integrace s TMDb/Trakt: ?q=...
    if 'q' in params:
        what = params['q']
        if not _is_placeholder(what) and what.strip():
            storesearch(what.strip(), profile, addon)

    # 4) Dialogové „Nové hledání“
    if 'ask' in params and (what is None):
        last = addon.getSetting('slast') or ''
        last = _sanitize_default(last)
        if 'what' in params and isinstance(params['what'], str) and params['what'].strip():
            last = _sanitize_default(params['what'])
        query = xbmcgui.Dialog().input(
            heading=addon.getLocalizedString(30007),
            defaultt=last,
            type=xbmcgui.INPUT_ALPHANUM
        )
        if query:
            what = query.strip()
            if what:
                storesearch(what, profile, addon)
                addon.setSetting('slast', what)
                updateListing = True
        else:
            xbmcplugin.endOfDirectory(handle, updateListing=True)
            return

    # 5) Spuštění hledání
    if what is None and 'what' in params:
        what = params['what']

    if what is not None:
        is_none = _is_placeholder(what)
        category = params['category'] if 'category' in params else categories[int(addon.getSetting('scategory'))]
        sort = params['sort'] if 'sort' in params else sorts[int(addon.getSetting('ssort'))]
        limit = int(params['limit']) if 'limit' in params else int(addon.getSetting('slimit'))
        offset = int(params['offset']) if 'offset' in params else 0

        if 'offset' not in params:
            if not is_none and isinstance(what, str) and what.strip():
                addon.setSetting('slast', what.strip())

        updateListing = True
        dosearch(token, what, category, sort, limit, offset, 'search',
                 addon, handle, get_url, api, revalidate, popinfo,
                 none_placeholder=none_placeholder)
        return

    # 6) Kořenové položky
    addon.setSetting('slast', '')
    history = loadsearch(profile)

    li = xbmcgui.ListItem(label=addon.getLocalizedString(30205))  # "Nové hledání"
    li.setArt({'icon': 'DefaultAddSource.png'})
    xbmcplugin.addDirectoryItem(handle, get_url(action='search', ask=1), li, True)

    li = xbmcgui.ListItem(label=addon.getLocalizedString(30208))  # "Nejnovější"
    li.setArt({'icon': 'DefaultAddonsRecentlyUpdated.png'})
    xbmcplugin.addDirectoryItem(handle, get_url(action='search', what=none_placeholder, sort=sorts[1]), li, True)

    li = xbmcgui.ListItem(label=addon.getLocalizedString(30209))  # "Největší"
    li.setArt({'icon': 'DefaultHardDisk.png'})
    xbmcplugin.addDirectoryItem(handle, get_url(action='search', what=none_placeholder, sort=sorts[3]), li, True)

    for search_text in history:
        safe_default = _sanitize_default(search_text)
        li = xbmcgui.ListItem(label=search_text)
        li.setArt({'icon': 'DefaultAddonsSearch.png'})
        commands = [(addon.getLocalizedString(30213), f"Container.Update({get_url(action='search', remove=search_text)})")]
        li.addContextMenuItems(commands)
        xbmcplugin.addDirectoryItem(handle, get_url(action='search', what=safe_default), li, True)

    xbmcplugin.endOfDirectory(handle, updateListing=updateListing)
