
# -*- coding: utf-8 -*-
# Module: offline_manager
# Author: mmirousek
# License: AGPL v3

import os
import re
import xbmc
import xbmcgui
import xbmcaddon
import xbmcvfs
from resources.lib.download_manager import download_file
from series_manager import SeriesManager
from resources.lib.ui_utils import clean_filename
from resources.lib.ui_utils import reset_cancel_flags, is_cancel_requested

ADDON_ID = 'plugin.video.mmirousek'
ADDON = xbmcaddon.Addon(ADDON_ID)

# --- cancel 'all' flag (Window property) ---
WIN_KEY_ALL = 'mm_cancel_download_all'


def _listdir_safe(path: str):
    """Bezpečně vrátí seznam souborů (pouze přes xbmcvfs – funguje i na SMB/NFS)."""
    if not path:
        return []
    try:
        dirs, files = xbmcvfs.listdir(path)
        return files  # vracíme jen soubory
    except Exception as e:
        xbmc.log(f"[OfflineManager] _listdir_safe error: {path} → {e}", xbmc.LOGDEBUG)
        return []
    
def _get_target_folder(series_name: str, season: str = None):
    """Vrátí cílovou složku pro seriál/sezónu – plně kompatibilní s SMB/NFS."""
    base_folder = ADDON.getSetting('dfolder') or ''
    if not base_folder:
        xbmcgui.Dialog().notification("Offline", "Chybí složka pro stahování (nastavení).", xbmcgui.NOTIFICATION_ERROR)
        return None

    base_real = xbmcvfs.translatePath(base_folder)
    if not base_real:
        return None

    # Zajistíme trailing slash
    if not base_real.endswith(('/', '\\')):
        base_real += '/'

    series_folder = f"{base_real}{clean_filename(series_name)}/"
    if season:
        series_folder += f"Sezona {season}/"

    # Vytvoříme složku přes xbmcvfs (funguje i na SMB/NFS)
    try:
        xbmcvfs.mkdirs(series_folder)
    except Exception as e:
        xbmc.log(f"[OfflineManager] Nelze vytvořit složku {series_folder}: {e}", xbmc.LOGERROR)

    return series_folder  # vracíme VFS cestu s trailing /

def save_episode(series_name: str, season: str, episode: str, series_manager: SeriesManager, prefer_quality=None):
    """
    Přidá epizodu do persistentní fronty stahování.
    """

    if is_cancel_requested():
        xbmc.log(f"[OfflineManager] Cancel active before queue add S{season}E{episode}", xbmc.LOGINFO)
        return

    # Načti data epizody
    data = series_manager.load_series_data(series_name)
    if not data:
        xbmcgui.Dialog().notification("Offline", "Data seriálu nenalezena.", xbmcgui.NOTIFICATION_ERROR)
        return

    ep = data.get('seasons', {}).get(str(season), {}).get(str(episode))
    if not ep or not ep.get('streams'):
        xbmcgui.Dialog().notification("Offline", "Streamy pro epizodu nenalezeny.", xbmcgui.NOTIFICATION_WARNING)
        return

    streams = ep['streams']

    # --- výběr kvality (beze změny) ---
    def _has_quality(st, tag):
        blob = " ".join([st.get('quality', ''), st.get('name', ''), st.get('ainfo', '')]).lower()
        feats = [f.lower() for f in st.get('features', []) if isinstance(f, str)]
        if tag == '2160':
            return ('2160' in blob or '4k' in blob) or any('2160p' in f or '4k' in f for f in feats)
        return (tag in blob) or any(tag in f for f in feats)

    def _pick_stream_by_quality(streams, prefer):
        for st in streams:
            if prefer and _has_quality(st, prefer):
                return st
        if prefer == '2160':
            for tag in ('1080', '720'):
                for st in streams:
                    if _has_quality(st, tag):
                        return st
        elif prefer == '1080':
            for tag in ('720',):
                for st in streams:
                    if _has_quality(st, tag):
                        return st
        return streams[0] if streams else None

    if prefer_quality is None:
        if any(_has_quality(st, '2160') for st in streams):
            prefer_quality = '2160'
        elif any(_has_quality(st, '1080') for st in streams):
            prefer_quality = '1080'
        elif any(_has_quality(st, '720') for st in streams):
            prefer_quality = '720'

    best_stream = _pick_stream_by_quality(streams, prefer_quality or '')
    if not best_stream or not best_stream.get('ident'):
        xbmcgui.Dialog().notification("Offline", "Nepodařilo se vybrat stream.", xbmcgui.NOTIFICATION_ERROR)
        return

    ws_ident = best_stream.get('ident')
    expected_size = int(best_stream.get('size') or 0) if best_stream.get('size') else None

    # Cílová složka
    folder = _get_target_folder(series_name, str(season))
    if not folder:
        return

    # Název souboru
    base_name = clean_filename(ep.get('name') or ep.get('title') or ep.get('source_name') or f"S{season}E{episode}")

    # Získej link (raw pro hlavičky)
    from resources.lib.webshare_auth import WebshareClient
    ws = WebshareClient(ADDON_ID)
    ws.revalidate()
    link_raw = ws.getlink(ws_ident, dtype='download')
    if not link_raw:
        xbmcgui.Dialog().notification("Offline", "Nepodařilo se získat link.", xbmcgui.NOTIFICATION_ERROR)
        return

    url = link_raw.split('\n', 1)[0]
    _, ext = os.path.splitext(url.rsplit('/', 1)[-1].split('?', 1)[0])
    _, base_ext = os.path.splitext(base_name)
    filename = base_name if base_ext else (base_name + (ext or ""))
    target_path = os.path.join(folder, filename)

    # ►► PŘIDÁNÍ DO FRONTY včetně season/episode ◄◄
    from resources.lib.download_queue import add_to_queue
    add_to_queue({
        'ident': ws_ident,
        'name': base_name,
        'series_name': series_name,
        'season': str(season),            # <--- NOVĚ
        'episode': str(episode),          # <--- NOVĚ
        'size': expected_size,
        'target_path': target_path,
        'url': url,
        'link_raw': link_raw,
        'prefer_quality': prefer_quality
    })

    xbmc.log(f"[OfflineManager] Přidáno do fronty: {series_name} S{season}E{episode}", xbmc.LOGINFO)

def save_season(series_name: str, season: str, series_manager: SeriesManager):
    """
    Přidá všechny epizody sezóny do persistentní fronty stahování s jednotnou kvalitou.
    Přeskakuje epizody, které jsou offline nebo už ve frontě.
    Nově: zobrazuje DialogProgress s možností zrušení.
    """

    if is_cancel_requested():
        xbmc.log(f"[OfflineManager] Cancel active before queue add season S{season}", xbmc.LOGINFO)
        reset_cancel_flags()  # nově z ui_utils
        return

    data = series_manager.load_series_data(series_name)
    if not data:
        xbmcgui.Dialog().notification("Offline", "Data seriálu nenalezena.", xbmcgui.NOTIFICATION_ERROR)
        return

    episodes = data.get('seasons', {}).get(str(season), {})
    if not episodes:
        xbmcgui.Dialog().notification("Offline", "Sezóna nenalezena.", xbmcgui.NOTIFICATION_WARNING)
        return

    # Výběr kvality pro sezónu
    has_2160, has_1080, has_720 = False, False, False
    for ep in episodes.values():
        for st in ep.get('streams', []):
            blob = f"{st.get('quality','')} {st.get('name','')} {st.get('ainfo','')}".lower()
            if '2160' in blob or '4k' in blob: has_2160 = True
            if '1080' in blob: has_1080 = True
            if '720' in blob: has_720 = True

    prefer_quality = None
    options, mapping = [], []
    if has_2160: options.append("4K (2160p) – nejvyšší kvalita"); mapping.append('2160')
    if has_1080: options.append("1080p – vyvážená kvalita"); mapping.append('1080')
    if has_720: options.append("720p – menší soubor"); mapping.append('720')

    if options:
        idx = xbmcgui.Dialog().select("Vyber kvalitu pro sezónu", options)
        if idx < 0:
            xbmc.log("[OfflineManager] Uživatel zrušil volbu kvality sezóny.", xbmc.LOGINFO)
            return
        prefer_quality = mapping[idx]

    # Přidávání epizod s progress dialogem
    added = 0
    season_folder = _get_target_folder(series_name, str(season))
    files = _listdir_safe(season_folder) if season_folder and xbmcvfs.exists(season_folder) else []
    from resources.lib.download_queue import load_queue
    q = load_queue() or []

    dlg = xbmcgui.DialogProgress()
    dlg.create("Offline", f"Přidávám sezónu {season} do fronty...")
    total = len(episodes)
    count = 0

    for ep_num in sorted(episodes.keys(), key=int):
        if dlg.iscanceled():
            xbmc.log(f"[OfflineManager] Uživatel zrušil přidávání sezóny S{season}", xbmc.LOGINFO)
            break

        if is_cancel_requested():
            xbmc.log(f"[OfflineManager] Global cancel ALL hit during S{season}E{ep_num} → stopping", xbmc.LOGINFO)
            _clear_cancel_all_flag()
            break

        ep = episodes.get(ep_num)
        ep_title = ep.get('name') or ep.get('title') or ep.get('source_name') or ''
        expected_bases = [clean_filename(ep_title).lower()] if ep_title else []
        expected_bases.append(f"S{int(season):02d}E{int(ep_num):02d}".lower())

                # Offline check
        found = False
        for fname in files:
            try:
                base_noext = os.path.splitext(fname)[0].lower()
                if any(base_noext.startswith(b) or base_noext == b for b in expected_bases):
                    found = True
                    break
            except Exception:
                continue

        if found:
            xbmc.log(f"[OfflineManager] Skip S{season}E{ep_num} (offline exists)", xbmc.LOGINFO)
            count += 1
            dlg.update(int(count * 100 / total), f"Přeskakuji offline S{season}E{ep_num}")
            continue

        # Queue check
        in_queue = any(i.get('series_name') == series_name and i.get('season') == str(season) and i.get('episode') == str(ep_num) for i in q)
        if in_queue:
            xbmc.log(f"[OfflineManager] Skip S{season}E{ep_num} (already in queue)", xbmc.LOGINFO)
            count += 1
            dlg.update(int(count * 100 / total), f"Ve frontě: S{season}E{ep_num}")
            continue

        try:
            save_episode(series_name, str(season), str(ep_num), series_manager, prefer_quality)
            added += 1
            count += 1
            dlg.update(int(count * 100 / total), f"Přidáno {count}/{total} epizod")
        except Exception as e:
            xbmc.log(f"[OfflineManager] CHYBA při přidávání S{season}E{ep_num} do fronty: {e}", xbmc.LOGERROR)
            count += 1
            dlg.update(int(count * 100 / total), f"Chyba u S{season}E{ep_num}")

    dlg.close()
