
# -*- coding: utf-8 -*-
import xbmc
import xbmcaddon
import xbmcgui
import sys
import os
import xbmcvfs
import urllib.parse
import time
from datetime import datetime
try:
    from resources.lib.trakt.trakt_service import trakt_progress_cache_refresh_full_ui, get_last_sync_time, set_last_sync_time
except ImportError as e:
    xbmc.log(f'[Service] TraktService import error: {e}', xbmc.LOGERROR)
    # Voláme sys.exit(1), protože bez Traktu je service k ničemu.
    # Nicméně pro Kodi environment to může být příliš agresivní, necháme log a service se pokusí běžet dál.
    pass
try:
    from resources.lib.utils.background_enrich import background_enrich_omdb_ratings, get_last_omdb_enrich_time, set_last_omdb_enrich_time, background_enrich_tmdb_titles
except ImportError as e:
    xbmc.log(f'[Service] OMDb Enrich import error: {e}', xbmc.LOGERROR)
    pass

ADDON_ID = 'plugin.video.mmirousek_v2'

# Guard proti více instancím
try:
    win = xbmcgui.Window(10000)
    if win.getProperty('mm_scrobble_service_running') == '1':
        xbmc.log('[Service] Už běží – ukončuji druhý start', xbmc.LOGINFO)
        sys.exit(0)
    win.setProperty('mm_scrobble_service_running', '1')
    xbmc.log('[Service] Guard OK', xbmc.LOGINFO)
except Exception as e:
    xbmc.log(f'[Service] Guard property error: {e}', xbmc.LOGWARNING)

try:
    __addon__ = xbmcaddon.Addon(ADDON_ID)
    __addon_path__ = xbmcvfs.translatePath(__addon__.getAddonInfo('path'))
    sys.path.append(os.path.join(__addon_path__, 'resources', 'lib'))
    from resources.lib.utils.cloud_sync import CloudSync
    from scrobble import scrobble_stop, scrobble_pause
    from resources.lib.down.download_queue import load_queue, start_all, stop_all_downloads
    xbmc.log('##### SERVICE.PY BĚŽÍ #####', xbmc.LOGINFO)

except Exception as e:
    xbmc.log(f'[Service] FATÁLNÍ CHYBA při importu: {e}', xbmc.LOGERROR)
    sys.exit(1)


# ----------------------------------------------------------------------
# CloudSync — spuštění synchronizace při startu doplňku
# ----------------------------------------------------------------------
try:
    cs = CloudSync(addon)
    cs.on_start()
except Exception as e:
    xbmc.log(f"[CloudSync] on_start error: {e}", xbmc.LOGERROR)


# --- Pomocné funkce ---
def in_time_window(start_str, end_str):
    try:
        now = datetime.now().time()
        sh, sm = map(int, start_str.split(':'))
        eh, em = map(int, end_str.split(':'))
        start_t = datetime.now().replace(hour=sh, minute=sm, second=0).time()
        end_t = datetime.now().replace(hour=eh, minute=em, second=0).time()
        return (start_t <= now <= end_t) if start_t <= end_t else (now >= start_t or now <= end_t)
    except Exception:
        return False

def _set_auto_session(active: bool):
    try:
        xbmcgui.Window(10000).setProperty('mm_auto_dl_session', '1' if active else '')
        xbmc.log(f"[AutoDownload] session={'ON' if active else 'OFF'}", xbmc.LOGDEBUG)
    except Exception:
        pass

def _is_auto_session() -> bool:
    try:
        return xbmcgui.Window(10000).getProperty('mm_auto_dl_session') == '1'
    except Exception:
        return False

def _any_downloading() -> bool:
    try:
        q = load_queue() or []
        return any((i or {}).get('status') == 'downloading' for i in q)
    except Exception:
        return False

def _any_pending() -> bool:
    try:
        q = load_queue() or []
        return any((i or {}).get('status') == 'pending' for i in q)
    except Exception:
        return False


# --- Player monitor: reaguje jen na přehrávání z našeho doplňku ---
class PlayerMonitor(xbmc.Player):
    def __init__(self):
        super().__init__()
        xbmc.log('##### PlayerMonitor Inicializován #####', xbmc.LOGINFO)

    def _is_ours(self) -> bool:
        """
        Vrátí True, pokud aktuální přehrávání pochází z našeho doplňku.
        Kritéria (primární):
          - ListItem.Property('tmdb_id') je neprázdné
          - ListItem.Property('mediatype') ∈ {'movie','episode','tvshow'}
        Fallback (sekundární, pro jistotu):
          - getPlayingFile() obsahuje 'plugin.video.mmirousek_v2' nebo 'webshare.cz'
        """
        try:
            tmdb_id = xbmc.getInfoLabel('ListItem.Property(tmdb_id)') or ''
            mediatype = (xbmc.getInfoLabel('ListItem.Property(mediatype)') or '').lower()
            if tmdb_id and mediatype in ('movie', 'episode', 'tvshow'):
                return True
            # fallback přes cestu souboru (např. pipe headers u Webshare)
            try:
                path = self.getPlayingFile() or ''
            except Exception:
                path = xbmc.getInfoLabel('Player.FilenameAndPath') or ''
            if 'plugin.video.mmirousek_v2' in path or 'webshare.cz' in path:
                return True
        except Exception:
            pass
        return False

    def onPlayBackStarted(self):
        if not self._is_ours():
            xbmc.log('[Service] onPlayBackStarted: ignored (not our plugin)', xbmc.LOGINFO)
            return
        xbmc.log('[Service] onPlayBackStarted – spouštím scrobble_delay.py', xbmc.LOGINFO)
        try:
            # __addon_path__ je inicializováno výše v service.py (viz tvůj kód)
            script_path = os.path.join(__addon_path__, 'scrobble_delay.py')
            encoded_path = urllib.parse.quote_plus(__addon_path__)
            xbmc.executebuiltin(f'RunScript("{script_path}",1,"{encoded_path}")')
        except Exception as e:
            xbmc.log(f'[Service] onPlayBackStarted error: {e}', xbmc.LOGERROR)


    def onPlayBackStopped(self):
        """
        STOP: primárně rozhodujeme podle locku (trakt_payload_locked), protože
        po skončení přehrávání Kodi často ListItem.Property(...) už nevrátí.
        Pokud lock není aktivní, jako fallback použijeme původní _is_ours().
        """
        try:
            import xbmcaddon
            ADDON = xbmcaddon.Addon('plugin.video.mmirousek_v2')
            lock_active = (ADDON.getSetting('trakt_payload_locked') == 'true')
        except Exception:
            lock_active = False

        if lock_active:
            xbmc.log('[Service] onPlayBackStopped: lock_active -> volám scrobble_stop()', xbmc.LOGINFO)
        else:
            if not self._is_ours():
                xbmc.log('[Service] onPlayBackStopped: ignored (not our plugin, no lock)', xbmc.LOGDEBUG)
                return

        xbmc.log('[Service] onPlayBackStopped – scrobble_stop', xbmc.LOGINFO)
        try:
            from scrobble import scrobble_stop
            scrobble_stop()
        except Exception as e:
            xbmc.log(f'[Service] onPlayBackStopped error: {e}', xbmc.LOGERROR)



    def onPlayBackPaused(self):
        """
        PAUSE: stejná strategie jako u STOP — lock má prioritu.
        Pokud není lock aktivní, ověříme ještě _is_ours() a případně přeskočíme.
        """
        try:
            import xbmcaddon
            ADDON = xbmcaddon.Addon('plugin.video.mmirousek_v2')
            lock_active = (ADDON.getSetting('trakt_payload_locked') == 'true')
        except Exception:
            lock_active = False

        if lock_active:
            xbmc.log('[Service] onPlayBackPaused: lock_active -> volám scrobble_pause()', xbmc.LOGDEBUG)
        else:
            if not self._is_ours():
                xbmc.log('[Service] onPlayBackPaused: ignored (not our plugin, no lock)', xbmc.LOGDEBUG)
                return

        try:
            from scrobble import scrobble_pause
            scrobble_pause()
        except Exception as e:
            xbmc.log(f'[Service] onPlayBackPaused error: {e}', xbmc.LOGERROR)


# --- Hlavní smyčka ---
if __name__ == '__main__':

    try:
        # Kontrola, zda existuje Trakt modul a funkce
        if 'trakt_progress_cache_refresh_full_ui' in locals() or 'trakt_progress_cache_refresh_full_ui' in globals():
            
            # 1. Načtení nastavení (používáme existující __addon__ objekt)
            enabled = __addon__.getSettingBool('trakt_auto_sync_enabled')
            interval_hours = float(__addon__.getSetting('trakt_auto_sync_interval_h') or '24.0')
            INTERVAL_SECONDS = interval_hours * 3600
            
            xbmc.log(f'[TraktService] Auto-sync povolen: {enabled}, interval: {interval_hours}h ({INTERVAL_SECONDS}s)', xbmc.LOGINFO)

            if enabled:
                last_sync_ts = get_last_sync_time()
                current_ts = time.time()
                
                # 2. Kontrola, zda uplynul interval
                if current_ts - last_sync_ts > INTERVAL_SECONDS:
                    xbmc.log('[TraktService] Spouštím full synchronizaci: Uplynul interval.', xbmc.LOGINFO)
                    
                    # --- VOLÁNÍ SYNCHRONIZACE ---
                    trakt_progress_cache_refresh_full_ui()
                    
                    # 3. Uložení nového času po dokončení
                    set_last_sync_time()
                    
                else:
                    time_remaining_h = (INTERVAL_SECONDS - (current_ts - last_sync_ts)) / 3600
                    xbmc.log(f'[TraktService] Synchronizace přeskočena: Interval ({time_remaining_h:.2f} h) ještě neuplynul.', xbmc.LOGINFO)
            
        else:
            xbmc.log('[TraktService] TraktService moduly nejsou k dispozici, sync přeskočena.', xbmc.LOGWARNING)


    except Exception as e:
        xbmc.log(f'[TraktService] Chyba při spouštění jednorázové sync: {e}', xbmc.LOGERROR)

    try:
        if 'background_enrich_omdb_ratings' in locals() or 'background_enrich_omdb_ratings' in globals():
            
            # 1. Načtení nastavení
            enabled = __addon__.getSettingBool('omdb_enrich_enabled')
            interval_hours = float(__addon__.getSetting('omdb_enrich_interval_h') or '24.0')
            INTERVAL_SECONDS = interval_hours * 3600
            
            xbmc.log(f'[OMDB Enrich Service] Auto-enrich povolen: {enabled}, interval: {interval_hours}h ({INTERVAL_SECONDS}s)', xbmc.LOGINFO)

            if enabled:
                last_sync_ts = get_last_omdb_enrich_time()
                current_ts = time.time()
                
                # 2. Kontrola, zda uplynul interval
                if current_ts - last_sync_ts > INTERVAL_SECONDS:
                    xbmc.log('[OMDB Enrich Service] Spouštím doplňování dat OMDb: Uplynul interval.', xbmc.LOGINFO)
                    
                    # --- VOLÁNÍ FUNKCE ---
                    background_enrich_tmdb_titles()
                    background_enrich_omdb_ratings()
                    # 3. Uložení nového času po dokončení
                    set_last_omdb_enrich_time()
                    
                else:
                    time_remaining_h = (INTERVAL_SECONDS - (current_ts - last_sync_ts)) / 3600
                    xbmc.log(f'[OMDB Enrich Service] Doplňování přeskočeno: Interval ({time_remaining_h:.2f} h) ještě neuplynul.', xbmc.LOGINFO)
            
        else:
            xbmc.log('[OMDB Enrich Service] Moduly pro OMDb doplňování nejsou k dispozici, přeskočeno.', xbmc.LOGWARNING)


    except Exception as e:
        xbmc.log(f'[OMDB Enrich Service] Chyba při spouštění OMDb enrich: {e}', xbmc.LOGERROR)

# --- Konec jednorázové logiky ---


    player_monitor = PlayerMonitor()
    monitor = xbmc.Monitor()
    last_auto_check = 0
    xbmc.log('[Service] Hlavní smyčka start', xbmc.LOGINFO)

    try:
        while not monitor.abortRequested():
            try:
                # --- SCROBBLE UPDATE ---
                from resources.lib.scrobble import _update_progress
                _update_progress()

                # --- AUTO DOWNLOAD každých 5 minut ---
                now_ts = time.time()
                if now_ts - last_auto_check >= 300:  # 5 minut
                    last_auto_check = now_ts
                    enable_auto = __addon__.getSettingBool('enable_auto_download')
                    start_time = __addon__.getSetting('auto_download_start') or '22:00'
                    end_time   = __addon__.getSetting('auto_download_end')   or '06:00'
                    in_window  = in_time_window(start_time, end_time)

                    xbmc.log(f"[AutoDownload] enable={enable_auto}, window={start_time}-{end_time}, in_window={in_window}", xbmc.LOGDEBUG)

                    if not enable_auto:
                        if _is_auto_session():
                            xbmc.log("[AutoDownload] Disable detected → clearing session", xbmc.LOGINFO)
                            _set_auto_session(False)
                    else:
                        if in_window:
                            if _any_pending() and not _any_downloading():
                                xbmc.log("[AutoDownload] Starting queue (in window)", xbmc.LOGINFO)
                                xbmcgui.Dialog().notification("AutoDownload", "Automatické stahování zahájeno", xbmcgui.NOTIFICATION_INFO, 2500)
                                _set_auto_session(True)
                                start_all()
                        else:
                            if _is_auto_session() and _any_downloading():
                                xbmc.log("[AutoDownload] Outside window → stopping auto session downloads", xbmc.LOGINFO)
                                stop_all_downloads()
                                xbmcgui.Dialog().notification("AutoDownload", "Stahování ukončeno mimo časové okno", xbmcgui.NOTIFICATION_WARNING, 2500)
                                _set_auto_session(False)
                            elif _is_auto_session():
                                xbmc.log("[AutoDownload] Outside window and no downloading → clearing session", xbmc.LOGDEBUG)
                                _set_auto_session(False)

                if monitor.waitForAbort(1):
                    break
            except Exception as e:
                xbmc.log(f'[Service] Tik chyba: {e}', xbmc.LOGWARNING)

    finally:
        try:
            xbmcgui.Window(10000).setProperty('mm_scrobble_service_running', '')
        except Exception:
            pass
        xbmc.log('##### SERVICE UKONČEN #####', xbmc.LOGINFO)
        
        # CloudSync – provést sync při ukončení, pokud povoleno
        try:
            cs.on_exit()
        except Exception as e:
            xbmc.log(f"[CloudSync] on_exit error: {e}", xbmc.LOGERROR)

