# -*- coding: utf-8 -*-
#
#  CloudSync – FTP synchronizace JSON souborů a metadata.db
#  Převod z Kodi VFS → reálné cesty + použití open() pro binární operace.
#
# ---------------------------------------------------------------------

import os
import time
import ftplib
import xbmc
import xbmcaddon
import xbmcvfs


class CloudSync:

    FTP_HOST = "repo.ido.cz"
    FTP_PORT = 21
    FTP_USER = "mirahoresrepo"
    FTP_PASS = "bovh#DPF29"
    FTP_ROOT = "/web/databaze"

    def __init__(self, addon):
        """addon = xbmcaddon.Addon() instance"""

        self.addon = addon

        self.enabled = addon.getSettingBool("cloud_sync_enabled")
        self.username = addon.getSetting("cloud_sync_user") or ""
        self.sync_json = addon.getSettingBool("cloud_sync_json")
        self.sync_sql = addon.getSettingBool("cloud_sync_sql")
        self.sync_on_start = addon.getSettingBool("cloud_sync_on_start")
        self.sync_on_exit = addon.getSettingBool("cloud_sync_on_exit")

        self.ftp = None

        # Lokální cesty (překládané)
        self.local_series_dir = xbmcvfs.translatePath(
            "special://profile/addon_data/plugin.video.mmirousek_v2/series_db/"
        )

        self.local_sql_file = xbmcvfs.translatePath(
            "special://profile/addon_data/plugin.video.mmirousek_v2/metadata.db"
        )

        xbmc.log(f"[CloudSync] Init enabled={self.enabled} user={self.username}", xbmc.LOGINFO)

    # ---------------------------------------------------------------------
    # Lazy FTP connect
    # ---------------------------------------------------------------------
    def _ensure_ftp(self):
        if self.ftp:
            return

        try:
            self.ftp = ftplib.FTP()
            self.ftp.connect(self.FTP_HOST, self.FTP_PORT, timeout=10)
            self.ftp.login(self.FTP_USER, self.FTP_PASS)
            xbmc.log("[CloudSync] FTP connected", xbmc.LOGINFO)
        except Exception as e:
            xbmc.log(f"[CloudSync] FTP connect FAILED: {e}", xbmc.LOGERROR)
            self.ftp = None

    # ---------------------------------------------------------------------
    # Vzdálené složky
    # ---------------------------------------------------------------------
    def _ensure_remote_dirs(self):
        if not self.username:
            xbmc.log("[CloudSync] Missing username - skip mkdir", xbmc.LOGERROR)
            return

        base = f"{self.FTP_ROOT}/{self.username}"
        sdb = base + "/series_db"

        for folder in (base, sdb):
            try:
                self.ftp.mkd(folder)
            except Exception:
                pass

    # ---------------------------------------------------------------------
    # Remote paths
    # ---------------------------------------------------------------------
    def _remote_json(self, filename):
        return f"{self.FTP_ROOT}/{self.username}/series_db/{filename}"

    def _remote_sql(self):
        return f"{self.FTP_ROOT}/{self.username}/metadata.db"

    # ---------------------------------------------------------------------
    # Local timestamp
    # ---------------------------------------------------------------------
    def _local_ts(self, path):
        try:
            return os.path.getmtime(path)
        except Exception:
            return 0

    # ---------------------------------------------------------------------
    # Remote timestamp (FTP MDTM)
    # ---------------------------------------------------------------------
    def _remote_ts(self, remote_path):
        try:
            resp = self.ftp.sendcmd("MDTM " + remote_path)
            if resp.startswith("213"):
                ts = time.mktime(time.strptime(resp[4:], "%Y%m%d%H%M%S"))
                return ts
        except Exception:
            pass
        return None

    # ---------------------------------------------------------------------
    # DOWNLOAD
    # ---------------------------------------------------------------------
    def _download(self, remote_path, local_path):
        try:
            # ensure phys. path
            local_path = xbmcvfs.translatePath(local_path)
            xbmc.log(f"[CloudSync-DEBUG] local_path = {local_path}", xbmc.LOGINFO)

            # ensure folder exist
            local_dir = os.path.dirname(local_path)
            if not xbmcvfs.exists(local_dir):
                ok = xbmcvfs.mkdirs(local_dir)
                xbmc.log(f"[CloudSync] mkdirs {local_dir} -> {ok}", xbmc.LOGINFO)

            # download
            with open(local_path, "wb") as out_f:
                self.ftp.retrbinary("RETR " + remote_path, out_f.write)

            xbmc.log(f"[CloudSync] Download OK: {remote_path} -> {local_path}", xbmc.LOGINFO)
            return True

        except Exception as e:
            xbmc.log(f"[CloudSync] Download FAIL {remote_path}: {e}", xbmc.LOGERROR)
            return False

    # ---------------------------------------------------------------------
    # UPLOAD
    # ---------------------------------------------------------------------
    def _upload(self, local_path, remote_path):
        try:
            local_path = xbmcvfs.translatePath(local_path)

            with open(local_path, "rb") as f:
                # ensure remote folder
                try:
                    remote_dir = os.path.dirname(remote_path)
                    self.ftp.mkd(remote_dir)
                except Exception:
                    pass

                self.ftp.storbinary("STOR " + remote_path, f)

                # Po uložení na FTP zjistit remote timestamp a nastavit ho lokálnímu souboru
                remote_ts = self._remote_ts(remote_path)
                if remote_ts:
                    try:
                        os.utime(local_path, (remote_ts, remote_ts))
                        xbmc.log(f"[CloudSync] Local timestamp updated to match remote: {remote_ts}", xbmc.LOGINFO)
                    except Exception as e:
                        xbmc.log(f"[CloudSync] Failed to update local timestamp: {e}", xbmc.LOGERROR)

            xbmc.log(f"[CloudSync] Upload OK: {remote_path}", xbmc.LOGINFO)
            return True

        except Exception as e:
            xbmc.log(f"[CloudSync] Upload FAIL {remote_path}: {e}", xbmc.LOGERROR)
            return False

    # ---------------------------------------------------------------------
    # JSON sync (all files)
    # ---------------------------------------------------------------------

    def _sync_json_files(self):
        xbmc.log("[CloudSync] Sync JSON start", xbmc.LOGINFO)

        # Lokální soubory
        try:
            _dirs, local_files = xbmcvfs.listdir(self.local_series_dir)
            local_set = {f for f in local_files if f.lower().endswith(".json")}
        except Exception as e:
            xbmc.log(f"[CloudSync] listdir failed: {e}", xbmc.LOGERROR)
            local_set = set()

        # Vzdálené soubory (FTP)
        remote_set = set(self._list_remote_jsons())

        # Sjednocená množina → projdeme VŠE (lokální i z cloudu)
        all_names = sorted(local_set | remote_set)

        # Pokud nic, není co řešit
        if not all_names:
            xbmc.log("[CloudSync] No JSON files (local|remote) found", xbmc.LOGINFO)
            return

        for fname in all_names:
            local_path = os.path.join(self.local_series_dir, fname)
            remote_path = self._remote_json(fname)

            local_ts = self._local_ts(local_path)
            remote_ts = self._remote_ts(remote_path)

            xbmc.log(f"[CloudSync-DEBUG] {fname} local_ts={local_ts} remote_ts={remote_ts}", xbmc.LOGDEBUG)

            # 1) Na cloudu neexistuje → pokud je lokální, nahrajeme (mirror chování)
            if remote_ts is None:
                if local_ts > 0:
                    xbmc.log(f"[CloudSync] JSON upload (remote missing): {fname}", xbmc.LOGINFO)
                    self._upload(local_path, remote_path)
                else:
                    # neexistuje ani lokálně (jen ve sjednocené množině kvůli chybějícímu NLST? nic)
                    xbmc.log(f"[CloudSync] JSON skip (absent both local & remote): {fname}", xbmc.LOGDEBUG)
                continue

            # 2) Lokálně chybí (local_ts == 0) → stáhnout
            if local_ts == 0:
                xbmc.log(f"[CloudSync] JSON download (missing locally): {fname}", xbmc.LOGINFO)
                self._download(remote_path, local_path)
                # zarovnat čas lokálního souboru na remote (předejdeme cyklům)
                try:
                    if remote_ts:
                        os.utime(local_path, (remote_ts, remote_ts))
                except Exception:
                    pass
                continue

            # 3) Porovnání timestampů (po naší předchozí úpravě _upload() už umíme srovnat časy)
            if local_ts > remote_ts:
                xbmc.log(f"[CloudSync] JSON upload (local newer): {fname}", xbmc.LOGINFO)
                self._upload(local_path, remote_path)
            elif remote_ts > local_ts:
                xbmc.log(f"[CloudSync] JSON download (remote newer): {fname}", xbmc.LOGINFO)
                self._download(remote_path, local_path)
                try:
                    if remote_ts:
                        os.utime(local_path, (remote_ts, remote_ts))
                except Exception:
                    pass
            else:
                xbmc.log(f"[CloudSync] JSON OK (in sync): {fname}", xbmc.LOGDEBUG)

    def _sync_single_json(self, fname):
        local_path = os.path.join(self.local_series_dir, fname)
        remote_path = self._remote_json(fname)

        local_ts = self._local_ts(local_path)
        remote_ts = self._remote_ts(remote_path)
        xbmc.log(f"[CloudSync-DEBUG] {fname} local_ts={local_ts} remote_ts={remote_ts}", xbmc.LOGINFO)

        if remote_ts is None:
            xbmc.log(f"[CloudSync] JSON upload (remote missing): {fname}", xbmc.LOGINFO)
            self._upload(local_path, remote_path)
            return

        if local_ts > remote_ts:
            xbmc.log(f"[CloudSync] JSON upload (local newer): {fname}", xbmc.LOGINFO)
            self._upload(local_path, remote_path)

        elif remote_ts > local_ts:
            xbmc.log(f"[CloudSync] JSON download (remote newer): {fname}", xbmc.LOGINFO)
            self._download(remote_path, local_path)

    # ---------------------------------------------------------------------
    # SQL sync
    # ---------------------------------------------------------------------
    def _sync_sql_file(self):
        local_path = self.local_sql_file
        remote_path = self._remote_sql()

        local_ts = self._local_ts(local_path)
        remote_ts = self._remote_ts(remote_path)

        if remote_ts is None:
            xbmc.log("[CloudSync] SQL upload (remote missing)", xbmc.LOGINFO)
            self._upload(local_path, remote_path)
            return

        if local_ts > remote_ts:
            xbmc.log("[CloudSync] SQL upload (local newer)", xbmc.LOGINFO)
            self._upload(local_path, remote_path)

        elif remote_ts > local_ts:
            xbmc.log("[CloudSync] SQL download (remote newer)", xbmc.LOGINFO)
            self._download(remote_path, local_path)

    # ---------------------------------------------------------------------
    # FULL SYNC
    # ---------------------------------------------------------------------
    def sync_all(self):
        if not self.enabled:
            xbmc.log("[CloudSync] Disabled → skip sync", xbmc.LOGINFO)
            return

        self._ensure_ftp()
        if not self.ftp:
            return

        self._ensure_remote_dirs()

        if self.sync_json:
            self._sync_json_files()

        if self.sync_sql:
            self._sync_sql_file()

        xbmc.log("[CloudSync] Sync ALL complete", xbmc.LOGINFO)

    # ---------------------------------------------------------------------
    # Hooks for service
    # ---------------------------------------------------------------------
    def on_start(self):
        if self.enabled and self.sync_on_start:
            xbmc.log("[CloudSync] on_start → sync_all()", xbmc.LOGINFO)
            self.sync_all()

    def on_exit(self):
        if self.enabled and self.sync_on_exit:
            xbmc.log("[CloudSync] on_exit → sync_all()", xbmc.LOGINFO)
            self.sync_all()

    def _list_remote_jsons(self):
        """
        Vrátí seznam názvů souborů *.json ve složce
        /web/databaze/<user>/series_db na FTP.
        Používá NLST (je rychlé a dostatečné).
        """
        try:
            base = f"{self.FTP_ROOT}/{self.username}/series_db"
            # zajistit, že složka existuje (pokud ne, vrátíme prázdný list)
            try:
                self.ftp.cwd(base)
            except Exception:
                return []

            names = self.ftp.nlst()
            # NLST vrací buď názvy nebo cesty – normalizujeme jen na "file.json"
            out = []
            for n in names:
                fn = n.rsplit('/', 1)[-1]
                if fn.lower().endswith(".json"):
                    out.append(fn)
            return sorted(set(out))
        except Exception as e:
            xbmc.log(f"[CloudSync] _list_remote_jsons error: {e}", xbmc.LOGWARNING)
            return []
