
# -*- coding: utf-8 -*-
"""
Univerzální aplikační log pro plugin.video.mmirousek_v2:
- thread-safe zápis (LOCK),
- rotace při dosažení velikosti (MAX_BYTES),
- textové logování (úroveň, modul, zpráva),
- JSON Lines logování (událost, kontext),
- utilita tail() pro rychlé čtení posledních řádků.
"""

import os
import time
import json
import gzip
import threading

import xbmcaddon
import xbmcvfs

ADDON = xbmcaddon.Addon('plugin.video.mmirousek_v2')
PROFILE = xbmcvfs.translatePath(ADDON.getAddonInfo('profile'))

# Cesty k logům
LOG_PATH = os.path.join(PROFILE, 'app.log')
# Pozn.: JSON log přesměrujeme do stejného souboru jako NDJSON řádky (jednotný log).
# Pokud bys chtěl oddělené soubory, přidej LOG_JSON_PATH = os.path.join(PROFILE, 'app.jsonl').

# Zámek pro bezpečný zápis
LOCK = threading.Lock()

# Parametry rotace
MAX_BYTES = 2 * 1024 * 1024   # 2 MB na soubor
MAX_BACKUPS = 4               # app.log.1 .. app.log.4.gz

def _ensure_profile():
    """Zajistí existenci profilu (addon_data), aby open() nepadal."""
    try:
        if not xbmcvfs.exists(PROFILE):
            xbmcvfs.mkdirs(PROFILE)
    except Exception:
        # Best-effort; open() to obvykle stejně vytvoří dle translatePath
        pass

def _rotate_if_needed():
    """Velikostní rotace: current → .1 (plain), .2+ → gzip."""
    try:
        if os.path.exists(LOG_PATH) and os.path.getsize(LOG_PATH) >= MAX_BYTES:
            # posuň staré verze (plain .1, ostatní připravíme pro gzip)
            for i in reversed(range(1, MAX_BACKUPS)):
                src = f"{LOG_PATH}.{i}"
                dst = f"{LOG_PATH}.{i+1}"
                if os.path.exists(src):
                    os.replace(src, dst)
            # aktuální → .1
            os.replace(LOG_PATH, f"{LOG_PATH}.1")

            # .2 .. .MAX_BACKUPS → gzip
            for i in range(2, MAX_BACKUPS + 1):
                path_i = f"{LOG_PATH}.{i}"
                gz_i = f"{path_i}.gz"
                if os.path.exists(path_i):
                    try:
                        with open(path_i, 'rb') as fin, gzip.open(gz_i, 'wb', compresslevel=6) as fout:
                            fout.write(fin.read())
                        os.remove(path_i)
                    except Exception:
                        # Nevadí, když gzip selže; ponecháme plain
                        pass
    except Exception:
        # Rotace nesmí shodit doplněk
        pass

def write(level: str, msg: str, module: str = ''):
    """
    Zápis jedné textové řádky.
    level: 'DEBUG' | 'INFO' | 'WARN' | 'ERROR'
    module: jméno modulu/procesu (např. 'tmdb', 'trakt', 'queue', 'service')
    """
    ts = time.strftime('%Y-%m-%d %H:%M:%S', time.localtime())
    line = f"[{ts}] [{(level or 'INFO').upper()}] [{module}] {msg}\n"
    with LOCK:
        try:
            _ensure_profile()
            _rotate_if_needed()
            # Na translatePath máme lokální cestu => použití standardního open() je ok.
            with open(LOG_PATH, 'a', encoding='utf-8', errors='ignore') as f:
                f.write(line)
        except Exception:
            # Best-effort: logování nesmí vyhodit chybu do aplikace
            pass

def write_json(event: str, data: dict = None, level: str = 'INFO', module: str = ''):
    """
    Zápis JSON Lines (NDJSON):
    {"ts": "...", "level": "...", "event": "...", "module": "...", "data": {...}}
    """
    rec = {
        "ts": time.strftime('%Y-%m-%dT%H:%M:%S', time.gmtime()),  # UTC ISO-8601 bez offsetu
        "level": (level or 'INFO').upper(),
        "event": event,
        "module": module or '',
        "data": data or {}
    }
    line = json.dumps(rec, ensure_ascii=False) + "\n"
    with LOCK:
        try:
            _ensure_profile()
            _rotate_if_needed()
            with open(LOG_PATH, 'a', encoding='utf-8', errors='ignore') as f:
                f.write(line)
        except Exception:
            pass

def tail(max_lines: int = 2000, max_bytes: int = 256 * 1024) -> str:
    """
    Vrátí poslední část logu jako text pro rychlé zobrazení (TextViewer).
    - preferuje čtení posledních max_bytes,
    - následně ořízne na max_lines konce.
    """
    try:
        if not os.path.exists(LOG_PATH):
            return ""
        size = os.path.getsize(LOG_PATH)
        start = max(0, size - int(max_bytes))
        content = ""
        with open(LOG_PATH, 'r', encoding='utf-8', errors='ignore') as f:
            if start > 0:
                f.seek(start)
                # zahodit případnou rozseklou první řádku
                f.readline()
            content = f.read()
        # na závěr limit na počet řádků
        lines = content.splitlines()
        return "\n".join(lines[-int(max_lines):])
    except Exception:
        return ""

# (EOF-padding-1)
# (EOF-padding-2)
# (EOF-padding-3# (EOF-padding-3)
