# -*- coding: utf-8 -*-
import xbmc
import xbmcgui
import xbmcplugin
import xbmcaddon
import sys
import urllib.parse
from resources.lib.utils.ui_utils import pick_icon as ui_pick_icon, add_menu_item as ui_add_item
from resources.lib.utils.ui_utils import build_trakt_context
from resources.lib.tmdb.tmdb_utils import get_english_title          # ← ZACHOVÁNO

# Importujeme TMDb klienta pro získání dat
from resources.lib.tmdb.tmdb_module import TMDb, API_URL, IMAGE_BASE_URL_POSTER  # ← TADY TO PŘIDEJ!
from resources.lib.tmdb.omdb import get_omdb_info, extract_metrics, build_label2
# from resources.lib.cache_store import get_imdb_id, put_imdb_id
# --- IMPORTY (OPRAVENÉ PRO /lib/tmdb/) ---
from resources.lib.tmdb.tmdb_series_detail import show_seasons, show_episodes
# Zbytek importů zůstává stejný
from resources.lib.tmdb.omdb import get_omdb_info, extract_metrics, build_label2
from resources.lib.utils.ui_utils import build_trakt_context
from resources.lib.metadata_db import db  # ← NOVÉ – centrální DB
from typing import Optional

# Společné utilitky pro ikonky a položky s plot

# --- TMDb Konstanty pro obrázky ---
IMAGE_BASE_URL = 'https://image.tmdb.org/t/p/w500' 
FANART_BASE_URL = 'https://image.tmdb.org/t/p/original' 

# Standardní Kodi setup pro práci s doplňkem
addon_handle = int(sys.argv[1])
addon = xbmcaddon.Addon('plugin.video.mmirousek_v2')
base_url = sys.argv[0]

# --- build_url: původní funkce ---
def build_url(query):
    """Pomocná funkce pro sestavení URL, kterou Kodi pošle zpět do routeru."""
    return base_url + '?' + urllib.parse.urlencode(query)

# --- NOVÝ WRAPPER ---
def build_url_kwargs(**kwargs):
    """Wrapper kolem build_url, který akceptuje **kwargs (dict-like)."""
    return build_url(kwargs)

# Kategorie (zachovány přesně)
MOVIE_CATEGORIES = {
    'movie_popular': 'Populární filmy',
    'movie_top_rated': 'Nejlépe hodnocené filmy',
    'movie_upcoming': 'Připravované filmy',
}

TV_CATEGORIES = {
    'tv_popular': 'Populární seriály',
    'tv_top_rated': 'Nejlépe hodnocené seriály',
    'tv_on_the_air': 'Právě vysílané (týden)',
    'tv_airing_today': 'Vysílané dnes',
}
PERSON_CATEGORIES = {
    'person_popular': 'Populární osoby (herci, tvůrci)',
    'person_trending_day': 'Trendy osoby dnes',
    'person_trending_week': 'Trendy osoby týdne',
}


def set_video_info(li, info_labels: dict):
    """
    Bezpečně nastaví video metadata. Převezme existující slovník info_labels
    a nastaví jednotlivá pole přes InfoTagVideo, aby zmizel deprekační warning.
    Funguje pro filmy/seriály/epizody (typuje se podle 'mediatype').
    """
    try:
        tag = li.getVideoInfoTag()
    except Exception:
        # fallback (nemělo by nastat v nových Kodi), ale ať nic nespadne
        try:
            li.setInfo('Video', info_labels or {})
        except Exception:
            pass
        return

    if not isinstance(info_labels, dict):
        return

    # --- Title / Plot / PlotOutline / Tagline ---
    title = info_labels.get('title')
    plot = info_labels.get('plot')
    plotoutline = info_labels.get('plotoutline')
    tagline = info_labels.get('tagline')

    if title:
        tag.setTitle(str(title))
    if plot:
        tag.setPlot(str(plot))
    elif plotoutline:
        # pokud máš jen outline, použij ho jako plot
        tag.setPlot(str(plotoutline))
    # Tagline – některé skiny ho zobrazují
    if tagline:
        try:
            tag.setTagLine(str(tagline))
        except Exception:
            # starší verze nemusí mít setter – ignoruj
            pass

    # --- Media type ---
    mtype = info_labels.get('mediatype')
    if isinstance(mtype, str) and mtype:
        tag.setMediaType(mtype)

    # --- Year / Rating / Votes ---
    year = info_labels.get('year')
    rating = info_labels.get('rating')
    votes = info_labels.get('votes') or info_labels.get('imdbvotes')

    try:
        if isinstance(year, int):
            tag.setYear(year)
        elif isinstance(year, str) and year.isdigit():
            tag.setYear(int(year))
    except Exception:
        pass

    try:
        if rating is not None:
            tag.setRating(float(rating))
    except Exception:
        pass

    try:
        if votes is not None:
            tag.setVotes(int(votes))
    except Exception:
        pass

    # --- Season/Episode (pokud je to epizoda) ---
    season = info_labels.get('season')
    episode = info_labels.get('episode')
    if season is not None:
        try:
            tag.setSeason(int(season))
        except Exception:
            pass
    if episode is not None:
        try:
            tag.setEpisode(int(episode))
        except Exception:
            pass

    # --- Duration (minuty) ---
    duration = info_labels.get('duration')
    try:
        if isinstance(duration, int) and duration > 0:
            tag.setDuration(duration)
        elif isinstance(duration, str) and duration.isdigit():
            tag.setDuration(int(duration))
    except Exception:
        pass

    # --- Datum (správný setter podle typu) ---
    # Pozn.: pro epizody používáme setFirstAired(),
    # pro filmy/TV show setPremiered() (pokud někdy vyloženě předáváš 'aired')
    aired = info_labels.get('aired')      # 'YYYY-MM-DD'
    premiered = info_labels.get('premiered') or info_labels.get('released')  # někdy jiný klíč
    try:
        if mtype == 'episode' and aired:
            tag.setFirstAired(aired)           # epizody
        elif premiered:
            tag.setPremiered(premiered)        # filmy/TV show
        elif mtype == 'movie' and aired:
            tag.setPremiered(aired)            # fallback pro filmy
    except Exception:
        pass

    # --- Genres / Directors / Writers (pokud je někde předáváš do dictu) ---
    genres = info_labels.get('genre') or info_labels.get('genres')
    if isinstance(genres, (list, tuple)) and genres:
        try:
            tag.setGenres([str(g) for g in genres if g])
        except Exception:
            pass
    elif isinstance(genres, str) and genres:
        try:
            tag.setGenres([g.strip() for g in genres.split(',') if g.strip()])
        except Exception:
            pass

    directors = info_labels.get('director') or info_labels.get('directors')
    if isinstance(directors, (list, tuple)) and directors:
        try:
            tag.setDirectors([str(d) for d in directors if d])
        except Exception:
            pass
    elif isinstance(directors, str) and directors:
        try:
            tag.setDirectors([d.strip() for d in directors.split(',') if d.strip()])
        except Exception:
            pass

    writers = info_labels.get('writer') or info_labels.get('writers')
    if isinstance(writers, (list, tuple)) and writers:
        try:
            tag.setWriters([str(w) for w in writers if w])
        except Exception:
            pass
    elif isinstance(writers, str) and writers:
        try:
            tag.setWriters([w.strip() for w in writers.split(',') if w.strip()])
        except Exception:
            pass

    # --- IMDB number (pokud někde v dictu máš) ---
    imdb_id = info_labels.get('imdbnumber') or info_labels.get('imdb_id')
    if isinstance(imdb_id, str) and imdb_id:
        try:
            tag.setIMDBNumber(imdb_id)
        except Exception:
            pass

    # Hotovo: žádné li.setInfo() se nevolá → zmizí warning a hodnoty jsou
    # nastavené přes InfoTagVideo per-field settery.


# ----------------------------------------------------------------------
# 1. HLAVNÍ MENU FUNKCE
# ----------------------------------------------------------------------
def show_tmdb_menu():
    """Vytvoří základní menu: Trendy, Filmy a Seriály (sjednocený look)."""
    # xbmc.log("TMDb INFO: Generování hlavního menu (Trendy/Filmy/Seriály).", xbmc.LOGINFO)

    ui_add_item(
        handle=addon_handle,
        build_url_fn=build_url,
        label='Vyhledat na TMDb',
        plot='Zadej název filmu nebo seriálu a zobraz výsledky vyhledávání přímo z TMDb.',
        action='tmdb_search',
        art_icon=ui_pick_icon(addon, 'trending_day.png', 'DefaultAddonsSearch.png', 'tmdb'),
        is_folder=True
    )

    ui_add_item(
        handle=addon_handle,
        build_url_fn=build_url,
        label='Trendy dnes (Filmy & Seriály)',
        plot='Kombinovaný žebříček toho, co je dnes na TMDb nejvíce trendy.',
        action='tmdb_list',
        art_icon=ui_pick_icon(addon, 'trending_day.png', 'DefaultFavourites.png', 'tmdb'),
        is_folder=True,
        category='trending_day',
        page=1
    )

    ui_add_item(
        handle=addon_handle,
        build_url_fn=build_url,
        label='Filmy',
        plot='Procházení filmových kategorií: Populární, Nejlépe hodnocené, Připravované.',
        action='tmdb_movie_categories',
        art_icon=ui_pick_icon(addon, 'movie_popular.png', 'DefaultMovies.png', 'tmdb'),
        is_folder=True
    )

    ui_add_item(
        handle=addon_handle,
        build_url_fn=build_url,
        label='Seriály',
        plot='Procházení seriálových kategorií: Populární, Nejlépe hodnocené, Právě vysílané, Dnes v TV.',
        action='tmdb_tv_categories',
        art_icon=ui_pick_icon(addon, 'tv_popular.png', 'DefaultTVShows.png', 'tmdb'),
        is_folder=True
    )
    ui_add_item(
        handle=addon_handle,
        build_url_fn=build_url,
        label='Osoby (Herci, Tvůrci)',
        plot='Procházení populárních a trendy osobností na TMDb a jejich vyhledávání.',
        action='tmdb_person_categories', # NOVÁ AKCE!
        art_icon=ui_pick_icon(addon, 'person_popular.png', 'DefaultActors.png', 'tmdb'),
        is_folder=True
    )

    xbmcplugin.endOfDirectory(addon_handle)


def show_movie_categories():
    items = [
        ('movie_popular',   'Populární filmy',         'Filmy, které jsou právě nejvíce populární.',         'movie_popular.png'),
        ('movie_top_rated', 'Nejlépe hodnocené filmy', 'Filmy s nejvyšším hodnocením na TMDb.',               'movie_top_rated.png'),
        ('movie_upcoming',  'Připravované filmy',      'Nadcházející premiéry a brzké uvedení do kin.',       'movie_upcoming.png'),
    ]

    for key, label, plot, icon in items:
        ui_add_item(
            handle=addon_handle,
            build_url_fn=build_url,
            label=label,
            plot=plot,
            action='tmdb_list',
            art_icon=ui_pick_icon(addon, icon, 'DefaultFolder.png', 'tmdb'),
            is_folder=True,
            category=key,
            page=1
        )

    # --- NOVĚ: Pokročilý filtr (TMDb Discover) ---
    ui_add_item(
        handle=addon_handle,
        build_url_fn=build_url,
        label='Pokročilý filtr (TMDb)',
        plot=('Filtrování filmů podle žánru, volitelného roku/rozpětí a minimálního ratingu.\n'
              'Výsledky se zobrazí se stránkováním jen s „Další strana“.'),
        action='tmdb_advanced_filter_menu',  # → otevře dialogy a po výběru přesměruje na tmdb_discover_list
        art_icon=ui_pick_icon(addon, 'filter_advanced.png', 'DefaultMovieTitle.png', 'tmdb'),
        is_folder=True
    )
    ui_add_item(
        handle=addon_handle,
        build_url_fn=build_url,
        label='Pokračovat s posledním filtrem (TMDb)',
        plot='Znovu otevře TMDb Discover s naposledy použitými parametry.',
        action='tmdb_discover_resume',
        art_icon=ui_pick_icon(addon, 'filter_advanced.png', 'DefaultMovieTitle.png', 'tmdb'),
        is_folder=True
    )

    xbmcplugin.endOfDirectory(addon_handle)

def show_tv_categories():
    items = [
        ('tv_popular',       'Populární seriály',          'Seriály, které jsou právě nejvíce populární.',            'tv_popular.png'),
        ('tv_top_rated',     'Nejlépe hodnocené seriály',  'Seriály s nejvyšším hodnocením na TMDb.',                 'tv_top_rated.png'),
        ('tv_on_the_air',    'Právě vysílané (týden)',     'Epizody, které se aktuálně vysílají v tomto týdnu.',      'tv_on_the_air.png'),
        ('tv_airing_today',  'Vysílané dnes',              'Seriály, které mají novou epizodu právě dnes.',           'tv_airing_today.png'),
    ]
    for key, label, plot, icon in items:
        ui_add_item(
            handle=addon_handle,
            build_url_fn=build_url,
            label=label,
            plot=plot,
            action='tmdb_list',
            art_icon=ui_pick_icon(addon, icon, 'DefaultFolder.png', 'tmdb'),
            is_folder=True,
            category=key,
            page=1
        )
    xbmcplugin.endOfDirectory(addon_handle)

def tmdb_search_menu():
    kb = xbmc.Keyboard('', 'Vyhledat na TMDb')
    kb.doModal() 
    if kb.isConfirmed():
        search_query = kb.getText()
        if not search_query:
            xbmcplugin.endOfDirectory(addon_handle, succeeded=False)
            return
        xbmc.log(f"TMDb INFO: Vyhledávání dotazu (Keyboard): {search_query}", xbmc.LOGINFO)
        url = build_url({
            'action': 'tmdb_list', 
            'category': 'search', 
            'q': search_query,
            'page': 1
        })
        xbmc.executebuiltin(f'Container.Update({url}, replace)')
    xbmcplugin.endOfDirectory(addon_handle)

# ----------------------------------------------------------------------
# 2. ZOBRAZENÍ SEZNAMU POLOŽEK (Filmy/Seriály) – VYLEPŠENO O HROMADNÉ NAČÍTÁNÍ
# ----------------------------------------------------------------------
# resources/lib/tmdb_kodi.py (Nová funkce, přidejte ji pod show_tv_categories)

def show_person_categories():
    """Vykreslí podkategorie pro objevování osob (Populární, Trendy)."""
    items = [
        ('person_popular',      'Populární osoby',      'Aktuálně nejpopulárnější herci a tvůrci na TMDb.',    'person_popular.png'),
        ('person_trending_day', 'Trendy osoby dnes',    'Osoby s největším nárůstem popularity během 24h.',   'trending_day.png'),
        ('person_trending_week','Trendy osoby týdne',   'Osoby s největším nárůstem popularity za poslední týden.', 'trending_week.png'),
    ]
    
    # První položka je vyhledávání
    ui_add_item(
        handle=addon_handle,
        build_url_fn=build_url,
        label='Vyhledat osobu na TMDb',
        plot='Zadej jméno osoby a zobraz výsledky vyhledávání.',
        action='tmdb_search_person', # NOVÁ AKCE PRO VYHLEDÁVÁNÍ OSOB
        art_icon=ui_pick_icon(addon, 'trending_day.png', 'DefaultAddonsSearch.png', 'tmdb'),
        is_folder=True
    )

    for key, label, plot, icon in items:
        ui_add_item(
            handle=addon_handle,
            build_url_fn=build_url,
            label=label,
            plot=plot,
            action='tmdb_list', # Použijeme stávající 'tmdb_list'
            art_icon=ui_pick_icon(addon, icon, 'DefaultFolder.png', 'tmdb'),
            is_folder=True,
            category=key
        )
    xbmcplugin.endOfDirectory(addon_handle)
# resources/lib/tmdb_kodi.py (Nová funkce, přidejte ji pod tmdb_search_menu)

def tmdb_search_person_menu():
    """Spustí dialog pro vyhledávání osoby a přesměruje na tmdb_list."""
    kb = xbmc.Keyboard('', 'Vyhledat osobu na TMDb')
    kb.doModal() 
    if kb.isConfirmed():
        search_query = kb.getText()
        if not search_query:
            xbmcplugin.endOfDirectory(addon_handle, succeeded=False)
            return
        xbmc.log(f"TMDb INFO: Vyhledávání osoby (Keyboard): {search_query}", xbmc.LOGINFO)
        url = build_url({
            'action': 'tmdb_list', 
            'category': 'search_person', # NOVÁ KATEGORIE!
            'q': search_query
        })
        xbmc.executebuiltin(f'Container.Update({url}, replace)')
    xbmcplugin.endOfDirectory(addon_handle)    

def show_tmdb_list(params):
    category = params.get('category')
    query = params.get('q')
    try:
        page = int(params.get('page', 1))
    except Exception:
        page = 1

    # --- Ověření TMDb API klíče ---
    api_key = addon.getSetting('tmdb_api_key')
    if not api_key:
        xbmcgui.Dialog().notification('TMDb chyba', 'Chybí TMDb API klíč v nastavení doplňku.',
                                      xbmcgui.NOTIFICATION_WARNING, 3000)
        xbmcplugin.endOfDirectory(addon_handle, succeeded=False)
        return

    tmdb = TMDb(api_key)

    # --- Načtení jedné stránky dat (paging) ---
    try:
        items, total_pages, page_size = tmdb.get_list_page(category, query=query, page=page)
    except Exception as e:
        xbmc.log(f"TMDb ERROR: Chyba při získávání seznamu/vyhledávání: {e}", xbmc.LOGERROR)
        xbmcgui.Dialog().notification('TMDb chyba', f'Chyba při volání API: {e}',
                                      xbmcgui.NOTIFICATION_ERROR, 3000)
        xbmcplugin.endOfDirectory(addon_handle, succeeded=False)
        return

    if not items:
        xbmc.log(f"TMDb INFO: Načteno 0 položek pro kategorii {category} (q={query}) page={page}", xbmc.LOGINFO)
        xbmcplugin.endOfDirectory(addon_handle)
        return

    # --- Nastavení content typu podle obsahu stránky ---
    has_tv = any((it.get('media_type') or '') == 'tv' for it in items)
    has_movies = any((it.get('media_type') or '') == 'movie' for it in items)
    if has_tv and not has_movies:
        xbmcplugin.setContent(addon_handle, 'tvshows')
    elif has_movies and not has_tv:
        xbmcplugin.setContent(addon_handle, 'movies')
    else:
        xbmcplugin.setContent(addon_handle, 'videos')

    # --- Helper: bezpečný build „zlatého“ OMDb headeru ---
    def _build_gold_header_from_metrics(metrics_dict: dict) -> str:
        try:
            parts = []

            # IMDb
            r = metrics_dict.get('imdbRating')
            v = metrics_dict.get('imdbVotes')
            if r not in (None, "", 0, "N/A"):
                try:
                    r_val = float(r)
                    star = f"[COLOR=FFFFD700]★ {r_val:.1f}[/COLOR]"
                except Exception:
                    star = f"[COLOR=FFFFD700]★ {r}[/COLOR]"
                votes_txt = None
                if v not in (None, "", 0, "N/A"):
                    try:
                        v_clean = int(str(v).replace(",", "").replace(" ", ""))
                        votes_txt = f"{v_clean:,}".replace(",", " ")
                    except Exception:
                        votes_txt = str(v)
                parts.append("IMDb: " + (f"{star} ({votes_txt})" if votes_txt else star))

            # Metascore
            m = metrics_dict.get('Metascore')
            if m not in (None, "", 0, "N/A"):
                parts.append(f"Metascore: [B]{m}[/B]")

            # RottenTomatoes
            rt = metrics_dict.get('RottenTomatoes')
            if rt not in (None, "", "N/A"):
                parts.append(f"RT: [B]{rt}[/B]")

            # Box Office
            bo = metrics_dict.get('BoxOffice')
            if bo not in (None, "", "N/A"):
                parts.append(f"Box Office: [B]{bo}[/B]")

            safe_parts = [p for p in parts if isinstance(p, str) and p.strip()]
            if safe_parts:
                return f"[COLOR FFFFFF00]{' • '.join(safe_parts)}[/COLOR]"
        except Exception as e:
            xbmc.log(f"[OMDb] build gold header error: {e}", xbmc.LOGWARNING)
        return ""

    # --- Helper: bezpečné vložení headeru do plotu ---
    def _inject_header_into_plot(li: xbmcgui.ListItem, info_labels: dict, header: str, fallback_plot: str):
        try:
            if not header:
                return
            plot_existing = info_labels.get('plot')
            if not isinstance(plot_existing, str):
                plot_existing = str(plot_existing) if plot_existing is not None else ""
            if not plot_existing:
                plot_existing = str(fallback_plot or "")
            info_labels['plot'] = f"{header}\n\n{plot_existing}".strip()
            set_video_info(li, info_labels)
        except Exception as e:
            xbmc.log(f"[OMDb] inject header into plot error: {e}", xbmc.LOGWARNING)

    # --- Jednoduchá větev pro OSOBY (paging + bezpečný artwork) ---
    is_person_list = items[0].get('media_type') == 'person'
    if is_person_list:
        xbmcplugin.setPluginCategory(addon_handle, (PERSON_CATEGORIES.get(category) or 'Vyhledávání osob'))
        xbmcplugin.setContent(addon_handle, 'actors')

        for item in items:
            person_id = item.get('id')
            name = item.get('label') or 'Neznámá osoba'
            profile_path = item.get('profile_path')
            known_for_str = ", ".join([it.get('title') or it.get('name') or ''
                                       for it in item.get('known_for', [])[:3] if it]) or ''

            li = xbmcgui.ListItem(label=name)
            li.setProperty('mediatype', 'actor')
            if profile_path:
                try:
                    p = str(profile_path or '').strip()
                    if p:
                        li.setArt({'icon': IMAGE_BASE_URL + p, 'thumb': IMAGE_BASE_URL + p})
                except Exception as e:
                    xbmc.log(f"[TMDb] Person art concat error person_id={person_id}: {e}", xbmc.LOGWARNING)
            else:
                li.setArt({'icon': 'DefaultActor.png', 'thumb': 'DefaultActor.png'})

            info_labels = {'title': name, 'plot': f"Známý pro: {known_for_str}".strip(), 'mediatype': 'actor'}
            set_video_info(li, info_labels)

            url = build_url({'action': 'person_detail', 'person_id': str(person_id or '')})
            xbmcplugin.addDirectoryItem(addon_handle, url, li, isFolder=True)

        # Další strana (dole)
        display_total = (str(total_pages) if total_pages else '?')

        # 1) Tlačítko DALŠÍ STRANA (Next Page)
        # Podmínka: Pokud aktuální stránka je menší než celkový počet
        if total_pages and page < total_pages:
            next_page = page + 1
            
            # Musíme sestavit URL se všemi parametry (category, q), aby vyhledávání pokračovalo
            url_next = build_url({
                'action': 'tmdb_list',
                'category': str(category or ''),
                'q': str(query or ''),
                'page': str(next_page)
            })
            
            li_next = xbmcgui.ListItem(label=f"[COLOR gold]>>> Další strana ({next_page}/{display_total}) >>>[/COLOR]")
            li_next.setArt({'icon': 'DefaultVideoPlaylists.png'})
        
        xbmcplugin.addDirectoryItem(addon_handle, url_next, li_next, isFolder=True)


        xbmcplugin.endOfDirectory(addon_handle)
        return  # ukončit person větev (bez OMDb)

    # --- OMDb konfigurace (na stránku) ---
    addon_settings = xbmcaddon.Addon('plugin.video.mmirousek_v2')
    try:
        omdb_enabled = addon_settings.getSettingBool('omdb_enable')
    except Exception:
        omdb_enabled = True
    try:
        max_omdb = int(addon.getSetting('omdb_max_per_page') or '20')
    except Exception:
        max_omdb = 20
    try:
        timeout = int(addon.getSetting('omdb_timeout') or '4')
    except Exception:
        timeout = 4

    # --- Bulk external_ids pro tuto stránku ---
    tmdb_ids_movie = [it['id'] for it in items if it.get('media_type') == 'movie']
    tmdb_ids_tv = [it['id'] for it in items if it.get('media_type') == 'tv']
    external_movie = tmdb.get_external_ids_bulk(tmdb_ids_movie, 'movie') if tmdb_ids_movie else {}
    external_tv = tmdb.get_external_ids_bulk(tmdb_ids_tv, 'tv') if tmdb_ids_tv else {}
    all_external = {**external_movie, **external_tv}

    # --- Render položek (bezpečně) ---
    for idx, item in enumerate(items):
        try:
            media_type = item.get('media_type')
            tmdb_id = item.get('id')

            cached = db.get(tmdb_id) or {}
            title_cs = cached.get('title_cs')
            title_en = cached.get('title_en')
            title_original = cached.get('title_original')
            year = cached.get('year') or item.get('year')
            poster_path = cached.get('poster_path') or item.get('poster_path')
            backdrop_path = cached.get('backdrop_path') or item.get('backdrop_path')
            original_plot = cached.get('overview_cs') or item.get('plot') or ''
            original_language = item.get('original_language') or 'en'
            trakt_watched = cached.get('trakt_watched')

            # Fallback detail (cs-CZ) jen když je opravdu potřeba
            need_detail = (not (title_cs or title_en or title_original)) or (not poster_path) or (not original_plot)
            if need_detail and tmdb_id:
                try:
                    detail = tmdb._call_api_raw(f"{API_URL}/{media_type}/{tmdb_id}",
                                                {'language': 'cs-CZ'}).json()
                    title_cs = detail.get('title') or detail.get('name') or title_cs
                    title_en = detail.get('original_title') or detail.get('original_name') or title_en
                    title_original = detail.get('original_title') or detail.get('original_name') or title_original
                    year = (detail.get('release_date') or detail.get('first_air_date') or '')[:4] or year
                    poster_path = detail.get('poster_path') or poster_path
                    backdrop_path = detail.get('backdrop_path') or backdrop_path
                    plot_cs = detail.get('overview')
                    if plot_cs:
                        original_plot = plot_cs
                    if original_language not in ['en', 'cs']:
                        english_label = get_english_title(media_type, tmdb_id)
                        if english_label:
                            title_cs = english_label
                            xbmc.log(f"[Fallback] Použit EN název pro TMDb {tmdb_id}: {english_label}", xbmc.LOGINFO)
                    db.update(tmdb_id, {
                        'media_type': media_type,
                        'title_cs': title_cs,
                        'title_en': title_en,
                        'title_original': title_original,
                        'year': year,
                        'poster_path': poster_path,
                        'backdrop_path': backdrop_path,
                        'overview_cs': original_plot
                    }, source='tmdb_detail')
                except Exception as e:
                    xbmc.log(f"[TMDb] Fallback název selhal pro {tmdb_id}: {e}", xbmc.LOGWARNING)

            # Sestava labelu
            label = title_cs or title_en or title_original or item.get('label') or ''
            fallback_title = title_en or title_original or label
            secondary_label = None
            if original_language == 'en' and title_original and str(title_original) != str(label):
                secondary_label = str(title_original)
            if original_language not in ['en', 'cs'] and tmdb_id:
                try:
                    english_label = get_english_title(media_type, tmdb_id)
                    if english_label:
                        label = english_label
                        fallback_title = english_label
                except Exception:
                    pass

            prefix = "[COLOR orange]Seriál[/COLOR] " if media_type == 'tv' else "[COLOR purple]Film[/COLOR] "
            watched_prefix = "[COLOR green]● [/COLOR]" if (trakt_watched and int(trakt_watched) >= 1) else ""
            label = "" if label is None else str(label)
            year_str = str(year) if year not in (None, "") else ""
            display_label = f"{watched_prefix}{prefix}{label}"
            if secondary_label and secondary_label != label:
                display_label += f" [{secondary_label}]"
            if year_str:
                display_label += f" ({year_str})"
            rating_val = item.get('rating', 0) or 0.0
            try:
                rating_val = float(rating_val)
            except Exception:
                rating_val = 0.0
            if rating_val > 0.0:
                display_label += f" [COLOR=FFFFFFFF]★[/COLOR][COLOR=FFFFD700]{rating_val:.1f}[/COLOR]"

            # URL po kliku
            if media_type == 'tv':
                url = build_url({
                    'action': 'tmdb_seasons',
                    'tmdb_id': str(tmdb_id or ''),
                    'title': str(label or ''),
                    'fallback_title': str(fallback_title or '')
                })
                is_folder = True
            else:
                url = build_url({
                    'action': 'show_search_dialog',
                    'cz_title': str(title_cs or ''),
                    'eng_title': str(title_en or ''),
                    'year': str(year or ''),
                    'tmdb_id': str(tmdb_id or ''),
                    'mediatype': 'movie'
                })
                is_folder = True

            # ListItem + bezpečný artwork
            li = xbmcgui.ListItem(label=display_label)
            if poster_path:
                try:
                    p = str(poster_path or '').strip()
                    if p:
                        li.setArt({'icon': IMAGE_BASE_URL + p, 'thumb': IMAGE_BASE_URL + p})
                except Exception as e:
                    xbmc.log(f"[TMDb] Poster path concat error tmdb_id={tmdb_id}: {e}", xbmc.LOGWARNING)
            if backdrop_path:
                try:
                    b = str(backdrop_path or '').strip()
                    if b:
                        li.setArt({'fanart': FANART_BASE_URL + b})
                except Exception as e:
                    xbmc.log(f"[TMDb] Fanart path concat error tmdb_id={tmdb_id}: {e}", xbmc.LOGWARNING)

            info_labels = {
                'title': label,
                'plot': original_plot or '',
                'year': int(year) if isinstance(year, str) and year.isdigit() else (year if isinstance(year, int) else None),
                'rating': rating_val,
                'mediatype': 'tvshow' if media_type == 'tv' else 'movie',
            }
            info_labels = {k: v for k, v in info_labels.items() if v not in (None, '')}
            set_video_info(li, info_labels)
            if tmdb_id:
                li.setProperty('tmdb_id', str(tmdb_id))
                li.setProperty('mediatype', 'tvshow' if media_type == 'tv' else 'movie')

            # OMDb: získání imdb_id
            imdb_id = (all_external.get(tmdb_id, {}) or {}).get('imdb_id') or cached.get('imdb_id')
            if tmdb_id and not imdb_id:
                try:
                    ext_func = tmdb.get_tv_external_ids if media_type == 'tv' else tmdb.get_movie_external_ids
                    ext = ext_func(tmdb_id)
                    imdb_id = ext.get('imdb_id')
                    try:
                        put_imdb_id(tmdb_id, media_type, imdb_id)  # může být neimportováno → ošetřeno
                    except NameError:
                        pass
                except Exception as e:
                    xbmc.log(f"TMDb WARN: external_ids selhalo pro {tmdb_id}: {e}", xbmc.LOGWARNING)

            # DB větev OMDb
            row = None
            try:
                row = db.conn.execute("""
                  SELECT imdb_rating, imdb_votes, metascore, rt_rating, box_office
                  FROM media WHERE tmdb_id = ?
                """, (tmdb_id,)).fetchone()
            except Exception as e:
                xbmc.log(f"[OMDb] DB SELECT failed for tmdb_id={tmdb_id}: {e}", xbmc.LOGWARNING)

            metrics_from_db = None
            if row:
                vals = list(row)
                if len(vals) < 5:
                    vals += [None] * (5 - len(vals))
                metrics_from_db = {
                    'imdbRating': vals[0],
                    'imdbVotes': vals[1],
                    'Metascore': vals[2],
                    'RottenTomatoes': vals[3],
                    'BoxOffice': vals[4],
                }

            use_omdb_api = True
            if metrics_from_db and metrics_from_db.get('imdbRating') not in (None, "", 0, "N/A"):
                label2 = build_label2(metrics_from_db)
                if label2:
                    try:
                        li.setLabel2(label2)
                    except Exception:
                        xbmc.log("[OMDb] setLabel2 failed — skin nepodporuje label2", xbmc.LOGDEBUG)

                colored = f"[B][COLOR=FFFFD700]{label2}[/COLOR][/B]" if label2 else ""
                if colored:
                    existing_tagline = info_labels.get('tagline') or ''
                    if str(existing_tagline).strip() != colored:
                        info_labels['tagline'] = colored
                        set_video_info(li, info_labels)

                # GOLD header do plotu
                header = _build_gold_header_from_metrics(metrics_from_db)
                _inject_header_into_plot(li, info_labels, header, original_plot)

                use_omdb_api = False

            # API větev OMDb (limitováno na prvních N položek stránky)
            if omdb_enabled and use_omdb_api and tmdb_id and idx < max_omdb:
                omdb_title = (fallback_title or label) if not imdb_id else None
                omdb_type = 'series' if media_type == 'tv' else 'movie'
                omdb_info = get_omdb_info(
                    title=omdb_title,
                    year=year if not imdb_id else None,
                    imdb_id=imdb_id,
                    media_type=omdb_type,
                    timeout=timeout
                )
                metrics = extract_metrics(omdb_info)
                label2 = build_label2(metrics)
                if metrics:
                    try:
                        db.update(tmdb_id, {
                            'imdb_rating': metrics.get('imdbRating'),
                            'imdb_votes': metrics.get('imdbVotes'),
                            'metascore': metrics.get('Metascore'),
                            'rt_rating': metrics.get('RottenTomatoes'),
                            'box_office': metrics.get('BoxOffice'),
                        }, source='omdb')
                    except Exception as e:
                        xbmc.log(f"[OMDb] DB save error: {e}", xbmc.LOGWARNING)

                if label2:
                    try:
                        li.setLabel2(label2)
                    except Exception:
                        xbmc.log("[OMDb] setLabel2 failed — skin nepodporuje label2", xbmc.LOGDEBUG)
                    colored = f"[B][COLOR=FFFFD700]{label2}[/COLOR][/B]"
                    existing_tagline = info_labels.get('tagline') or ''
                    if str(existing_tagline).strip() != colored:
                        info_labels['tagline'] = colored
                        set_video_info(li, info_labels)

                    # GOLD header do plotu
                    header = _build_gold_header_from_metrics(metrics)
                    _inject_header_into_plot(li, info_labels, header, original_plot)

            # Kontextové menu (Trakt + ČSFD)
            cm = build_trakt_context(
                get_url_fn=build_url_kwargs,
                title=label,
                tmdb_id=tmdb_id,
                media_type='shows' if media_type == 'tv' else 'movies',
                context_type='default'
            )
            csfd_url = build_url_kwargs(action='csfd_lookup_enhanced', series_name=label)
            csfd_menu_item = ("ČSFD detail (rozšířený)", f"RunPlugin({csfd_url})")
            if isinstance(cm, list):
                cm.append(csfd_menu_item)
            else:
                cm = [csfd_menu_item] if not cm else list(cm) + [csfd_menu_item]
            if cm:
                li.addContextMenuItems(cm, replaceItems=False)

            xbmcplugin.addDirectoryItem(addon_handle, url, li, isFolder=is_folder)

        except Exception as e:
            import traceback
            xbmc.log(f"[TMDb LIST RENDER] FATAL tmdb_id={item.get('id')} media_type={item.get('media_type')}: {traceback.format_exc()}",
                     xbmc.LOGERROR)
            continue  # pokračuj na další položku

    # --- COMMIT nově získaných metadat ---
    try:
        db.conn.commit()
    except Exception as e:
        xbmc.log(f"[TMDb] Database COMMIT error: {e}", xbmc.LOGERROR)

    # --- Další strana (dole seznamu) ---
    # Další strana (dole)
    display_total = (str(total_pages) if total_pages else '?')

    # 1) Tlačítko DALŠÍ STRANA (Next Page)
    # Podmínka: Pokud aktuální stránka je menší než celkový počet
    if total_pages and page < total_pages:
        next_page = page + 1
        
        # Musíme sestavit URL se všemi parametry (category, q), aby vyhledávání pokračovalo
        url_next = build_url({
            'action': 'tmdb_list',
            'category': str(category or ''),
            'q': str(query or ''),
            'page': str(next_page)
        })
        
        li_next = xbmcgui.ListItem(label=f"[COLOR gold]>>> Další strana ({next_page}/{display_total}) >>>[/COLOR]")
        li_next.setArt({'icon': 'DefaultVideoPlaylists.png'})
    
    xbmcplugin.addDirectoryItem(addon_handle, url_next, li_next, isFolder=True)
    # Finální ukončení listu
    xbmcplugin.endOfDirectory(addon_handle)

# from typing import Optional # MUSÍ BÝT ZAJIŠTĚN IMPORT
# Tato funkce je definována uvnitř show_person_detail, proto ji zde zopakuji pro kontext
def _year_from_str(s: Optional[str]) -> int:
    try:
        t = (s or '').strip()
        return int(t[:4]) if len(t) >= 4 else 0
    except Exception:
        return 0
        
def show_person_detail(params):
    """
    Zobrazí detail osoby a její filmografii získanou z TMDb.
    Voláno z routeru akcí 'person_detail'.
    """
    person_id = params.get('person_id')
    if not person_id:
        xbmc.log("TMDb ERROR: Chybí person_id pro show_person_detail", xbmc.LOGERROR)
        return

    handle = int(sys.argv[1])
    api_key = xbmcaddon.Addon().getSetting('tmdb_api_key')
    tmdb = TMDb(api_key)

    # --- 1) Detail osoby (bio CZ/EN, fotka, credits) ---
    info = tmdb.get_person_detail(int(person_id))
    if not info:
        xbmcplugin.endOfDirectory(handle, cacheToDisc=False)
        return

    name = info.get('name') or 'Neznámé jméno'
    bio = (info.get('biography') or '').strip()
    profile_path = info.get('profile_path')
    birthday = info.get('birthday')

    xbmcplugin.setPluginCategory(handle, f"TMDb • {name}")
    xbmcplugin.setContent(handle, 'actors')
    
    # Položka s informacemi o osobě
    colored_name = f"[COLOR red]{name}[/COLOR]"
    li_info = xbmcgui.ListItem(label=colored_name)    
    
    art = {'icon': 'DefaultActor.png'} # Výchozí ikona
    if profile_path:
        thumb = IMAGE_BASE_URL_POSTER + str(profile_path)
        art.update({'thumb': thumb, 'icon': thumb})
    li_info.setArt(art)
    
    plot_text = f"Narozen: {birthday}\n\n{bio}" if birthday else bio
    li_info.setInfo('Video', {'title': name, 'plot': plot_text or 'Biografie není k dispozici.'})
    li_info.setProperty('mediatype', 'actor')
    
    xbmcplugin.addDirectoryItem(handle, build_url_kwargs(action='noop'), li_info, isFolder=False)

    # --- 2) Filmografie (z combined_credits) ---
    combined_credits = info.get('combined_credits', {})
    
    # 🚨 SLOUČENÍ: Zahrnujeme obsazení (cast) i štáb (crew)
    all_credits = combined_credits.get('cast', []) + combined_credits.get('crew', []) 

    # Slovník pro kontrolu unikátnosti (zabránění duplicitám)
    unique_titles = {} 
    
    # TMDb Genre ID pro 'Dokumentární film'
    DOCUMENTARY_GENRE_ID = 99 

    movies, shows = [], []
    for c in all_credits:
        media_type = (c.get('media_type') or '').lower()
        if media_type not in ('movie', 'tv'):
            continue

        title_id = c.get('id')
        title_name = c.get('title') if media_type == 'movie' else c.get('name')
        
        # 1. Kontrola duplicit (použijeme TMDb ID + typ jako klíč)
        unique_key = f"{title_id}_{media_type}"
        
        if unique_key in unique_titles:
            existing_item = unique_titles[unique_key]
            
            # Pokud titul už máme, přidáme novou roli/pozici ('job' pro crew, 'character' pro cast)
            new_role = c.get('character') or c.get('job') or ''
            
            if new_role and new_role not in existing_item['character']:
                # Přidáme novou roli/pozici k existující, oddělenou |
                existing_item['character'] += f" | {new_role}"
            continue 

        # 2. Filtrace dokumentů (pouze u filmů)
        if media_type == 'movie':
            genre_ids = c.get('genre_ids', [])
            if DOCUMENTARY_GENRE_ID in genre_ids:
                continue 

        # 3. Zpracování unikátní položky
        date = c.get('release_date') if media_type == 'movie' else c.get('first_air_date')
        year = _year_from_str(date)
        popularity = float(c.get('popularity') or 0.0)
        
        # Získání primární role: 'character' pro cast, 'job' pro crew
        primary_role = c.get('character') or c.get('job') or ''
        original_title = c.get('original_title') if media_type == 'movie' else c.get('original_name')
        
        item = {
            'title': title_name,
            'year': year,
            'popularity': popularity,
            'character': primary_role, # Primární role/pozice
            'poster_path': c.get('poster_path'),
            'tmdb_mid': title_id,
            'media_type': media_type,
            'plot': c.get('overview', 'Shrnutí není dostupné.'),
            'original_title': original_title, # 🟢 PŘIDÁNO
        }
        
        # Uložíme do slovníku pro kontrolu duplicit
        unique_titles[unique_key] = item
        
        if media_type == 'movie':
            movies.append(item)
        else:
            shows.append(item)

    # --- 3) Řazení: rok DESC, sekundárně popularita DESC ---
    movies.sort(key=lambda x: (x['year'], x['popularity']), reverse=True)
    shows.sort(key=lambda x: (x['year'], x['popularity']), reverse=True)

    # Volitelný limit na sekci
    MAX_PER_SECTION = 60
    movies = movies[:MAX_PER_SECTION]
    shows = shows[:MAX_PER_SECTION]
    
    # --- 4) Vykreslení sekce Filmy ---
    if movies:
        colored_header = f"[COLOR purple][Filmy] • {len(movies)}[/COLOR]"
        header = xbmcgui.ListItem(label=colored_header)
        # header = xbmcgui.ListItem(label=f"[Filmy] • {len(movies)}")
        xbmcplugin.addDirectoryItem(handle, build_url_kwargs(action='noop'), header, isFolder=False)

        for m in movies:
            base_label = f"{m['title']} ({m['year']})" if m['year'] else m['title']
            orig_name = m.get('original_title')
            if orig_name and orig_name != m['title']:
                # Barva nastavena na šedou (grey)
                base_label = f"{base_label} [COLOR grey][{orig_name}][/COLOR]"
            # Zobrazení všech rolí/pozic
            if m['character']:
                base_label += f" — {m['character']}"

            li = xbmcgui.ListItem(label=base_label)
            li.setProperty('mediatype', 'movie') 

            # Artwork
            if m['poster_path']:
                purl = IMAGE_BASE_URL_POSTER + str(m['poster_path'])
                li.setArt({'thumb': purl, 'icon': purl, 'poster': purl})

            # Klik: AKCE 'search' pro Webshare
            query = m['title']
            if m['year']:
                query += f" {m['year']}"
            # url = build_url_kwargs(action='search', q=query.strip())
            title_cs = m['title']
            title_en = orig_name
            url = build_url({'action': 'show_search_dialog', 
                'cz_title': title_cs, 
                'eng_title': title_en, 
                'year': year,
                'tmdb_id': "", # 🚨 Přidat
                'mediatype': 'movie'})  # 🚨 Přidat
            
            xbmcplugin.addDirectoryItem(handle, url, li, isFolder=True)
            
    # --- 5) Vykreslení sekce Seriály ---
    if shows:
        colored_header = f"[COLOR orange][Seriály] • {len(shows)}[/COLOR]"
        header = xbmcgui.ListItem(label=colored_header)
        # header = xbmcgui.ListItem(label=f"[Seriály] • {len(shows)}")
        xbmcplugin.addDirectoryItem(handle, build_url_kwargs(action='noop'), header, isFolder=False)

        for s in shows:
            base_label = f"{s['title']} ({s['year']})" if s['year'] else s['title']
            orig_name = s.get('original_title')
            if orig_name and orig_name != s['title']:
                # Barva nastavena na šedou (grey)
                base_label = f"{base_label} [COLOR grey][{orig_name}][/COLOR]"
            if s['character']:
                base_label += f" — {s['character']}"

            li = xbmcgui.ListItem(label=base_label)
            li.setProperty('mediatype', 'tvshow') 

            # Artwork
            if s['poster_path']:
                purl = IMAGE_BASE_URL_POSTER + str(s['poster_path'])
                li.setArt({'thumb': purl, 'icon': purl, 'poster': purl})

            # Klik: AKCE 'tmdb_seasons' pro detail seriálu
            if s['tmdb_mid']:
                url = build_url_kwargs(action='tmdb_seasons', tmdb_id=s['tmdb_mid'], title=s['title'])
                xbmcplugin.addDirectoryItem(handle, url, li, isFolder=True)
            else:
                xbmcplugin.addDirectoryItem(handle, build_url_kwargs(action='noop'), li, isFolder=False)

    xbmcplugin.endOfDirectory(handle, cacheToDisc=False)


import xbmc
import xbmcgui
import xbmcplugin
import xbmcaddon

from .tmdb_module import TMDb   # uprav import podle tvé struktury
# Předpokládá existující helpers: build_url/get_url, IMAGE_BASE_URL/FANART_BASE_URL apod.





def show_advanced_filter_menu():
    """
    Dialogy pro výběr žánru, (volitelně) roku/rozpětí a minimálního ratingu.
    Po potvrzení uloží 'poslední filtr' (SQL pokud je k dispozici, jinak addon settings)
    a otevře 'tmdb_discover_list' s page=1 přes ActivateWindow (robustní napříč skiny).
    """
    # --- IMPORTY / ENV ---
    import sys
    handle = None
    try:
        handle = int(sys.argv[1])
    except Exception:
        pass

    LOG_PFX = "[TMDb Advanced]"
    try:
        xbmc.log(f"{LOG_PFX} ENTER function", xbmc.LOGINFO)
    except Exception:
        pass

    # --- ŽÁNRY ---
    try:
        genres = [
            ('Akční', 28), ('Dobrodružný', 12), ('Animovaný', 16), ('Komediální', 35),
            ('Krimi', 80), ('Dokumentární', 99), ('Drama', 18), ('Rodinný', 10751),
            ('Fantasy', 14), ('Historický', 36), ('Horor', 27), ('Hudební', 10402),
            ('Mysteriózní', 9648), ('Romantický', 10749), ('Sci‑Fi', 878), ('Thriller', 53),
            ('Válečný', 10752), ('Western', 37)
        ]
        idx = xbmcgui.Dialog().select("Vyber žánr", [g[0] for g in genres])
        if idx < 0:
            xbmc.log(f"{LOG_PFX} cancelled at genre select", xbmc.LOGINFO)
            if handle is not None:
                xbmcplugin.endOfDirectory(handle, succeeded=False)
            return
        selected_genre = genres[idx][1]
        xbmc.log(f"{LOG_PFX} genre_id={selected_genre}", xbmc.LOGINFO)
    except Exception as e:
        xbmc.log(f"{LOG_PFX} GENRE DIALOG FAIL: {e}", xbmc.LOGERROR)
        if handle is not None:
            xbmcplugin.endOfDirectory(handle, succeeded=False)
        return

    # --- ROK / ROZPĚTÍ (volitelné) ---
    year = from_year = to_year = None
    try:
        mode = xbmcgui.Dialog().select("Rok / Rozpětí (volitelné)", ["Bez omezení", "Jeden rok…", "Rozpětí let…"])
        if mode == 1:
            y = xbmcgui.Dialog().numeric(0, "Zadejte rok (YYYY)")
            if y and y.isdigit():
                year = int(y)
        elif mode == 2:
            fy = xbmcgui.Dialog().numeric(0, "Od roku (YYYY)")
            ty = xbmcgui.Dialog().numeric(0, "Do roku (YYYY)")
            if fy and fy.isdigit():
                from_year = int(fy)
            if ty and ty.isdigit():
                to_year = int(ty)
        xbmc.log(f"{LOG_PFX} year={year} from={from_year} to={to_year}", xbmc.LOGINFO)
    except Exception as e:
        xbmc.log(f"{LOG_PFX} YEAR/RANGE DIALOG FAIL: {e}", xbmc.LOGERROR)
        if handle is not None:
            xbmcplugin.endOfDirectory(handle, succeeded=False)
        return

    # --- MINIMÁLNÍ RATING (volitelné) ---
    min_rating = None
    try:
        r = xbmcgui.Dialog().input("Minimální rating TMDb (např. 7.5)")
        if r:
            try:
                min_rating = float(str(r).replace(",", "."))
            except Exception:
                min_rating = None
        xbmc.log(f"{LOG_PFX} min_vote={min_rating}", xbmc.LOGINFO)
    except Exception as e:
        xbmc.log(f"{LOG_PFX} RATING INPUT FAIL: {e}", xbmc.LOGERROR)
        if handle is not None:
            xbmcplugin.endOfDirectory(handle, succeeded=False)
        return

    # --- ULOŽ POSLEDNÍ FILTR (SQL pokud existuje KV, jinak settings) ---
    addon_local = xbmcaddon.Addon('plugin.video.mmirousek_v2')
    last_filter = {
        'genre_id': str(selected_genre),
        'year': str(year or ''),
        'from_year': str(from_year or ''),
        'to_year': str(to_year or ''),
        'min_vote': str(min_rating or ''),
        'sort_by': 'popularity.desc',
        'media_type': 'movie'
    }

    # Pokus o SQL uložení do existující KV tabulky (např. app_state) – bez vytváření
    import json
    saved_sql = False
    try:
        # zjisti, zda existuje tabulka app_state
        row = db.conn.execute(
            "SELECT name FROM sqlite_master WHERE type='table' AND name='app_state'"
        ).fetchone()
        if row and row[0] == 'app_state':
            db.conn.execute(
                "INSERT OR REPLACE INTO app_state(key, value, updated_at) VALUES (?, ?, datetime('now'))",
                ('tmdb_last_filter', json.dumps(last_filter, ensure_ascii=False))
            )
            db.conn.commit()
            saved_sql = True
            xbmc.log("[TMDb State] Saved last filter to SQL app_state", xbmc.LOGINFO)
    except Exception as e:
        xbmc.log(f"[TMDb State] SQL save skipped: {e}", xbmc.LOGDEBUG)

    # Fallback: addon settings (pokud SQL uložení neproběhlo)
    if not saved_sql:
        try:
            addon_local.setSetting('tmdb_last_genre_id', last_filter['genre_id'])
            addon_local.setSetting('tmdb_last_year', last_filter['year'])
            addon_local.setSetting('tmdb_last_from_year', last_filter['from_year'])
            addon_local.setSetting('tmdb_last_to_year', last_filter['to_year'])
            addon_local.setSetting('tmdb_last_min_vote', last_filter['min_vote'])
            addon_local.setSetting('tmdb_last_sort_by', last_filter['sort_by'])
            addon_local.setSetting('tmdb_last_media_type', last_filter['media_type'])
            xbmc.log("[TMDb State] Saved last filter to addon settings", xbmc.LOGINFO)
        except Exception as e:
            xbmc.log(f"[TMDb State] SAVE settings FAIL: {e}", xbmc.LOGERROR)

    # --- SESTAVENÍ URL + NAVIGACE ---
    try:
        url = build_url({'action': 'tmdb_discover_list', **last_filter, 'page': '1'})
        xbmc.log(f"{LOG_PFX} goto URL={url}", xbmc.LOGINFO)

        # ukonči aktuální container, ať Kodi navigaci přijme
        if handle is not None:
            try:
                xbmcplugin.endOfDirectory(handle, succeeded=True)
            except Exception:
                pass

        # robustně otevři výsledky Discoveru
        xbmc.executebuiltin(f'ActivateWindow(Videos,"{url}",return)')

    except NameError as ne:
        xbmc.log(f"{LOG_PFX} NameError (build_url scope): {ne}", xbmc.LOGERROR)
        xbmcgui.Dialog().notification("TMDb", "Interní chyba: chybí build_url v tmdb_kodi.py",
                                      xbmcgui.NOTIFICATION_ERROR, 3000)
        if handle is not None:
            try:
                xbmcplugin.endOfDirectory(handle, succeeded=False)
            except Exception:
                pass
    except Exception as e:
        xbmc.log(f"{LOG_PFX} NAVIGATE FAIL: {e}", xbmc.LOGERROR)
        if handle is not None:
            try:
                xbmcplugin.endOfDirectory(handle, succeeded=False)
            except Exception:
                pass


def show_tmdb_discover_list(params):
    """
    TMDb Discover (žánr, rok/rozpětí, min. rating) + paging.
    Primární název: CZ (pokud je latinkou) jinak EN fallback. Sekundární název: zobrazí se pouze CZ/EN a jen když se liší.
    Nezobrazuje se žádný jiný jazyk (KO/ZH apod.), aby skin neukazoval „čtverečky“.
    """
    # --- ENTER log ---
    xbmc.log("[TMDb Discover] ENTER: params=" + str(params), xbmc.LOGINFO)

    addon = xbmcaddon.Addon('plugin.video.mmirousek_v2')
    handle = int(sys.argv[1])

    # ------- pomocné validace skriptu (latinka) -------
    def is_latinish(s: str) -> bool:
        """
        Vrátí True, pokud je text převážně latinkou (včetně českých diakritik).
        Používá podíly znaků v Latin-1 Supplement a Latin Extended-A/B; filtruje Hangul/CJK/Cyrilici/Arabštinu.
        """
        if not s:
            return False
        total_letters = 0
        latin_letters = 0
        strong_nonlatin = 0
        for ch in s:
            code = ord(ch)
            # písmena: A-Z, a-z, diakritika v Latin-1 (00C0–00FF) + Latin Ext (0100–024F)
            is_latin = (0x0041 <= code <= 0x005A) or (0x0061 <= code <= 0x007A) or \
                       (0x00C0 <= code <= 0x00FF) or (0x0100 <= code <= 0x024F)
            # silné ne-latin skripty (typicky „čtverečky“ ve skinu)
            is_hangul = (0xAC00 <= code <= 0xD7AF) or (0x1100 <= code <= 0x11FF) or (0x3130 <= code <= 0x318F)
            is_cjk    = (0x4E00 <= code <= 0x9FFF) or (0x3400 <= code <= 0x4DBF)
            is_kata   = (0x30A0 <= code <= 0x30FF)
            is_hira   = (0x3040 <= code <= 0x309F)
            is_cyr    = (0x0400 <= code <= 0x04FF)
            is_arab   = (0x0600 <= code <= 0x06FF)
            # započítávej jen „písmena“; interpunkce a mezery nevadí
            if ch.isalpha():
                total_letters += 1
                if is_latin:
                    latin_letters += 1
                if is_hangul or is_cjk or is_kata or is_hira or is_cyr or is_arab:
                    strong_nonlatin += 1
        if total_letters == 0:
            return False
        # Pokud jsou přítomny „silné ne-latin“ bloky, odmítni
        if strong_nonlatin > 0:
            return False
        # Jinak vyžaduj aspoň ~70 % latinky
        return (latin_letters / float(total_letters)) >= 0.70

    def _to_int(s):
        try:
            return int(str(s))
        except Exception:
            return None

    # ------- parametry z URL -------
    genre_id  = params.get('genre_id')
    year      = _to_int(params.get('year')) if params.get('year') else None
    from_year = _to_int(params.get('from_year')) if params.get('from_year') else None
    to_year   = _to_int(params.get('to_year')) if params.get('to_year') else None

    try:
        min_vote = float(params.get('min_vote')) if params.get('min_vote') else None
    except Exception:
        min_vote = None

    try:
        page = int(params.get('page', 1))
    except Exception:
        page = 1

    # ------- API key check -------
    api_key = addon.getSetting('tmdb_api_key')
    if not api_key:
        xbmcgui.Dialog().notification('TMDb chyba', 'Chybí TMDb API klíč v nastavení doplňku.',
                                      xbmcgui.NOTIFICATION_WARNING, 3000)
        xbmcplugin.endOfDirectory(handle, succeeded=False)
        return

    tmdb = TMDb(api_key)

    # ------- Discover volání (tuple check) -------
    sort_by = params.get('sort_by') or 'popularity.desc'
    try:
        result = tmdb.discover_movies_page(
            genre_id=genre_id, year=year, from_year=from_year, to_year=to_year,
            min_vote=min_vote, page=page, language='cs-CZ', sort_by=sort_by
        )
    except Exception as e:
        xbmc.log(f"[TMDb Discover] ERROR: {e}", xbmc.LOGERROR)
        xbmcgui.Dialog().notification('TMDb chyba', f'Chyba Discover: {e}', xbmcgui.NOTIFICATION_ERROR, 3000)
        xbmcplugin.endOfDirectory(handle, succeeded=False)
        return

    if not result or not isinstance(result, tuple) or len(result) != 3:
        xbmc.log("[TMDb Discover] ERROR: discover_movies_page() did not return (items, total_pages, page_size)",
                 xbmc.LOGERROR)
        xbmcgui.Dialog().notification('TMDb', 'Interní chyba Discover. Viz log.', xbmcgui.NOTIFICATION_ERROR, 3000)
        xbmcplugin.endOfDirectory(handle, succeeded=False)
        return

    items, total_pages, page_size = result

    if not items:
        xbmc.log(f"[TMDb Discover] EMPTY: genre={genre_id} year={year} from={from_year} to={to_year} vote>={min_vote} page={page}", xbmc.LOGINFO)
        xbmcgui.Dialog().notification('TMDb', 'Žádné výsledky pro zadaný filtr.', xbmcgui.NOTIFICATION_INFO, 2500)
        xbmcplugin.endOfDirectory(handle)
        return

    xbmcplugin.setContent(handle, 'movies')

    # ------- Render položek -------
    for item in items:
        tmdb_id         = item.get('id')
        label_cz_raw    = item.get('label') or ''                  # lokalizace z cs-CZ Discover
        label_origin    = item.get('original_label') or ''
        origin_lang     = (item.get('original_language') or 'en').lower()
        year_txt        = item.get('year')
        rating_val      = item.get('rating') or 0.0
        try:
            rating_val = float(rating_val)
        except Exception:
            rating_val = 0.0

        # EN fallback (načteme dopředu)
        en_title = None
        try:
            en_title = get_english_title('movie', tmdb_id)
        except Exception as e:
            xbmc.log(f"[Fallback] get_english_title selhalo tmdb_id={tmdb_id}: {e}", xbmc.LOGWARNING)

        # --- výběr primárního a sekundárního názvu ---
        label_cz = label_cz_raw.strip() if label_cz_raw else ''
        label_en = (str(en_title).strip() if en_title else '')

        # Primární: CZ, ale jen pokud je latinkou. Jinak EN.
        if label_cz and is_latinish(label_cz):
            primary_label = label_cz
            # Sekundárně zobraz EN pouze, když existuje a liší se (CZ/EN only)
            secondary_label = label_en if (label_en and label_en != primary_label) else None
        else:
            # CZ není latinkou (nebo prázdné) → použij EN
            if label_cz and not is_latinish(label_cz):
                xbmc.log(f"[TitleFilter] CZ label není latinkou → EN fallback (tmdb_id={tmdb_id}): '{label_cz}'", xbmc.LOGINFO)
            primary_label = label_en or (label_origin if is_latinish(label_origin) else 'Bez názvu')
            # Sekundárně CZ pouze, když je latinkou a liší se; nezobrazuj KO/ZH apod.
            secondary_label = label_cz if (label_cz and is_latinish(label_cz) and label_cz != primary_label) else None

        # Sestavení display labelu (CZ/EN jen)
        display_label = primary_label
        if secondary_label and secondary_label != primary_label:
            display_label += f" [COLOR grey][{secondary_label}][/COLOR]"
        if year_txt:
            display_label += f" ({year_txt})"
        if rating_val > 0.0:
            display_label += f" [COLOR=FFFFFFFF]★[/COLOR][COLOR=FFFFD700]{rating_val:.1f}[/COLOR]"

        li = xbmcgui.ListItem(label=display_label)

        # Artwork
        poster_path   = item.get('poster_path')
        backdrop_path = item.get('backdrop_path')
        if poster_path:
            p = str(poster_path or '').strip()
            if p:
                li.setArt({'icon': IMAGE_BASE_URL + p, 'thumb': IMAGE_BASE_URL + p})
        if backdrop_path:
            b = str(backdrop_path or '').strip()
            if b:
                li.setArt({'fanart': FANART_BASE_URL + b})

        # InfoTagVideo
        info_labels = {
            'title': primary_label,
            'plot': item.get('plot') or '',
            'year': int(year_txt) if isinstance(year_txt, str) and year_txt.isdigit() else None,
            'rating': rating_val,
            'mediatype': 'movie'
        }
        info_labels = {k: v for k, v in info_labels.items() if v not in (None, '')}
        set_video_info(li, info_labels)

        # Zápis do DB (jen CZ/EN; jiné jazyky se do title_* neukládají)
        try:
            title_cs_for_db = primary_label if origin_lang == 'cs' else (label_cz if is_latinish(label_cz) else None)
            title_en_for_db = label_en or (primary_label if origin_lang == 'en' else None)
            db.update(
                tmdb_id,
                {
                    'media_type': 'movie',
                    'title_cs': title_cs_for_db,
                    'title_en': title_en_for_db,
                    'year': year_txt or None,
                    'poster_path': poster_path or None,
                    'backdrop_path': backdrop_path or None
                },
                source='tmdb_discover'
            )
        except Exception as e:
            xbmc.log(f"[TMDb Discover] DB update failed tmdb_id={tmdb_id}: {e}", xbmc.LOGWARNING)

        # Klik → show_search_dialog (předávej jen CZ/EN; jiné jazyky vynecháme)
        url = build_url({
            'action': 'show_search_dialog',
            'cz_title': (title_cs_for_db or ''),
            'eng_title': (title_en_for_db or ''),
            'year': year_txt or '',
            'tmdb_id': str(tmdb_id or ''),
            'mediatype': 'movie'
        })
        xbmcplugin.addDirectoryItem(handle, url, li, isFolder=True)

    # COMMIT strany
    try:
        db.conn.commit()
    except Exception as e:
        xbmc.log(f"[TMDb Discover] Database COMMIT error: {e}", xbmc.LOGERROR)

    # Paging
    display_total = (str(total_pages) if total_pages else '?')
    has_more = False
    try:
        has_more = int(page) < int(total_pages or 1)
    except Exception:
        try:
            has_more = len(items) >= int(page_size or 20)
        except Exception:
            has_more = len(items) >= 20

    if has_more:
        next_url = build_url({
            'action': 'tmdb_discover_list',
            'genre_id': str(genre_id or ''),
            'year': str(year or ''),
            'from_year': str(from_year or ''),
            'to_year': str(to_year or ''),
            'min_vote': str(min_vote or ''),
            'page': str(page + 1)
        })
        li_next = xbmcgui.ListItem(label=f"Další strana » ({page+1}/{display_total})")
        li_next.setArt({'icon': 'DefaultFolder.png'})
        xbmcplugin.addDirectoryItem(handle, next_url, li_next, isFolder=True)

    # Ukončení containeru (updateListing=True, fallback pro starší Kodi)
    xbmc.log("[TMDb Discover] END: calling endOfDirectory(updateListing=True)", xbmc.LOGINFO)
    try:
        xbmcplugin.endOfDirectory(handle, succeeded=True, cacheToDisc=True, updateListing=True)
    except TypeError:
        xbmcplugin.endOfDirectory(handle)

def tmdb_discover_resume(params=None):
    """
    Načte poslední filtr (SQL app_state -> settings fallback),
    a místo navigace přímo zavolá show_tmdb_discover_list(payload),
    aby se listing vykreslil v aktuálním kontejneru (správné chování Back).
    """
    import sys, json
    handle = int(sys.argv[1])

    # 1) Pokus o SQL load (pokud máš KV tabulku 'app_state')
    data = None
    try:
        row = db.conn.execute("SELECT value FROM app_state WHERE key = ?", ('tmdb_last_filter',)).fetchone()
        if row and row[0]:
            data = json.loads(row[0])
            xbmc.log("[TMDb Resume] Loaded last filter from SQL app_state", xbmc.LOGINFO)
    except Exception as e:
        xbmc.log(f"[TMDb Resume] SQL load skipped: {e}", xbmc.LOGDEBUG)

    # 2) Fallback do addon settings
    if not data:
        addon_local = xbmcaddon.Addon('plugin.video.mmirousek_v2')
        data = {
            'genre_id': addon_local.getSetting('tmdb_last_genre_id') or '',
            'year': addon_local.getSetting('tmdb_last_year') or '',
            'from_year': addon_local.getSetting('tmdb_last_from_year') or '',
            'to_year': addon_local.getSetting('tmdb_last_to_year') or '',
            'min_vote': addon_local.getSetting('tmdb_last_min_vote') or '',
            'sort_by': addon_local.getSetting('tmdb_last_sort_by') or 'popularity.desc',
            'media_type': addon_local.getSetting('tmdb_last_media_type') or 'movie'
        }
        xbmc.log("[TMDb Resume] Loaded last filter from addon settings", xbmc.LOGINFO)

    # 3) Pokud je vše prázdné, nic neotvírat
    if not any(str(v).strip() for v in data.values()):
        xbmcgui.Dialog().notification('TMDb', 'Není uložen žádný poslední filtr.', xbmcgui.NOTIFICATION_INFO, 2500)
        xbmcplugin.endOfDirectory(handle, succeeded=False)
        return

    # 4) Normalize payload pro show_tmdb_discover_list()
    payload = {
        'genre_id': str(data.get('genre_id') or ''),
        'year': str(data.get('year') or ''),
        'from_year': str(data.get('from_year') or ''),
        'to_year': str(data.get('to_year') or ''),
        'min_vote': str(data.get('min_vote') or ''),
        'page': '1'
    }

    xbmc.log(f"[TMDb Resume] DIRECT CALL show_tmdb_discover_list with payload={payload}", xbmc.LOGINFO)

    # 5) ŽÁDNÁ NAVIGACE. PŘÍMÉ VYKRESLENÍ:
    try:
        show_tmdb_discover_list(payload)
    except Exception as e:
        xbmc.log(f"[TMDb Resume] DIRECT CALL FAIL: {e}", xbmc.LOGERROR)
        xbmcgui.Dialog().notification('TMDb', 'Nepodařilo se vykreslit poslední filtr.', xbmcgui.NOTIFICATION_ERROR, 3000)
        xbmcplugin.endOfDirectory(handle, succeeded=False)
        return

    # --- extra padding (proti usekávání přenosu) ---
    # (konec tmdb_discover_resume)
    # (EOF-block-padding-1)
    # (EOF-block-padding-2)
