
# -*- coding: utf-8 -*-
import json
import requests
import xbmc
import xbmcaddon
import hashlib
from requests.exceptions import RequestException
from collections import OrderedDict

OMDB_BASE_URL = 'https://www.omdbapi.com/'

# ---------- 1. Cache v paměti (LRU) ----------
_cache = OrderedDict()
_MAX_CACHE = 256

def _cache_put(k, v):
    """Vloží položku do LRU cache v paměti."""
    _cache[k] = v or {}
    _cache.move_to_end(k)
    if len(_cache) > _MAX_CACHE:
        _cache.popitem(last=False)


# ---------- 3. Načtení API klíče ----------
def _read_api_key():
    """Načte OMDb API klíč z nastavení addonu (ponecháno původní Addon ID)."""
    try:
        addon = xbmcaddon.Addon('plugin.video.mmirousek_v2')
        return addon.getSetting('omdb_api_key') or ''
    except Exception as e:
        xbmc.log(f"[OMDb] Nelze načíst nastavení: {e}", xbmc.LOGERROR)
        return ''


# ---------- 4. Hlavní funkce – pouze paměťový cache ----------
def get_omdb_info(title=None, year=None, imdb_id=None, media_type=None, api_key=None, timeout=4):
    """
    Vrátí OMDb JSON dict. Při chybě vrací {}.
    Preferuje lookup přes imdb_id, jinak title(+year). 'media_type' může být 'movie' nebo 'series'.
    """
    key = (api_key or _read_api_key()).strip()
    if not key:
        xbmc.log("[OMDb] API klíč není nastaven – přeskočeno.", xbmc.LOGINFO)
        return {}

    params = {'apikey': key, 'r': 'json', 'plot': 'short'}
    if imdb_id:
        params['i'] = str(imdb_id).strip()
    elif title:
        params['t'] = str(title).strip()
        if year:
            params['y'] = str(year).strip()
    if media_type in ('movie', 'series'):
        params['type'] = media_type

    cache_key = hashlib.md5(json.dumps(params, sort_keys=True).encode('utf-8')).hexdigest()

    # 1) Cache v paměti
    cached = _cache.get(cache_key)
    if cached is not None:
        return cached

    # 2) Volání API
    try:
        r = requests.get(OMDB_BASE_URL, params=params, timeout=timeout)
        r.raise_for_status()
        data = r.json() or {}

        if str(data.get('Response', 'True')).lower() == 'true':
            _cache_put(cache_key, data)
            return data
        else:
            xbmc.log(f"[OMDb] Nenalezeno: {data.get('Error')}", xbmc.LOGINFO)
            return {}

    except RequestException as e:
        xbmc.log(f"[OMDb] Chyba požadavku: {e}", xbmc.LOGERROR)
        return {}


# ---------- 5. Extrakce metrik ----------
def extract_metrics(data):
    """Extrahuje metriky pro label2. Při prázdných datech vrací prázdný dict."""
    if not data:
        return {}

    rt = None
    for r in (data.get('Ratings') or []):
        if r.get('Source') == 'Rotten Tomatoes':
            rt = r.get('Value')
            break

    return {
        'Metascore': data.get('Metascore'),
        'imdbRating': data.get('imdbRating'),
        'imdbVotes': data.get('imdbVotes'),
        'BoxOffice': data.get('BoxOffice'),
        'RottenTomatoes': rt
    }


# ---------- 6. Sestavení label2 (bezpečné, vždy str) ----------
def build_label2(metrics):
    """
    Vytvoří kompaktní řetězec pro label2.
    - Vždy vrací STRING (prázdný, když nejsou data).
    - Bezpečně formátuje čísla (žádné 'str + int').
    """
    if not metrics:
        return ""

    # --- Pomocné převody ---
    def _to_float(val):
        try:
            if val in (None, "", "N/A"):
                return None
            return float(str(val).replace(",", "."))
        except Exception:
            return None

    def _to_int(val):
        try:
            if val in (None, "", "N/A"):
                return None
            s = str(val).replace(",", "").replace(" ", "")
            return int(s)
        except Exception:
            return None

    parts = []

    # IMDb rating + votes
    r = _to_float(metrics.get('imdbRating'))
    v = _to_int(metrics.get('imdbVotes'))
    if r is not None:
        if v is not None:
            votes_str = f"{v:,}"  # např. '123,456' (pro CZ styl lze nahradit čárku za \u202F)
            parts.append(f"IMDb {r:.1f}/10 • {votes_str} hlasů")
        else:
            parts.append(f"IMDb {r:.1f}/10")

    # Metascore
    m = metrics.get('Metascore')
    if m not in (None, "", "N/A"):
        parts.append(f"Metascore {m}")

    # Rotten Tomatoes
    rt = metrics.get('RottenTomatoes')
    if rt not in (None, "", "N/A"):
        rt_str = str(rt).strip()
        if rt_str.isdigit():
            rt_str = f"{rt_str}%"
        parts.append(f"RT {rt_str}")

    # BoxOffice
    bo = metrics.get('BoxOffice')
    if bo not in (None, "", "N/A"):
        parts.append(f"BoxOffice {bo}")

    return " · ".join(parts) if parts else ""





# # -*- coding: utf-8 -*-
# import json
# import requests
# import xbmc
# import xbmcaddon
# import hashlib
# from time import time
# from requests.exceptions import RequestException
# from collections import OrderedDict

# OMDB_BASE_URL = 'https://www.omdbapi.com/'

# # ---------- 1. Cache v paměti (LRU) ----------
# _cache = OrderedDict()
# _MAX_CACHE = 256

# def _cache_put(k, v):
#     """Vloží položku do LRU cache v paměti."""
#     _cache[k] = v
#     _cache.move_to_end(k)
#     if len(_cache) > _MAX_CACHE:
#         _cache.popitem(last=False)


# # ---------- 3. Načtení API klíče ----------
# def _read_api_key():
#     try:
#         addon = xbmcaddon.Addon('plugin.video.mmirousek_v2')
#         return addon.getSetting('omdb_api_key') or ''
#     except Exception as e:
#         xbmc.log(f"[OMDb] Nelze načíst nastavení: {e}", xbmc.LOGERROR)
#         return ''

# # ---------- 4. Hlavní funkce – pouze paměťový cache ----------
# def get_omdb_info(title=None, year=None, imdb_id=None, media_type=None, api_key=None, timeout=4):
#     key = (api_key or _read_api_key()).strip()
#     if not key:
#         xbmc.log("[OMDb] API klíč není nastaven – přeskočeno.", xbmc.LOGINFO)
#         return None

#     params = {'apikey': key, 'r': 'json', 'plot': 'short'}
#     if imdb_id:
#         params['i'] = imdb_id
#     elif title:
#         params['t'] = title
#         if year:
#             params['y'] = str(year)
#     if media_type in ('movie', 'series'):
#         params['type'] = media_type

#     cache_key = hashlib.md5(json.dumps(params, sort_keys=True).encode('utf-8')).hexdigest()

#     # 1. Cache v paměti
#     if cache_key in _cache:
#         return _cache[cache_key]

#     # 2. Volání API
#     try:
#         r = requests.get(OMDB_BASE_URL, params=params, timeout=timeout)
#         r.raise_for_status()
#         data = r.json()

#         if data.get('Response') == 'True':
#             # Ukládáme POUZE do paměti – disk už nepotřebujeme
#             _cache_put(cache_key, data)
#             return data
#         else:
#             xbmc.log(f"[OMDb] Nenalezeno: {data.get('Error')}", xbmc.LOGINFO)
#             return None

#     except RequestException as e:
#         xbmc.log(f"[OMDb] Chyba požadavku: {e}", xbmc.LOGERROR)
#         return None


# # ---------- 5. Zbytek beze změny ----------
# def extract_metrics(data):
#     """Extrahuje metriky pro label2."""
#     if not data:
#         return {}

#     rt = None
#     for r in data.get('Ratings', []) or []:
#         if r.get('Source') == 'Rotten Tomatoes':
#             rt = r.get('Value')
#             break

#     return {
#         'Metascore': data.get('Metascore'),
#         'imdbRating': data.get('imdbRating'),
#         'imdbVotes': data.get('imdbVotes'),
#         'BoxOffice': data.get('BoxOffice'),
#         'RottenTomatoes': rt
#     }


# def build_label2(metrics):
#     """Vytvoří kompaktní řetězec pro label2."""
#     if not metrics:
#         return None

#     parts = []
#     if metrics.get('imdbRating') and metrics['imdbRating'] != 'N/A':
#         parts.append(f"IMDb {metrics['imdbRating']}")
#     if metrics.get('imdbVotes') and metrics['imdbVotes'] != 'N/A':
#         parts.append(metrics['imdbVotes'] + " hlasů")
#     if metrics.get('Metascore') and metrics['Metascore'] != 'N/A':
#         parts.append("Metascore " + metrics['Metascore'])
#     if metrics.get('RottenTomatoes'):
#         parts.append("RT " + metrics['RottenTomatoes'])
#     if metrics.get('BoxOffice') and metrics['BoxOffice'] != 'N/A':
#         parts.append("BoxOffice " + metrics['BoxOffice'])

#     return " · ".join(parts) if parts else None


