# -*- coding: utf-8 -*-
# resources/lib/scrobble.py
# Scrobbling do Trakt + lokální historie
# - jednotný log prefix [SCROBBLE]
# - TMDb parsování epizod přes tmdb_utils
# - ukládání stream_url do lokální historie
# - žádné přímé čtení Trakt client_id/token (jen get_trakt_headers())

import json
import requests
import xbmc
import xbmcgui
import xbmcaddon
import xbmcvfs
from datetime import datetime
import time  # používá se v progress_scrobble()
from resources.lib import local_history
from resources.lib.tmdb import tmdb_utils
from resources.lib.trakt.trakt_service import get_trakt_headers  # používáme JEN tohle
from resources.lib.metadata_db import db # 🚨 NOVÝ IMPORT: Předpokládáme tuto cestu k databázi metadat
from typing import Any, Dict, Optional

PREFIX = "[SCROBBLE]"
ADDON_ID = 'plugin.video.mmirousek_v2'
TRAKT_API_SCROBBLE_URL = 'https://api.trakt.tv/scrobble'

# stav (info/debug)
_last_known_progress = 0.0
_progress_updated = False
_ADDON_INSTANCE = None


# ============================== UTIL / COMMON ===============================

def _get_addon_instance():
    """Lazy načtení Addon instance s ošetřením chyb."""
    global _ADDON_INSTANCE
    if _ADDON_INSTANCE is None:
        try:
            _ADDON_INSTANCE = xbmcaddon.Addon(ADDON_ID)
        except Exception as e:
            xbmc.log(f'{PREFIX}[UTIL] CHYBA inicializace ADDON: {e}', xbmc.LOGERROR)
            _ADDON_INSTANCE = False
    return _ADDON_INSTANCE if _ADDON_INSTANCE is not False else None


def _get_log_tag():
    """Sestaví čitelný tag s názvem addonu."""
    ADDON = _get_addon_instance()
    name = ADDON.getAddonInfo("name") if ADDON else "StreamManager"
    return f'{PREFIX}[{name}]'

# resources/lib/scrobble.py

# ... (na konci souboru, nebo nad _clean_settings)

def build_payload_from_id(tmdb_id: str, mediatype: str) -> Optional[Dict[str, Any]]:
    """
    Sestaví Trakt payload z TMDB ID. Data získá z Kodi ListItem (nejrychlejší) 
    nebo lokální DB (fallback).
    """
    try:
        if not tmdb_id:
            return None
        
        tmdb_id = str(tmdb_id)
        
        # 1. Získání informací z Kodi ListItem (nejrychlejší)
        # Zde jsou data, která nastavuje vaše listovací funkce (Trakt/TMDb moduly)
        title = xbmc.getInfoLabel('ListItem.Title')
        year = xbmc.getInfoLabel('ListItem.Year')

        # 2. Fallback: z lokální DB
        if not title or title in ('', 'ListItem.Title', '0'):
            # Použijeme lokální DB pro získání CZ/ENG názvů a roku
            metadata = db.get_metadata_by_tmdb_id(tmdb_id)
            if metadata:
                title = metadata.get('eng_title') or metadata.get('cz_title')
                year = metadata.get('year')

        if not title:
            xbmc.log(f'{PREFIX} Nenalezen název pro ID {tmdb_id}. Payload nelze sestavit.', xbmc.LOGWARNING)
            return None

        # 3. Sestavení payloadu (Filmy)
        if mediatype == 'movie':
            payload = {
                'movie': {
                    'title': title,
                    'year': int(year) if year and str(year).isdigit() else None,
                    'ids': {'tmdb': int(tmdb_id)}
                }
            }
            return payload

        # 4. Sestavení payloadu (Seriály/Epizody) - Pro budoucí implementaci
        elif mediatype == 'episode':
            # Potřebujeme S/E, které musí být uloženo ve vlastnostech ListItem!
            season_num = xbmc.getInfoLabel('ListItem.Season')
            episode_num = xbmc.getInfoLabel('ListItem.Episode')
            show_title = xbmc.getInfoLabel('ListItem.TVShowTitle')

            if not season_num or not episode_num:
                xbmc.log(f'{PREFIX} Chyba: Pro epizodu {tmdb_id} chybí S/E v ListItem.', xbmc.LOGERROR)
                return None
            
            payload = {
                'show': {
                    'title': show_title or title, # Titul seriálu
                    'year': int(year) if year and str(year).isdigit() else None,
                    'ids': {'tmdb': int(tmdb_id)} # Zde by mělo být TMDB ID seriálu, nikoli epizody
                },
                'episode': {
                    'season': int(season_num),
                    'number': int(episode_num),
                    'ids': {'tmdb': int(tmdb_id)} # Zde by mělo být TMDB ID epizody
                }
            }
            # POZN: Pro serialy se musí TMDB ID epizody a serialu získat ze dvou různých ListItem.Property, 
            # ale toto je základ pro refactoring. Vratme se k tomu, až vyřešíme filmy.
            return payload

        return None
    
    except Exception as e:
        xbmc.log(f'{PREFIX} CHYBA při sestavování payloadu z ID: {e}', xbmc.LOGERROR)
        return None


# --- NOVÉ: jednotné uzamknutí payloadu do settings ---
def lock_payload(payload: dict, last_stream_url: Optional[str] = None, init_progress: float = 0.0) -> None:
    """
    Uloží (uzamkne) Trakt payload do settings + volitelně zapíše URL streamu.
    """
    try:
        addon = _get_addon_instance()
        if not addon:
            xbmc.log(f'{PREFIX}[LOCK] Nelze získat Addon instance', xbmc.LOGERROR)
            return
        addon.setSetting('trakt_last_payload', json.dumps(payload))
        addon.setSetting('trakt_payload_locked', 'true')
        addon.setSetting('trakt_last_progress', f'{init_progress:.1f}')
        addon.setSetting('trakt_progress_updated', 'true')
        if last_stream_url:
            addon.setSetting('last_stream_url', last_stream_url)
        xbmc.log(f'{PREFIX}[LOCK] Payload uzamknut, init_progress={init_progress:.1f}', xbmc.LOGINFO)
    except Exception as e:
        xbmc.log(f'{PREFIX}[LOCK] CHYBA: {e}', xbmc.LOGERROR)

# --- NOVÉ: builder payloadu pro EPIZODU z dodaných ID ---
def build_episode_payload(show_tmdb_id: int,
                          episode_tmdb_id: int,
                          season_num: int,
                          episode_num: int,
                          show_title: Optional[str] = None,
                          show_year: Optional[int] = None,
                          episode_title: Optional[str] = None) -> Optional[Dict[str, Any]]:
    """
    Sestaví Trakt payload pro epizodu. Vloží TMDb ID seriálu i epizody na správná místa.
    """
    try:
        if not (show_tmdb_id and episode_tmdb_id and season_num and episode_num):
            return None
        payload = {
            "show": {
                "title": show_title or "",
                "year": int(show_year) if show_year and str(show_year).isdigit() else None,
                "ids": {"tmdb": int(show_tmdb_id)}
            },
            "episode": {
                "season": int(season_num),
                "number": int(episode_num),
                "title": episode_title or "",
                "ids": {"tmdb": int(episode_tmdb_id)}
            }
        }
        return payload
    except Exception as e:
        xbmc.log(f'{PREFIX}[EP PAYLOAD] CHYBA: {e}', xbmc.LOGERROR)
        return None

# --- NOVÉ: builder payloadu pro EPIZODU přímo z SeriesManager JSONu ---
def build_episode_payload_from_db(series_name: str, season_num: int, episode_num: int) -> Optional[Dict[str, Any]]:
    """
    Načte data ze SeriesManager cache a sestaví přesný Trakt payload pro epizodu.
    Použije tmdb.id (show) a tmdb.episode_id (epizoda), názvy, rok (pokud dohledatelný).
    """
    try:
        from resources.lib.series import series_manager
        ADDON = xbmcaddon.Addon(ADDON_ID)
        profile_path = xbmcvfs.translatePath(ADDON.getAddonInfo('profile'))
        sm = series_manager.SeriesManager(ADDON, profile_path)
        data = sm.load_series_data(series_name)
        if not data:
            xbmc.log(f'{PREFIX}[EP PAYLOAD] Data seriálu nenalezena: {series_name}', xbmc.LOGWARNING)
            return None

        show_tmdb_id = (data.get('tmdb') or {}).get('id')
        show_title = (data.get('tmdb') or {}).get('title') or series_name
        show_year = (data.get('tmdb') or {}).get('year')

        season = (data.get('seasons') or {}).get(str(season_num), {})
        ep = season.get(str(episode_num), {}) if isinstance(season, dict) else {}
        tm = ep.get('tmdb', {}) if isinstance(ep, dict) else {}

        episode_tmdb_id = tm.get('episode_id')  # POZOR: v SeriesManager je klíč 'episode_id'
        episode_title = tm.get('title') or ep.get('name') or ep.get('title') or ''

        if not (show_tmdb_id and episode_tmdb_id):
            xbmc.log(f'{PREFIX}[EP PAYLOAD] Chybí TMDb ID (show/episode) | show={show_tmdb_id}, ep={episode_tmdb_id}', xbmc.LOGWARNING)
            return None

        return build_episode_payload(show_tmdb_id, episode_tmdb_id, int(season_num), int(episode_num),
                                     show_title=show_title, show_year=show_year, episode_title=episode_title)
    except Exception as e:
        xbmc.log(f'{PREFIX}[EP PAYLOAD] CHYBA build from DB: {e}', xbmc.LOGERROR)
        return None


def _clean_settings():
    """Vyčistí dočasné položky scrobble stavu v settings."""
    try:
        addon = _get_addon_instance()
        if addon:
            addon.setSetting('trakt_last_payload', '')
            addon.setSetting('trakt_last_progress', '')
            addon.setSetting('trakt_progress_updated', '')
    except Exception as e:
        xbmc.log(f'{PREFIX}[UTIL] Chyba při čištění nastavení: {e}', xbmc.LOGERROR)


# ============================= DATA O PŘEHRÁVANÉM ===========================

def get_item_data() -> Optional[Dict[str, Any]]:
    """
    Načte scrobble payload. Primárně se spoléhá na předem uzamčený payload 
    uložený modulem scrobble_delay.py, který získal spolehlivé ID z ListItem.
    """
    addon = _get_addon_instance()
    locked_payload_json = addon.getSetting('trakt_last_payload')
    is_locked = addon.getSetting('trakt_payload_locked') == 'true'

    if locked_payload_json and is_locked:
        try:
            payload = json.loads(locked_payload_json)
            # Logování pro potvrzení úspěšného načtení
            xbmc.log(f'{PREFIX} Načítám uzamčený payload z nastavení. Typ: {payload.get("movie", {}).get("title") or payload.get("show", {}).get("title")}', xbmc.LOGINFO)
            return payload
        except Exception as e:
            xbmc.log(f'{PREFIX} Chyba parsování uzamčeného payloadu: {e}', xbmc.LOGERROR)

    # Fallback pro případ, že scrobble_delay selhal nebo není spuštěn
    # Nyní zde už nemáme nespolehlivý regex, takže se vrátí None
    xbmc.log(f'{PREFIX} Uzamčený payload nenalezen. Přehrávání NEBUDE scroblováno na Trakt.', xbmc.LOGWARNING)
    return None
# ================================== TIK =====================================


def _update_progress():
    """
    Každou sekundu (volá service.py):
      - Ukládá aktuální progress (last_progress) a flag progress_updated=true.
      - Lock/payload je ošetřen v scrobble_delay.py.
    """
    addon = xbmcaddon.Addon(ADDON_ID)

    # Odstraněna veškerá stará fallback logika pro payload/lock
    if addon.getSetting('trakt_payload_locked') == 'true':
        xbmc.log(f'{PREFIX}[TIK] Payload zamknutý – neřeším payload.', xbmc.LOGDEBUG)
    else:
        xbmc.log(f'{PREFIX}[TIK] Lock není aktivní. Zajišťuje scrobble_delay.', xbmc.LOGDEBUG)

    # Ulož progres
    player = xbmc.Player()
    if player.isPlaying():
        try:
            current = player.getTime()
            total = player.getTotalTime()
            if total > 0:
                progress = (current / total) * 100.0
                addon.setSetting('trakt_last_progress', str(progress))
                addon.setSetting('trakt_progress_updated', 'true')
                global _last_known_progress, _progress_updated
                _last_known_progress = progress
                _progress_updated = True
                xbmc.log(f'{PREFIX}[TIK] Progress uložen: {progress:.1f}%', xbmc.LOGDEBUG)
        except Exception as e:
            xbmc.log(f'{PREFIX}[TIK] Chyba při zjišťování času: {e}', xbmc.LOGWARNING)
    else:
        # Nic nehraje → nic neukládáme
        pass
    
def progress_scrobble():
    """
    Spouští se každých 5s, pokud se přehrává.
    - Ošetřuje lock, aby nedocházelo k chybám.
    - Aktualizuje globální/in-memory progress.
    - Odesílá Trakt /progress každých 60s (nebo po >1 % posunu).
    """
    LOG = _get_log_tag()
    
    addon = _get_addon_instance()
    if addon is None:
        return

    # Kontrola zámku
    if addon.getSetting('trakt_payload_locked') != 'true':
        # Spouští se scrobble až po získání spolehlivého ID (scrobble_delay)
        xbmc.log(f'{LOG} [TIK] Payload není zamknut – přeskočeno.', xbmc.LOGDEBUG)
        return

    # 🚨 ZMĚNA: Používáme UZAMKNUTÝ payload z nastavení!
    item = get_item_data() 
    if not item:
        # Tichý přeskok, scrobble_delay nespustil scrobble start správně
        xbmc.log(f'{LOG} [TIK] Chyba při přípravě payloadu: get_item_data() nenačetlo data - tichý přeskok.', xbmc.LOGWARNING)
        return

    # Zbytek logiky scrobblingu zůstává
    try:
        player = xbmc.Player()
        if not player.isPlaying():
            xbmc.log(f'{LOG} [TIK] Přehrávač již neběží – přeskok.', xbmc.LOGDEBUG)
            return

        current_time = player.getTime()
        total_time = player.getTotalTime()

        if total_time <= 0:
            xbmc.log(f'{LOG} [TIK] Nulová celková délka – přeskok.', xbmc.LOGWARNING)
            return

        progress = (current_time / total_time) * 100.0
        
        # Uložit do settings, aby se scrobble_stop mohl spolehnout
        addon.setSetting('trakt_last_progress', f'{progress:.1f}')
        addon.setSetting('trakt_progress_updated', 'true')
        
        # Aktualizovat in-memory progress
        global _last_known_progress, _progress_updated
        _last_known_progress = progress
        _progress_updated = True
        
        # Logování
        progress_diff = abs(progress - addon.getSettingInt('trakt_last_scrobble_progress'))
        xbmc.log(f'{LOG} [TIK] Progress: {progress:.1f}%. Rozdíl: {progress_diff:.1f}%', xbmc.LOGDEBUG)

        # Kontrola podmínek pro odeslání /progress
        last_scrobble_time = addon.getSettingInt('trakt_last_scrobble_time')
        elapsed_time = time.time() - last_scrobble_time
        
        send_scrobble = False
        
        if elapsed_time > 60: # Pošli každých 60s
            send_scrobble = True
            
        if progress_diff >= 1.0: # Pošli pokud se progress posunul o 1 % (kvůli zaokrouhlování Traktu)
            send_scrobble = True

        if send_scrobble:
            trakt_progress_scrobble(item, progress)
            addon.setSetting('trakt_last_scrobble_progress', f'{progress:.1f}')
            addon.setSetting('trakt_last_scrobble_time', str(int(time.time())))
        
    except Exception as e:
        xbmc.log(f'{LOG} [TIK] Chyba scrobbling smyčky: {e}', xbmc.LOGERROR)

# ================================= START ====================================

def scrobble_start(fallback_title=None):
    """
    Připraví payload (z historie nebo z ListItemu), uloží ho do settings a pošle Trakt /start.
    """
    LOG = _get_log_tag()
    ADDON = xbmcaddon.Addon(ADDON_ID)
    item = None

    # 1) Start z lokální historie (pokud přehráno z menu „Naposledy sledované“)
    history_item_json = ADDON.getSetting('trakt_launch_history_item')
    if history_item_json:
        ADDON.setSetting('trakt_launch_history_item', '')  # vyčistit po použití
        history_item = json.loads(history_item_json)
        trakt_payload = history_item.get('trakt_payload')
        if trakt_payload:
            item = history_item.copy()
            item.update(trakt_payload)
            xbmc.log(f'{LOG} [START] Rekonstruován Trakt payload z lokální historie.', xbmc.LOGINFO)
        else:
            xbmc.log(f'{LOG} [START] Chybí trakt_payload v lokální historii, pokračuji z ListItem.', xbmc.LOGWARNING)

    # 2) Fallback: z ListItemu
    if not item:
        # Použijeme NOVOU verzi get_item_data, která čte uzamčený payload z nastavení!
        item = get_item_data() 
        if not item:
            xbmc.log(f'{LOG} [START] Žádná data – přeskočeno (Payload nebyl uložen v settings).', xbmc.LOGWARNING)
            return
    # 2a) Získat stream URL pro lokální historii
    try:
        player = xbmc.Player()
        if player.isPlaying():
            stream_url = player.getPlayingFile()
            if stream_url:
                item['stream_url'] = stream_url
                xbmc.log(f'{LOG} [START] Získán stream URL pro historii.', xbmc.LOGINFO)
    except Exception as e:
        xbmc.log(f'{LOG} [START] Chyba při získávání stream URL: {e}', xbmc.LOGWARNING)

    # 3) Uložit payload + init progress
    addon_inst = _get_addon_instance()
    if not addon_inst:
        xbmc.log(f'{LOG} [START] Nelze získat instanci addonu – konec.', xbmc.LOGERROR)
        return
    try:
        addon_inst.setSetting('trakt_last_payload', json.dumps(item))
        addon_inst.setSetting('trakt_last_progress', '0.0')
        addon_inst.setSetting('trakt_progress_updated', 'true')
        xbmc.log(f'{LOG} [START] Payload a progress init uložen.', xbmc.LOGINFO)
        xbmc.sleep(300)  # malá prodleva
    except Exception as e:
        xbmc.log(f'{LOG} [START] Chyba uložení payloadu/init: {e}', xbmc.LOGERROR)

    # 4) Odeslat START na Trakt (pokud povoleno a autorizováno)
    if not ADDON.getSettingBool('trakt_enabled'):
        xbmc.log(f'{LOG} [START] Trakt je zakázán – ignoruji.', xbmc.LOGINFO)
        return

    headers = get_trakt_headers()
    if not headers:
        xbmc.log(f'{LOG} [START] Trakt není autorizován (get_trakt_headers() vrátilo None).', xbmc.LOGINFO)
        return

    # Očistit payload pro Trakt (bez lokálních klíčů)
    trakt_payload = item.copy()
    for k in ('stream_url', 'name', 'tmdb_id'):
        trakt_payload.pop(k, None)

    data = {"progress": 0.1, **trakt_payload}

    try:
        r = requests.post(f"{TRAKT_API_SCROBBLE_URL}/start", json=data, headers=headers, timeout=10)
        if r.status_code in (201, 409):
            xbmc.log(f'{LOG} [START] ÚSPĚCH', xbmc.LOGINFO)
        else:
            xbmc.log(f'{LOG} [START] CHYBA HTTP {r.status_code}', xbmc.LOGERROR)
    except Exception as e:
        xbmc.log(f'{LOG} [START] Síťová chyba: {e}', xbmc.LOGERROR)


# ================================= PAUSE ====================================

def scrobble_pause():
    """Odešle Trakt /pause (pokud povoleno a autorizováno)."""
    LOG = _get_log_tag()

    # Načíst uložený payload
    item = None
    try:
        addon = xbmcaddon.Addon(ADDON_ID)
        payload_json = addon.getSetting('trakt_last_payload')
        if payload_json:
            item = json.loads(payload_json)
            xbmc.log(f'{LOG} [PAUSE] Payload načten.', xbmc.LOGINFO)
    except Exception as e:
        xbmc.log(f'{LOG} [PAUSE] Chyba načtení payloadu: {e}', xbmc.LOGWARNING)

    if not item:
        xbmc.log(f'{LOG} [PAUSE] Žádný payload – přeskočeno.', xbmc.LOGWARNING)
        return

    addon = _get_addon_instance()
    if addon is None or not addon.getSettingBool('trakt_enabled'):
        xbmc.log(f'{LOG} [PAUSE] Trakt je zakázán – ignoruji.', xbmc.LOGINFO)
        return

    headers = get_trakt_headers()
    if not headers:
        xbmc.log(f'{LOG} [PAUSE] Trakt není autorizován – ignoruji.', xbmc.LOGINFO)
        return

    trakt_payload = item.copy()
    # Ponechává jen Trakt-specifické klíče (movie/show/episode)
    for k in ('stream_url', 'name', 'tmdb_id', 'trakt_payload', 'media_type', 'watched_at'):
        trakt_payload.pop(k, None)    
    
    data = {"progress": 0.0, **trakt_payload}

    try:
        r = requests.post(f"{TRAKT_API_SCROBBLE_URL}/pause", json=data, headers=headers, timeout=10)
        if r.status_code in (201, 409):
            xbmc.log(f'{LOG} [PAUSE] ÚSPĚCH', xbmc.LOGINFO)
        else:
            xbmc.log(f'{LOG} [PAUSE] CHYBA HTTP {r.status_code}', xbmc.LOGERROR)
    except Exception as e:
        xbmc.log(f'{LOG} [PAUSE] Chyba: {e}', xbmc.LOGERROR)


# ================================= STOP =====================================

def scrobble_stop():
    """
    Při stopu:
    - Načte payload + progress z settings.
    - progress < 90 % → uloží jako rozkoukáno (pokud >5 %), VŽDY uloží do lokální historie.
    - progress >= 90 % → pošle Trakt /stop (pokud povoleno) a uloží zhlédnuto do lokální historie.
    - epizody: uloží progress do SeriesManager + pošle Trakt STOP.
    - vyčistí dočasné hodnoty a odemkne lock.
    """
    LOG = _get_log_tag()
    xbmc.log(f'{LOG} [STOP] Spouštím...', xbmc.LOGINFO)

    addon = _get_addon_instance()
    if addon is None:
        xbmc.log(f'{LOG} [STOP] Nelze získat instanci Addonu – konec.', xbmc.LOGERROR)
        return

    # --- Payload z settings ---
    item = None
    fresh_addon = xbmcaddon.Addon(ADDON_ID)
    try:
        payload_json = fresh_addon.getSetting('trakt_last_payload')
        xbmc.log(f'{PREFIX}[STOP] Načteno z disku: {payload_json}', xbmc.LOGDEBUG)
        if payload_json and payload_json.strip():
            item = json.loads(payload_json)
            xbmc.log(f'{PREFIX}[STOP] Payload načten.', xbmc.LOGINFO)
        else:
            xbmc.log(f'{PREFIX}[STOP] Payload prázdný – přeskočeno.', xbmc.LOGWARNING)
    except Exception as e:
        xbmc.log(f'{PREFIX}[STOP] Chyba načítání payloadu: {e}', xbmc.LOGERROR)

    # 🚨 ZMĚNA: ODSTRANĚN NESPELJIVÝ FALLBACK
    if not item:
        xbmc.log(f'{PREFIX}[STOP] Žádná data (Trakt payload nebyl nalezen). Čistím settings a končím.', xbmc.LOGERROR)
        # Nyní MUSÍME vyčistit settings
        _clean_settings()
        try:
            # Explicitně odemkneme, aby se systém nerozbil
            fresh_addon.setSetting('trakt_payload_locked', 'false')
            fresh_addon.setSetting('trakt_last_payload', '')
            fresh_addon.setSetting('trakt_last_progress', '0.0')
            fresh_addon.setSetting('trakt_progress_updated', 'false')
        except Exception:
            pass
        return

    # --- Progress --- (Logika zůstává stejná, jen používá fresh_addon)
    progress = 0.0
    updated = False
    try:
        xbmc.sleep(200)  # malá prodleva, aby se stihl uložit poslední tik
        prog_str = fresh_addon.getSetting('trakt_last_progress')
        upd_str = fresh_addon.getSetting('trakt_progress_updated')
        if prog_str and upd_str == 'true':
            progress = float(prog_str)
            updated = True
            xbmc.log(f'{PREFIX}[STOP] Progress načten (settings): {progress:.1f}%', xbmc.LOGINFO)
    except Exception as e:
        xbmc.log(f'{PREFIX}[STOP] Chyba čtení progressu: {e}', xbmc.LOGWARNING)

    global _last_known_progress, _progress_updated
    if not updated and _progress_updated:
        progress = _last_known_progress
        updated = True
        xbmc.log(f'{PREFIX}[STOP] Progress převzat z in-memory: {progress:.1f}%', xbmc.LOGINFO)

    if not updated:
        try:
            p = xbmc.Player()
            if p.isPlaying():
                current = p.getTime()
                total = p.getTotalTime()
                if total > 0:
                    progress = (current / total) * 100.0
                    updated = True
                    xbmc.log(f'{PREFIX}[STOP] Progress dopočítán z Playeru: {progress:.1f}%', xbmc.LOGINFO)
        except Exception as e:
            xbmc.log(f'{PREFIX}[STOP] Chyba fallbacku z Playeru: {e}', xbmc.LOGWARNING)

    if not updated:
        xbmc.log(f'{PREFIX}[STOP] Progress nebyl aktualizován – STOP přeskočen.', xbmc.LOGWARNING)
        _clean_settings()
        fresh_addon.setSetting('trakt_payload_locked', 'false') # Odblokovat
        return

    # --- Připrav payload pro lokální DB / Trakt ---
    
    # NOVÝ ZPŮSOB: item je Trakt payload, neparsujeme!
    item_for_local_db = item.copy()
    item_for_local_db['progress'] = progress

    # Sestavit Trakt payload (čištění lokálních klíčů, které Trakt nechce)
    trakt_payload = item.copy()
    for k in ('stream_url', 'name', 'tmdb_id', 'trakt_payload', 'media_type', 'watched_at'):
        trakt_payload.pop(k, None) 
    
    # Doplníme čistý trakt_payload do položky pro lokální DB
    item_for_local_db['trakt_payload'] = trakt_payload
    
    # Získání typu média (movie/episode)
    mt = (item_for_local_db.get('media_type') 
          or ('episode' if ('show' in item_for_local_db and 'episode' in item_for_local_db) else None)
          or ('movie' if 'movie' in item_for_local_db else 'movie'))

    # 🚨 ZMĚNA: ZAJIŠTĚNÍ TMDB ID pro lokální DB z payloadu
    tmdb_id_for_db = None
    if mt == 'movie':
        tmdb_id_for_db = trakt_payload.get('movie', {}).get('ids', {}).get('tmdb')
    elif mt == 'episode':
        tmdb_id_for_db = trakt_payload.get('episode', {}).get('ids', {}).get('tmdb')
        
    if tmdb_id_for_db:
        # Použijeme string, protože DB to obvykle ukládá jako string
        item_for_local_db['tmdb_id'] = str(tmdb_id_for_db)
    else:
        # Fallback na '0' pro lokální DB, pokud ID nemáme (nemělo by se stát)
        item_for_local_db['tmdb_id'] = '0'
        xbmc.log(f'{PREFIX}[STOP] CHYBA: TMDB ID pro lokální DB nenalezeno v payloadu. Ukládám s ID=0.', xbmc.LOGERROR)
    
    # 🚨 ZMĚNA: ODSTRANĚN CELÝ PŮVODNÍ BLOK ID EXTRACT!

    # ✅ EPIZODY: SeriesManager + Trakt STOP
    if mt == 'episode':
        try:
            # Nový payload nemá 'name', musíme použít show title
            series_title = item.get('show', {}).get('title') or 'Neznámý seriál'
            season = item.get('episode', {}).get('season')
            episode = item.get('episode', {}).get('number')
            
            if season and episode:
                from datetime import datetime as dt
                current_time_iso = dt.utcnow().strftime('%Y-%m-%dT%H:%M:%S')
                xbmc.log(f"[SCROBBLE][DEBUG] Připraven k předání Watched_At: {current_time_iso}", xbmc.LOGINFO)
                from resources.lib.series import series_manager
                import xbmcvfs # pro translatePath
                profile_path = xbmcvfs.translatePath(xbmcaddon.Addon(ADDON_ID).getAddonInfo('profile'))
                sm = series_manager.SeriesManager(xbmcaddon.Addon(ADDON_ID), profile_path)
                sm.mark_episode_progress(series_title, int(season), int(episode), progress, set_by='auto', watched_at=current_time_iso)
                xbmc.log(f"[SCROBBLE][STOP] Progress uložen do SeriesManager: {series_title} S{season}E{episode} -> {progress}%", xbmc.LOGINFO)

                # Trakt STOP pro epizodu (pokud povoleno)
                if addon.getSettingBool('trakt_enabled'):
                    headers = get_trakt_headers()
                    if headers:
                        ep_payload = trakt_payload.copy() # Použijeme již očištěný
                        data = {"progress": min(progress, 99.9), **ep_payload}
                        try:
                            resp = requests.post(f"{TRAKT_API_SCROBBLE_URL}/stop", json=data, headers=headers, timeout=10)
                            if resp.status_code in (201, 409):
                                xbmc.log("[SCROBBLE][STOP] Trakt STOP úspěšně odeslán pro epizodu.", xbmc.LOGINFO)
                            else:
                                xbmc.log(f"[SCROBBLE][STOP] Trakt STOP chyba HTTP {resp.status_code}", xbmc.LOGERROR)
                        except Exception as e:
                            xbmc.log(f"[SCROBBLE][STOP] Síťová chyba při odeslání Trakt STOP: {e}", xbmc.LOGERROR)
                    else:
                        xbmc.log("[SCROBBLE][STOP] Trakt není autorizován – STOP přeskočen.", xbmc.LOGWARNING)
            else:
                xbmc.log("[SCROBBLE][STOP] Chybí S/E pro epizodu – progress nezapsán.", xbmc.LOGWARNING)
        except Exception as e:
            xbmc.log(f"[SCROBBLE][STOP] CHYBA zápisu progressu do SeriesManager: {e}", xbmc.LOGERROR)

    # FILMY: Ukládáme jen rozkoukáno (5–89 %)
    if mt == 'movie':
        
        try:
            addon = xbmcaddon.Addon(ADDON_ID)
            if addon.getSettingBool('trakt_enabled'):
                headers = get_trakt_headers()
            else:
                headers = None

            movie_payload = trakt_payload.copy()  # {'movie': {...}}
            from datetime import datetime as dt
            watched_at_iso = dt.utcnow().strftime('%Y-%m-%dT%H:%M:%S')

            if headers:
                data = {"progress": min(progress, 99.9), **movie_payload}
                # Volitelné: sjednotit čas konce na Traktu — odkomentuj podle potřeby:
                # data["watched_at"] = watched_at_iso

                try:
                    resp = requests.post(f"{TRAKT_API_SCROBBLE_URL}/stop",
                                        json=data, headers=headers, timeout=10)
                    if resp.status_code in (201, 409):
                        xbmc.log("[SCROBBLE][STOP] Trakt STOP úspěšně odeslán pro film.", xbmc.LOGINFO)
                    else:
                        xbmc.log(f"[SCROBBLE][STOP] Trakt STOP chyba HTTP {resp.status_code}", xbmc.LOGERROR)
                except Exception as e:
                    xbmc.log(f"[SCROBBLE][STOP] Síťová chyba při odeslání Trakt STOP (film): {e}", xbmc.LOGERROR)
            else:
                xbmc.log("[SCROBBLE][STOP] Trakt není autorizován/aktivní – STOP (film) přeskočen.", xbmc.LOGWARNING)
        except Exception as e:
            xbmc.log(f"[SCROBBLE][STOP] Chyba při přípravě Trakt STOP (film): {e}", xbmc.LOGERROR)

        # --- 👇 Tady hned pod STOPEM nech svůj původní blok lokální historie: ---
       
        if 5.0 < progress < 90.0:
            try:
                from resources.lib import local_history
                from datetime import datetime
                
                # Načteme původní payload (je v proměnné 'item')
                original_payload = item  # ← to je ten payload z play()
                
                # Vytáhneme čistý název
                clean_title = (original_payload.get('movie', {}).get('title') or 
                             original_payload.get('show', {}).get('title') or 
                             'Neznámý film')
                last_url = ""
                try:
                    addon = xbmcaddon.Addon('plugin.video.mmirousek_v2')
                    last_url = addon.getSetting('last_stream_url') or ""
                except:
                    pass
                
                # Aktualizovaná položka pro rozkoukáno
                rozkoukano_item = {
                    "name": clean_title,
                    "media_type": "movie",
                    "tmdb_id": str(tmdb_id_for_db or 0),
                    "progress": round(progress, 2),
                    "watched_at": datetime.utcnow().strftime('%Y-%m-%dT%H:%M:%S'),
                    "stream_url": last_url, 
                    "trakt_payload": trakt_payload,  # čistý pro Trakt
                    "movie": original_payload.get('movie')  # pro kompatibilitu s UI
                }
                
                local_history.store_item(rozkoukano_item)
                xbmc.log(f'{PREFIX}[STOP] Rozkoukáno aktualizováno: {clean_title} – {progress:.1f}%', xbmc.LOGINFO)
            except Exception as e:
                xbmc.log(f'{PREFIX}[STOP] Chyba při ukládání rozkoukána: {e}', xbmc.LOGERROR)
        else:
            # DOKONČENÝ FILM (≥90 %) – ukládáme JEN lokálně (Trakt STOP už proběhl v play())
            try:
                from resources.lib import local_history
                from datetime import datetime
                
                original_payload = item
                clean_title = (original_payload.get('movie', {}).get('title') or 'Neznámý film')
                
                last_url = ""
                try:
                    addon = xbmcaddon.Addon('plugin.video.mmirousek_v2')
                    last_url = addon.getSetting('last_stream_url') or ""
                except:
                    pass

                dokonceny_item = {
                    "name": clean_title,
                    "media_type": "movie",
                    "tmdb_id": str(tmdb_id_for_db or 0),
                    "progress": 100.0,
                    "watched_at": datetime.utcnow().strftime('%Y-%m-%dT%H:%M:%S'),
                    "stream_url": last_url,
                    "trakt_payload": trakt_payload,
                    "movie": original_payload.get('movie')
                }
                
                local_history.store_item(dokonceny_item)
                xbmc.log(f'{PREFIX}[STOP] Dokončený film uložen do historie (jen lokálně): {clean_title}', xbmc.LOGINFO)
            except Exception as e:
                xbmc.log(f'{PREFIX}[STOP] Chyba při ukládání dokončeného filmu: {e}', xbmc.LOGERROR)

    # --- Vyčistit nastavení + odemknout ---
    _clean_settings()
    try:
        fresh_addon.setSetting('trakt_payload_locked', 'false')
        fresh_addon.setSetting('trakt_last_payload', '')
        fresh_addon.setSetting('trakt_last_progress', '0.0')
        fresh_addon.setSetting('trakt_progress_updated', 'false')
        xbmc.log(f'{PREFIX}[STOP] Zámek a dočasná data vyčištěna.', xbmc.LOGINFO)
    except Exception as e:
        xbmc.log(f'{PREFIX}[STOP] Chyba při čištění dočasných dat: {e}', xbmc.LOGWARNING)