# -*- coding: utf-8 -*-
# resources/lib/metadata_db.py
# Centrální SQLite DB pro plugin.video.mmirousek_v2
# Autor: mmirousek + Grok 2025

import os
import sqlite3
import json
import time
import xbmc
import xbmcvfs
import xbmcaddon
import xbmcgui
from datetime import datetime, timedelta
from typing import List, Optional, Dict, Any

ADDON = xbmcaddon.Addon('plugin.video.mmirousek_v2')
PROFILE = xbmcvfs.translatePath(ADDON.getAddonInfo('profile'))
DB_PATH = os.path.join(PROFILE, 'metadata.db')

# Zajistíme existenci profilu
if not xbmcvfs.exists(PROFILE):
    xbmcvfs.mkdirs(PROFILE)

class MetadataDB:
    def __init__(self):
        self.conn = None
        self._connect()
        self._create_tables()
        self._create_indexes()

    def _connect(self):
        try:
            self.conn = sqlite3.connect(DB_PATH, timeout=30.0, check_same_thread=False)
            self.conn.execute("PRAGMA foreign_keys = ON")
            self.conn.execute("PRAGMA journal_mode = WAL")
            self.conn.execute("PRAGMA synchronous = NORMAL")
        except Exception as e:
            xbmc.log(f"[MetadataDB] Connect error: {e}", xbmc.LOGERROR)

    def _create_tables(self):
            # ----------------------------------------------------------------------
            # Krok 1: Hlavní tabulka 'media' - Zajištění plné sady sloupců pro nové instalace
            # ----------------------------------------------------------------------
            sql = """
            CREATE TABLE IF NOT EXISTS media (
                tmdb_id             INTEGER PRIMARY KEY,
                media_type          TEXT NOT NULL CHECK(media_type IN ('movie','tv')),
                
                -- TMDb
                title_cs            TEXT,
                title_en            TEXT,
                title_original      TEXT,
                overview_cs         TEXT,
                overview_en         TEXT,
                year                INTEGER,
                poster_path         TEXT,
                backdrop_path       TEXT,
                vote_average        REAL,
                vote_count          INTEGER,
                popularity          REAL,
                genres              TEXT,        -- JSON
                runtime             INTEGER,
                
                -- OMDb
                imdb_id             TEXT UNIQUE,
                imdb_rating         REAL,
                imdb_votes          INTEGER,
                metascore           INTEGER,
                rt_rating           TEXT,
                box_office          TEXT,
                
                -- Trakt
                trakt_id            INTEGER UNIQUE,
                trakt_slug          TEXT,
                trakt_watched       INTEGER DEFAULT 0,
                trakt_last_watched_at TEXT,  -- NOVÝ SLOUPEC: Datum posledního zhlédnutí
                trakt_collected     INTEGER DEFAULT 0,
                trakt_watchlist     INTEGER DEFAULT 0,
                trakt_rating        INTEGER,
                trakt_episodes_aired     INTEGER,
                trakt_episodes_completed INTEGER,
                trakt_full_data     TEXT,
                trakt_is_fully_watched INTEGER DEFAULT 0,
                trakt_next_episode_id TEXT DEFAULT NULL,
                trakt_next_episode_title     TEXT,
                trakt_total_episodes INTEGER DEFAULT 0,
                -- Lokální
                webshare_ident      TEXT,
                webshare_size       INTEGER,
                
                -- Metadata
                last_updated        TEXT DEFAULT (datetime('now')),
                last_accessed       TEXT DEFAULT (datetime('now')),
                source_flags        INTEGER DEFAULT 0,    -- bitová maska: 1=TMDb, 2=OMDb, 4=Trakt,
                ws_watchlist_ts REAL DEFAULT NULL
            );
            """
            try:
                # Vytvoří tabulku, pokud neexistuje (pro nové uživatele)
                self.conn.execute(sql)
                self.conn.commit()
            except Exception as e:
                xbmc.log(f"[MetadataDB] Create table error: {e}", xbmc.LOGERROR)
                # Pokud se to nepodaří, nemá smysl pokračovat

            # ----------------------------------------------------------------------
            # Krok 2: Inicializace a migrace
            # TOTO JE KLÍČOVÁ SEKCE PRO SNADNÉ PŘIDÁVÁNÍ SLOUPCŮ BEZ ZTRÁTY DAT
            # ----------------------------------------------------------------------
            
            # Vytvoříme pomocnou tabulku pro sledování verze databáze
            self.conn.execute("""
                CREATE TABLE IF NOT EXISTS db_version (
                    id INTEGER PRIMARY KEY,
                    version INTEGER NOT NULL
                )
            """)
            
            # Zjistíme aktuální verzi (výchozí verze je 0 pro databáze, kde db_version ještě neexistuje)
            try:
                cursor = self.conn.execute("SELECT version FROM db_version WHERE id = 1")
                db_version = cursor.fetchone()[0]
            except Exception:
                db_version = 0

            # --- Migrace Verze 1 (Přidání trakt_last_watched_at) ---
            if db_version < 1:
                xbmc.log("[MetadataDB Migrace] Spouštím migraci na verzi 1: trakt_last_watched_at.", xbmc.LOGINFO)
                try:
                    # 1. Přidání trakt_last_watched_at (Text pro ISO datum)
                    # Používáme tabulku 'media' (dle tvého schématu)
                    self.conn.execute("ALTER TABLE media ADD COLUMN trakt_last_watched_at TEXT")
                    xbmc.log("[MetadataDB Migrace] Sloupec trakt_last_watched_at přidán.", xbmc.LOGINFO)
                except sqlite3.OperationalError as e:
                    # Ošetření: Pokud už sloupec existuje
                    if 'duplicate column name' not in str(e):
                        raise # Vyhodit neočekávanou chybu
                    xbmc.log("[MetadataDB Migrace] Sloupec trakt_last_watched_at již existuje (OK).", xbmc.LOGINFO)
                
                # Nastavíme novou verzi
                db_version = 1
                self.conn.execute("INSERT OR REPLACE INTO db_version (id, version) VALUES (1, 1)")
                xbmc.log("[MetadataDB Migrace] Migrace na verzi 1 dokončena.", xbmc.LOGINFO)

            # --- Migrace Verze 2 (Šablona pro budoucí rozšíření) ---
            if db_version < 2:
                xbmc.log("[MetadataDB Migrace] Spouštím migraci na verzi 2: NOVÝ SLOUPEC.", xbmc.LOGINFO)
                try:
                    self.conn.execute("ALTER TABLE media ADD COLUMN trakt_full_data TEXT")
                    xbmc.log("[MetadataDB Migrace] Sloupec trakt_full_data přidán.", xbmc.LOGINFO)
                except sqlite3.OperationalError as e:
                    if 'duplicate column name' not in str(e):
                        raise
                db_version = 2
                self.conn.execute("INSERT OR REPLACE INTO db_version (id, version) VALUES (1, 2)")
            
            self.conn.commit()

            # --- Migrace Verze 3 (Nové sloupce pro progress) ---
            if db_version < 3:
                xbmc.log("[MetadataDB Migrace] Spouštím migraci na verzi 3: PROGRESS SLOUPECE.", xbmc.LOGINFO)
                
                # 1. Sloupec pro kontrolu kompletnosti shlédnutí
                try:
                    self.conn.execute("ALTER TABLE media ADD COLUMN trakt_is_fully_watched INTEGER DEFAULT 0")
                    xbmc.log("[MetadataDB Migrace] Sloupec trakt_is_fully_watched přidán.", xbmc.LOGINFO)
                except sqlite3.OperationalError as e:
                    if 'duplicate column name' not in str(e):
                        raise
                        
                # 2. Sloupec pro ID následující epizody
                try:
                    self.conn.execute("ALTER TABLE media ADD COLUMN trakt_next_episode_id TEXT DEFAULT NULL")
                    xbmc.log("[MetadataDB Migrace] Sloupec trakt_next_episode_id přidán.", xbmc.LOGINFO)
                except sqlite3.OperationalError as e:
                    if 'duplicate column name' not in str(e):
                        raise
                        
                db_version = 3
                self.conn.execute("INSERT OR REPLACE INTO db_version (id, version) VALUES (1, 3)")
            
            self.conn.commit()

            # --- Migrace Verze 4 (Nové sloupce pro progress) ---
            if db_version < 4:
                xbmc.log("[MetadataDB Migrace] Spouštím migraci na verzi 4: PROGRESS SLOUPECE.", xbmc.LOGINFO)
                
                # 1. Sloupec pro kontrolu kompletnosti shlédnutí
                try:
                    self.conn.execute("ALTER TABLE media ADD COLUMN trakt_episodes_aired INTEGER DEFAULT 0")
                    xbmc.log("[MetadataDB Migrace] Sloupec trakt_episodes_aired přidán.", xbmc.LOGINFO)
                except sqlite3.OperationalError as e:
                    if 'duplicate column name' not in str(e):
                        raise
                        
                # 2. Sloupec pro ID následující epizody
                try:
                    self.conn.execute("ALTER TABLE media ADD COLUMN trakt_episodes_completed INTEGER DEFAULT 0")
                    xbmc.log("[MetadataDB Migrace] Sloupec trakt_episodes_completed přidán.", xbmc.LOGINFO)
                except sqlite3.OperationalError as e:
                    if 'duplicate column name' not in str(e):
                        raise
                        
                db_version = 4
                self.conn.execute("INSERT OR REPLACE INTO db_version (id, version) VALUES (1, 4)")
            
            self.conn.commit()

            # --- Migrace Verze 5 (Nové sloupce pro progress) ---
            if db_version < 5:
                xbmc.log("[MetadataDB Migrace] Spouštím migraci na verzi 4: PROGRESS SLOUPECE.", xbmc.LOGINFO)
                
                # 1. Sloupec pro kontrolu kompletnosti shlédnutí
                try:
                    self.conn.execute("ALTER TABLE media ADD COLUMN trakt_next_episode_title TEXT DEFAULT NULL")
                    xbmc.log("[MetadataDB Migrace] Sloupec trakt_next_episode_title přidán.", xbmc.LOGINFO)
                except sqlite3.OperationalError as e:
                    if 'duplicate column name' not in str(e):
                        raise
                       
                db_version = 5
                self.conn.execute("INSERT OR REPLACE INTO db_version (id, version) VALUES (1, 5)")
            
            self.conn.commit()

            # --- Migrace Verze 6 (Nové sloupce pro progress) ---
            if db_version < 6:
                xbmc.log("[MetadataDB Migrace] Spouštím migraci na verzi 6: PROGRESS SLOUPECE.", xbmc.LOGINFO)
                
                # 1. Sloupec pro kontrolu kompletnosti shlédnutí
                try:
                    self.conn.execute("ALTER TABLE media ADD COLUMN trakt_total_episodes INTEGER DEFAULT 0")
                    xbmc.log("[MetadataDB Migrace] Sloupec trakt_total_episodes přidán.", xbmc.LOGINFO)
                except sqlite3.OperationalError as e:
                    if 'duplicate column name' not in str(e):
                        raise
                       
                db_version = 6
                self.conn.execute("INSERT OR REPLACE INTO db_version (id, version) VALUES (1, 6)")
            
            self.conn.commit()

            # --- Migrace Verze 7 (Nové sloupce pro progress) ---
            if db_version < 7:
                xbmc.log("[MetadataDB Migrace] Spouštím migraci na verzi 6: PROGRESS SLOUPECE.", xbmc.LOGINFO)
                
                try:
                    self.conn.execute("ALTER TABLE media ADD COLUMN ws_watchlist_ts REAL DEFAULT NULL")
                    xbmc.log("[MetadataDB Migrace] Sloupec ws_watchlist_ts.", xbmc.LOGINFO)
                except sqlite3.OperationalError as e:
                    if 'duplicate column name' not in str(e):
                        raise
                       
                db_version = 7
                self.conn.execute("INSERT OR REPLACE INTO db_version (id, version) VALUES (1, 7)")
            
            self.conn.commit()

            # ----------------------------------------------------------------------
            # Krok 3: Indexy (volání tvé původní metody)
            # ----------------------------------------------------------------------
            self._create_indexes()

    def _create_indexes(self):
        indexes = [
            "CREATE INDEX IF NOT EXISTS idx_imdb ON media(imdb_id)",
            "CREATE INDEX IF NOT EXISTS idx_trakt ON media(trakt_id)",
            "CREATE INDEX IF NOT EXISTS idx_title ON media(title_cs, title_en)",
            "CREATE INDEX IF NOT EXISTS idx_accessed ON media(last_accessed)",
            "CREATE INDEX IF NOT EXISTS idx_type_year ON media(media_type, year)"
        ]
        for idx in indexes:
            try:
                self.conn.execute(idx)
            except:
                pass
        self.conn.commit()

    def get(self, tmdb_id):
        """Vrátí záznam nebo None"""
        try:
            cur = self.conn.cursor()
            cur.execute("SELECT * FROM media WHERE tmdb_id = ?", (tmdb_id,))
            row = cur.fetchone()
            if row:
                cols = [col[0] for col in cur.description]
                return dict(zip(cols, row))
        except Exception as e:
            xbmc.log(f"[MetadataDB] get error: {e}", xbmc.LOGERROR)
        return None

    def get_or_create(self, tmdb_id: int, media_type: str, metadata: Dict[str, Any]) -> Optional[Dict[str, Any]]:
        """
        Pokusí se získat záznam podle tmdb_id. Pokud neexistuje, vytvoří jej s minimálními metadaty.
        Vrací existující nebo nově vytvořený záznam.
        """
        if not tmdb_id:
            return None

        # 1. Zkusíme načíst existující
        existing = self.get_by_tmdb_id(tmdb_id, columns=['tmdb_id'])
        
        if existing:
            return existing
        
        # 2. Neexistuje - vytvoříme nový záznam s minimem dat
        try:
            # Připravíme data pro nový záznam
            new_data = {
                'tmdb_id': int(tmdb_id),
                'media_type': media_type,
                'title_cs': metadata.get('cz_title') or metadata.get('eng_title'),
                'title_en': metadata.get('eng_title'),
                'title_original': metadata.get('eng_title'), # Pro jednoduchost
                'year': metadata.get('year'),
                # Všechny ostatní sloupce (plot, poster, ts) budou NULL nebo defaultní (což je správně)
            }
            
            # Vložení nového záznamu (použijeme metodu update pro insert s tmdb_id)
            # Metoda update je již robustní a volá insert/commit.
            self.update(tmdb_id, new_data, source='trakt_lazy_load')

            # Vracíme záznam s tmdb_id, který stačí pro build_trakt_context
            return {'tmdb_id': tmdb_id} 

        except Exception as e:
            xbmc.log(f"[MetadataDB] CHYBA při lazy-load filmu {tmdb_id}: {e}", xbmc.LOGERROR)
            return None

    def get_by_tmdb_id(self, tmdb_id: int, columns: List[str] = None) -> Optional[dict]:
        """
        Vyhledá a vrátí záznam pro film nebo seriál na základě tmdb_id.
        Volitelně omezuje vrácené sloupce.
        """
        if not tmdb_id:
            return None

        col_str = ', '.join(columns) if columns else '*'
        row = None
        
        try:
            with self.conn: # 🚨 KRITICKÁ ÚPRAVA C
                cur = self.conn.cursor()
                cur.execute(f"SELECT {col_str} FROM media WHERE tmdb_id = ?", (int(tmdb_id),))
                row = cur.fetchone()
            
            if row:
                cols = [col[0] for col in cur.description]
                return dict(zip(cols, row))
        except Exception as e:
            xbmc.log(f"[MetadataDB] get_by_tmdb_id error pro {tmdb_id}: {e}", xbmc.LOGERROR)
            
        return None
        
    def update(self, tmdb_id, data, source='tmdb'):
        """Aktualizuje nebo vloží záznam"""
        try:
            existing = self.get(tmdb_id)
            if existing:
                sql = "UPDATE media SET "
                params = []
                for k, v in data.items():
                    if k in ('last_updated', 'last_accessed', 'source_flags'):
                        continue
                    sql += f"{k} = ?, "
                    params.append(v)
                sql += "last_accessed = datetime('now'), "
                sql += "last_updated = datetime('now') WHERE tmdb_id = ?"
                params.append(tmdb_id)
                self.conn.execute(sql, params)
            else:
                keys = ['tmdb_id'] + list(data.keys())
                placeholders = ','.join(['?'] * len(keys))
                sql = f"INSERT INTO media ({', '.join(keys)}) VALUES ({placeholders})"
                values = [tmdb_id] + list(data.values())
                self.conn.execute(sql, values)
            self.conn.commit()
        except Exception as e:
            xbmc.log(f"[MetadataDB] update error: {e}", xbmc.LOGERROR)

    def cleanup(self, days=30):
        """Smaže staré záznamy"""
        try:
            cutoff = (datetime.now() - timedelta(days=days)).strftime('%Y-%m-%d %H:%M:%S')
            self.conn.execute("DELETE FROM media WHERE last_accessed < ?", (cutoff,))
            self.conn.commit()
            xbmc.log(f"[MetadataDB] Cleanup: {self.conn.total_changes} records removed", xbmc.LOGINFO)
        except Exception as e:
            xbmc.log(f"[MetadataDB] cleanup error: {e}", xbmc.LOGERROR)

    def close(self):
        if self.conn:
            self.conn.close()

# Globální instance
db = MetadataDB()

# ========================================
# AUTOMATICKÁ MIGRACE ZE STARÝCH JSONŮ (pouze jednou)
# ========================================
import shutil

def _migrate_old_caches():
    import xbmcvfs
    import json
    import os
    # from resources.lib.cache_store import get_imdb_id as old_get_imdb_id

    migrated = False
    ADDON_ID_OLD = 'plugin.video.mmirousek'        # starý doplněk
    ADDON_ID_V2  = 'plugin.video.mmirousek_v2'     # nový doplněk
    profile_old = xbmcvfs.translatePath(f'special://profile/addon_data/{ADDON_ID_OLD}')
    profile_v2  = xbmcvfs.translatePath(f'special://profile/addon_data/{ADDON_ID_V2}')

    # 1. IMDb ID cache (cache_store.py)
    try:
        old_cache = os.path.join(profile_old, 'omdb_cache.json')
        if xbmcvfs.exists(old_cache):
            with open(old_cache, 'r', encoding='utf-8') as f:
                data = json.load(f)
                for key, val in data.get('imdb_ids', {}).items():
                    if ':' in key:
                        mt, tid = key.split(':', 1)
                        tid = int(tid)
                        imdb = val.get('imdb_id')
                        if imdb:
                            db.update(tid, {'imdb_id': imdb, 'media_type': mt}, source='migration')
                            migrated = True
    except Exception as e:
        xbmc.log(f"[Migration] IMDb cache error: {e}", xbmc.LOGWARNING)

    # 2. tmdb_cache.json
    try:
        old_tmdb = os.path.join(profile_old, 'tmdb_cache.json')
        if xbmcvfs.exists(old_tmdb):
            with open(old_tmdb, 'r', encoding='utf-8') as f:
                cache = json.load(f)
                for item in cache.values():
                    tid = item.get('tmdb_id')
                    if not tid: continue
                    data = {
                        'media_type': item.get('media_type', 'movie'),
                        'title_cs': item.get('title_cs'),
                        'title_en': item.get('title_en'),
                        'title_original': item.get('original_title') or item.get('original_name'),
                        'overview_cs': item.get('overview_cs'),
                        'overview_en': item.get('overview_en'),
                        'year': item.get('year'),
                        'poster_path': item.get('poster_path'),
                        'backdrop_path': item.get('backdrop_path'),
                        'vote_average': item.get('vote_average'),
                        'imdb_id': item.get('imdb_id')
                    }
                    db.update(tid, {k: v for k, v in data.items() if v}, source='migration')
                    migrated = True
    except Exception as e:
        xbmc.log(f"[Migration] tmdb_cache error: {e}", xbmc.LOGWARNING)

    # 3. omdb_results_cache.json (volitelně – IMDb už máme)
    try:
        old_omdb = os.path.join(profile_old, 'omdb_results_cache.json')
        if xbmcvfs.exists(old_omdb):
            xbmc.log("[Migration] OMDb cache nalezena – přeskočeno (IMDb už máme)", xbmc.LOGINFO)
    except Exception as e:
        xbmc.log(f"[Migration] omdb_cache error: {e}", xbmc.LOGWARNING)

    if migrated:
        xbmcgui.Dialog().notification("mmirousek_v2", "Migrace cache → SQLite dokončena!", xbmcgui.NOTIFICATION_INFO, 5000)
        xbmc.log("[V2] Migrace starých cache dokončena", xbmc.LOGINFO)

# ========================================
# MIGRACE STARÝCH CACHE – SPUSTÍ SE JEN JEDNOU (trvale)
# ========================================
def _run_migration_once():
    # Používáme nastavení doplňku jako příznak – přežije restart
    if ADDON.getSetting('v2_migration_done') == 'true':
        return  # už byla provedena

    xbmc.log("[V2] Spouštím migraci starých cache → SQLite...", xbmc.LOGINFO)
    _migrate_old_caches()

    # Označíme jako hotové – už se to nikdy nespustí
    ADDON.setSetting('v2_migration_done', 'true')
    xbmcgui.Dialog().notification("mmirousek_v2", "Migrace cache → SQLite dokončena!", xbmcgui.NOTIFICATION_INFO, 5000)
    xbmc.log("[V2] Migrace dokončena a označena jako hotová", xbmc.LOGINFO)

# Spustíme migraci (jen pokud ještě nebyla)
_run_migration_once()