# -*- coding: utf-8 -*-
"""
EPG updater pro TV Program (Kodi doplněk plugin.video.mmirousek_v2)

- Defaultní seznam kanálů je zabudován níže (bez duplicity 'filmbox').
- Pokud v epg/ neexistuje kanaly_id.json, vytvoří se automaticky z DEFAULT_CHANNELS.
- Aktualizace tahá dny 0..3 (dnes + 3 dopředu), mezi požadavky dělá krátkou pauzu.
- Zápis epg_all.json probíhá atomicky (temp -> rename).
- Volitelná integrace s Kodi: progress update + cancel callback.

Pozn.: Používá stejné API a mapování ID -> názvů jako původní skript.
"""

from __future__ import annotations
import os
import json
import time
from datetime import datetime, timedelta

try:
    import xbmc
    import xbmcgui
except Exception:
    xbmc = None
    xbmcgui = None

import requests


# ---- Konfigurace ----

API_URL = "https://api-cz.webtv.tv/epg/channel"
USER_AGENT = (
    "Mozilla/5.0 (Windows NT 10.0; Win64; x64) "
    "AppleWebKit/537.36 (KHTML, like Gecko) "
    "Chrome/141.0.0.0 Safari/537.36"
)

HEADERS = {
    "accept": "application/json, text/plain, */*",
    "accept-language": "cs",
    "appversion": "6.6.0",
    "cache-control": "no-cache",
    "content-type": "application/json",
    "origin": "https://webtv.tv",
    "pragma": "no-cache",
    "referer": "https://webtv.tv/",
    "user-agent": USER_AGENT,
}

# Mapování některých historických/alternativních ID kanálů na zobrazovaný název
CHANNEL_NAME_MAP = {
    "smichov": "Nova Fun",
    "tv_fanfa": "Nova Cinema",
    "telka": "Nova Krimi",
    "prima_news": "CNN Prima News",
}

# Výchozí seznam kanálů (bez duplicit "filmbox")
DEFAULT_CHANNELS = [
    "nova",
    "prima",
    "prima_cool",
    "prima_krimi",
    "prima_max",
    "nova_cinema",
    "hbo",
    "hbo_2",
    "hbo_3",
    "cinemax",
    "ct_1_sm",
    "seznam_tv",
    "prima_comedy_central",
    "warner",
    "joj_cinema",
    "joj_family",
    "cinemax_2",
    "filmbox_extra_hd",
    "filmbox_premium",
    "filmbox_family",
    "filmbox_plus",
    "amc",
    "film_plus",
    "cs_film",
    "axn",
    "axn_white",
    "axn_black",
]


# ---- Pomocné logování ----

def _log(msg, level="INFO"):
    txt = f"[EPGUpdater] {msg}"
    try:
        if xbmc:
            lvl = {
                "INFO":  xbmc.LOGINFO,
                "WARN":  xbmc.LOGWARNING,
                "ERROR": xbmc.LOGERROR,
                "DEBUG": xbmc.LOGDEBUG,
            }.get(level, xbmc.LOGINFO)
            xbmc.log(txt, lvl)
        else:
            print(txt)
    except Exception:
        print(txt)


# ---- Práce se soubory ----

def _ensure_dir(path_dir: str):
    if not os.path.isdir(path_dir):
        os.makedirs(path_dir, exist_ok=True)


def _atomic_write_json(target_path: str, data_obj):
    """Zapisuje JSON atomicky: .tmp -> rename."""
    folder = os.path.dirname(target_path)
    _ensure_dir(folder)
    tmp_path = target_path + ".tmp"
    with open(tmp_path, "w", encoding="utf-8") as f:
        json.dump(data_obj, f, ensure_ascii=False, indent=2)
        f.flush()
        os.fsync(f.fileno())
    os.replace(tmp_path, target_path)  # atomic rename na většině platforem


# ---- Kanály ----

def ensure_default_channels(epg_dir: str) -> list[str]:
    """
    Vrátí seznam kanálů. Pokud epg/kanaly_id.json neexistuje, vytvoří ho z DEFAULT_CHANNELS.
    """
    _ensure_dir(epg_dir)
    channels_path = os.path.join(epg_dir, "kanaly_id.json")

    if not os.path.exists(channels_path):
        _log("Soubor kanaly_id.json neexistuje – vytvářím z DEFAULT_CHANNELS…")
        _atomic_write_json(channels_path, DEFAULT_CHANNELS)
        return list(DEFAULT_CHANNELS)

    # Načti uživatelský seznam
    try:
        with open(channels_path, "r", encoding="utf-8") as f:
            data = json.load(f)
        if isinstance(data, list):
            # Odfiltruj prázdné/dublikované kanály při běhu (nepřepisujeme uživ. soubor)
            seen = set()
            out = []
            for ch in data:
                chs = str(ch).strip()
                if not chs:
                    continue
                if chs not in seen:
                    seen.add(chs)
                    out.append(chs)
            return out
        else:
            _log("kanaly_id.json není JSON pole – použiji DEFAULT_CHANNELS", "WARN")
            return list(DEFAULT_CHANNELS)
    except Exception as e:
        _log(f"Chyba čtení kanaly_id.json: {e} – použiji DEFAULT_CHANNELS", "ERROR")
        return list(DEFAULT_CHANNELS)


# ---- Aktualizace EPG ----

import threading
import time
import requests
import xbmc
import xbmcgui

def update_epg(
    epg_dir: str,
    on_progress=None,
    is_canceled=None,
    days_forward: int = 5,
    start_offset: int = 0,
    pause_seconds: float = 0.5,
    timeout: int = 10,
) -> str:
    """
    Stáhne EPG pro dny <start_offset>..<start_offset+days_forward> pro všechny kanály
    a zapíše epg_all.json atomicky. Vrací cestu k výslednému souboru.

    on_progress(percent:int, line1:str, line2:str) -> callback pro progress bar
    is_canceled() -> True, pokud uživatel zrušil
    """
    from datetime import datetime, timedelta
    import threading
    import requests
    import xbmc
    import xbmcgui
    from resources.lib.epg_updater import _ensure_dir, ensure_default_channels, _atomic_write_json, API_URL, HEADERS, CHANNEL_NAME_MAP

    _ensure_dir(epg_dir)
    channels = ensure_default_channels(epg_dir)
    out_path = os.path.join(epg_dir, "epg_all.json")

    total_days = days_forward - start_offset + 1
    total_steps = max(1, len(channels) * total_days)

    # Pokud jsou callbacky předané, použij původní logiku
    if callable(on_progress) and callable(is_canceled):
        step = 0
        sess = requests.Session()
        sess.headers.update(HEADERS)
        output = []
        utc_midnight_today = datetime.utcnow().replace(hour=0, minute=0, second=0, microsecond=0)
        cancelled = False

        for ch in channels:
            if is_canceled():
                cancelled = True
                break
            channel_started = False
            for off in range(start_offset, start_offset + days_forward + 1):
                if is_canceled():
                    cancelled = True
                    break
                date_obj = utc_midnight_today + timedelta(days=off)
                date_str = date_obj.strftime("%Y-%m-%dT00:00:00.000Z")
                payload = {"ofset": "0", "limit": "2000", "channel_id": ch, "date": date_str}

                step += 1
                percent = (step * 100.0) / total_steps
                if on_progress(percent, f"Kanál: {ch}", f"Den: {date_str[:10]} ({step}/{total_steps})") is False:
                    cancelled = True
                    break

                try:
                    resp = sess.post(API_URL, json=payload, timeout=timeout)
                    data = resp.json()
                    if isinstance(data, dict) and "id" in data and isinstance(data.get("content"), list):
                        chan_id = data.get("id") or ch
                        chan_name = CHANNEL_NAME_MAP.get(chan_id, chan_id)
                        if not channel_started:
                            output.append(chan_name)
                            output.append([])
                            channel_started = True
                        output[-1].extend(data["content"])
                except Exception as e:
                    xbmc.log(f"[EPG] Chyba: {e}", xbmc.LOGERROR)

                time.sleep(pause_seconds)

            if cancelled:
                break

        _atomic_write_json(out_path, output)
        return out_path

    # --- Fallback: vlákno + DialogProgress ---
    state = {"percent": 0, "message": "Inicializuji...", "done": False, "error": None, "cancelled": False}

    def worker():
        try:
            sess = requests.Session()
            sess.headers.update(HEADERS)
            output = []
            utc_midnight_today = datetime.utcnow().replace(hour=0, minute=0, second=0, microsecond=0)
            step = 0
            for ch in channels:
                if state["cancelled"]:
                    return
                channel_started = False
                for off in range(start_offset, start_offset + days_forward + 1):
                    if state["cancelled"]:
                        return
                    date_obj = utc_midnight_today + timedelta(days=off)
                    date_str = date_obj.strftime("%Y-%m-%dT00:00:00.000Z")
                    payload = {"ofset": "0", "limit": "2000", "channel_id": ch, "date": date_str}

                    step += 1
                    state["percent"] = int(step * 100.0 / total_steps)
                    state["message"] = f"Kanál: {ch} • Den: {date_str[:10]}"

                    try:
                        resp = sess.post(API_URL, json=payload, timeout=timeout)
                        data = resp.json()
                        if isinstance(data, dict) and "id" in data and isinstance(data.get("content"), list):
                            chan_id = data.get("id") or ch
                            chan_name = CHANNEL_NAME_MAP.get(chan_id, chan_id)
                            if not channel_started:
                                output.append(chan_name)
                                output.append([])
                                channel_started = True
                            output[-1].extend(data["content"])
                    except Exception as e:
                        state["error"] = str(e)
                        return

                    time.sleep(pause_seconds)

            _atomic_write_json(out_path, output)
            state["done"] = True
            state["message"] = "EPG aktualizováno."
        except Exception as e:
            state["error"] = str(e)

    t = threading.Thread(target=worker)
    t.start()

    dp = xbmcgui.DialogProgress()
    dp.create("TV Program", "Aktualizuji EPG…vyčkejte prosím cca 1.minutu.")

    while not state["done"] and not state["error"]:
        if dp.iscanceled():
            state["cancelled"] = True
            break
        dp.update(state["percent"], state["message"])
        xbmc.sleep(200)

    dp.close()

    if state["cancelled"]:
        xbmcgui.Dialog().notification("TV Program", "Aktualizace EPG zrušena.", xbmcgui.NOTIFICATION_WARNING, 4000)
    elif state["error"]:
        xbmcgui.Dialog().notification("TV Program", f"Chyba: {state['error']}", xbmcgui.NOTIFICATION_ERROR, 5000)
    else:
        xbmcgui.Dialog().notification("TV Program", "EPG úspěšně aktualizováno.", xbmcgui.NOTIFICATION_INFO, 4000)

    return out_path
