# -*- coding: utf-8 -*-
# Module: download_manager
# Author: mmirousek (upraveno dle yawsp pro VFS)
# License: AGPL v3

import os
import io
import re
import traceback
import requests
import xbmc
import xbmcgui
import xbmcaddon
import xbmcvfs
from urllib.parse import parse_qsl
from resources.lib.webshare_auth import WebshareClient
from resources.lib.utils.ui_utils import clean_filename
from resources.lib.utils.ui_utils import reset_cancel_flags, set_cancel_flags, is_cancel_requested

ADDON_ID = 'plugin.video.mmirousek_v2'
ADDON = xbmcaddon.Addon(ADDON_ID)
UA = "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/81.0.4044.138 Safari/537.36"
BASE = 'https://webshare.cz'
_session = requests.Session()
_session.headers.update({'User-Agent': UA, 'Referer': BASE})

# --- Cancel flag: lokální (pro běžící instanci) + globální přes Window(10000) ---
CANCEL_DOWNLOAD = False
WIN_KEY_TRANSFER = 'mm_cancel_download'       # cancel současného přenosu (epizody)
WIN_KEY_ALL      = 'mm_cancel_download_all'   # cancel celé sezóny / všech přenosů

def cancel_download():
    """Nastaví globální cancel flagy. Volá se z routeru (jiná RunPlugin instance)."""
    set_cancel_flags()  # nově z ui_utils
    xbmc.log("[DOWNLOAD_MANAGER] Cancel flags set (transfer + all)", xbmc.LOGINFO)

def reset_cancel_flag():
    """Resetuje lokální cancel flag (pro nově spouštěná stahování)."""
    reset_cancel_flags()  # nově z ui_utils
    xbmc.log("[DOWNLOAD_MANAGER] Cancel flags reset", xbmc.LOGDEBUG)

def popinfo(message,
            heading=ADDON.getAddonInfo('name'),
            icon=xbmcgui.NOTIFICATION_INFO,
            time=3000,
            sound=False):
    """Wrapper pro notifikace (defaultně bez zvuku)."""
    try:
        xbmcgui.Dialog().notification(heading, message, icon, time, sound=sound)
    except Exception:
        xbmcgui.Dialog().notification(heading, message, icon, time)

def download(params):
    """
    Stáhne soubor z Webshare podle identu.
    - Podporuje lokální i síťové (VFS) cesty.
    """
    try:
        xbmc.log(f"[DOWNLOAD] params={params}", xbmc.LOGDEBUG)
        reset_cancel_flag()

        if not params or not params.get('ident'):
            xbmc.log("[DOWNLOAD] Missing ident in params", xbmc.LOGERROR)
            popinfo(ADDON.getLocalizedString(30107), icon=xbmcgui.NOTIFICATION_WARNING)
            return

        ws = WebshareClient(ADDON_ID)
        _ = ws.revalidate()
        link = ws.getlink(params['ident'], dtype='download')
        xbmc.log(f"[DOWNLOAD] getlink returned: {link}", xbmc.LOGDEBUG)
        if not link:
            popinfo(ADDON.getLocalizedString(30107), icon=xbmcgui.NOTIFICATION_WARNING)
            return

        url = link
        _orig_session_headers = None
        if '\n' in link:
            try:
                url, hdrs = link.split('\n', 1)
                pairs = dict(parse_qsl(hdrs, encoding='utf-8'))
                _orig_session_headers = dict(_session.headers)
                _session.headers.update(pairs)
            except Exception as e:
                xbmc.log(f"[DOWNLOAD] failed parsing headers: {e}", xbmc.LOGWARNING)

        # Cílová složka
        dfolder = ADDON.getSetting('dfolder') or ''
        if dfolder:
            dest_real = xbmcvfs.translatePath(dfolder)
            # Změna: xbmcvfs.mkdirs funguje i pro síťové cesty, os.makedirs jen pro lokální
            if not xbmcvfs.exists(dest_real):
                try:
                    xbmcvfs.mkdirs(dest_real)
                except Exception as e:
                    xbmc.log(f"[DOWNLOAD] cannot create destination {dest_real}: {e}", xbmc.LOGERROR)
                    # Fallback na výběr složky
                    destination = xbmcgui.Dialog().browse(3, ADDON.getLocalizedString(30302), 'files')
                    if not destination:
                        return
                    dest_real = xbmcvfs.translatePath(destination)
        else:
            destination = xbmcgui.Dialog().browse(3, ADDON.getLocalizedString(30302), 'files')
            if not destination:
                return
            dest_real = xbmcvfs.translatePath(destination)

        # Název souboru
        raw_name = params.get('name') or url.split('/')[-1].split('?')[0] or params.get('ident')
        base_name = clean_filename(raw_name)
        _, ext = os.path.splitext(url.rsplit('/', 1)[-1].split('?', 1)[0])
        _, base_ext = os.path.splitext(base_name)
        filename = base_name if base_ext else (base_name + (ext or ""))
        
        # Sestavení finální cesty (join funguje univerzálně pro stringy)
        if dest_real.endswith('/') or dest_real.endswith('\\'):
            target_path = dest_real + filename
        else:
            target_path = dest_real + '/' + filename
            
        xbmc.log(f"[DOWNLOAD] final target_path={target_path}", xbmc.LOGINFO)

        try:
            download_file(url, target_path, expected_size=None)
        finally:
            if _orig_session_headers is not None:
                _session.headers.clear()
                _session.headers.update(_orig_session_headers)
    except Exception as e:
        xbmc.log(f"[DOWNLOAD] unexpected error: {e}\n{traceback.format_exc()}", xbmc.LOGERROR)
        popinfo("Chyba při zahájení stahování", icon=xbmcgui.NOTIFICATION_ERROR)

def download_file(url, name, expected_size=None):
    """
    Stáhne URL na zadanou cestu.
    - Rozlišuje lokální a síťové (VFS) cesty.
    - Lokální cesty: Používá io.open, podporuje RESUME.
    - Síťové cesty: Používá xbmcvfs.File, nepodporuje RESUME (vždy od začátku).
    """
    response = None
    bf = None # Buffer file object
    
    try:
        xbmc.log(f"[DOWNLOAD_FILE] start url={url} target={name}", xbmc.LOGINFO)

        # Nastavení notifikací
        notify_enabled = ADDON.getSettingBool('dnotify')
        interval_raw = ADDON.getSetting('dnevery') or '10%'
        try:
            interval = int(str(interval_raw).replace('%', '').strip())
            if interval <= 0 or interval > 100:
                interval = 10
        except Exception:
            interval = 10

        # Přelož cestu (pro jistotu, i když už by měla být)
        real_path = xbmcvfs.translatePath(name)
        
        # Detekce, zda se jedná o lokální cestu (viditelnou pro OS)
        # Zkontrolujeme rodičovský adresář, protože soubor ještě nemusí existovat
        parent_dir = os.path.dirname(real_path)
        is_local = os.path.exists(parent_dir)

        # --- Logika RESUME a velikosti ---
        current_size = 0
        
        if is_local:
            # Pro lokální soubory zkusíme resume
            if os.path.exists(real_path):
                try:
                    current_size = os.path.getsize(real_path)
                except Exception:
                    current_size = 0
        else:
            # Pro síťové soubory (VFS) začínáme vždy od nuly (resume je riskantní/složité přes VFS)
            current_size = 0
            # Pokud soubor existuje na síti, bude přepsán
            
        # HTTP Range pro resume (jen pokud je local a máme nějaká data)
        headers = {}
        if current_size > 0:
            headers['Range'] = f"bytes={current_size}-"
            xbmc.log(f"[DOWNLOAD_FILE] Resume (Local) od {current_size} bajtů", xbmc.LOGINFO)
        elif not is_local:
            xbmc.log(f"[DOWNLOAD_FILE] VFS (Network) download - always fresh start", xbmc.LOGINFO)

        # HTTP GET
        response = _session.get(url, headers=headers, stream=True, timeout=60, allow_redirects=True)
        status = getattr(response, "status_code", None)
        
        # Kontrola status kódu (200 OK, 206 Partial Content)
        if status not in (200, 206):
            if notify_enabled:
                popinfo("Stahování selhalo (server error)", icon=xbmcgui.NOTIFICATION_ERROR)
            return

        # Celková velikost
        total_size = 0
        try:
            total_size = int(response.headers.get("content-length") or 0)
        except Exception:
            total_size = 0
            
        if total_size <= 0 and expected_size:
            try:
                total_size = int(expected_size)
            except Exception:
                total_size = 0

        # Pokud máme resume (status 206), přičti current_size k total_size
        if status == 206:
            total_size += current_size
        elif status == 200 and current_size > 0:
            # Server nepodporuje Range, stahuje se od začátku -> reset current_size
            current_size = 0

        chunk_size = 4 * 1024 * 1024 # 4 MB chunks
        downloaded = current_size

        # Inicializace prahu pro notifikace
        if total_size > 0:
            pct_now = int(downloaded * 100 / total_size)
            next_threshold = ((pct_now // interval) * interval) + interval
            if next_threshold <= pct_now:
                next_threshold += interval
            next_threshold = min(next_threshold, 100)
        else:
            next_threshold = interval

        # --- Otevření souboru pro zápis ---
        if is_local:
            # Lokální disk: io.open s 'ab' (append) nebo 'wb' (write)
            mode = 'ab' if current_size > 0 else 'wb'
            bf = io.open(real_path, mode)
        else:
            # Síťový disk: xbmcvfs.File (vždy write 'w')
            bf = xbmcvfs.File(real_path, 'w')

        # --- Smyčka stahování ---
        try:
            for chunk in response.iter_content(chunk_size=chunk_size):
                if is_cancel_requested():
                    xbmc.log("[DOWNLOAD_FILE] Cancel requested", xbmc.LOGINFO)
                    if notify_enabled:
                        popinfo("Stahování zrušeno", icon=xbmcgui.NOTIFICATION_INFO, time=2000)
                    raise KeyboardInterrupt("Cancelled by user")

                if not chunk:
                    continue

                bf.write(chunk)
                downloaded += len(chunk)
                # Notifikace
                if notify_enabled and total_size > 0:
                    pct = int(downloaded * 100 / total_size)
                    while pct >= next_threshold and next_threshold <= 100:
                        # Krátký label: SxxExx nebo zkrácený název
                        short_label = None
                        try:
                            # Pokud máme season/episode v params (předané z queue)
                            if params.get('season') and params.get('episode'):
                                short_label = f"S{int(params['season']):02d}E{int(params['episode']):02d}"
                        except Exception:
                            pass
                        if not short_label:
                            short_label = os.path.basename(real_path)[:15]  # fallback
                        popinfo(f"{short_label} – {next_threshold}%", time=1500, sound=False)
                        next_threshold += interval
        finally:
            # Bezpečné uzavření souboru
            if bf:
                bf.close()
            xbmc.log(f"[DOWNLOAD_FILE] finished downloaded={downloaded}", xbmc.LOGINFO)
            if notify_enabled:
                # Vyčisti předchozí notifikace (trik: prázdná notifikace na 1 ms)
                xbmc.executebuiltin("Notification(Clean, , 1)")
                # Krátká finální notifikace
                popinfo(f"Staženo: {os.path.basename(real_path)}", icon=xbmcgui.NOTIFICATION_INFO,)

    except KeyboardInterrupt:
        xbmc.log("[DOWNLOAD_FILE] cancelled by user", xbmc.LOGINFO)
        # U VFS (xbmcvfs) často zůstane soubor nevalidní/nedokončený, to je OK chování
        pass

    except Exception as e:
        xbmc.log(f"[DOWNLOAD_FILE] error: {e}\n{traceback.format_exc()}", xbmc.LOGERROR)
        if notify_enabled:
            popinfo("Chyba při stahování", icon=xbmcgui.NOTIFICATION_ERROR)

    finally:
        try:
            if response:
                response.close()
        except Exception:
            pass