# resources/lib/background_enrich.py (UPRAVENÝ KÓD)
# Zajišťuje, že ADDON_ID souhlasí s ADDON_ID v service.py

import xbmc
import xbmcgui
import time
import xbmcaddon
from datetime import datetime

from resources.lib.metadata_db import db
from resources.lib.tmdb.omdb import get_omdb_info, extract_metrics
from resources.lib.tmdb.tmdb_module import TMDb, API_URL

ADDON_ID = 'plugin.video.mmirousek_v2' # Mělo by souhlasit s ADDON_ID v service.py
ADDON = xbmcaddon.Addon(ADDON_ID)

# === UTILITY FUNKCE PRO ČASOVÁNÍ (Stejná logika jako u Trakt) ===
def get_last_omdb_enrich_time() -> float:
    """Načte čas poslední OMDb synchronizace (Unix timestamp)."""
    try:
        ts = float(ADDON.getSetting('omdb_last_enrich_ts') or '0.0')
        return ts
    except Exception:
        return 0.0

def set_last_omdb_enrich_time():
    """Uloží aktuální Unix timestamp jako čas poslední OMDb synchronizace."""
    ADDON.setSetting('omdb_last_enrich_ts', str(time.time()))
# ================================================================

def background_enrich_omdb_ratings():
    """
    Hlavní funkce pro spouštění doplňování OMDb ratingů.
    Používá DialogProgressBG.
    """
    
    # 🚨 Ponecháváme xbmc.sleep(1200) uvnitř smyčky pro respektování limitů OMDb API.
    # Původní zpoždění (xbmc.sleep(10000)) bylo odstraněno, protože spuštění řídí service.py
    
    xbmc.log("[V2] Background OMDb enrich: STARTING PROCESS", xbmc.LOGINFO)

    try:
        rows = db.conn.execute("""
            SELECT tmdb_id, imdb_id FROM media 
            WHERE imdb_id IS NOT NULL AND imdb_id != ''
              AND (imdb_rating IS NULL OR imdb_rating = '')
            LIMIT 800
        """).fetchall()

        if not rows:
            xbmc.log("[V2] Background enrich: nic k doplnění", xbmc.LOGINFO)
            return

        total = len(rows)
        progress = xbmcgui.DialogProgressBG()
        progress.create("mmirousek_v2", f"Doplňuji OMDb ratingy... ({total})")

        for idx, (tmdb_id, imdb_id) in enumerate(rows):
            # 🚨 PŮVODNÍ CHYBNÝ KÓD ODEBRÁN:
            # if progress.iscanceled(): 
            #     xbmc.log("[V2] Background enrich: přerušeno uživatelem", xbmc.LOGINFO)
            #     break
                
            progress.update(int((idx+1)/total*100), message=f"{idx+1}/{total}")
            
            try:
                # ... zbytek logiky pro získání a uložení dat
                data = get_omdb_info(imdb_id=imdb_id, timeout=10)
                if data:
                    m = extract_metrics(data)
                    db.update(tmdb_id, {
                        'imdb_rating': m.get('imdbRating'),
                        'imdb_votes': m.get('imdbVotes'),
                        'metascore': m.get('Metascore'),
                        'rt_rating': m.get('RottenTomatoes'),
                        'box_office': m.get('BoxOffice')
                    }, source='bg_omdb')
            except Exception as e:
                xbmc.log(f"[OMDB Enrich] Chyba pro {imdb_id}: {e}", xbmc.LOGWARNING)
                
            xbmc.sleep(1200) # Pauza 1.2 sekundy mezi API voláními
            
        progress.close()
        xbmc.log("[V2] Background OMDb enrich: FINISHED", xbmc.LOGINFO)

    except Exception as e:
        xbmc.log(f"[V2] Background enrich FATAL ERROR: {e}", xbmc.LOGERROR)

# resources/lib/background_enrich.py (NOVÁ FUNKCE)

def background_enrich_tmdb_titles():
    """Doplňuje chybějící české názvy a přehledy z TMDb, voláno po OMDb."""

    tmdb_api_key = ADDON.getSetting('tmdb_api_key')
    if not tmdb_api_key:
        xbmc.log("[V2] Background TMDb Title Enrich: Chybí TMDb API klíč", xbmc.LOGWARNING)
        return

    tmdb_client = TMDb(tmdb_api_key)
    xbmc.log("[V2] Background TMDb Title Enrich: STARTING PROCESS", xbmc.LOGINFO)

    try:
        # POUZE POLOŽKY, KTERÉ MAJÍ tmdb_id a CHYBÍ jim title_cs
        rows = db.conn.execute("""
            SELECT tmdb_id, media_type FROM media
            WHERE title_cs IS NULL OR title_cs = ''
            LIMIT 500
        """).fetchall()

        if not rows:
            xbmc.log("[V2] Background TMDb Title Enrich: nic k doplnění", xbmc.LOGINFO)
            return

        total = len(rows)
        progress = xbmcgui.DialogProgressBG()
        progress.create("", f"Doplňuji české názvy z TMDb... ({total})")

        for idx, (tmdb_id, media_type) in enumerate(rows):
                
            progress.update(int((idx+1)/total*100), message=f"{idx+1}/{total}")
            
            try:
                # Volání TMDb API s explicitním požadavkem na 'cs-CZ' jazyk
                url_path = f"/{media_type}/{tmdb_id}" # media_type je 'movie' nebo 'tv'
                response = tmdb_client._call_api_raw(API_URL + url_path, {'language': 'cs-CZ'})
                details_cs = response.json()
                
                # Extrakce českých dat
                title_cs = details_cs.get('title') or details_cs.get('name')
                overview_cs = details_cs.get('overview')

                if title_cs and title_cs.strip():
                    # Uložení do DB
                    db.update(tmdb_id, {
                        'title_cs': title_cs,
                        'overview_cs': overview_cs 
                    }, source='bg_tmdb_title') 
                
            except Exception as e:
                xbmc.log(f"[TMDb Enrich] Chyba pro {tmdb_id}: {e}", xbmc.LOGWARNING)
                
            xbmc.sleep(500) # Pauza 0.5 sekundy pro uvolnění procesoru a dodržení limitů TMDb
            
        progress.close()
        xbmc.log("[V2] Background TMDb Title Enrich: FINISHED", xbmc.LOGINFO)

    except Exception as e:
        xbmc.log(f"[V2] Background TMDb Title Enrich FATAL ERROR: {e}", xbmc.LOGERROR)


def background_audit_fill_tmdb_metadata():
    """
    Projde celou DB (tabulka 'media') a:
      - validuje title_cs/title_en na latinku (odstraní KO/ZH apod. 'čtverečky'),
      - doplní/napraví title_cs (z TMDb 'language=cs-CZ') pokud chybí nebo je ne-latinkou,
      - doplní/napraví title_en (z TMDb 'language=en-US') pokud chybí nebo je ne-latinkou,
      - doplní overview_cs/overview_en, year, poster_path, backdrop_path (jen pokud chybí),
      - zapisuje pouze změněná pole (minimální UPDATE),
      - průběžně zobrazuje pozadí progress a commituje po dávkách.
    """
    import time

    LOG_PFX = "[V2][TMDb AUDIT]"
    xbmc.log(f"{LOG_PFX} START", xbmc.LOGINFO)

    # --- helper: latinka (shodná logika jako v show_tmdb_discover_list) ---
    def is_latinish(s: str) -> bool:
        if not s:
            return False
        total_letters = 0
        latin_letters = 0
        strong_nonlatin = 0
        for ch in s:
            code = ord(ch)
            is_latin = (0x0041 <= code <= 0x005A) or (0x0061 <= code <= 0x007A) or \
                       (0x00C0 <= code <= 0x00FF) or (0x0100 <= code <= 0x024F)
            is_hangul = (0xAC00 <= code <= 0xD7AF) or (0x1100 <= code <= 0x11FF) or (0x3130 <= code <= 0x318F)
            is_cjk    = (0x4E00 <= code <= 0x9FFF) or (0x3400 <= code <= 0x4DBF)
            is_kata   = (0x30A0 <= code <= 0x30FF)
            is_hira   = (0x3040 <= code <= 0x309F)
            is_cyr    = (0x0400 <= code <= 0x04FF)
            is_arab   = (0x0600 <= code <= 0x06FF)
            if ch.isalpha():
                total_letters += 1
                if is_latin:
                    latin_letters += 1
                if is_hangul or is_cjk or is_kata or is_hira or is_cyr or is_arab:
                    strong_nonlatin += 1
        if total_letters == 0:
            return False
        if strong_nonlatin > 0:
            return False
        return (latin_letters / float(total_letters)) >= 0.70

    # --- API key + klient ---
    tmdb_api_key = ADDON.getSetting('tmdb_api_key')
    if not tmdb_api_key:
        xbmc.log(f"{LOG_PFX} Chybí TMDb API klíč – audit zrušen.", xbmc.LOGWARNING)
        return
    tmdb_client = TMDb(tmdb_api_key)

    # --- načti celou DB (jen movie/tv) ---
    try:
        rows = db.conn.execute("""
            SELECT tmdb_id, media_type, title_cs, title_en, title_original,
                   overview_cs, overview_en, year, poster_path, backdrop_path
            FROM media
            WHERE tmdb_id IS NOT NULL
              AND media_type IN ('movie','tv')
            ORDER BY tmdb_id
        """).fetchall()
    except Exception as e:
        xbmc.log(f"{LOG_PFX} DB SELECT FATAL: {e}", xbmc.LOGERROR)
        return

    total = len(rows or [])
    if total == 0:
        xbmc.log(f"{LOG_PFX} Nic k auditování.", xbmc.LOGINFO)
        return

    progress = xbmcgui.DialogProgressBG()
    progress.create("mmirousek_v2", f"Audit & doplnění TMDb metadat… ({total})")

    changed_count = 0
    batch = 0

    def _fetch_detail(media_type: str, tmdb_id: int, lang: str) -> dict:
        """
        Bezpečné volání TMDb detailu s daným jazykem ('cs-CZ' / 'en-US').
        Vrátí {} při chybě. Throttle ~350 ms per call.
        """
        try:
            url_path = f"/{media_type}/{tmdb_id}"
            resp = tmdb_client._call_api_raw(API_URL + url_path, {'language': lang})
            data = resp.json() if resp is not None else {}
            return data if isinstance(data, dict) else {}
        except Exception as e:
            xbmc.log(f"{LOG_PFX} TMDb detail FAIL tmdb_id={tmdb_id} lang={lang}: {e}", xbmc.LOGWARNING)
            return {}
        finally:
            xbmc.sleep(350)  # jemné škrcení API

    def _year_from(detail: dict, media_type: str, fallback_year: int) -> int:
        try:
            if media_type == 'movie':
                ds = (detail.get('release_date') or '')[:4]
            else:
                ds = (detail.get('first_air_date') or '')[:4]
            if ds.isdigit():
                return int(ds)
        except Exception:
            pass
        # zachovej původní
        try:
            return int(fallback_year) if fallback_year not in (None, '', 0, '0') else 0
        except Exception:
            return 0

    for idx, (tmdb_id, media_type, title_cs, title_en, title_original,
              overview_cs, overview_en, year, poster_path, backdrop_path) in enumerate(rows, start=1):
        try:
            progress.update(int(idx / float(total) * 100), message=f"{idx}/{total}")

            # --- co je potřeba doplnit/napravit? ---
            need_fix_title_cs = (not title_cs) or (isinstance(title_cs, str) and not is_latinish(title_cs))
            need_fix_title_en = (not title_en) or (isinstance(title_en, str) and not is_latinish(title_en))
            need_overview_cs  = not overview_cs
            need_overview_en  = not overview_en
            need_year         = (year in (None, '', 0, '0'))
            need_poster       = not poster_path
            need_backdrop     = not backdrop_path

            # Když není co řešit, běž dál
            if not (need_fix_title_cs or need_fix_title_en or need_overview_cs or need_overview_en
                    or need_year or need_poster or need_backdrop):
                continue

            # stáhni jen to, co je potřeba
            detail_cs = _fetch_detail(media_type, tmdb_id, 'cs-CZ') if (need_fix_title_cs or need_overview_cs) else {}
            detail_en = _fetch_detail(media_type, tmdb_id, 'en-US') if (need_fix_title_en or need_overview_en or need_year or need_poster or need_backdrop) else {}

            # --- pole z detailů (bezpečně podle typu) ---
            def _pick_title(d: dict) -> str:
                return (d.get('title') or d.get('name') or '').strip()

            new_title_cs = _pick_title(detail_cs) if detail_cs else None
            new_title_en = _pick_title(detail_en) if detail_en else None

            # validace latinky (title_cs nesmí být KO/ZH apod.)
            if new_title_cs and not is_latinish(new_title_cs):
                new_title_cs = None
            if new_title_en and not is_latinish(new_title_en):
                new_title_en = None

            # overview
            new_overview_cs = (detail_cs.get('overview') or '').strip() if detail_cs else None
            new_overview_en = (detail_en.get('overview') or '').strip() if detail_en else None

            # artwork (stejné v obou jazycích – vezmeme cokoliv, co přijde k dispozici)
            # preferuj EN detail (často kompletnější), jinak CZ
            def _pick_val(*vals):
                for v in vals:
                    if v:
                        return v
                return None

            d_art = detail_en if detail_en else detail_cs
            new_poster = _pick_val(d_art.get('poster_path') if d_art else None)
            new_backdrop = _pick_val(d_art.get('backdrop_path') if d_art else None)

            # year
            d_year_src = detail_en if detail_en else detail_cs
            new_year = _year_from(d_year_src, media_type, year) if (need_year and (detail_en or detail_cs)) else None

            # --- sestav změny (pouze co dává smysl a co se má měnit) ---
            update_data = {}

            # title_cs: oprav pouze, když chybí/neplatný → pokud máme validní nový CZ title
            if need_fix_title_cs and new_title_cs:
                update_data['title_cs'] = new_title_cs

            # title_en: oprav pouze, když chybí/neplatný → pokud máme validní nový EN title
            if need_fix_title_en and new_title_en:
                update_data['title_en'] = new_title_en

            # overviews – jen doplň chybějící
            if need_overview_cs and new_overview_cs:
                update_data['overview_cs'] = new_overview_cs
            if need_overview_en and new_overview_en:
                update_data['overview_en'] = new_overview_en

            # year – jen doplň
            if need_year and isinstance(new_year, int) and new_year > 0:
                update_data['year'] = new_year

            # artwork – jen doplň
            if need_poster and new_poster:
                update_data['poster_path'] = new_poster
            if need_backdrop and new_backdrop:
                update_data['backdrop_path'] = new_backdrop

            # Pokud jsme nic nevymysleli (např. CZ titulek stále není dostupný/latinkou), pokračuj
            if not update_data:
                continue

            # nezapomeň uložit media_type (pro jistotu, pokud by řádek vznikl starou migrací)
            if media_type in ('movie', 'tv') and 'media_type' not in update_data:
                update_data['media_type'] = media_type

            # zápis
            try:
                db.update(tmdb_id, update_data, source='bg_tmdb_audit')
                changed_count += 1
                batch += 1
            except Exception as e:
                xbmc.log(f"{LOG_PFX} DB UPDATE error tmdb_id={tmdb_id}: {e}", xbmc.LOGWARNING)

            # dávkový commit (omez růst WAL + viditelnost pro ostatní vlákna)
            if batch >= 50:
                try:
                    db.conn.commit()
                except Exception as e:
                    xbmc.log(f"{LOG_PFX} COMMIT error: {e}", xbmc.LOGERROR)
                finally:
                    batch = 0

        except Exception as e:
            xbmc.log(f"{LOG_PFX} LOOP item fail tmdb_id={tmdb_id}: {e}", xbmc.LOGERROR)
            continue

    # finální commit + progress close
    try:
        db.conn.commit()
    except Exception as e:
        xbmc.log(f"{LOG_PFX} Final COMMIT error: {e}", xbmc.LOGERROR)

    try:
        progress.close()
    except Exception:
        pass

    xbmc.log(f"{LOG_PFX} FINISHED • changed={changed_count} / scanned={total}", xbmc.LOGINFO)
    xbmcgui.Dialog().notification("TMDb audit", f"Hotovo: změněno {changed_count} / prohlédnuto {total}", xbmcgui.NOTIFICATION_INFO, 3500)


# --- EOF padding (kvůli dřívějším ořezům přenosu) ---
# (padding-1)
# (padding-2)
# (padding-3)
