
# -*- coding: utf-8 -*-
import os
import traceback
import xbmc
import xbmcgui
import xbmcplugin
import unidecode

from resources.lib.utils.ui_utils import make_static_item, clean_filename, path_exists, listdir_safe
from resources.lib.series import series_manager
from resources.lib.down.download_queue import load_queue
from resources.lib.utils.app_log import write as app_log, write_json as app_log_json

# ──────────────────────────────────────────────────────────────────────────────
# Helper pro název vyhledávání
# ──────────────────────────────────────────────────────────────────────────────
def _safe_search_name(data: dict, fallback: str) -> str:
    """
    Vrátí vhodný název pro vyhledávání:
    - Preferuje original_title, pokud po unidecode zůstane čitelné (latinka)
    - Jinak použije title nebo fallback
    """
# 🛑 ZMĚNA 1: Nejvyšší priorita je uložený název, který fungoval.
    search_name = data.get('search_query_name')
    if search_name:
        xbmc.log(f"[SafeSearchName] Používám prioritní název z cache: '{search_name}'", xbmc.LOGDEBUG)
        return search_name    
    tmdb = data.get('tmdb', {}) if isinstance(data, dict) else {}
    orig = tmdb.get('original_title') or ''
    title = tmdb.get('title') or fallback
    if orig:
        ascii_version = unidecode.unidecode(orig).strip()
        if ascii_version and len(ascii_version) >= 3:
            xbmc.log(f"[SafeSearchName] Používám original_title (ASCII): '{ascii_version}'", xbmc.LOGDEBUG)
            return ascii_version
    xbmc.log(f"[SafeSearchName] Používám fallback title: '{title}'", xbmc.LOGDEBUG)    
    return title or fallback
# ──────────────────────────────────────────────────────────────────────────────
# Funkce pro vyhledávání seriálu
# ──────────────────────────────────────────────────────────────────────────────
# resources/lib/series/series_ui.py (KOMPLETNÍ FUNKCE series_search)

def series_search(params, addon, handle, profile, get_url, api, revalidate, popinfo):
    """Search for a TV series and organize it into seasons and episodes"""
    token = revalidate()
    
    # 1. Získání názvů
    series_name = params.get('what') or xbmcgui.Dialog().input("Zadejte název seriálu")
    fallback_series_name = params.get('fallback_series_name')
    xbmc.log(f"[SeriesSearch DEBUG] Params received. Primary: '{series_name}', Fallback: '{fallback_series_name}'", xbmc.LOGINFO)

    if not series_name:
        xbmcplugin.endOfDirectory(handle, succeeded=False)
        return

    sm = series_manager.SeriesManager(addon, profile)
    data = sm.load_series_data(series_name) or {}

    # --- NOVÉ: nabídka volby, jestli hledat CZ nebo EN název ---
    tmdb = data.get('tmdb', {}) if isinstance(data, dict) else {}
    cz_title = tmdb.get('title') or (data.get('name') if isinstance(data, dict) else None) or series_name
    en_title = tmdb.get('original_title')
    if not en_title:
        en_title = (params.get('fallback_series_name') or '').strip() or None

    xbmc.log(f"[DEBUG SeriesSearch] CZ='{cz_title}' ENG='{en_title}'", xbmc.LOGINFO)

    chosen_search_name = None  # výsledek volby uživatele (nebo parametru); None -> použije se autodetekce

    # Pokud není prefer_lang, nabídneme volbu (jen když máme oba názvy a liší se)
    if chosen_search_name is None:
        try:
            if en_title and en_title.strip() and en_title.strip().lower() != (cz_title or '').strip().lower():
                choice = xbmcgui.Dialog().select(
                    "Vyhledat podle názvu",
                    ["Česky: " + (cz_title or ''), "Anglicky: " + en_title]
                )
                if choice == 0:
                    chosen_search_name = cz_title or series_name
                    xbmc.log(f"[SeriesSearch] Volba uživatele: CZ → '{chosen_search_name}'", xbmc.LOGINFO)
                elif choice == 1:
                    chosen_search_name = en_title
                    xbmc.log(f"[SeriesSearch] Volba uživatele: EN → '{chosen_search_name}'", xbmc.LOGINFO)
                # choice < 0 -> zavřel dialog; necháme chosen_search_name = None (autodetekce níže)
        except Exception:
            # dialog nepodporován / jiná drobná chyba -> pokračujeme autodetekcí
            pass

    # 2. Určení primárního názvu pro vyhledávání (z TMDB dat)
    search_name_primary = chosen_search_name or _safe_search_name(data, series_name)
    xbmc.log(f"[SeriesSearch] Používám primární název pro vyhledávání: '{search_name_primary}' (původní: '{series_name}')", xbmc.LOGINFO)

    # 🟢 Sběr všech potenciálních názvů (Stejná logika jako v series_refresh, ale pro první hledání)
    search_queries = []
    
    # 1. Priorita: Zvolený název uživatelem / safe search name
    search_queries.append(search_name_primary) 

    # 2. Český název (title), pokud se liší a je platný
    if cz_title and cz_title != search_name_primary:
        search_queries.append(cz_title)
    
    # 3. Originální název (original_title / fallback_series_name), pokud se liší a je platný
    en_or_fallback = en_title if en_title else fallback_series_name
    if en_or_fallback and en_or_fallback != search_name_primary and en_or_fallback != cz_title:
        search_queries.append(en_or_fallback)

    # Odstranění duplicit a prázdných řetězců
    unique_queries = [q for q in dict.fromkeys(search_queries) if q] 
    xbmc.log(f"[SeriesSearch] Potenciální dotazy k testování: {unique_queries}", xbmc.LOGINFO)

    progress = xbmcgui.DialogProgress()
    progress.create('StreamManager', f'Vyhledávám seriál {search_name_primary}...')
    
    try:
        series_data = None
        found_any = False

        # 🟢 ITERACE PŘES VŠECHNY UNIKÁTNÍ NÁZVY
        for i, query in enumerate(unique_queries):
            progress_val = int((i + 1) / len(unique_queries) * 100)
            
            xbmc.log(f"[DEBUG SEARCH] Pokus {i+1}/{len(unique_queries)}: Hledám podle '{query}'", xbmc.LOGINFO)
            progress.update(progress_val, f'Vyhledávám seriál (pokus {i+1}/{len(unique_queries)}): {query}...')
            
            # Volání search_series. Pro první hledání není potřeba refresh=True.
            current_data = sm.search_series(
                query, 
                api, 
                token
            )
            
            if current_data and current_data.get('seasons'):
                series_data = current_data # Uložíme poslední úspěšně nalezená data
                found_any = True
                # Pokračujeme, aby se potenciálně našlo více streamů i pod jinými názvy

        if not series_data or not series_data['seasons']:
            progress.close()
            popinfo('Nenalezeny žádné epizody tohoto seriálu', icon=xbmcgui.NOTIFICATION_WARNING)
            xbmcplugin.endOfDirectory(handle, succeeded=False)
            return
        
        progress.close()
        
        # Zbytek logiky pro úspěšné nalezení
        dialog = xbmcgui.Dialog()
        message = (f"Seriál '{series_name}' byl úspěšně nalezen a uložen.\n\n"
                   f"Nalezeno {sum(len(season) for season in series_data['seasons'].values())} epizod "
                   f"v {len(series_data['seasons'])} sezónách.")
        dialog.ok(addon.getAddonInfo('name'), message)
        app_log("INFO", f"[SeriesSearch] '{series_name}' nalezeno: {sum(len(s) for s in series_data['seasons'].values())} epizod", module="series")
        
        # Container.Update jen pro ruční vyhledávání
        # Použijeme series_name, pod kterým je seriál uložen
        if not params.get('fallback_series_name'):
             xbmc.executebuiltin(f'Container.Update({get_url(action="series_detail", series_name=series_name)})')

    except Exception as e:
        progress.close()
        traceback.print_exc()
        popinfo(f'Chyba: {str(e)}', icon=xbmcgui.NOTIFICATION_ERROR)


def series_refresh_smart(params, addon, handle, profile, get_url, api, revalidate, popinfo):
    """
    Smart refresh:
    - Pass A: pouze UHD (2160p/4K/HDR)
    - Pass B: pouze HD (1080p + 720p), bez UHD
    - Pass C: name-only (SxxExx) cíleně pro chybějící epizody
    - Pass D (volitelně): široké S/E dotazy, pokud C nic nenašlo
    """
    series_name = params.get('series_name')
    if not series_name:
        xbmcplugin.endOfDirectory(handle, succeeded=False)
        return

    token = revalidate()
    sm = series_manager.SeriesManager(addon, profile)
    data = sm.load_series_data(series_name)
    if not data:
        popinfo(f"Seriál '{series_name}' není v databázi.", icon=xbmcgui.NOTIFICATION_WARNING)
        return

    # Připrav názvy a query (stejně jako v series_refresh)
    tmdb = data.get('tmdb', {}) if isinstance(data, dict) else {}
    orig_title = tmdb.get('original_title') or ''
    cz_title = tmdb.get('title') or series_name

    search_name_primary = _safe_search_name(data, series_name)
    search_queries = [search_name_primary]
    if cz_title and cz_title != search_name_primary:
        search_queries.append(cz_title)
    if orig_title and orig_title not in (search_name_primary, cz_title):
        search_queries.append(orig_title)

    unique_queries = [q for q in dict.fromkeys(search_queries) if q]
    xbmc.log(f"[SmartRefresh] Dotazy: {unique_queries}", xbmc.LOGINFO)

    progress = xbmcgui.DialogProgress()
    progress.create('StreamManager', f"Aktualizuji (smart) {series_name}…")

    try:
        found_any = False
        passes = [
            # (popis, kwargs pro search_series, max_streams)
            ("UHD (4K/HDR)", dict(refresh=True, only_uhd=True), 10),
            ("HD only (1080p + 720p)", dict(refresh=True, force_hd_only=True, skip_high_quality=True), 10),
            ("Name-only (SxxExx)", dict(refresh=True, name_only=True), 10),
        ]

        # Spustit A/B
        total_steps = len(passes) * max(1, len(unique_queries))
        step = 0

        for label, opts, max_streams in passes[:2]:  # A/B
            for i, q in enumerate(unique_queries):
                step += 1
                pct = int(step / total_steps * 100)
                progress.update(pct, f"{label}: {q}")
                xbmc.log(f"[SmartRefresh] {label} – hledám podle '{q}'", xbmc.LOGINFO)

                current_data = sm.search_series(
                    q, api, token,
                    **opts,
                    max_streams=max_streams
                )
                if current_data and current_data.get('seasons'):
                    found_any = True
                    data = current_data  # poslední snapshot

        # Pass C: name-only cíleně na chybějící epizody (běží i bez query iterace)
        step += 1
        pct = int(step / total_steps * 100)
        progress.update(pct, "Name-only (SxxExx): cílené chybějící epizody")
        xbmc.log("[SmartRefresh] Cílené name-only na chybějící epizody", xbmc.LOGINFO)

        current_data = sm.search_series(
            search_name_primary, api, token,
            refresh=True,
            name_only=True,
            max_streams=10
        )
        if current_data and current_data.get('seasons'):
            found_any = True
            data = current_data

        # Volitelný Pass D: široké S/E dotazy, pouze pokud nic nenašel Pass C
        # (Pokud chceš D vždy, odkomentuj níže.)
        if not found_any:
            step += 1
            pct = int(step / total_steps * 100)
            progress.update(pct, "Fallback: široké S/E dotazy")
            xbmc.log("[SmartRefresh] Fallback široké dotazy S/E", xbmc.LOGINFO)
            current_data = sm.search_series(
                search_name_primary, api, token,
                refresh=True,
                name_only=True,
                max_streams=10
            )
            if current_data and current_data.get('seasons'):
                found_any = True
                data = current_data

        if not found_any:
            progress.close()
            popinfo('Nenalezeny žádné epizody tohoto seriálu', icon=xbmcgui.NOTIFICATION_WARNING)
            xbmc.executebuiltin(f'Container.Update({get_url(action="series")})')
            return

        progress.close()
        episode_count = sum(len(season) for season in data.get('seasons', {}).values())
        popinfo(f"Smart aktualizace: {episode_count} epizod v {len(data.get('seasons', {}))} sezónách")
        xbmc.executebuiltin(f'Container.Update({get_url(action="series_detail", series_name=series_name)})')

    except Exception as e:
        progress.close()
        traceback.print_exc()
        popinfo(f"Chyba při smart aktualizaci: {e}", icon=xbmcgui.NOTIFICATION_ERROR)



# def series_search(params, addon, handle, profile, get_url, api, revalidate, popinfo):
#     """Search for a TV series and organize it into seasons and episodes"""
#     token = revalidate()
    
#     # 1. Získání názvů
#     series_name = params.get('what') or xbmcgui.Dialog().input("Zadejte název seriálu")
#     fallback_series_name = params.get('fallback_series_name') # ⬅️ NOVÝ ZÁLOŽNÍ NÁZEV
#     xbmc.log(f"[SeriesSearch DEBUG] Params received. Primary: '{series_name}', Fallback: '{fallback_series_name}'", xbmc.LOGINFO)

#     if not series_name:
#         xbmcplugin.endOfDirectory(handle, succeeded=False)
#         return

#     sm = series_manager.SeriesManager(addon, profile)
#     data = sm.load_series_data(series_name) or {}

#     # --- NOVÉ: nabídka volby, jestli hledat CZ nebo EN název ---
#     tmdb = data.get('tmdb', {}) if isinstance(data, dict) else {}
#     cz_title = tmdb.get('title') or (data.get('name') if isinstance(data, dict) else None) or series_name
#     en_title = tmdb.get('original_title')
#     if not en_title:
#         en_title = (params.get('fallback_series_name') or '').strip() or None

#     xbmc.log(f"[DEBUG SeriesSearch] CZ='{cz_title}' ENG='{en_title}'", xbmc.LOGINFO)

#     chosen_search_name = None  # výsledek volby uživatele (nebo parametru); None -> použije se autodetekce

#     # Pokud není prefer_lang, nabídneme volbu (jen když máme oba názvy a liší se)
#     if chosen_search_name is None:
#         try:
#             if en_title and en_title.strip() and en_title.strip().lower() != (cz_title or '').strip().lower():
#                 choice = xbmcgui.Dialog().select(
#                     "Vyhledat podle názvu",
#                     ["Česky: " + (cz_title or ''), "Anglicky: " + en_title]
#                 )
#                 if choice == 0:
#                     chosen_search_name = cz_title or series_name
#                     xbmc.log(f"[SeriesSearch] Volba uživatele: CZ → '{chosen_search_name}'", xbmc.LOGINFO)
#                 elif choice == 1:
#                     chosen_search_name = en_title
#                     xbmc.log(f"[SeriesSearch] Volba uživatele: EN → '{chosen_search_name}'", xbmc.LOGINFO)
#                 # choice < 0 -> zavřel dialog; necháme chosen_search_name = None (autodetekce níže)
#         except Exception:
#             # dialog nepodporován / jiná drobná chyba -> pokračujeme autodetekcí
#             pass

#     # 2. Určení primárního názvu pro vyhledávání (z TMDB dat)
#     search_name_primary = chosen_search_name or _safe_search_name(data, series_name)
#     xbmc.log(f"[SeriesSearch] Používám primární název pro vyhledávání: '{search_name_primary}' (původní: '{series_name}')", xbmc.LOGINFO)

#     progress = xbmcgui.DialogProgress()
#     progress.create('StreamManager', f'Vyhledávám seriál {search_name_primary}...')
#     try:
#         # POKUS 1: Primární vyhledávací název (CS název / bezpečný název)
#         xbmc.log(f"[DEBUG SEARCH] Hledám seriál: '{search_name_primary}' | Chosen: '{chosen_search_name}'", xbmc.LOGINFO)
#         series_data = sm.search_series(search_name_primary, api, token)
        
#         # POKUS 2: Záložní vyhledávací název (předaný EN název)
#         # Pouze pokud první pokus selhal A máme platný záložní název, který jsme ještě nezkoušeli.
#         if (not series_data or not series_data['seasons']) and fallback_series_name:
#             search_name_fallback = fallback_series_name
#             # Kontrola, zda už jsme náhodou nezkoušeli stejný název
#             if search_name_fallback != search_name_primary:
#                 xbmc.log(f"[SeriesSearch] Nenalezeno podle '{search_name_primary}', zkouším fallback: '{search_name_fallback}'", xbmc.LOGINFO)
#                 progress.update(50, f'Vyhledávám záložní název {search_name_fallback}...')
#                 series_data = sm.search_series(search_name_fallback, api, token)


#         if not series_data or not series_data['seasons']:
#             progress.close()
#             popinfo('Nenalezeny žádné epizody tohoto seriálu', icon=xbmcgui.NOTIFICATION_WARNING)
#             xbmcplugin.endOfDirectory(handle, succeeded=False)
#             return
        
#         progress.close()
#         # Zbytek logiky pro úspěšné nalezení
#         dialog = xbmcgui.Dialog()
#         message = (f"Seriál '{series_name}' byl úspěšně nalezen a uložen.\n\n"
#                    f"Nalezeno {sum(len(season) for season in series_data['seasons'].values())} epizod "
#                    f"v {len(series_data['seasons'])} sezónách.")
#         dialog.ok(addon.getAddonInfo('name'), message)
        
#         # Container.Update jen pro ruční vyhledávání
#         if not params.get('fallback_series_name'):
#              xbmc.executebuiltin(f'Container.Update({get_url(action="series_detail", series_name=series_name)})')

#     except Exception as e:
#         progress.close()
#         traceback.print_exc()
#         popinfo(f'Chyba: {str(e)}', icon=xbmcgui.NOTIFICATION_ERROR)

# ──────────────────────────────────────────────────────────────────────────────
# Hlavní menu Seriály
# ──────────────────────────────────────────────────────────────────────────────
def series_menu(params, addon, handle, profile, get_url, api, revalidate, popinfo):
    xbmcplugin.setPluginCategory(handle, addon.getAddonInfo('name') + " \\\\ Seriály")
    sm = series_manager.SeriesManager(addon, profile)

    # (1) Hledat nový seriál
    li_new = make_static_item(
        label="Hledat nový seriál",
        plot=("Vyhledá nový seriál na Webshare.\n"
              "Uloží data do Správce seriálů pro další práci."),
        addon=addon,
        icon_name='search.png',
        default_icon='DefaultAddonsSearch.png'
    )
    li_new.setLabel2("Vyhledá nový seriál • uloží do Správce")
    xbmcplugin.addDirectoryItem(handle, get_url(action='series_search'), li_new, False)

    # (1a) Zrušit stahování + stav fronty
    try:
        q = load_queue() or []
        downloading_count = sum(1 for i in q if (i or {}).get('status') == 'downloading')
        pending_count = sum(1 for i in q if (i or {}).get('status') == 'pending')
    except Exception:
        downloading_count = 0
        pending_count = 0

    status_tag = f"[COLOR lime]↓ {downloading_count}[/COLOR] • [COLOR gray]ve frontě: {pending_count}[/COLOR]"
    li_cancel = make_static_item(
        label=f"Zrušit stahování {status_tag}",
        plot=("Okamžitě zastaví všechny probíhající stahovací úlohy.\n"
              "Použijte pouze v případě potřeby."),
        addon=addon,
        icon_name='clear_queue.png',
        default_icon='DefaultScript.png'
    )
    li_cancel.setLabel2("Zastaví všechny stahovací úlohy")
    xbmcplugin.addDirectoryItem(handle, get_url(action='cancel_download'), li_cancel, False)

    # (2) Rozkoukané
    li_inp = make_static_item(
        label="Rozkoukané",
        plot=("Zobrazí uložené seriály, které ještě nejsou plně dosledované.\n"
              "Umožní rychlý vstup do sezón a epizod."),
        addon=addon,
        icon_name='progress.png',
        default_icon='DefaultRecentlyAddedEpisodes.png'
    )
    li_inp.setLabel2("Seznam nedosledovaných seriálů")
    xbmcplugin.addDirectoryItem(handle, get_url(action='series_filtered', status='in_progress'), li_inp, True)

    # (3) Dosledované
    li_comp = make_static_item(
        label="Dosledované",
        plot=("Zobrazí uložené seriály, které jsou označené jako plně zhlédnuté.\n"
              "Přehled hotových titulů."),
        addon=addon,
        icon_name='finish.png',
        default_icon='DefaultTvShows.png'
    )
    li_comp.setLabel2("Seznam plně dosledovaných seriálů")
    xbmcplugin.addDirectoryItem(handle, get_url(action='series_filtered', status='completed'), li_comp, True)
    
    li_next = make_static_item(
        label="Epizody ke sledování", # Použijte nový string pro "Následující epizody"
        plot=("Seznam přesně jedné epizody na řadě pro každý rozkoukaný seriál. "
              "Seřazeno podle toho, který seriál jste sledoval jako poslední."),
        addon=addon,
        icon_name='progress.png', 
        default_icon='DefaultRecentlyAddedEpisodes.png'
    )
    li_next.setLabel2("Seznam epizod na řadě (seřazený dle aktivity)")
    xbmcplugin.addDirectoryItem(handle, get_url(action='series_next_episodes'), li_next, True)

    try:
        xbmcplugin.setContent(handle, 'tvshows')
    except Exception:
        pass

    xbmcplugin.endOfDirectory(handle)

# ──────────────────────────────────────────────────────────────────────────────
# Detail seriálu (sezóny)
# ──────────────────────────────────────────────────────────────────────────────
def series_detail(params, addon, handle, profile, get_url, api, revalidate, popinfo):
    xbmcplugin.setPluginCategory(handle, addon.getAddonInfo('name') + " \\\\ " + params['series_name'])
    sm = series_manager.SeriesManager(addon, profile)
    series_manager.create_seasons_menu(sm, handle, params['series_name'], build_url_fn=get_url)

# ──────────────────────────────────────────────────────────────────────────────
# Detail sezóny (epizody)
# ──────────────────────────────────────────────────────────────────────────────
def series_season(params, addon, handle, profile, get_url, api, revalidate, popinfo):
    xbmcplugin.setPluginCategory(handle, addon.getAddonInfo('name') + " \\\\ " +
                                 params['series_name'] + f" \\\\ Sezona {params['season']}")
    sm = series_manager.SeriesManager(addon, profile)
    series_manager.create_episodes_menu(sm, handle, params['series_name'], params['season'], build_url_fn=get_url)

def create_next_episodes_menu(series_manager, handle, addon, get_url, profile):
    """
    Zobrazí seznam následujících epizod (Next-Up) pro všechny rozehrané seriály.
    Integruje logiku offline detekce, kontextového menu a stavu sledování.
    """
    import xbmcplugin
    import xbmcgui
    import xbmc
    import xbmcvfs
    import xbmcaddon
    import os
    
    # Předpokládáme, že tyto helpery jsou dostupné z importů na začátku series_ui.py nebo yawsp.py
    builder = get_url
    ADDON = xbmcaddon.Addon()
    color_tags = ADDON.getSetting('series_progress_color_tags') == 'true'

    # 1. Načtení dat (volání metody z series_manager.py)
    sm = series_manager
    next_episodes = sm.get_next_episodes_list()
    
    # 2. Nastavení kategorie a typu obsahu
    title = "Následující epizody" 
    xbmcplugin.setPluginCategory(handle, title)
    xbmcplugin.setContent(handle, 'episodes')
    
    # 3. Příprava pro offline detekci
    dfolder = ADDON.getSetting('dfolder') or ''
    base_real = None
    if dfolder:
        # POUŽITÍ KODI FUNKCE: Správná cesta k OFFLINE SLOŽCE
        base = xbmcvfs.translatePath(dfolder)
        if base and not base.endswith(('/', '\\')):
             base += '/'
        base_real = base

    # 4. Iterace přes nalezené Next-Up epizody
    for ep in next_episodes:
        series_name = ep['series_name']
        s_num = ep.get('season_num', 0)
        e_num = ep.get('episode_num', 0)
        
        # Načíst PLNOU DATA SERIÁLU pro získání TMDB metadat a overview (pro každou iteraci)
        series_data = sm.load_series_data(series_name)
        tm_series = series_data.get('tmdb', {}) if isinstance(series_data, dict) else {}
        
        series_poster = tm_series.get('poster_url')
        series_fanart = tm_series.get('backdrop_url')
        series_overview = tm_series.get('overview') or ''
        series_year = tm_series.get('year')
        show_runtime = tm_series.get('show_runtime')

        tm_ep = ep.get('tmdb', {}) if isinstance(ep, dict) else {}
        ep_title = ep.get('name') or ep.get('title') or ep.get('source_name') or ''
        ep_plot = tm_ep.get('overview') or series_overview
        ep_still = tm_ep.get('still_url') or tm_ep.get('still_path')
        ep_air = tm_ep.get('air_date')
        ep_runtime = tm_ep.get('runtime') or show_runtime

        # Stav sledování (OK / 75% / New)
        # 🚨 Důležité: _label_suffix_for_episode je metoda SeriesManagera
        suffix = series_manager._label_suffix_for_episode(ep, color_tags) 

        # OFFLINE DETEKCE + NALEZENÍ SKUTEČNÉHO SOUBORU (Logika z create_episodes_menu)
        local_file_path = None
        is_offline = False
        
        season_folder = None
        if base_real:
            # Použít clean_filename, který musí být dostupný v rámci series_ui.py
            # Musíte zajistit, že clean_filename je importován/dostupný!
            season_folder = f"{base_real}{clean_filename(series_name)}/Sezona {s_num}/" 

        if season_folder and path_exists(season_folder):
             try:
                 # Musíte zajistit, že listdir_safe je importován/dostupný!
                 files = listdir_safe(season_folder) 
                 patterns = []
                 if ep_title:
                     # Musíte zajistit, že clean_filename je importován/dostupný!
                     patterns.append(clean_filename(ep_title).lower())
                 patterns.append(f"s{int(s_num):02d}e{int(e_num):02d}".lower())
                 
                 for f in files:
                     name_noext = os.path.splitext(f)[0].lower()
                     if any(name_noext == p or name_noext.startswith(p) for p in patterns):
                         local_file_path = season_folder + f
                         is_offline = True
                         break
             except Exception as e:
                 xbmc.log(f"[NextEpisodes] Offline detekce selhala pro {series_name}: {e}", xbmc.LOGDEBUG)

        # Finální label s offline indikací
        offline_tag = " [COLOR limegreen]offline[/COLOR]" if is_offline else ""
        label = f"S{int(s_num):02d}E{int(e_num):02d} | {series_name} - {ep_title} {suffix}{offline_tag}"
        li = xbmcgui.ListItem(label=label)

        # Artwork
        art = {}
        if ep_still:
            art.update({'thumb': ep_still, 'icon': ep_still})
        else:
            art['icon'] = 'DefaultVideo.png'
        if is_offline:
            art['thumb'] = 'DefaultHardDisk.png'
        if series_fanart:
            art['fanart'] = series_fanart
        if series_poster:
            art['poster'] = series_poster
        li.setArt(art)

        # InfoLabels
        info = {
            'title': ep_title or f"Epizoda {e_num}",
            'tvshowtitle': series_name, # Přidáno
            'plot': ep_plot,
            'aired': ep_air or '',
            'season': int(s_num),
            'episode': int(e_num),
            'mediatype': 'episode'
        }
        if isinstance(ep_runtime, int) and ep_runtime > 0:
            info['duration'] = ep_runtime
        if isinstance(series_year, int):
            info['year'] = series_year
        li.setInfo('video', info)

        # Kontextové menu (přesně jako v create_episodes_menu)
        cm = [
            ("Označit jako zhlédnuté",
             f"RunPlugin({builder(action='series_mark_episode', series_name=series_name, season=s_num, episode=e_num, progress=100)})"),
            ("Označit jako rozkoukané",
             f"RunPlugin({builder(action='series_mark_episode', series_name=series_name, season=s_num, episode=e_num, progress=75)})"),
            ("Vymazat stav",
             f"RunPlugin({builder(action='series_mark_episode', series_name=series_name, season=s_num, episode=e_num, progress=0)})"),
            ("Smazat epizodu",
             f"RunPlugin({builder(action='series_remove_episode', series_name=series_name, season=s_num, episode=e_num)})"),
            ("Přidat do fronty",
             f"RunPlugin({builder(action='series_download_episode', series_name=series_name, season=s_num, episode=e_num)});Container.Refresh"),
            ("Spustit frontu",
             f"RunPlugin({builder(action='queue_start_all')})"),
        ]
        if tm_series.get('id'):
            cm.append(
                ("Odeslat na Trakt…",
                 f"RunPlugin({builder(action='series_trakt_mark_episode_prompt', tmdb_id=int(tm_series.get('id')), season=int(s_num), episode=int(e_num))})")
            )
        li.addContextMenuItems(cm, replaceItems=True)

        # HLAVNÍ LOGIKA: offline → přehrává rovnou, online → výběr streamu
        if is_offline and local_file_path:
            li.setProperty('IsPlayable', 'true')
            xbmcplugin.addDirectoryItem(handle, local_file_path, li, isFolder=False)
        else:
            li.setProperty('IsPlayable', 'false')
            url = builder(
                action='play', # Zde používáme 'series_episode_play' z routeru, ne 'play'
                series_name=series_name,
                season=str(s_num),
                episode=str(e_num),
                select='1' # Pro přeskočení menu streamů, pokud je jen jeden (stejně jako v 'play')
            )
            xbmcplugin.addDirectoryItem(handle, url, li, isFolder=False) 

    # Nastavení zobrazení a konec adresáře
    # 🚨 Důležité: NEVYNUCOVAT ViewMode, ale pouze nastavit typ obsahu.
    xbmcplugin.endOfDirectory(handle)

# ──────────────────────────────────────────────────────────────────────────────
# Odstranění seriálu
# ──────────────────────────────────────────────────────────────────────────────
def series_remove(params, addon, handle, profile, get_url, api, revalidate, popinfo):
    series_name = params['series_name']
    status = (params.get('status') or '').strip().lower()
    dialog = xbmcgui.Dialog()
    win = xbmcgui.Window(10000)
    guard_key = 'mm_last_series_removed'
    last_removed = win.getProperty(guard_key)
    if not dialog.yesno('StreamManager', f'Opravdu chcete odebrat seriál "{series_name}"?'):
        return
    sm = series_manager.SeriesManager(addon, profile)
    ok = sm.remove_series(series_name)
    if ok:
        if last_removed != series_name:
            xbmcgui.Dialog().notification(addon.getAddonInfo('name'), 'Seriál odstraněn.', xbmcgui.NOTIFICATION_INFO, 2500)
            win.setProperty(guard_key, series_name)
    else:
        xbmcgui.Dialog().notification(addon.getAddonInfo('name'), 'Nelze odstranit seriál.', xbmcgui.NOTIFICATION_WARNING, 2500)
    try:
        if status in ('in_progress', 'completed'):
            xbmc.executebuiltin(f'Container.Update({get_url(action="series_filtered", status=status)})')
        else:
            xbmc.executebuiltin(f'Container.Update({get_url(action="series")})')
    except Exception:
        xbmc.executebuiltin(f'Container.Update({get_url(action="series")})')

# ──────────────────────────────────────────────────────────────────────────────
# Odstranění sezóny
# ──────────────────────────────────────────────────────────────────────────────
def series_remove_season(params, addon, handle, profile, get_url, api, revalidate, popinfo):
    series_name = params.get('series_name')
    season_key = params.get('season')
    if not series_name or not season_key:
        xbmcgui.Dialog().notification(addon.getAddonInfo('name'), "Chybí parametry", xbmcgui.NOTIFICATION_ERROR)
        return
    sm = series_manager.SeriesManager(addon, profile)
    data = sm.load_series_data(series_name)
    if not data:
        xbmcgui.Dialog().notification(addon.getAddonInfo('name'), "Data seriálu nenalezena", xbmcgui.NOTIFICATION_ERROR)
        return
    if not xbmcgui.Dialog().yesno("StreamManager", f"Opravdu smazat sezónu {season_key} ze seriálu '{series_name}'?"):
        return
    try:
        if season_key in data.get('seasons', {}):
            del data['seasons'][season_key]
        if season_key in data.get('season_state', {}):
            del data['season_state'][season_key]
        safe_name = sm._safe_filename(series_name, data.get('tmdb', {}).get('id'))
        file_path = os.path.join(sm.series_db_path, f"{safe_name}.json")
        series_manager._atomic_write_json(file_path, data)
        xbmcgui.Dialog().notification(addon.getAddonInfo('name'), f"Sezóna {season_key} odstraněna", xbmcgui.NOTIFICATION_INFO, 2500)
    except Exception as e:
        xbmc.log(f"[SeriesRemoveSeason] CHYBA: {e}", xbmc.LOGERROR)
        xbmcgui.Dialog().notification(addon.getAddonInfo('name'), "Chyba při mazání sezóny", xbmcgui.NOTIFICATION_ERROR)
    xbmc.executebuiltin(f'Container.Update({get_url(action="series_detail", series_name=series_name)})')

# ──────────────────────────────────────────────────────────────────────────────
# Odstranění epizody
# ──────────────────────────────────────────────────────────────────────────────
def series_remove_episode(params, addon, handle, profile, get_url, api, revalidate, popinfo):
    series_name = params.get('series_name')
    season_key = str(params.get('season'))
    episode_key = str(params.get('episode'))
    if not series_name or not season_key or not episode_key:
        xbmcgui.Dialog().notification(addon.getAddonInfo('name'), "Chybí parametry", xbmcgui.NOTIFICATION_ERROR)
        return
    sm = series_manager.SeriesManager(addon, profile)
    data = sm.load_series_data(series_name)
    if not data:
        xbmcgui.Dialog().notification(addon.getAddonInfo('name'), "Data seriálu nenalezena", xbmcgui.NOTIFICATION_ERROR)
        return
    if not xbmcgui.Dialog().yesno("StreamManager",
                                  f"Opravdu smazat epizodu S{season_key}E{episode_key} ze seriálu '{series_name}'?"):
        return
    try:
        if season_key in data.get('seasons', {}) and episode_key in data['seasons'][season_key]:
            del data['seasons'][season_key][episode_key]
        sm._recalc_season_state(data, season_key)
        safe_name = sm._safe_filename(series_name, data.get('tmdb', {}).get('id'))
        file_path = os.path.join(sm.series_db_path, f"{safe_name}.json")
        series_manager._atomic_write_json(file_path, data)
        xbmcgui.Dialog().notification(addon.getAddonInfo('name'),
                                      f"Epizoda S{season_key}E{episode_key} odstraněna",
                                      xbmcgui.NOTIFICATION_INFO, 2500)
    except Exception as e:
        xbmc.log(f"[SeriesRemoveEpisode] CHYBA: {e}", xbmc.LOGERROR)
        xbmcgui.Dialog().notification(addon.getAddonInfo('name'), "Chyba při mazání epizody", xbmcgui.NOTIFICATION_ERROR)
    xbmc.executebuiltin(
        f'Container.Update({get_url(action="series_season", series_name=series_name, season=season_key)})'
    )
