
# -*- coding: utf-8 -*-
# Module: series_manager
# Author: user extension (+ curated streams/TMDb enrichment by mm)
# Updated: 2025-11-27 (refactor: clean_filename/path_exists/listdir_safe from ui_utils)
# License: AGPL v.3 https://www.gnu.org/licenses/agpl-3.0.html

import os
import io
import re
import sys
import json
import shutil
import tempfile
import xbmc
import xbmcaddon
import xbmcgui
import xml.etree.ElementTree as ET
import xbmcplugin
import traceback
import datetime
from resources.lib.utils.ui_utils import make_static_item
from resources.lib.utils.ui_utils import clean_filename, path_exists, listdir_safe
from urllib.parse import quote
from typing import List, Dict, Any
import unicodedata

# Konstanty pro uložení stavu
PROP_SEASON_FILTER_BASE = '__YAWSP_SEASON_FILTER_{}__' 
WINDOW = xbmcgui.Window(xbmcgui.getCurrentWindowId())

# Helper: bezpečný builder URL, pokud nepředáš get_url z yawsp.py
def _default_build_url(**kwargs):
    """
    Fallback builder plugin URL, když create_*_menu není volána s build_url_fn.
    Použije sys.argv[0] jako base a urlencode s UTF-8.
    """
    try:
        import sys
        from urllib.parse import urlencode
        base = sys.argv[0]
        return "{}?{}".format(base, urlencode(kwargs, encoding="utf-8"))
    except Exception:
        # poslední možnost – minimální kompatibilita
        return "plugin://plugin.video.mmirousek_v2/?" + "&".join(
            [f"{k}={v}" for k, v in kwargs.items()]
        )

# --- Atomický zápis JSON ------------------------------------------------------
def _atomic_write_json(file_path: str, data: dict) -> None:
    """
    Zapíše JSON atomicky: nejdřív do temp souboru, pak rename.
    Zajišťuje odolnost proti souběžnému zápisu.
    """
    try:
        dir_name = os.path.dirname(file_path)
        if dir_name and not os.path.exists(dir_name):
            os.makedirs(dir_name)
        fd, tmp_path = tempfile.mkstemp(dir=dir_name, prefix="tmp_", suffix=".json")
        with os.fdopen(fd, 'w', encoding='utf-8') as f:
            json.dump(data, f, ensure_ascii=False, indent=2)
        shutil.move(tmp_path, file_path)
    except Exception as e:
        xbmc.log(f'YaWSP SeriesManager: Chyba atomického zápisu: {e}', xbmc.LOGERROR)

def format_date_cz(date_str: str) -> str:
    """
    Převádí 'YYYY-MM-DD' nebo 'YYYY-MM-DDTHH:MM:SS' na 'dd.mm.yyyy'.
    Při chybě vrátí původní řetězec.
    """
    if not isinstance(date_str, str) or len(date_str) < 10:
        return date_str
    try:
        # vezmeme prvních 10 znaků (YYYY-MM-DD)
        y, m, d = date_str[:10].split('-')
        return f"{int(d):02d}.{int(m):02d}.{int(y):04d}"
    except Exception:
        return date_str

def _derive_latest_episode_timestamp(series_data: dict) -> str:
    """
    Vrátí nejnovější 'last_watched_at' napříč epizodami daného seriálu
    ve formátu 'YYYY-MM-DDTHH:MM:SS'. Pokud nic nenajde, vrátí ''.
    """
    if not isinstance(series_data, dict):
        return ''

    seasons = series_data.get('seasons') or {}
    latest = ''

    for season in seasons.values():
        if not isinstance(season, dict):
            continue
        for ep in season.values():
            if not isinstance(ep, dict):
                continue
            ts = (ep.get('last_watched_at') or '').strip()
            if not ts:
                continue
            # Lexikografické porovnání funguje pro formát YYYY-MM-DDTHH:MM:SS
            if not latest or ts > latest:
                latest = ts

    return latest

# --- Pomocník: převod features na stream_info (video + audio) ---
def _features_to_stream_info(features: list, runtime_min: int = None):
    """
    Ze seznamu 'features' (např. ['2160p','HEVC','HDR10','Atmos','5.1']) složí
    stream info pro video + audio. runtime_min (minuty) se převede na sekundy.
    Vrací tuple (video_info_dict, audio_info_dict).
    """
    feats = [str(f).lower() for f in (features or [])]

    # Rozlišení
    width, height = 0, 0
    if '2160p' in feats or '4k' in feats:
        width, height = 3840, 2160
    elif '1080p' in feats:
        width, height = 1920, 1080
    elif '720p' in feats:
        width, height = 1280, 720

    # Video codec + HDR
    vcodec = 'h265' if ('hevc' in feats or 'x265' in feats) else ('h264' if ('h.264' in feats or 'x264' in feats) else '')
    hdrtype = 'dv' if 'dv' in feats else ('hdr10' if 'hdr10' in feats else ('hdr' if 'hdr' in feats else ''))
    aspect = 1.78  # bezpečný default 16:9
    duration = int(runtime_min) * 60 if isinstance(runtime_min, int) and runtime_min > 0 else 0

    video_info = {
        'codec': vcodec,
        'width': width,
        'height': height,
        'aspect': aspect,
        'duration': duration,
        'hdrtype': hdrtype
    }

    # Audio codec + kanály
    channels = 2
    if any('5.1' in f for f in feats): channels = 6
    if any('7.1' in f for f in feats): channels = 8

    acodec = ''
    if 'atmos' in feats:        acodec = 'eac3'   # rozumný default
    elif 'dts' in feats:        acodec = 'dts'
    elif 'dolby' in feats or 'ac3' in feats: acodec = 'ac3'
    elif 'aac' in feats:        acodec = 'aac'

    audio_info = {
        'codec': acodec,
        'channels': channels,
        # language můžeš doplnit z tvých tagů (CZ/EN), pokud ji parsuješ
    }
    return video_info, audio_info
       
# --- Py2/3 kompatibilní urlencode --------------------------------------------
try:
    from urllib.parse import urlencode
except ImportError:
    from urllib import urlencode  # Py2 fallback (Kodi Leia)

# --- Robustnější detekce S/E (CZ-friendly) -----------------------------------
RE_SxxEyy = re.compile(r'(?i)\bS(\d{1,2})\s*E(\d{1,3})\b')
RE_1x02   = re.compile(r'(?i)\b(\d{1,2})\s*[x×]\s*(\d{1,3})\b')
RE_CZ     = re.compile(r'(?i)\b(sezona|série|řada)\s*(\d{1,2}).*\b(díl|epizoda|ep)\s*(\d{1,3})\b')
RE_EPONLY = re.compile(r'(?i)\b(ep|epizoda)\s*(\d{1,3})\b')
SEASON_PACK_HINTS = re.compile(r'(?i)\b(pack|complete|season\s*pack|s\d{1,2}-s\d{1,2})\b')

PREF_LANG_ORDER = ("cs", "cz", "sk", "czsk", "en", "eng")

def _norm(s):
    return (s or "").strip()

def _light_clean(title):
    """Lehké očištění: ponechá SxxEyy/1x02, odstraní šum typu rozlišení/kodek apod."""
    if not title:
        return ""
    s = title.replace('.', ' ')
    s = re.sub(
        r'(?i)\b(480p|720p|1080p|2160p|4k|webrip|web[\- ]?dl|b[dr]rip|bluray|remux|hdtv|hdr|dv|x264|x265|hevc)\b',
        '', s
    )
    s = re.sub(r'\s{2,}', ' ', s).strip(" -.()[]")
    return s


class SeriesManager:
    def __init__(self, addon, profile):
        self.addon = addon
        self.profile = profile
        self.series_db_path = os.path.join(profile, 'series_db')
        self.ensure_db_exists()
        self.page_size = 100
        self.max_pages = 5  # dle dohody

    # --- NOVÉ/UPRAVENÉ METODY ROBUSTNÍHO NAČÍTÁNÍ ----------------------------
    
    def get_next_episodes_list(self) -> List[Dict[str, Any]]:
        """
        Vrátí seznam následujících (přesně jednu) epizod pro všechny rozkoukané seriály, 
        seřazený dle posledního sledování (last_watched_at_utc).
        """
        import xbmc
        PREFIX = "[NextEpisodes]"
        xbmc.log(f"{PREFIX} Spouštím hledání následujících epizod.", xbmc.LOGINFO)

        # 1. Načíst data VŠECH seriálů (vzor pro procházení a načítání dat)
        series_name_list = self.get_all_series() 
        xbmc.log(f"{PREFIX} Krok 1: Načteno {len(series_name_list)} seriálů (objektů z DB).", xbmc.LOGINFO)
        
        all_series_data = []
        for s in series_name_list:
            series_name = s['name']
            data = self.load_series_data(series_name)
            
            # Předfiltrujeme jen ty, které mají sezóny
            if data and data.get('seasons'): 
                data['name'] = series_name
                all_series_data.append(data)
            elif not data:
                xbmc.log(f"{PREFIX} SKIP: {series_name} - Data seriálu nenalezena/prázdná.", xbmc.LOGDEBUG)
            elif not data.get('seasons'):
                xbmc.log(f"{PREFIX} SKIP: {series_name} - Chybí klíč 'seasons'.", xbmc.LOGDEBUG)

        xbmc.log(f"{PREFIX} Krok 2: Zůstalo {len(all_series_data)} seriálů s načtenými sezónami.", xbmc.LOGINFO)
        
        # 2. Filtrace POUZE rozkoukaných seriálů s platným časem


        in_progress_series = []
        for d in all_series_data:
            series_name = d['name']
            is_in_progress = not d.get('completed', False)

            # 1) Kořenové timestampy (Trakt / hypotetický lokální)
            last_utc   = d.get('last_watched_at_utc')
            last_local = d.get('last_watched_at')  # na kořeni zpravidla není

            # 2) Fallback: odvodit z epizod (nejnovější ep['last_watched_at'])
            derived_ts = _derive_latest_episode_timestamp(d)

            # 3) Výběr timestampu pro filtr/řazení (priorita: UTC > local > derived)
            ts_for_filter = last_utc or last_local or derived_ts

            if is_in_progress and ts_for_filter:
                d['_nextup_sort_key'] = ts_for_filter
                in_progress_series.append(d)
                xbmc.log(
                    f"[NextEpisodes] FILTER OK: {series_name} "
                    f"(timestamp={ts_for_filter}, source="
                    f"{'utc' if last_utc else ('local' if last_local else 'derived')})",
                    xbmc.LOGDEBUG
                )
            else:
                xbmc.log(
                    f"[NextEpisodes] FILTER FAIL: {series_name} - completed={d.get('completed')}, "
                    f"UTC={bool(last_utc)}, ROOT_LOCAL={bool(last_local)}, DERIVED={bool(derived_ts)}",
                    xbmc.LOGDEBUG
                )

        
        # Řazení: Využíváme klíč pro sestupné řazení (nejnovější nahoře)
        
        in_progress_series.sort(
            key=lambda x: x.get('_nextup_sort_key', '1970'),
            reverse=True)

        next_episodes = []

        # 3. Hledání první nezhlédnuté epizody (Next-up)
        for data in in_progress_series:
            series_name = data['name']
            found_ep = None
            
            # Změna: Aby se zamezilo chybě, kde se sezóny/epizody jmenují "speciální", používáme robustnější řazení
            sorted_seasons = sorted(data.get('seasons', {}).items(), key=lambda item: int(item[0]) if str(item[0]).isdigit() else 9999)

            xbmc.log(f"{PREFIX} Krok 4: Hledám Next-up epizodu pro: {series_name}", xbmc.LOGDEBUG)

            for s_key, season_data in sorted_seasons:
                if not season_data: continue

                sorted_episodes = sorted(season_data.items(), key=lambda item: int(item[0]) if str(item[0]).isdigit() else 9999)

                for e_key, ep in sorted_episodes:
                    
                    if not ep.get('watched', False):
                        found_ep = ep
                        
                        # Obohacení dat epizody o metadata pro UI
                        ep_data = found_ep.copy() 
                        ep_data['series_name'] = series_name
                        ep_data['series_tmdb_id'] = data.get('tmdb_id')
                        ep_data['last_watched_at_utc'] = data.get('last_watched_at_utc')
                        ep_data['season_num'] = int(s_key) if str(s_key).isdigit() else s_key
                        ep_data['episode_num'] = int(e_key) if str(e_key).isdigit() else e_key
                        
                        next_episodes.append(ep_data)
                        xbmc.log(f"{PREFIX} NALEZENO: {series_name} -> S{s_key}E{e_key} ({ep_data.get('title')})", xbmc.LOGINFO)
                        break 
                
                if found_ep: break # Nalezena epizoda, ukončit procházení sezón
            
            if not found_ep:
                xbmc.log(f"{PREFIX} CHYBA: Pro seriál {series_name} nebyly nalezeny žádné nezhlédnuté epizody, i když je 'in_progress'.", xbmc.LOGWARNING)

        xbmc.log(f"{PREFIX} Krok 5: CELKEM NALEZENO: {len(next_episodes)} následujících epizod. Končím.", xbmc.LOGINFO)
        return next_episodes

    def validate_series_streams(self, series_name, api_fn, token):
        """
        Validuje dostupnost všech streamů v seriálu přes Webshare API.
        Odstraní neplatné streamy z JSON, vrátí seznam epizod pro rescan.
        """
        import time
        import xbmcgui
        from xml.etree import ElementTree as ET

        data = self.load_series_data(series_name)
        if not data:
            return {"ok": 0, "invalid": 0, "errors": [], "rescan": []}

        ok_count, invalid_count = 0, 0
        errors = []
        rescan_eps = []

        dlg = xbmcgui.DialogProgress()
        dlg.create("Kontrola streamů", f"Seriál: {series_name}")
        total_streams = sum(len(ep.get("streams", [])) for s in data.get("seasons", {}).values() for ep in s.values())
        checked = 0

        for s_key, season in (data.get("seasons") or {}).items():
            for e_key, ep in (season or {}).items():
                streams = ep.get("streams", [])
                valid_streams = []
                for st in streams:
                    if dlg.iscanceled():
                        dlg.close()
                        return {"ok": ok_count, "invalid": invalid_count, "errors": errors, "rescan": rescan_eps}

                    ident = st.get("ident")
                    checked += 1
                    progress_value = int((checked / total_streams) * 100)
                    line1 = f"S{int(s_key):02d}E{int(e_key):02d}"
                    line2 = f"Zkontrolováno: {checked}/{total_streams}"

                    # Kodi 21 podporuje více řádků, fallback pro starší verze
                    try:
                        dlg.update(progress_value, line1, line2)
                    except TypeError:
                        dlg.update(progress_value)

                    try:
                        resp = api_fn("file_info", {"ident": ident, "wst": token})
                        xml = ET.fromstring(resp.content)
                        if xml.findtext("status") == "OK":
                            ok_count += 1
                            valid_streams.append(st)
                        else:
                            invalid_count += 1
                            tag = f"S{int(s_key):02d}E{int(e_key):02d}"
                            if tag not in errors:
                                errors.append(tag)
                    except Exception:
                        invalid_count += 1
                        tag = f"S{int(s_key):02d}E{int(e_key):02d}"
                        if tag not in errors:
                            errors.append(tag)

                    time.sleep(0.17)  # ~6 req/s

                if len(valid_streams) != len(streams):
                    ep["streams"] = valid_streams
                    rescan_eps.append((s_key, e_key))

        # Atomický zápis JSON
        safe_name = self._safe_filename(series_name, data.get("tmdb", {}).get("id"))
        file_path = os.path.join(self.series_db_path, f"{safe_name}.json")
        _atomic_write_json(file_path, data)

        dlg.close()
        return {"ok": ok_count, "invalid": invalid_count, "errors": errors, "rescan": rescan_eps}


    def _resolve_series_file_path(self, series_name_or_path: str) -> str:
        """
        Robustně určí cestu k JSON souboru seriálu.
        Přijme:
        - plnou cestu k .json (vrátí ji, pokud existuje),
        - název seriálu (CZ/EN),
        - 'tmdb_12345' (safe name z filename).
        Vrátí absolutní path nebo None.
        """
        if not isinstance(series_name_or_path, str) or not series_name_or_path.strip():
            return None
        key = series_name_or_path.strip()

        # 1) Je to přímo path k JSON souboru?
        if key.endswith('.json') and (os.path.sep in key):
            return key if os.path.exists(key) else None

        # 2) Je to safe name z filename? (např. 'tmdb_12345')
        if key.startswith('tmdb_'):
            try:
                tmdb_id = int(key[5:])
                fp = os.path.join(self.series_db_path, f"tmdb_{tmdb_id}.json")
                if os.path.exists(fp):
                    return fp
            except Exception:
                pass  # pokračuj dál

        # 3) Slug podle názvu seriálu (bez TMDb)
        slug = self._safe_filename(key)
        fp_slug = os.path.join(self.series_db_path, f"{slug}.json")
        if os.path.exists(fp_slug):
            return fp_slug

        # 4) Najdi odpovídající tmdb_<id>.json podle obsahu (name)
        try:
            for fn in os.listdir(self.series_db_path):
                if not fn.startswith('tmdb_') or not fn.endswith('.json'):
                    continue
                fp = os.path.join(self.series_db_path, fn)
                try:
                    with io.open(fp, 'r', encoding='utf8') as f:
                        d = json.load(f)
                    if not isinstance(d, dict):
                        continue
                    nm = (d.get('name') or '').strip().lower()
                    if nm and nm == key.strip().lower():
                        return fp
                except Exception:
                    continue
        except Exception:
            pass
        return None

    def load_series_data(self, series_name_or_path):
        """
        Načte data seriálu — název i path jsou podporované.
        Preferuje soubor tmdb_<id>.json, pokud existuje.
        """
        file_path = self._resolve_series_file_path(series_name_or_path)
        if not file_path or not os.path.exists(file_path):
            return None
        try:
            with io.open(file_path, 'r', encoding='utf8') as f:
                data = f.read()
            series_data = json.loads(data)  # Py3-safe
            return self._migrate_schema(series_data)
        except Exception as e:
            xbmc.log(f'YaWSP Series Manager: Error loading series data: {e}', level=xbmc.LOGERROR)
            return None

    def _get_tmdb_id(self, series_name: str) -> int:
        """
        Vrátí TMDb ID pro daný seriál — robustně.
        Zkusí načíst data přes _resolve_series_file_path a přečte tmdb.id.
        Pokud nenajde, vrátí None.
        """
        try:
            # 1) Zkus najít soubor podle názvu/safe name
            fp = self._resolve_series_file_path(series_name)
            if fp and os.path.exists(fp):
                with io.open(fp, 'r', encoding='utf8') as f:
                    d = json.load(f)
                if isinstance(d, dict):
                    tid = (d.get("tmdb") or {}).get("id")
                    if isinstance(tid, int) and tid > 0:
                        return tid

            # 2) Fallback: projdi tmdb_* soubory a porovnej name
            for fn in os.listdir(self.series_db_path):
                if not fn.startswith('tmdb_') or not fn.endswith('.json'):
                    continue
                try:
                    with io.open(os.path.join(self.series_db_path, fn), 'r', encoding='utf8') as f:
                        d = json.load(f)
                    if not isinstance(d, dict):
                        continue
                    nm = (d.get('name') or '').strip().lower()
                    if nm and nm == (series_name or '').strip().lower():
                        tid = (d.get("tmdb") or {}).get("id")
                        if isinstance(tid, int) and tid > 0:
                            return tid
                except Exception:
                    continue
            return None
        except Exception:
            return None

    def mark_episode_progress(self, series_name: str, season: int, episode: int, progress: float,
                              set_by: str = 'manual', watched_at: str = None) -> None:
        """Označí jednu epizodu jako zhlédnutou/rozkoukanou nebo resetuje stav."""
        xbmc.log(f"[SeriesManager] DEBUG: {series_name} - Progress: {progress}, Watched_At Arg: {watched_at}", xbmc.LOGINFO) # 🚨 PŘIDAT TENTO ŘÁDEK
        try:
            data = self.load_series_data(series_name)
            if not data:
                xbmc.log(f"[SeriesManager] mark_episode_progress: Data nenalezena pro {series_name}", xbmc.LOGWARNING)
                return

            s_key, e_key = str(season), str(episode)
            ep = data.get('seasons', {}).get(s_key, {}).get(e_key)
            if not ep:
                xbmc.log(f"[SeriesManager] mark_episode_progress: Epizoda {s_key}/{e_key} nenalezena", xbmc.LOGWARNING)
                return

            ep['progress'] = round(progress, 1)
            ep['watched'] = progress >= 90.0
            if progress > 0.0 and watched_at:
                ep['watched_at'] = watched_at
                ep['set_by'] = set_by
            else:
                # Pokud je progress 0.0 (reset), smazat záznam
                ep.pop('watched_at', None)
                # set_by necháme, pokud by to bylo relevantní i pro 0%

            if ep['watched']:
                ep['play_count'] = ep.get('play_count', 0) + 1
                ep['last_watched_at'] = xbmc.getInfoLabel('System.Date') + 'T' + xbmc.getInfoLabel('System.Time')
                ep['set_by'] = set_by
                

            # Přepočítat stav sezóny
            self._recalc_season_state(data, s_key)
            if watched_at:
                data['last_watched_at_utc'] = watched_at
    
            # 🚨 ZDE MUSÍ BÝT VOLÁNÍ PRO ULOŽENÍ DAT NA DISK
            self._save_series_data(series_name, data)

            # Uložit změny
            safe_name = self._safe_filename(series_name, data.get('tmdb', {}).get('id'))
            file_path = os.path.join(self.series_db_path, f"{safe_name}.json")
            _atomic_write_json(file_path, data)

            xbmc.log(f"[SeriesManager] mark_episode_progress: {series_name} S{s_key}E{e_key} -> {progress}%", xbmc.LOGINFO)
        except Exception as e:
            xbmc.log(f"[SeriesManager] CHYBA mark_episode_progress: {e}", xbmc.LOGERROR)

    def recalc_and_save_season_state(self, data: dict, season_key: str):
        """
        Přepočítá stav sezóny a uloží změny do souboru.
        """
        try:
            # Přepočítat stav sezóny
            self._recalc_season_state(data, season_key)

            # Uložit změny atomicky
            safe_name = self._safe_filename(data.get('name'), data.get('tmdb', {}).get('id'))
            file_path = os.path.join(self.series_db_path, f"{safe_name}.json")
            _atomic_write_json(file_path, data)

            xbmc.log(f"[SeriesManager] Season {season_key} state recalculated and saved", xbmc.LOGINFO)
        except Exception as e:
            xbmc.log(f"[SeriesManager] CHYBA recalc_and_save_season_state: {e}", xbmc.LOGERROR)

    # --- Přepočet stavu sezóny ------------------------------------------------
    def _recalc_season_state(self, data: dict, season_key: str) -> None:
        """
        Spočítá watched_count, total a watched flag pro sezónu.
        Uloží do data['season_state'][season_key].
        """
        try:
            season = data.get('seasons', {}).get(season_key, {})
            total = len(season)
            watched_count = sum(1 for ep in season.values() if ep.get('watched'))
            watched_flag = (watched_count == total and total > 0)

            season_state = data.setdefault('season_state', {})
            season_state[season_key] = {
                'watched': watched_flag,
                'watched_count': watched_count,
                'total': total,
                'last_updated': xbmc.getInfoLabel('System.Date')
            }
        except Exception as e:
            xbmc.log(f'YaWSP SeriesManager: _recalc_season_state CHYBA: {e}', xbmc.LOGERROR)

    def _label_suffix_for_episode(self, ep: dict, color: bool) -> str:
        """
        Vrátí suffix pro stav epizody: [OK], [75%], [New] s/bez barvy.
        """
        try:
            progress = float(ep.get('progress', 0.0) or 0.0)
            if ep.get('watched'):
                return '[COLOR lime]ok[/COLOR]' if color else '[ok]'
            elif progress > 5.0:
                p = int(round(progress))
                return f"[COLOR lime]{p}%[/COLOR]" if color else f"[{p}%]"
            else:
                return '[COLOR dimgray]New[/COLOR]' if color else '[New]'
        except Exception:
            return ''

    # --- FS zajištění ---------------------------------------------------------
    def ensure_db_exists(self):
        """Ensure that the series database directory exists"""
        try:
            if not os.path.exists(self.profile):
                os.makedirs(self.profile)
            if not os.path.exists(self.series_db_path):
                os.makedirs(self.series_db_path)
        except Exception as e:
            xbmc.log(f'YaWSP Series Manager: Error creating directories: {e}', level=xbmc.LOGERROR)

    # --- Vyhledávání + kurátorování + TMDb enrichment -------------------------
    def search_series(self, series_name, api_function, token, **kwargs):
        refresh = bool(kwargs.get('refresh', False)) # Získat flag z kwargs (series_refresh)
        # --- režimové přepínače (kvůli quality režimům) ---
        skip_high_quality = bool(kwargs.get('skip_high_quality', False))
        force_hd_only     = bool(kwargs.get('force_hd_only', False))
        name_only         = bool(kwargs.get('name_only', False))
        only_uhd          = bool(kwargs.get('only_uhd', False))

        if skip_high_quality:
            xbmc.log("[SeriesManager] skip_high_quality=True → kurátorování bez 4K", xbmc.LOGINFO)
        if force_hd_only:
            xbmc.log("[SeriesManager] force_hd_only=True → pouze 1080p/720p", xbmc.LOGINFO)
        if name_only:
            xbmc.log("[SeriesManager] name_only=True → ignoruji kvalitu, zachovám pořadí API", xbmc.LOGINFO)
        if only_uhd:
            xbmc.log("[SeriesManager] only_uhd=True → filtruji striktně na 2160p/4K", xbmc.LOGINFO)

        # Init dat struktury
        series_data = self._init_series_data(series_name)
        series_data['search_query_name'] = series_name
        xbmc.log(f"[SeriesManager] Uložen primární vyhledávací název: '{series_name}'", xbmc.LOGDEBUG)
        saved_slug = series_data.get('tmdb', {}).get('slug')
        xbmc.log(f"[SeriesManager] DEBUG: Načtený saved_slug z cache: '{saved_slug}'", xbmc.LOGDEBUG)        
        # ⬅️ DŮLEŽITÉ: zkusit získat TMDb hned na začátku (aby filtr měl ID)
        try:
            self._enrich_with_tmdb(series_data, series_name)
        except Exception as e:
            xbmc.log(f'YaWSP Series Manager: TMDb enrich (early) skipped: {e}', xbmc.LOGWARNING)

        # 🎯 ZDE JE OPRAVA: Vytvoříme referenční slug z aktuálního search_name a uložíme ho
        # do dat seriálu, čímž přepíšeme případný starý, chybný slug z cache.
        slug_series = _normalize_and_slugify(series_name)
        if series_data.get('tmdb'):
            # 🛑 ZMĚNA 2: Ponechat saved_slug, pokud se jedná o refresh a máme uloženou hodnotu
            if refresh and saved_slug:
                series_data['tmdb']['slug'] = saved_slug
                xbmc.log(f"[SeriesManager] Přepis slug ZRUŠEN/OBNOVEN na: '{saved_slug}' (Refresh protection)", xbmc.LOGDEBUG)
            else:
                # Použít nově vygenerovaný slug (původní chování pro první vyhledávání)
                series_data['tmdb']['slug'] = slug_series
                xbmc.log(f"[SeriesManager] Přepisuji referenční slug v datech na: '{slug_series}'", xbmc.LOGDEBUG)
        known_idents = set(series_data.get("known_idents", []))
        main_tmdb_slug = series_data.get('tmdb', {}).get('slug')
        czech_name = series_data.get('name') 
        original_title = series_data.get('tmdb', {}).get('original_title')
        
        valid_slugs = set()
        
        if main_tmdb_slug:
            valid_slugs.add(main_tmdb_slug)
            
        # Přidání slugu z českého názvu (používáme re.sub místo slugify)
        if czech_name:
            # POUŽITÍ VAŠEHO STÁVAJÍCÍHO VZORU PRO SLUG
            czech_slug = re.sub(r'[^a-z0-9]', '', czech_name.lower())
            valid_slugs.add(czech_slug)
                
        # Přidání slugu z originálního názvu
        if original_title:
            original_slug = re.sub(r'[^a-z0-9]', '', original_title.lower())
            valid_slugs.add(original_slug)

        for alias in series_data.get('aliases', []):
            alias_slug = re.sub(r'[^a-z0-9]', '', alias.lower())
            valid_slugs.add(alias_slug)
            
        xbmc.log(f"[SeriesManager] Valid Slugs for filtering: {valid_slugs}", xbmc.LOGDEBUG)
        
        # 🎯 NOVÁ LOGIKA: Získání chybějících epizod, pokud jde o aktualizaci v režimu name_only
        if refresh and name_only:
            tmdb_seasons = series_data.get('tmdb', {}).get('seasons')
            existing_streams = series_data.get('seasons', {})
            
            # 🚨 DIAGNOSTIKA 1: Kontrola dostupnosti TMDb dat
            if not tmdb_seasons:
                xbmc.log("[SM DEBUG TMDB] TMDb sezónní metadata chybí, nelze spustit cílené hledání chybějících epizod.", xbmc.LOGWARNING)

            missing_queries = set()
            if tmdb_seasons:
                for s_num_str, season_meta in tmdb_seasons.items():
                    # Zpracovávat pouze, pokud existuje episode_count pro sezónu
                    num_episodes = season_meta.get('episode_count')
                    if not num_episodes or int(num_episodes) == 0:
                        xbmc.log(f"[SM DEBUG TMDB] Sezóna {s_num_str} nemá definovaný počet epizod.", xbmc.LOGWARNING)
                        continue # Přeskočit sezónu bez metadat
                    
                    s_num = int(s_num_str)
                    
                    # 🚨 DIAGNOSTIKA 2: Kontrola rozsahu cyklu
                    xbmc.log(f"[SM DEBUG RANGE] Start kontroly S{s_num} (počet epizod: {num_episodes})", xbmc.LOGINFO)

                    for e_num in range(1, num_episodes + 1):
                        s_str = str(s_num)
                        e_str = str(e_num)
                        
                        episode_streams = existing_streams.get(s_str, {}).get(e_str, {}).get('streams', [])
                        
                        # 🚨 DIAGNOSTIKA 3: Kontrola, zda se epizoda vyhodnocuje jako nalezená
                        if episode_streams:
                             xbmc.log(f"[SM DEBUG FOUND] S{s_str}E{e_str} se jeví jako nalezená ({len(episode_streams)} streamů)", xbmc.LOGINFO)

                        if not episode_streams:
                            # Pokud streamy chybí nebo je seznam prázdný, hledáme.
                            missing_queries.add(f"{series_name} S{s_num:02d}E{e_num:02d}")

            

            if missing_queries:
                xbmc.log(f"[SeriesManager] NameOnly/Refresh: Hledám {len(missing_queries)} chybějících/nekompletních epizod", xbmc.LOGINFO)
                queries = list(missing_queries)
            else:
                # Pokud nic nechybí, nebo nemáme TMDb data (nebo je vše nalezeno)
                # Použijeme standardní, široké hledání pro SxxExx režim
                queries = [f"{series_name} S", f"{series_name} E", f"{series_name} S01"]
                xbmc.log("[SeriesManager] NameOnly/Refresh: Žádné zjevně chybějící epizody. Standardní hledání Sxx/Exx.", xbmc.LOGINFO)

        else:
            # Standardní/výchozí dotazy pro široké hledání
            queries = [series_name, f"{series_name} s01", f"{series_name} episode", f"{series_name} season"]

        # queries = [series_name, f"{series_name} s01", f"{series_name} episode", f"{series_name} season"]
        raw_items, seen_idents = [], set()

        for q in queries:
            for page in range(self.max_pages):
                results = self._perform_search(q, api_function, token,
                                               offset=page * self.page_size,
                                               limit=self.page_size)
                if not results:
                    break
                for r in results:
                    name = r.get('name') or ''
                    ident = r.get('ident') or ''
                    if not ident or ident in seen_idents:
                        continue
                    # Vyřazení season-packů
                    if SEASON_PACK_HINTS.search(name or ''):
                        continue
                    # Slug kontrola
                    is_match = False
                    
                    if name_only:
                        # Režim name_only (SxxExx only): Ignoruje slug
                        s, e = self._detect_episode_info(name, series_name)
                        if s is not None and e is not None:
                            is_match = True 
                    else:
                        # 2. Vytvoření file_slug (VAŠE STÁVAJÍCÍ SPRÁVNÁ LOGIKA)
                        file_slug = _normalize_and_slugify(name)
                        
                        is_slug_match = False
                        if valid_slugs: 
                            for vs in valid_slugs:
                                # Kontrola, zda je některý z platných slugů (anglický, český, alias) v názvu souboru
                                if vs in file_slug:
                                    is_slug_match = True
                                    break
                        else:
                            # Fallback pro staré série
                            is_slug_match = file_slug.startswith(slug_series)

                        # Match vyžaduje detekci epizody A shodu slugu
                        is_match = self._is_likely_episode(name, series_name) and is_slug_match
                        
                        if not is_match:
                            xbmc.log(f"[SeriesManager] Vyřazeno (Slug/SxxExx Mismatch): {name} (File Slug: {file_slug})", xbmc.LOGDEBUG)

                    if is_match:
                        raw_items.append(r)
                        # Logování pro debug:
                        xbmc.log(f"[SeriesManager] Zahrnuto: {name} [NameOnly={name_only}]", xbmc.LOGDEBUG)
                    else:
                        # Původní log pro vyřazení:
                        xbmc.log(f"[SeriesManager] Vyřazeno (slug/SxxExx mismatch): {name} [NameOnly={name_only}]", xbmc.LOGDEBUG)
                    seen_idents.add(ident)

        xbmc.log(f"[SeriesManager] Po slug filtru zůstává {len(raw_items)} kandidátů", xbmc.LOGINFO)

        # Seskupení do bucketů (S,E)
        buckets = {}
        for item in raw_items:
            s, e = self._detect_episode_info(item.get('name', ''), series_name)
            if s is None or e is None:
                continue
            buckets.setdefault((s, e), []).append(item)

        # ⬅️ TMDb validace sezón/epizod — teď už máme tmdb.id → filtr nebude přeskakovat
        xbmc.log(f"[SeriesManager] Buckets před TMDb filtrem: {len(buckets)}", xbmc.LOGINFO)
        try:
            buckets = filter_invalid_seasons(series_data, buckets)
        except Exception as ex:
            xbmc.log(f"[SeriesManager] Filtr TMDb selhal: {ex}", xbmc.LOGWARNING)
        xbmc.log(f"[SeriesManager] Buckets po TMDb filtru: {len(buckets)}", xbmc.LOGINFO)

        # Uložení kurátorovaných streamů (bez řazení, zachovat pořadí API)
        for (s, e), cand_list in buckets.items():
            if not cand_list:
                continue
            curated_streams = self._curate_streams(
                cand_list,
                skip_high_quality=skip_high_quality,
                force_hd_only=force_hd_only,
                name_only=name_only,
                only_uhd=only_uhd,
                max_per_episode=int(kwargs.get('max_streams', 6))
            )
            if not curated_streams:
                continue

            s_str, e_str = str(s), str(e)
            if s_str not in series_data['seasons']:
                series_data['seasons'][s_str] = {}
            entry = series_data['seasons'][s_str].get(e_str, {})
            entry.setdefault('source_name', curated_streams[0].get('name') or '')
            entry.setdefault('name', '')
            entry.setdefault('tmdb', {})
            entry['streams'] = curated_streams
            entry['curated'] = True
            entry['locked'] = entry.get('locked', False)
            entry['updated_at'] = xbmc.getInfoLabel('System.Date')
            series_data['seasons'][s_str][e_str] = entry

            for st in curated_streams:
                if st.get('ident'):
                    known_idents.add(st['ident'])
            series_data['known_idents'] = sorted(list(known_idents))

        # Doplň finální TMDb metadata (tituly epizod, stills, runtime, poster/backdrop)
        try:
            self._enrich_with_tmdb(series_data, series_name)
        except Exception as e:
            xbmc.log(f'YaWSP Series Manager: TMDb enrich skipped: {e}', xbmc.LOGWARNING)

        # 🛑 ZMĚNA 3: Finální kontrola a obnova slugu PŘED ULOŽENÍM
        # Protože druhé obohacení mohlo slug znovu přepsat.
        if refresh and saved_slug:
            final_slug = series_data.get('tmdb', {}).get('slug')
            if final_slug != saved_slug:
                series_data['tmdb']['slug'] = saved_slug
                xbmc.log(f"[SeriesManager] 💾 Finální RESTORE: Slug obnoven před uložením na: '{saved_slug}'", xbmc.LOGDEBUG)

        self._save_series_data(series_name, series_data)
        return self._migrate_schema(series_data)

    def _is_likely_episode(self, filename, series_name):
        """Check if a filename is likely to be an episode of the series"""
        if not filename:
            return False
        # povolit i varianty názvu (diakritika/oddělovače)
        if series_name and re.search(re.escape(series_name), filename, re.IGNORECASE) is None:
            if re.sub(r'\W+', '', series_name.lower()) not in re.sub(r'\W+', '', filename.lower()):
                return False
        t = _light_clean(filename)
        if RE_SxxEyy.search(t) or RE_1x02.search(t) or RE_CZ.search(t) or RE_EPONLY.search(t):
            return True
        return False

    def _perform_search(self, search_query, api_function, token, offset=0, limit=100):
        """Perform the actual search using the provided API function with paging"""
        results = []
        try:
            response = api_function('search', {
                'what': search_query,
                'category': 'video',
                'sort': 'recent',
                'limit': int(limit),
                'offset': int(offset),
                'wst': token,
                'maybe_removed': 'true'
            })
            try:
                xml = ET.fromstring(response.content)
            except Exception as ex:
                xbmc.log(f'YaWSP Series Manager: XML parse error: {ex}', xbmc.LOGWARNING)
                return []
            status = xml.find('status')
            if status is not None and (status.text or '').upper() == 'OK':
                for file in xml.iter('file'):
                    item = {}
                    for elem in file:
                        item[elem.tag] = elem.text
                    results.append(item)
            return results
        except Exception as e:
            xbmc.log(f'YaWSP Series Manager: search error: {e}', xbmc.LOGERROR)
            return []

    def _detect_episode_info(self, filename, series_name):
        """Detect S/E from filename (robust, CZ friendly)."""
        if not filename:
            return None, None
        t = _light_clean(filename)
        m = RE_SxxEyy.search(t)
        if m:
            return int(m.group(1)), int(m.group(2))
        m = RE_1x02.search(t)
        if m:
            return int(m.group(1)), int(m.group(2))
        m = RE_CZ.search(t)
        if m:
            # groups: (sezona|série|řada) (S) (díl|epizoda|ep) (E)
            try:
                return int(m.group(2)), int(m.group(4))
            except Exception:
                return None, None
        m = RE_EPONLY.search(t)
        if m:
            return 1, int(m.group(2))
        return None, None

    def _save_series_data(self, series_name, series_data):
        """Save series data to the database, filename based on TMDb ID"""
        try:
            tmdb_id = series_data.get("tmdb", {}).get("id")
            safe_name = self._safe_filename(series_name, tmdb_id)
            file_path = os.path.join(self.series_db_path, f"{safe_name}.json")
            _atomic_write_json(file_path, series_data)
            xbmc.log(f"YaWSP SeriesManager: uložen soubor {file_path}", xbmc.LOGINFO)
        except Exception as e:
            xbmc.log(f"YaWSP SeriesManager: Error saving series data: {e}", xbmc.LOGERROR)

    def get_all_series(self):
        """Get a list of all saved series"""
        series_list = []
        try:
            for filename in os.listdir(self.series_db_path):
                if filename.endswith('.json'):
                    safe_name = os.path.splitext(filename)[0]
                    # Zobrazíme uživatelské jméno dle obsahu, pokud jde načíst
                    display_name = safe_name.replace('_', ' ')
                    fp = os.path.join(self.series_db_path, filename)
                    try:
                        with io.open(fp, 'r', encoding='utf8') as f:
                            d = json.load(f)
                        nm = (d.get('name') or '').strip()
                        if nm:
                            display_name = nm
                    except Exception:
                        pass
                    series_list.append({
                        'name': display_name,
                        'filename': filename,
                        'safe_name': safe_name
                    })
        except Exception as e:
            xbmc.log(f'YaWSP Series Manager: Error listing series: {e}', level=xbmc.LOGERROR)
        return series_list

    def _safe_filename(self, series_name: str, tmdb_id: int = None) -> str:
        """
        Vrátí bezpečný název souboru pro seriál.
        Preferuje TMDb ID, jinak fallback na očištěný název.
        """
        if tmdb_id:
            return f"tmdb_{tmdb_id}"
        return re.sub(r'[^a-z0-9]+', '_', (series_name or '').lower()).strip('_')

    # --- schéma v3 + migrace --------------------------------------------------
    def _init_series_data(self, series_name):
        data = self.load_series_data(series_name) or {}
        if not data:
            data = {
                "schema_version": 3,
                "name": series_name,
                "aliases": [],
                "last_updated": xbmc.getInfoLabel('System.Date'),
                "source": "webshare",
                "known_idents": [],
                "tmdb": {},
                "extras": {"season_packs": []},
                "seasons": {}
            }
        else:
            data.setdefault("schema_version", 3)
            data.setdefault("aliases", [])
            data.setdefault("source", "webshare")
            data.setdefault("known_idents", [])
            data.setdefault("tmdb", {})
            data.setdefault("extras", {"season_packs": []})
            data.setdefault("seasons", {})
            data["last_updated"] = xbmc.getInfoLabel('System.Date')
        return data

    def _migrate_schema(self, data):
        if not isinstance(data, dict):
            return data
        data.setdefault("schema_version", 3)
        data.setdefault("aliases", [])
        data.setdefault("source", "webshare")
        data.setdefault("known_idents", [])
        data.setdefault("tmdb", {})
        data.setdefault("extras", {"season_packs": []})
        seasons = data.setdefault("seasons", {})

        # migrace starého "ident" -> streams[0]
        for s in list(seasons.keys()):
            for e in list(seasons[s].keys()):
                ep = seasons[s][e]
                if isinstance(ep, dict) and "streams" not in ep:
                    if "ident" in ep:
                        streams = [{
                            "ident": ep.get("ident"),
                            "name": ep.get("name", ""),
                            "quality": ep.get("quality", ""),
                            "lang": ep.get("lang", ""),
                            "size": int(ep.get("size", "0") or 0),
                            "ainfo": ep.get("ainfo", ""),
                            "score": 0,
                            "added_at": xbmc.getInfoLabel('System.Date'),
                            "source": "webshare"
                        }]
                        # doplň features/label
                        for st in streams:
                            st['features'] = self._extract_features(st)
                            st['label'] = ', '.join(st['features']) if st['features'] else (_norm(st['quality']) or _norm(st['name']) or 'Stream')
                        ep["streams"] = streams
                        ep.setdefault("source_name", ep.get("name", ""))
                        ep.setdefault("tmdb", {})
                        ep.setdefault("curated", False)
                        ep.setdefault("locked", False)
                        ep.setdefault("updated_at", xbmc.getInfoLabel('System.Date'))
                        # ep['name'] bude později přepsán TMDb názvem
        return data

    # --- skórování a kurátorování --------------------------------------------
    def _quality_rank(self, q):
        ql = (q or "").lower()
        if "2160" in ql or "4k" in ql:
            return 100
        if "1080" in ql:
            return 80
        if "720" in ql:
            return 60
        if "540" in ql or "sd" in ql:
            return 30
        return 40

    def _lang_rank(self, lang):
        l = (lang or "").lower()
        for idx, tag in enumerate(PREF_LANG_ORDER):
            if tag in l:
                return (len(PREF_LANG_ORDER) - idx) * 10
        return 0

    def _candidate_score(self, item):
        q = (item.get('quality') or '') + ' ' + (item.get('ainfo') or '')
        lang = (item.get('lang') or '')
        score = self._quality_rank(q) + self._lang_rank(lang)
        ql = q.lower()
        if 'hdr' in ql or 'dolby vision' in ql or 'dv' in ql:
            score += 10
        if 'x265' in ql or 'hevc' in ql:
            score += 5
        try:
            size = float(item.get('size', 0) or 0)  # bytes
            score += min(size / (1024*1024*200), 10)  # až +10 podle velikosti
        except Exception:
            pass
        if re.search(r'(?i)\b(cam|telesync|ts|scr|screener)\b', q):
            score -= 50
        return score

    def _curate_streams(self, candidates, skip_high_quality=False, force_hd_only=False,
                        name_only=False, only_uhd=False, max_per_episode: int = 6):
        """
        Kurátorování streamů:
        - Zachová pořadí z Webshare API (Most relevant).
        - Použije jen filtry kvality podle režimu.
        - Žádné řazení podle velikosti ani skórování.
        """
        norm = []
        seen = set()
        if not name_only:
            if only_uhd:
                candidates = [c for c in candidates if re.search(r'(?i)(2160|4k)', (c.get('quality') or '') + (c.get('name') or ''))]
            if force_hd_only:
                candidates = [c for c in candidates if re.search(r'(?i)(1080|720)', (c.get('quality') or '') + (c.get('name') or ''))]
            if skip_high_quality:
                candidates = [c for c in candidates if not re.search(r'(?i)(2160|4k)', (c.get('quality') or '') + (c.get('name') or ''))]

        for c in candidates:
            ident = c.get('ident') or ''
            if not ident or ident in seen:
                continue
            seen.add(ident)
            item = {
                'ident': ident,
                'name': c.get('name') or '',
                'quality': c.get('quality') or c.get('ainfo') or '',
                'lang': c.get('lang') or '',
                'size': int(c.get('size', '0') or 0),
                'ainfo': c.get('ainfo') or '',
                'score': 0,
                'added_at': xbmc.getInfoLabel('System.Date'),
                'source': 'webshare'
            }
            feats = self._extract_features(item)
            item['features'] = feats
            item['label'] = ', '.join(feats) if feats else (_norm(item['quality']) or _norm(item['name']) or 'Stream')
            norm.append(item)
        return norm[:6]

    def _enrich_with_tmdb(self, series_data: dict, series_name: str):
        """
        Doplní detail seriálu (poster/backdrop/overview + show_runtime) a metadata epizod:
        series_data['tmdb'] = {
          "id", "title", "original_title", "year", "poster", "backdrop",
          "poster_url", "backdrop_url", "overview", "show_runtime": int|None,
          "imdb_id": str|None
        }
        Epizodám ukládá: title, overview, still_url, air_date, vote, runtime (fallback na show_runtime).
        """
        try:
            from resources.lib.tmdb import tmdb_utils as TM
        except Exception:
            TM = None

        # 1) Najdi/ověř show ID (přes search)
        if not series_data.get('tmdb') or not series_data['tmdb'].get('id'):
            tv_meta = None
            try:
                if TM and hasattr(TM, 'search_tmdb_tv'):
                    tv_meta = TM.search_tmdb_tv(series_name)
            except Exception:
                tv_meta = None
            if tv_meta:
                series_data.setdefault('tmdb', {})
                series_data['tmdb'].update({
                    "id": tv_meta.get("tmdb_id"),
                    "title": tv_meta.get("title"),
                    "original_title": tv_meta.get("original_title") or tv_meta.get("title"),
                    "year": tv_meta.get("year"),
                })
                series_data['name'] = tv_meta.get("title") or series_name

        tv_id = series_data.get('tmdb', {}).get('id')
        if not tv_id:
            return

        # 2) Detaily seriálu: get_tv_show_meta → show_runtime + poster/backdrop/overview
        show_meta = None
        if TM and hasattr(TM, 'get_tv_show_meta'):
            try:
                show_meta = TM.get_tv_show_meta(tv_id=int(tv_id), language='cs')
            except Exception:
                show_meta = None

        poster_path   = show_meta.get('poster_path')   if show_meta else None
        backdrop_path = show_meta.get('backdrop_path') if show_meta else None
        overview      = show_meta.get('overview')      if show_meta else None
        rt            = show_meta.get('episode_run_time') if show_meta else None

        if isinstance(rt, list) and rt:
            show_runtime = int(rt[0]) if isinstance(rt[0], (int, float)) else None
        elif isinstance(rt, int):
            show_runtime = rt
        else:
            show_runtime = None
        
        # 🎯 KONEČNÁ OPRAVA: ZÍSKÁNÍ A PŘÍPRAVA METADAT SEZÓN (episode_count)
        tmdb_seasons_data = {}
        
        # Zjišťujeme celkový počet sezón (TMDb vrací tento klíč i bez detailů sezón)
        num_seasons = show_meta.get('number_of_seasons') if show_meta else None
        
        if num_seasons and int(num_seasons) >= 1:
            xbmc.log(f"[SM Enrich] Zjištěno {num_seasons} sezón. Získávám detaily pro každou.", xbmc.LOGDEBUG)

            for s_num in range(1, int(num_seasons) + 1):
                s_num_str = str(s_num)
                try:
                    # Explicitní volání detailů sezóny pro získání přesného episode_count
                    season_meta = TM.get_tv_season_meta(int(tv_id), s_num, language='cs') or {}
                    
                    episode_count = season_meta.get('episode_count')
                    
                    if episode_count and int(episode_count) > 0:
                        tmdb_seasons_data[s_num_str] = {
                            'episode_count': episode_count
                        }
                    
                    xbmc.log(f"[SM Enrich] Sezóna {s_num:02d} - Episode Count: {episode_count}", xbmc.LOGDEBUG)

                except Exception as ex:
                    xbmc.log(f"[SM Enrich] Chyba při získávání meta S{s_num}: {ex}", xbmc.LOGWARNING)
                    continue

        # 🚨 NOVÝ DEBUG LOG: Ukážeme, co se podařilo extrahovat (se započtenými chybami)
        xbmc.log(f"[SM Enrich] Extrahováno {len(tmdb_seasons_data)} sezón. Data: {tmdb_seasons_data}", xbmc.LOGINFO)

        # 2a) Získání IMDb ID z externích identifikátorů
        imdb_id = None
        if TM and hasattr(TM, 'get_tv_external_ids'):
            try:
                ext_ids = TM.get_tv_external_ids(tv_id=int(tv_id))
                imdb_id = ext_ids.get('imdb_id')
            except Exception:
                imdb_id = None

        img_base_poster = "https://image.tmdb.org/t/p/w500"
        img_base_fanart = "https://image.tmdb.org/t/p/original"

        series_tm = series_data.setdefault('tmdb', {})
        series_tm.update({
            "poster": poster_path,
            "backdrop": backdrop_path,
            "poster_url": (img_base_poster + poster_path) if poster_path else None,
            "backdrop_url": (img_base_fanart + backdrop_path) if backdrop_path else None,
            "overview": overview,
            "show_runtime": show_runtime,
            "imdb_id": imdb_id
        })
        if tmdb_seasons_data:
            series_tm['seasons'] = tmdb_seasons_data # Ukládáme data sezón
        
        
        try:
            self._populate_tmdb_placeholders(series_data, tv_id=int(tv_id), language='cs')
            xbmc.log("[SM Enrich] TMDb placeholders doplněny pro všechny sezóny/epizody", xbmc.LOGDEBUG)
        except Exception as ex:
            xbmc.log(f"[SM Enrich] Placeholder populate failed: {ex}", xbmc.LOGWARNING)

        # 2b) Pokud nemáme 'year', zkus dopočítat z první epizody S01E01 (air_date[:4])
        if not series_tm.get('year'):
            try:
                s1 = series_data.get('seasons', {}).get('1', {})
                e1 = s1.get('1', {})
                air = (e1.get('tmdb') or {}).get('air_date') or (e1.get('air_date') if isinstance(e1, dict) else None)
                if air and len(air) >= 4:
                    series_tm['year'] = int(air[:4])
            except Exception:
                pass

        # 3) Epizody – title/overview/still_url/air_date/vote/runtime (+name)
        for s in list(series_data['seasons'].keys()):
            for e in list(series_data['seasons'][s].keys()):
                ep = series_data['seasons'][s][e]
                if ep.get('locked'):
                    continue
                ep.setdefault('tmdb', {})
                ep_meta = None
                if TM and hasattr(TM, 'get_tv_episode_meta'):
                    try:
                        ep_meta = TM.get_tv_episode_meta(tv_id=int(tv_id), season=int(s), episode=int(e), language='cs')
                    except Exception:
                        ep_meta = None
                if ep_meta:
                    ep_title   = ep_meta.get('title')
                    ep_overview= ep_meta.get('overview')
                    ep_still   = ep_meta.get('still_url') or None
                    ep_air     = ep_meta.get('air_date')
                    ep_vote    = ep_meta.get('vote')
                    ep_runtime = ep_meta.get('runtime')
                    if (ep_runtime is None or (isinstance(ep_runtime, int) and ep_runtime <= 0)) and isinstance(show_runtime, int) and show_runtime > 0:
                        ep_runtime = show_runtime
                    ep['tmdb'] = {
                        "episode_id": ep_meta.get("episode_id"),
                        "title": ep_title,
                        "overview": ep_overview,
                        "still_path": ep_meta.get('still_path'),
                        "still_url": ep_still,
                        "air_date": ep_air,
                        "vote": ep_vote,
                        "runtime": ep_runtime
                    }
                    if ep_title:
                        ep['title'] = ep_title
                        ep['name']  = ep_title
                    else:
                        if not ep.get('name'):
                            ep['name'] = ep.get('source_name') or ep.get('title') or ''
                else:
                    if not ep.get('name'):
                        ep['name'] = ep.get('title') or ep.get('source_name') or ''
    
    def _populate_tmdb_placeholders(self, series_data: dict, tv_id: int, language: str = 'cs') -> None:
        """
        Vytvoří (nebo doplní) placeholdery epizod podle TMDb pro všechny sezóny.
        - Nevytváří ani nepřepisuje streamy.
        - U 'locked' epizod nesahejte do title/overview (tj. pouze ponechá).
        - Na závěr přepočítá season_state pro každou sezónu.
        """
        try:
            from resources.lib.tmdb import tmdb_utils as TM
        except Exception:
            xbmc.log("[SM Placeholders] TMDb modul nedostupný, placeholders přeskočeny", xbmc.LOGWARNING)
            return

        tmdb = series_data.get('tmdb') or {}
        tmdb_seasons = tmdb.get('seasons') or {}
        # Zkusit určit seznam sezón (primárně z uložených tmdb.seasons; fallback na show meta)
        seasons_keys = []
        try:
            seasons_keys = sorted([int(k) for k in tmdb_seasons.keys()]) if tmdb_seasons else []
        except Exception:
            seasons_keys = []

        if not seasons_keys:
            show_meta = TM.get_tv_show_meta(int(tv_id), language=language) or {}
            num = show_meta.get('number_of_seasons') or 0
            try:
                seasons_keys = list(range(1, int(num) + 1))
            except Exception:
                seasons_keys = []

        series_data.setdefault('seasons', {})

        for s in seasons_keys:
            s_key = str(s)
            # Načíst seznam epizod TMDb pro sezónu
            season_meta = TM.get_tv_season_meta(int(tv_id), int(s), language=language) or {}
            episodes = season_meta.get('episodes') or []

            if s_key not in series_data['seasons']:
                series_data['seasons'][s_key] = {}

            for ep in episodes:
                e_num = ep.get('episode_number')
                if not isinstance(e_num, int):
                    continue
                e_key = str(e_num)

                entry = series_data['seasons'][s_key].get(e_key)
                if not entry:
                    # Nový placeholder epizody (bez streamů)
                    entry = {
                        'name': ep.get('name') or '',
                        'title': ep.get('name') or '',
                        'source_name': '',
                        'streams': [],            # důležité: prázdný list
                        'curated': False,
                        'locked': False,
                        'tmdb': {},
                        'updated_at': xbmc.getInfoLabel('System.Date')
                    }

                # Pokud není locked, obohať TMDb detailem epizody
                if not entry.get('locked'):
                    ep_meta = TM.get_tv_episode_meta(int(tv_id), int(s), int(e_num), language=language) or {}
                    title = ep_meta.get('title') or ep.get('name') or entry.get('name') or ''
                    entry['name'] = title or entry.get('name') or ''
                    entry['title'] = title or entry.get('title') or entry['name']

                    entry['tmdb'] = {
                        'episode_id': ep_meta.get('episode_id'),
                        'title': title,
                        'overview': ep_meta.get('overview'),
                        'still_path': ep_meta.get('still_path'),
                        'still_url': ep_meta.get('still_url'),
                        'air_date': ep_meta.get('air_date') or ep.get('air_date'),
                        'vote': ep_meta.get('vote'),
                        'runtime': ep_meta.get('runtime')
                    }

                # Zapiš/obnov na správném místě
                series_data['seasons'][s_key][e_key] = entry

            # Přepočet stavu sezóny (počty/flagy)
            try:
                self._recalc_season_state(series_data, s_key)
            except Exception as ex:
                xbmc.log(f"[SM Placeholders] Recalc season state selhal pro S{s_key}: {ex}", xbmc.LOGWARNING)
                        

    @staticmethod
    def _extract_features(st):
        """
        Vrátí seznam vlastností jako ['CZ-EN-Atmos-5.1', 'HDR10', 'HEVC', '2160p'].
        Vstupy bere z:
        - st['name'] (původní filename),
        - st['quality'] a st['ainfo'],
        - st['lang'] (pokud není v názvu).
        """
        name = (st.get('name') or '')
        q    = (st.get('quality') or '')
        ai   = (st.get('ainfo') or '')
        lang = (st.get('lang') or '')
        blob = ' '.join([name, q, ai]).lower()

        feats = []
        # 1) jazyk(y) + audio layout
        langs = []
        if re.search(r'\b(cz|cs)\b', blob) or 'cz' in lang or 'cs' in lang:
            langs.append('CZ')
        if re.search(r'\bsk\b', blob) or 'sk' in lang:
            if 'CZ' in langs:
                langs[-1] = 'CZ-SK'
            else:
                langs.append('SK')
        if re.search(r'\b(en|eng|english)\b', blob) or 'en' in lang:
            if langs and langs[-1] in ('CZ', 'CZ-SK', 'SK'):
                langs[-1] = f"{langs[-1]}-EN"
            else:
                langs.append('EN')

        audio = []
        if re.search(r'\batmos\b', blob):
            audio.append('Atmos')
        ch = re.search(r'\b(7\.1|5\.1|2\.0)\b', blob)
        if ch:
            audio.append(ch.group(1))

        lang_audio = '-'.join(x for x in [(' - '.join(langs) if langs else ''), *audio] if x)
        lang_audio = lang_audio.replace(' - ', '-') if lang_audio else ''
        if lang_audio:
            feats.append(lang_audio)

        # 2) HDR/DV
        if re.search(r'\bhdr10\b', blob):
            feats.append('HDR10')
        elif re.search(r'\bhdr\b', blob):
            feats.append('HDR')
        if re.search(r'\b(dv|dolby\s*vision)\b', blob):
            feats.append('DV')

        # 3) kodek
        if re.search(r'\b(hevc|x265)\b', blob):
            feats.append('HEVC')
        elif re.search(r'\b(x264|avc)\b', blob):
            feats.append('H.264')

        # 4) rozlišení
        if re.search(r'\b(2160p|4k)\b', blob):
            feats.append('2160p')
        elif re.search(r'\b1080p\b', blob):
            feats.append('1080p')
        elif re.search(r'\b720p\b', blob):
            feats.append('720p')

        # unikátní pořadí zachovat
        uniq = []
        for f in feats:
            if f not in uniq:
                uniq.append(f)
        return uniq

    # --- public helper: smazání seriálu --------------------------------------
    def remove_series(self, series_name):
        """
        Odstraní JSON soubor seriálu (tmdb_<id>.json i slug.json).
        Vrací True, pokud byl alespoň jeden soubor smazán.
        """
        try:
            safe_name = self._safe_filename(series_name)
            fp_slug = os.path.join(self.series_db_path, f"{safe_name}.json")
            # Pokus s tmdb_<id>
            tmdb_id = self._get_tmdb_id(series_name)
            fp_tmdb = os.path.join(self.series_db_path, f"tmdb_{tmdb_id}.json") if tmdb_id else None

            removed = False
            if fp_tmdb and os.path.exists(fp_tmdb):
                os.remove(fp_tmdb)
                removed = True
            if os.path.exists(fp_slug):
                os.remove(fp_slug)
                removed = True
            return removed
        except Exception as e:
            xbmc.log(f'YaWSP Series Manager: remove_series error: {e}', xbmc.LOGERROR)
            return False

# === Utility functions for the UI layer =======================================
import unicodedata

def _normalize_and_slugify(text):
    """Odstraní diakritiku, převede na malá písmena a odstraní nepovolené znaky."""
    if not text:
        return ''
    
    # Krok 1: Normalizace UNICODE (odstranění diakritiky)
    normalized = unicodedata.normalize('NFKD', text).encode('ascii', 'ignore').decode('utf-8')
    
    # Krok 2: Odstranění nepovolených znaků (stejně jako u názvu souboru)
    slug = re.sub(r'[^a-z0-9]', '', normalized.lower())
    return slug


def filter_invalid_seasons(series_data: dict, buckets: dict) -> dict:
    """
    Odfiltruje z buckets ty položky, které neodpovídají TMDb rozsahu sezón/epizod.
    Zahrnuje kontrolu slugu proti VŠEM známým verzím (CZ, EN, aliasy).
    """
    try:
        from resources.lib.tmdb import tmdb_utils as TM
        import re
    except Exception:
        xbmc.log("[SeriesManager] TMDb modul nebo re nedostupný, filtr přeskočen", xbmc.LOGWARNING)
        return buckets  # ← vrátit původní buckets

    tmdb_id = (series_data.get('tmdb') or {}).get('id')
    if not tmdb_id:
        xbmc.log("[SeriesManager] Chybí TMDb ID, filtr přeskočen", xbmc.LOGWARNING)
        return buckets  # ← vrátit původní buckets
    
    series_tmdb = series_data.get('tmdb', {})
    
    # 🎯 KROK 1: VÝPOČET SADY VŠECH PLATNÝCH SLUGŮ (OPRAVA 1)
    
    main_tmdb_slug = series_tmdb.get('slug') # Původní slug (např. anglický 'thelastfrontier')
    czech_name = series_data.get('name') 
    original_title = series_tmdb.get('original_title')
    
    valid_slugs = set()
    
    # 1. Hlavní (uložený) slug
    if main_tmdb_slug:
        valid_slugs.add(main_tmdb_slug)
        
    # 2. Slug z českého názvu
    if czech_name:
        czech_slug = _normalize_and_slugify(czech_name)
        valid_slugs.add(czech_slug)
            
    # 3. Slug z originálního názvu
    if original_title:
        original_slug = _normalize_and_slugify(original_title)
        valid_slugs.add(original_slug)

    # 4. Slugy z aliasů
    for alias in series_data.get('aliases', []):
        alias_slug = _normalize_and_slugify(alias)
        valid_slugs.add(alias_slug)

    xbmc.log(f"[SeriesManager] TMDb Filter - Valid Slugs for checking: {valid_slugs}", xbmc.LOGINFO)

    # KROK 2: Načtení detailů (number_of_seasons, year) přes TMDb volání
    show_meta = TM.get_tv_show_meta(int(tmdb_id), language='cs') or {}
    num_seasons = show_meta.get('number_of_seasons', 0)
    tmdb_year = show_meta.get('year') 
    
    if not isinstance(num_seasons, int) or num_seasons <= 0:
        xbmc.log("[SeriesManager] TMDb nevrátil validní počet sezón, filtr přeskočen", xbmc.LOGWARNING)
        return buckets  # ← vrátit původní buckets

    xbmc.log(f"[SeriesManager] TMDb filtr start: TMDb ID={tmdb_id}, seasons={num_seasons}", xbmc.LOGINFO)

    valid_buckets = {}
    for (s, e), streams in buckets.items():
        try:
            if s > num_seasons:
                xbmc.log(f"[SeriesManager] ❌ Sezóna {s} mimo rozsah TMDb ({num_seasons})", xbmc.LOGDEBUG)
                continue
            
            xbmc.log(f"[SM Filter DEBUG] Volám TM.get_tv_season_meta pro S{s:02d}", xbmc.LOGINFO)
            season_meta = TM.get_tv_season_meta(int(tmdb_id), s, language='cs') or {}
            if not season_meta:
                 xbmc.log(f"[SM Filter DEBUG] Získané season_meta pro S{s:02d} je PRÁZDNÉ.", xbmc.LOGWARNING)

            episodes_meta = season_meta.get('episodes') or []
            ep_numbers = {ep.get('episode_number') for ep in episodes_meta if isinstance(ep.get('episode_number'), int)}
            
            xbmc.log(f"[SM Filter DEBUG] S{s:02d} - Nalezené epizody v TMDb: {ep_numbers}", xbmc.LOGINFO)

            max_ep = max(ep_numbers) if ep_numbers else None

            if e not in ep_numbers:
                xbmc.log(f"[SeriesManager] ❌ Epizoda S{s:02d}E{e:02d} v TMDb sezóně neexistuje → skip", xbmc.LOGDEBUG)
                continue

            filtered_streams = []
            for st in streams:
                name = st.get('name') or ''
                name_slug = _normalize_and_slugify(name)
                
                # 🎯 OPRAVA 2: KONTROLA SLUGU PROTI SADĚ valid_slugs
                is_slug_match = False
                if valid_slugs:
                    for vs in valid_slugs:
                        if vs and vs in name_slug:
                            is_slug_match = True
                            break
                
                if not is_slug_match:
                    # Rozšířený log pro snadné ladění
                    xbmc.log(f"[SeriesManager] ❌ Slug mismatch. File Slug: {name_slug}, Valid Slugs: {valid_slugs} (vyřazeno: {name})", xbmc.LOGDEBUG)
                    continue

                if max_ep and e > max_ep:
                    xbmc.log(f"[SeriesManager] ❌ Epizoda {e} > max {max_ep}: {name}", xbmc.LOGDEBUG)
                    continue
                
                year_match = re.search(r'\b(19|20)\d{2}\b', name)
                if year_match and tmdb_year and int(year_match.group(0)) != int(tmdb_year):
                    xbmc.log(f"[SeriesManager] ❌ Rok mismatch {year_match.group(0)} vs {tmdb_year}: {name}", xbmc.LOGDEBUG)
                    continue
                    
                filtered_streams.append(st)

            if filtered_streams:
                valid_buckets[(s, e)] = filtered_streams
        except Exception as ex:
            xbmc.log(f"[SeriesManager] Filtr: chyba při validaci S{s}E{e}: {ex}", xbmc.LOGWARNING)
            # při chybě v jedné položce pokračuj dál

    xbmc.log(f"[SeriesManager] TMDb filtr dokončen: {len(valid_buckets)} validních položek z původních {len(buckets)}", xbmc.LOGINFO)
    return valid_buckets

def get_url(**kwargs):
    """Create a URL for calling the plugin recursively"""
    from yawsp import _url
    return '{0}?{1}'.format(_url, urlencode(kwargs, encoding='utf-8'))


def create_series_menu(series_manager, handle):
    """
    Menu seriálů:
    - Label: Název seriálu (+ offline souhrn, pokud existují stažené epizody)
    - Artwork: poster + fanart (+ offline ikona disku)
    - Kontext: Aktualizovat, Odebrat, Stáhnout celý seriál, Trakt…
    """
    import xbmcplugin
    import xbmcvfs
    import xbmcaddon

    ADDON = xbmcaddon.Addon()

    # SPRÁVNÉ ZÍSKÁNÍ A PŘÍPRAVA DFOLDER (funguje na SMB/NFS/Android/CoreELEC)
    dfolder = ADDON.getSetting('dfolder') or ''
    base_real = None
    if dfolder:
        base_real = xbmcvfs.translatePath(dfolder)
        if base_real and not base_real.endswith(('/', '\\')):
            base_real += '/'          # ← důležité!

    def _cz_plural_episodes(n: int) -> str:
        if n == 1:
            return "1 epizoda"
        if 2 <= n <= 4:
            return f"{n} epizody"
        return f"{n} epizod"

    def _count_offline_for_series(series_name: str, series_data: dict) -> int:
        """
        Spočítá celkový počet offline epizod napříč všemi sezónami daného seriálu.
        Používá path_exists + listdir_safe → funguje i na síťových discích.
        """
        if not base_real or not series_data or not isinstance(series_data, dict):
            return 0

        total_found = 0
        seasons = series_data.get('seasons', {}) or {}

        for season_key, season_dict in seasons.items():
            # SPRÁVNÉ SLOŽENÍ CESTY BEZ os.path.join → funguje i pro smb://
            season_folder = f"{base_real}{clean_filename(series_name)}/Sezona {season_key}/"

            if not path_exists(season_folder):
                continue

            files = listdir_safe(season_folder)
            if not files:
                continue

            expected_bases = set()
            for ep_key, ep in (season_dict.items() if isinstance(season_dict, dict) else []):
                ep_title = ep.get('name') or ep.get('title') or ep.get('source_name') or ''
                if ep_title:
                    expected_bases.add(clean_filename(ep_title).lower())
                try:
                    expected_bases.add(f"s{int(season_key):02d}e{int(ep_key):02d}".lower())
                except Exception:
                    expected_bases.add(f"s{season_key}e{ep_key}".lower())

            found = set()
            for fname in files:
                base_noext = os.path.splitext(fname)[0].lower()
                for b in expected_bases:
                    if base_noext == b or base_noext.startswith(b):
                        found.add(b)
                        break
            total_found += len(found)

        return total_found

    # Výpis všech uložených seriálů
    series_list = series_manager.get_all_series()
    for s in series_list:
        series_name = s['name']
        data = series_manager.load_series_data(series_name) or {}
        tm = data.get('tmdb', {}) if isinstance(data, dict) else {}

        # Rok (TMDb nebo z první epizody)
        year = tm.get('year')
        if not year:
            try:
                seasons = (data.get('seasons') or {}) if isinstance(data, dict) else {}
                s1 = seasons.get('1', {})
                e1 = s1.get('1', {})
                air = (e1.get('tmdb') or {}).get('air_date') or (e1.get('air_date') if isinstance(e1, dict) else None)
                if air and len(air) >= 4:
                    year = int(air[:4])
            except Exception:
                year = None

        # Počty sezón a epizod
        seasons = (data.get('seasons') or {}) if isinstance(data, dict) else {}
        try:
            seasons_count = len(seasons)
            episodes_count = sum(len(v) for v in seasons.values() if isinstance(v, dict))
        except Exception:
            seasons_count = episodes_count = 0

        # OFFLINE souhrn přes všechny sezóny
        offline_count = _count_offline_for_series(series_name, data)
        offline_tag = f" [COLOR green]offline: {_cz_plural_episodes(offline_count)}[/COLOR]" if offline_count > 0 else ""
        offline_thumb = 'DefaultHardDisk.png' if offline_count > 0 else None

        # Finální label
        label = data.get('name') or series_name
        if year:
            label += f" ({year})"
        label += f" [COLOR blue]{seasons_count} sez. / {episodes_count} ep.[/COLOR]"
        label += offline_tag

        # Artwork + plot
        poster_url = tm.get('poster_url')
        fanart_url = tm.get('backdrop_url')
        plot = tm.get('overview') or ''

        li = xbmcgui.ListItem(label=label)
        art = {'icon': 'DefaultFolder.png'}
        if poster_url:
            art['poster'] = poster_url
        if fanart_url:
            art['fanart'] = fanart_url
        if offline_thumb:
            art['thumb'] = offline_thumb
        li.setArt(art)
        if plot:
            li.setInfo('video', {'plot': plot})

        # Kontextové menu
        cm = []
        cm.append(("Aktualizovat seriál", f"RunPlugin({get_url(action='series_refresh_smart', series_name=series_name)})"))
        cm.append(("Odebrat seriál", f"RunPlugin({get_url(action='series_remove', series_name=series_name)})"))
        cm.append(("ČSFD detail (rozšířený)", f"RunPlugin({get_url(action='csfd_lookup_enhanced', series_name=series_name)})"))
        cm.append(("Stáhnout všechny sezóny", f"RunPlugin({get_url(action='series_download_show', series_name=series_name)})"))
        if tm.get('id'):
            cm.append(("Odeslat celý seriál na Trakt…",
                       f"RunPlugin({get_url(action='series_trakt_mark_show_prompt', tmdb_id=int(tm.get('id')))})"))
        cm.append(("Přidat do oblíbených", "Action(AddToFavorites)"))
        li.addContextMenuItems(cm, replaceItems=False)

        url = get_url(action='series_detail', series_name=series_name)
        xbmcplugin.addDirectoryItem(handle, url, li, isFolder=True)

    # Konec adresáře
    xbmcplugin.setPluginCategory(handle, "Seriály")
    xbmcplugin.setContent(handle, "tvshows")
    xbmcplugin.endOfDirectory(handle)

def is_series_completed(series_data: dict) -> bool:
    """
    Vrátí True, pokud všechny sezóny v season_state mají watched==True
    a total == watched_count (pokud jsou hodnoty k dispozici).
    Pokud season_state chybí, považujeme seriál za NEdosledovaný.
    """
    try:
        if not isinstance(series_data, dict):
            return False
        seasons = series_data.get('seasons') or {}
        season_state = series_data.get('season_state') or {}
        # Pokud nejsou žádné sezóny, neoznačujeme jako dosledované
        if not seasons:
            return False
        for s_key in seasons.keys():
            st = season_state.get(str(s_key)) or {}
            watched = bool(st.get('watched', False))
            total = st.get('total')
            watched_count = st.get('watched_count')
            # musí být watched True
            if not watched:
                return False
            # pokud máme čísla, musí sedět
            if isinstance(total, int) and isinstance(watched_count, int):
                if watched_count != total:
                    return False
        return True
    except Exception:
        return False


def create_series_menu_filtered(series_manager, handle, status: str):
    """
    Filtruje seriály podle stavu sledování:
    - status='completed'    → plně dosledované
    - status='in_progress'  → rozkoukané (nedosledované)

    Vše ostatní (vzhled, offline tagy, kontextové menu) je stejné jako v create_series_menu.
    """
    import xbmcplugin
    import xbmcgui
    import xbmcaddon
    import xbmcvfs

    ADDON = xbmcaddon.Addon()

    # SPRÁVNÉ ZÍSKÁNÍ DFOLDER – funguje i na síťových discích
    dfolder = ADDON.getSetting('dfolder') or ''
    base_real = None
    if dfolder:
        base_real = xbmcvfs.translatePath(dfolder)
        if base_real and not base_real.endswith(('/', '\\')):
            base_real += '/'          # ← důležité pro bezpečné složení cesty

    # České skloňování epizod
    def _cz_plural_episodes(n: int) -> str:
        if n == 1:
            return "1 epizoda"
        if 2 <= n <= 4:
            return f"{n} epizody"
        return f"{n} epizod"

    # Počítání offline epizod napříč všemi sezónami (funguje na SMB/NFS)
    def _count_offline_for_series(series_name: str, series_data: dict) -> int:
        if not base_real or not series_data or not isinstance(series_data, dict):
            return 0

        total_found = 0
        seasons = series_data.get('seasons', {}) or {}

        for season_key, season_dict in seasons.items():
            # SPRÁVNÉ SLOŽENÍ CESTY – BEZ os.path.join → funguje i na smb://
            season_folder = f"{base_real}{clean_filename(series_name)}/Sezona {season_key}/"

            if not path_exists(season_folder):
                continue

            files = listdir_safe(season_folder)
            if not files:
                continue

            expected_bases = set()
            for ep_key, ep in (season_dict.items() if isinstance(season_dict, dict) else []):
                ep_title = ep.get('name') or ep.get('title') or ep.get('source_name') or ''
                if ep_title:
                    expected_bases.add(clean_filename(ep_title).lower())
                try:
                    expected_bases.add(f"s{int(season_key):02d}e{int(ep_key):02d}".lower())
                except Exception:
                    expected_bases.add(f"s{season_key}e{ep_key}".lower())

            found = set()
            for fname in files:
                base_noext = os.path.splitext(fname)[0].lower()
                for b in expected_bases:
                    if base_noext == b or base_noext.startswith(b):
                        found.add(b)
                        break
            total_found += len(found)

        return total_found

    # Filtrace seriálů podle stavu sledování
    series_list = series_manager.get_all_series()
    filtered_series = []
    for s in series_list:
        series_name = s['name']
        data = series_manager.load_series_data(series_name) or {}
        completed = is_series_completed(data)  # předpokládám, že tato funkce existuje

        if (status == 'completed' and completed) or (status == 'in_progress' and not completed):
            filtered_series.append((series_name, data))

    # 🚨 NOVÁ LOGIKA PRO ŘAZENÍ ROZKOUKANÝCH
    if status == 'in_progress':
        # Řazení: Použít data (item[1]) a klíč 'last_watched_at_utc'
        # '1970' je fallback pro případy, kdy čas chybí, zajistí to, že se přesunou na konec.
        filtered_series.sort(
            key=lambda item: item[1].get('last_watched_at_utc') or '1970',
            reverse=True  # Sestupně (nejnovější nahoře)
        )

    # Titulek kategorie
    title = "Seriály · Dosledované" if status == 'completed' else "Seriály · Rozkoukané"
    xbmcplugin.setPluginCategory(handle, title)

    # Vykreslení filtrovaných seriálů
    for series_name, data in filtered_series:
        tm = data.get('tmdb', {}) if isinstance(data, dict) else {}

        # Rok (TMDb nebo z první epizody)
        year = tm.get('year')
        if not year:
            try:
                seasons = (data.get('seasons') or {}) if isinstance(data, dict) else {}
                s1 = seasons.get('1', {})
                e1 = s1.get('1', {})
                air = (e1.get('tmdb') or {}).get('air_date') or (e1.get('air_date') if isinstance(e1, dict) else None)
                if air and len(air) >= 4:
                    year = int(air[:4])
            except Exception:
                year = None

        # Počty sezón a epizod
        seasons = (data.get('seasons') or {}) if isinstance(data, dict) else {}
        try:
            seasons_count = len(seasons)
            episodes_count = sum(len(v) for v in seasons.values() if isinstance(v, dict))
        except Exception:
            seasons_count = episodes_count = 0

        # Offline souhrn
        offline_count = _count_offline_for_series(series_name, data)
        offline_tag = f" [COLOR green]offline: {_cz_plural_episodes(offline_count)}[/COLOR]" if offline_count > 0 else ""
        offline_thumb = 'DefaultHardDisk.png' if offline_count > 0 else None

        # Finální label
        label = data.get('name') or series_name
        if year:
            label += f" ({year})"
        label += f" [COLOR blue]{seasons_count} sez. / {episodes_count} ep.[/COLOR]"
        label += offline_tag

        # Artwork + plot
        poster_url = tm.get('poster_url')
        fanart_url = tm.get('backdrop_url')
        plot = tm.get('overview') or ''

        li = xbmcgui.ListItem(label=label)
        art = {'icon': 'DefaultFolder.png'}
        if poster_url:
            art['poster'] = poster_url
        if fanart_url:
            art['fanart'] = fanart_url
        if offline_thumb:
            art['thumb'] = offline_thumb
        li.setArt(art)
        if plot:
            li.setInfo('video', {'plot': plot})

        # Kontextové menu – přidáváme parametr status, aby se po smazání vrátilo do stejného filtru
        cm = []
        cm.append(("Aktualizovat seriál", f"RunPlugin({get_url(action='series_refresh_smart', series_name=series_name)})"))
        cm.append(("Odebrat seriál", f"RunPlugin({get_url(action='series_remove', series_name=series_name, status=status)})"))
        cm.append(("ČSFD detail (rozšířený)", f"RunPlugin({get_url(action='csfd_lookup_enhanced', series_name=series_name)})"))
        cm.append(("Stáhnout všechny sezóny", f"RunPlugin({get_url(action='series_download_show', series_name=series_name)})"))
        if tm.get('id'):
            cm.append(("Odeslat celý seriál na Trakt…",
                       f"RunPlugin({get_url(action='series_trakt_mark_show_prompt', tmdb_id=int(tm.get('id')))})"))
        cm.append(("Přidat do oblíbených", "Action(AddToFavorites)"))
        li.addContextMenuItems(cm, replaceItems=False)

        url = get_url(action='series_detail', series_name=series_name)
        xbmcplugin.addDirectoryItem(handle, url, li, isFolder=True)

    # Nastavení zobrazení
    xbmcplugin.setContent(handle, 'tvshows')
    xbmc.executebuiltin('Container.SetViewMode(50)')
    xbmcplugin.endOfDirectory(handle)

def create_seasons_menu(series_manager, handle, series_name, build_url_fn=None):
    """
    Menu sezón pro daný seriál:
    - Label: Sezona X [watched_count/total] (+ ok pokud vše zhlédnuto)
    - Offline indikace: [COLOR green]offline[/COLOR] + počet epizod + ikona disku
    - Artwork: poster + fanart
    - Kontext: Aktualizovat, Stáhnout sezónu, Trakt, atd.
    """
    import xbmcplugin
    import xbmcvfs
    import xbmcaddon
    import xbmcgui

    # Builder URL s fallbackem
    builder = build_url_fn or _default_build_url

    ADDON = xbmcaddon.Addon()
    color_tags = ADDON.getSetting('series_progress_color_tags') == 'true'

    # SPRÁVNÉ ZÍSKÁNÍ DFOLDER – funguje i na NASu
    dfolder = ADDON.getSetting('dfolder') or ''
    base_real = None
    if dfolder:
        base_real = xbmcvfs.translatePath(dfolder)
        if base_real and not base_real.endswith(('/', '\\')):
            base_real += '/'  # ← klíčové!

    def _cz_plural_episodes(n: int) -> str:
        if n == 1:
            return "1 epizoda"
        if 2 <= n <= 4:
            return f"{n} epizody"
        return f"{n} epizod"

    def _count_offline_eps_for_season(season_folder: str, season_dict: dict, season_num: int) -> int:
        """Spočítá offline epizody v dané sezóně – funguje na SMB/NFS."""
        if not season_folder or not path_exists(season_folder):
            return 0
        try:
            files = listdir_safe(season_folder)
        except Exception:
            return 0
        if not files:
            return 0

        expected_bases = set()
        for ep_key, ep in season_dict.items():
            ep_title = ep.get('name') or ep.get('title') or ep.get('source_name') or ''
            if ep_title:
                expected_bases.add(clean_filename(ep_title).lower())
            try:
                expected_bases.add(f"s{int(season_num):02d}e{int(ep_key):02d}".lower())
            except Exception:
                expected_bases.add(f"s{season_num}e{ep_key}".lower())

        found = set()
        for fname in files:
            base_noext = os.path.splitext(fname)[0].lower()
            for b in expected_bases:
                if base_noext == b or base_noext.startswith(b):
                    found.add(b)
                    break
        return len(found)

    # Načtení dat seriálu
    series_data = series_manager.load_series_data(series_name)
    if not series_data:
        xbmcgui.Dialog().notification('YaWSP', 'Data seriálu nenalezena', xbmcgui.NOTIFICATION_WARNING)
        xbmcplugin.endOfDirectory(handle, succeeded=False)
        return

    tm = series_data.get('tmdb', {}) if isinstance(series_data, dict) else {}
    poster_url = tm.get('poster_url')
    fanart_url = tm.get('backdrop_url')
    plot = tm.get('overview') or ''

    # Hlavička – název seriálu
    li_info = xbmcgui.ListItem(label=series_name)
    art_info = {'icon': 'DefaultFolder.png'}
    if poster_url:
        art_info['poster'] = poster_url
    if fanart_url:
        art_info['fanart'] = fanart_url
    li_info.setArt(art_info)
    if plot:
        li_info.setInfo('video', {'plot': plot})
    cm_info = [
        ("Aktualizovat seriál", f"RunPlugin({builder(action='series_refresh_smart', series_name=series_name)})"),
        ("Odebrat seriál", f"RunPlugin({builder(action='series_remove', series_name=series_name)})"),
    ]
    li_info.addContextMenuItems(cm_info, replaceItems=False)
    xbmcplugin.addDirectoryItem(handle, builder(action='noop'), li_info, False)

    # Explicitní tlačítka
    for label, action, icon in [
        ("Aktualizovat streamy", 'series_refresh_smart', 'DefaultAddonsSearch.png'),
        ("Ověřit dostupnost streamů", 'series_validate_streams', 'DefaultAddonService.png'),
        ("Zobrazit uložené streamy", 'series_streams_list', 'DefaultHardDisk.png'),  # ← NOVÉ
        ("Odebrat seriál", 'series_remove', 'DefaultIconError.png'),
    ]:
        li = xbmcgui.ListItem(label=label)
        li.setArt({'icon': icon, 'poster': poster_url or '', 'fanart': fanart_url or ''})
        xbmcplugin.addDirectoryItem(handle, builder(action=action, series_name=series_name),
                                    li, action != 'series_validate_streams')

    # Výpis sezón
    for season_key in sorted(series_data.get('seasons', {}).keys(), key=int):
        season_data = series_data['seasons'][season_key]
        season_state = series_data.get('season_state', {}).get(season_key, {})
        watched_count = season_state.get('watched_count', 0)
        total = season_state.get('total', len(season_data))
        watched_flag = season_state.get('watched', False)
        suffix = f"[{watched_count}/{total}]"
        if watched_flag:
            suffix += " " + ("[COLOR lime]ok[/COLOR]" if color_tags else "[ok]")


        # OFFLINE detekce – SPRÁVNĚ složená cesta! + chybějící streamy (bez budoucích)
        offline_tag = ""
        offline_thumb = None
        offline_count = 0

        # Připrav index souborů v sezónní složce (pro rychlé testy offline stavů epizod)
        season_folder = None
        season_files_bases = set()
        if base_real:
            season_folder = f"{base_real}{clean_filename(series_name)}/Sezona {season_key}/"
            if path_exists(season_folder):
                try:
                    files = listdir_safe(season_folder)  # vrací seznam názvů souborů
                    for fname in files:
                        base_noext = os.path.splitext(fname)[0].lower()
                        season_files_bases.add(base_noext)
                except Exception as ex:
                    xbmc.log(f"[SeasonsMenu] listdir selhal pro {season_folder}: {ex}", xbmc.LOGDEBUG)
                    season_folder = None  # aby se níže nespouštěla offline logika

        # Počet offline epizod (informativní zelený tag)
        if season_folder:
            try:
                offline_count = _count_offline_eps_for_season(season_folder, season_data, int(season_key))
                if offline_count > 0:
                    offline_tag = f" [COLOR green]offline[/COLOR] {_cz_plural_episodes(offline_count)}"
                    offline_thumb = 'DefaultHardDisk.png'
            except Exception as ex:
                xbmc.log(f"[SeasonsMenu] _count_offline_eps_for_season selhal: {ex}", xbmc.LOGDEBUG)

        # Spočítej chybějící streamy (bez budoucích a bez offline)
        missing_count = 0
        today = datetime.date.today()

        for ep_key, ep in season_data.items():
            # 1) má streamy?
            has_streams = bool((ep.get('streams') or []))
            if has_streams:
                continue

            # 2) je epizoda v budoucnu? (tmdb.air_date > dnes) → NEPOČÍTAT
            ep_air = (ep.get('tmdb') or {}).get('air_date') or ep.get('air_date')
            is_future = False
            if isinstance(ep_air, str) and len(ep_air) >= 10:
                try:
                    y, m, d = ep_air[:10].split('-')
                    air_dt = datetime.date(int(y), int(m), int(d))
                    is_future = air_dt > today
                except Exception:
                    is_future = False
            if is_future:
                continue

            # 3) je offline? (podle souborů v season_folder, vzory: clean(title) a SxxEyy)
            is_offline = False
            if season_files_bases:
                patterns = set()
                # vzor podle názvu epizody
                ep_title = ep.get('name') or ep.get('title') or ep.get('source_name') or ''
                if ep_title:
                    patterns.add(clean_filename(ep_title).lower())
                # vzor SxxEyy
                try:
                    patterns.add(f"s{int(season_key):02d}e{int(ep_key):02d}".lower())
                except Exception:
                    patterns.add(f"s{season_key}e{ep_key}".lower())

                # match – přesná shoda nebo prefix
                for base in season_files_bases:
                    if any(base == p or base.startswith(p) for p in patterns):
                        is_offline = True
                        break

            if not is_offline:
                missing_count += 1

        # Label sezóny + číselné sufixy
        total = len(season_data)
        suffix = f"[{watched_count}/{total}]"
        if watched_flag:
            suffix += " " + ("[COLOR lime]ok[/COLOR]" if color_tags else "[ok]")

        missing_suffix = f" [COLOR red]chybí: {missing_count}[/COLOR]" if missing_count > 0 else ""
        label = f"Sezona {season_key} {suffix}{offline_tag}{missing_suffix}"

        li = xbmcgui.ListItem(label=label)
        art_season = {'icon': 'DefaultFolder.png'}
        if poster_url:
            art_season['poster'] = poster_url
        if fanart_url:
            art_season['fanart'] = fanart_url
        if offline_thumb:
            art_season['thumb'] = offline_thumb
        li.setArt(art_season)
        if plot:
            li.setInfo('video', {'plot': plot})

        # Kontextové menu sezóny
        cm_season = [
            ("Aktualizovat seriál", f"RunPlugin({builder(action='series_refresh_smart', series_name=series_name)})"),
            ("Odebrat seriál", f"RunPlugin({builder(action='series_remove', series_name=series_name)})"),
            ("Přidat sezónu do fronty",
             f"RunPlugin({builder(action='series_download_season', series_name=series_name, season=season_key)})"),
            ("Spustit frontu (na pozadí)", f"RunPlugin({builder(action='queue_start_all')})"),
            ("Označit sezónu jako zhlédnutou",
             f"RunPlugin({builder(action='series_mark_season', series_name=series_name, tmdb_id=tm.get('id'), season=season_key, progress=100)})"),
            ("Vymazat stav sezóny",
             f"RunPlugin({builder(action='series_mark_season', series_name=series_name, season=season_key, progress=0)})"),
            ("Smazat sezónu",
             f"RunPlugin({builder(action='series_remove_season', series_name=series_name, season=season_key)})"),
        ]
        if tm.get('id'):
            try:
                cm_season.append(("Odeslat SEZÓNU na Trakt…",
                                  f"RunPlugin({builder(action='series_trakt_mark_season_prompt', tmdb_id=int(tm.get('id')), season=int(season_key))})"))
            except Exception:
                cm_season.append(("Odeslat SEZÓNU na Trakt…",
                                  f"RunPlugin({builder(action='series_trakt_mark_season_prompt', tmdb_id=tm.get('id'), season=season_key)})"))
        li.addContextMenuItems(cm_season, replaceItems=False)

        url = builder(action='series_season', series_name=series_name, season=season_key)
        xbmcplugin.addDirectoryItem(handle, url, li, isFolder=True)

    # Nastavení zobrazení
    xbmcplugin.setPluginCategory(handle, f"Sezóny: {series_name}")
    xbmcplugin.setContent(handle, "seasons")
    xbmc.executebuiltin("Container.SetViewMode(50)")
    xbmcplugin.endOfDirectory(handle)



def create_episodes_menu(series_manager, handle, series_name, season_num, build_url_fn=None):
    """
    Menu epizod pro danou sezónu:
    - Pokud je epizoda STAŽENÁ → přehrává se rovnou z disku (IsPlayable=true)
    - Pokud NENÍ stažená → otevře se klasický výběr streamu z Webshare
    """
    import xbmcplugin
    import xbmcvfs
    import xbmcgui
    import xbmcaddon

    # Builder URL s fallbackem
    builder = build_url_fn or _default_build_url

    ADDON = xbmcaddon.Addon()
    color_tags = ADDON.getSetting('series_progress_color_tags') == 'true'

    # Načtení dat seriálu
    series_data = series_manager.load_series_data(series_name)
    if not series_data or str(season_num) not in series_data.get('seasons', {}):
        xbmcgui.Dialog().notification('YaWSP', 'Data sezóny nenalezena', xbmcgui.NOTIFICATION_WARNING)
        xbmcplugin.endOfDirectory(handle, succeeded=False)
        return

    tm_series = series_data.get('tmdb', {}) if isinstance(series_data, dict) else {}
    series_poster = tm_series.get('poster_url')
    series_fanart = tm_series.get('backdrop_url')
    series_overview = tm_series.get('overview') or ''
    series_year = tm_series.get('year')
    show_runtime = tm_series.get('show_runtime')

    season_key = str(season_num)
    season = series_data['seasons'][season_key]

    # SPRÁVNÁ CESTA K OFFLINE SLOŽCE – funguje na SMB/NFS/Android
    dfolder = ADDON.getSetting('dfolder') or ''
    base_real = None
    if dfolder:
        base = xbmcvfs.translatePath(dfolder)
        if base and not base.endswith(('/', '\\')):
            base += '/'
        base_real = base

    season_folder = f"{base_real}{clean_filename(series_name)}/Sezona {season_key}/" if base_real else None

    # Procházení všech epizod
    for episode_num in sorted(season.keys(), key=int):
        ep = season[episode_num]
        tm_ep = ep.get('tmdb', {}) if isinstance(ep, dict) else {}
        ep_title = ep.get('name') or ep.get('title') or ep.get('source_name') or ''
        ep_plot = tm_ep.get('overview') or series_overview
        ep_still = tm_ep.get('still_url')
        ep_air = tm_ep.get('air_date') or (ep.get('air_date') if isinstance(ep, dict) else None)
        ep_runtime = tm_ep.get('runtime') or show_runtime
        ep_vote = tm_ep.get('vote') or ''
        
        # Stav sledování (OK / 75% / New)
        suffix = series_manager._label_suffix_for_episode(ep, color_tags)

        # OFFLINE DETEKCE + NALEZENÍ SKUTEČNÉHO SOUBORU
        local_file_path = None
        is_offline = False
        if season_folder and path_exists(season_folder):
            try:
                files = listdir_safe(season_folder)
                patterns = []
                if ep_title:
                    patterns.append(clean_filename(ep_title).lower())  # SPRÁVNĚ!
                patterns.append(f"s{int(season_num):02d}e{int(episode_num):02d}".lower())
                for f in files:
                    name_noext = os.path.splitext(f)[0].lower()
                    if any(name_noext == p or name_noext.startswith(p) for p in patterns):
                        local_file_path = season_folder + f
                        is_offline = True
                        break
            except Exception as e:
                xbmc.log(f"[EpisodesMenu] Offline detekce selhala: {e}", xbmc.LOGDEBUG)

        
        
        # 1) Je epizoda v budoucnu?
        is_future = False
        if isinstance(ep_air, str) and len(ep_air) >= 10:
            try:
                y, m, d = ep_air[:10].split('-')
                air_date_obj = datetime.date(int(y), int(m), int(d))
                is_future = air_date_obj > datetime.date.today()
            except Exception:
                is_future = False

        # 2) Má epizoda nějaké online streamy?
        has_streams = bool((ep.get('streams') or []))


        # 3) Prefix jen pro budoucí epizody, ve formátu dd.mm.yyyy
        date_prefix = f"[{format_date_cz(ep_air)}] " if (is_future and ep_air) else ""

        # 4) Rozhodnutí o barvě:
        #    - budoucí epizody → ORANŽOVĚ (varovná barva, ne chyba)
        #    - epizody bez streamu a ne-offline → ČERVENĚ (chyba/chybějící zdroj)
        color_code = None
        if is_future:
            color_code = "orange"
        elif (not has_streams) and (not is_offline):
            color_code = "red"

        # 5) Sestavení labelu + aplikace barvy pouze na tělo (prefix ponecháme bez barvy)
        offline_tag = " [COLOR limegreen]offline[/COLOR]" if is_offline else ""
        base_label  = f"S{int(season_num):02d}E{int(episode_num):02d} - {ep_title} {suffix}{offline_tag}"
        colored     = f"[COLOR {color_code}]{base_label}[/COLOR]" if color_code else base_label
        label       = f"{date_prefix}{colored}"
        
        li = xbmcgui.ListItem(label=label)

        # Artwork
        art = {}
        if ep_still:
            art.update({'thumb': ep_still, 'icon': ep_still})
        else:
            art['icon'] = 'DefaultVideo.png'
        if is_offline:
            art['thumb'] = 'DefaultHardDisk.png'
        if series_fanart:
            art['fanart'] = series_fanart
        if series_poster:
            art['poster'] = series_poster
        li.setArt(art)


        # === InfoTagVideo – moderní způsob bez deprekovaného setInfo() ===
        tag = li.getVideoInfoTag()
        # Titul a popis
        tag.setTitle(ep_title or f"Epizoda {int(episode_num)}")
        if ep_vote:
            tag.setRating(ep_vote)
        if ep_plot:
            tag.setPlot(ep_plot)
        # Datum odvysílání (string ISO 'YYYY-MM-DD')
        if ep_air:
            tag.setFirstAired(ep_air)
        # Sezóna a epizoda
        try:
            tag.setSeason(int(season_num))
        except Exception:
            pass
        try:
            tag.setEpisode(int(episode_num))
        except Exception:
            pass
        # Typ média
        tag.setMediaType('episode')
        # Délka v minutách
        if isinstance(ep_runtime, int) and ep_runtime > 0:
            tag.setDuration(ep_runtime)
        # Rok seriálu
        if isinstance(series_year, int):
            tag.setYear(series_year)

        best_stream = None
        for st in (ep.get('streams') or []):
            if isinstance(st, dict):
                best_stream = st
                break

        runtime_min = tm_ep.get('runtime') or show_runtime  # minuty

        if best_stream:
            feats = best_stream.get('features') or []
            vinfo, ainfo = _features_to_stream_info(feats, runtime_min)

            # Tohle dodá skinu podklady na spodní "krabičky"
            li.addStreamInfo('video', vinfo)
            li.addStreamInfo('audio', ainfo)
        else:
            # Nemáš streamy? Můžeš aspoň dodat runtime (sekundy) pro zobrazení délky:
            if isinstance(runtime_min, int) and runtime_min > 0:
                li.addStreamInfo('video', {'duration': runtime_min * 60})

        # Kontextové menu
        cm = [
            ("Označit jako zhlédnuté",
             f"RunPlugin({builder(action='series_mark_episode', series_name=series_name, season=season_num, episode=episode_num, progress=100)})"),
            ("Označit jako rozkoukané",
             f"RunPlugin({builder(action='series_mark_episode', series_name=series_name, season=season_num, episode=episode_num, progress=75)})"),
            ("Vymazat stav",
             f"RunPlugin({builder(action='series_mark_episode', series_name=series_name, season=season_num, episode=episode_num, progress=0)})"),
            ("Smazat epizodu",
             f"RunPlugin({builder(action='series_remove_episode', series_name=series_name, season=season_num, episode=episode_num)})"),
            ("Přidat do fronty",
             f"RunPlugin({builder(action='series_download_episode', series_name=series_name, season=season_num, episode=episode_num)});Container.Refresh"),
            ("Spustit frontu",
             f"RunPlugin({builder(action='queue_start_all')})"),
        ]
        if tm_series.get('id'):
            cm.append(
                ("Odeslat na Trakt…",
                 f"RunPlugin({builder(action='series_trakt_mark_episode_prompt', tmdb_id=int(tm_series.get('id')), season=int(season_num), episode=int(episode_num))})")
            )
        li.addContextMenuItems(cm, replaceItems=True)

        # HLAVNÍ LOGIKA: offline → přehrává rovnou, online → výběr streamu
        if is_offline and local_file_path:
            li.setProperty('IsPlayable', 'true')
            xbmcplugin.addDirectoryItem(handle, local_file_path, li, isFolder=False)
        else:
            li.setProperty('IsPlayable', 'false')
            url = builder(
                action='play',
                series_name=series_name,
                season=str(season_num),
                episode=str(episode_num),
                select='1'
            )
            xbmcplugin.addDirectoryItem(handle, url, li, isFolder=False)

    # Nastavení zobrazení
    xbmcplugin.setPluginCategory(handle, f"Epizody: Sezóna {season_num}")
    xbmcplugin.setContent(handle, "tvshows")
    xbmc.executebuiltin("Container.SetViewMode(50)")  # Wall view – nejlepší pro epizody
    xbmcplugin.endOfDirectory(handle)

# --- Nové pomocné API pro yawsp.py -------------------------------------------
def _format_stream_label(st):
    """Vytvoří label pro dialog: pouze podstatné vlastnosti (bez velikosti)."""
    feats = st.get('features')
    if isinstance(feats, list) and feats:
        return ', '.join(feats)
    # Fallback: spočítej jednoduše na místě
    try:
        name = st.get('name') or ''
        q    = st.get('quality') or ''
        ai   = st.get('ainfo') or ''
        lang = st.get('lang') or ''
        blob = ' '.join([name, q, ai, lang])
        tmp  = {'name': blob, 'quality': q, 'ainfo': ai, 'lang': lang}
        feats2 = SeriesManager._extract_features(tmp)
        return ', '.join(feats2) if feats2 else (_norm(st.get('label')) or _norm(st.get('quality')) or _norm(st.get('name')) or 'Stream')
    except Exception:
        return _norm(st.get('label')) or _norm(st.get('quality')) or _norm(st.get('name')) or 'Stream'


def get_episode_streams(series_manager_obj, series_name: str, season: str, episode: str):
    """Vrátí kurátorované streamy pro danou epizodu (seřazené)."""
    data = series_manager_obj.load_series_data(series_name)
    if not data:
        return []
    s = str(season); e = str(episode)
    ep = data.get('seasons', {}).get(s, {}).get(e)
    if not ep:
        return []
    return ep.get('streams', [])


def format_stream_two_lines(st: dict, runtime: int = None) -> (str, str):
    """
    Vrátí dvojici (line1, line2) pro dvouřádkový dialog výběru streamu.
    - line1: plný název streamu (původní filename z Webshare)
    - line2: klíčové vlastnosti + velikost GB + runtime v minutách (pokud je znám)
    """
    # 1) Line1 = celé jméno streamu (filename)
    line1 = (st.get('name') or '').strip()

    # 2) Line2 = slož z features; pokud chybí, dopočti on-the-fly
    feats = st.get('features')
    if not isinstance(feats, list) or not feats:
        try:
            tmp = {
                'name': st.get('name') or '',
                'quality': st.get('quality') or '',
                'ainfo': st.get('ainfo') or '',
                'lang': st.get('lang') or ''
            }
            feats = SeriesManager._extract_features(tmp)
        except Exception:
            feats = []

    # Velikost v GB (na konec řádku)
    size_gb = ''
    try:
        n = float(st.get('size') or 0.0)
        size_gb = f"{n/(1024*1024*1024):.1f} GB" if n > 0 else ''
    except Exception:
        size_gb = ''

    parts = []
    parts.extend(feats)
    if size_gb:
        parts.append(size_gb)
    if isinstance(runtime, int) and runtime > 0:
        parts.append(f"{runtime} min")

    line2 = ' • '.join(p.strip() for p in parts if p and p.strip())

    # Fallbacky
    if not line1:
        line1 = 'Stream'
    if not line2:
        line2 = (st.get('quality') or st.get('ident') or '').strip() or '—'

    return (line1, line2)


def _iter_saved_streams(series_data, season_only=None):
    """
    Vrací streamy napříč epizodami.
    Pokud season_only je číslo, vrací jen danou sezónu.
    """
    if not isinstance(series_data, dict):
        return
    seasons = series_data.get('seasons') or {}

    if isinstance(season_only, int):
        keys = [str(season_only)] if str(season_only) in seasons else []
    else:
        keys = sorted(seasons.keys(), key=lambda x: int(x) if str(x).isdigit() else x)

    for s_key in keys:
        season = seasons.get(s_key) or {}
        for e_key in sorted(season.keys(), key=lambda x: int(x) if str(x).isdigit() else x):
            ep = season.get(e_key) or {}
            for st in (ep.get('streams') or []):
                if isinstance(st, dict) and st.get('ident'):
                    try:
                        s_i = int(s_key)
                    except Exception:
                        s_i = s_key
                    try:
                        e_i = int(e_key)
                    except Exception:
                        e_i = e_key
                    yield (s_i, e_i, ep, st)

def _ask_season_filter():
    """
    Dotaz na číslo sezóny. Enter = všechny.
    """
    try:
        value = xbmcgui.Dialog().input(
            "Zadejte číslo sezóny (Enter = všechny)",
            type=xbmcgui.INPUT_NUMERIC
        )
    except Exception:
        value = xbmcgui.Dialog().input("Zadejte číslo sezóny (Enter = všechny)")

    if value is None:
        return None
    value = value.strip()
    if not value:
        return None
    return int(value) if value.isdigit() else None


def list_saved_streams(series_manager, handle, series_name: str, build_url_fn=None):
    """
    Vypíše všechny uložené streamy (napříč epizodami) do ListView.
    Kliknutí na položku => přidá stream do fronty (queue_add). Nepřehrává se.
    """
    builder = build_url_fn or _default_build_url
    data = series_manager.load_series_data(series_name)
    if not data:
        xbmcgui.Dialog().notification('YaWSP', 'Data seriálu nenalezena', xbmcgui.NOTIFICATION_WARNING)
        xbmcplugin.endOfDirectory(handle, succeeded=False)
        return

    tm = data.get('tmdb', {}) if isinstance(data, dict) else {}
    poster_url = tm.get('poster_url')
    fanart_url = tm.get('backdrop_url')
    series_overview = tm.get('overview') or ''
    series_year = tm.get('year')

    # Hlavička (neaktivní)

    title = data.get("name") or series_name
    header_label = (
        f"[COLOR lime]Uložené streamy[/COLOR] · "
        f"[COLOR deepskyblue]{title}[/COLOR]"
    )
    header = xbmcgui.ListItem(label=header_label)
    art = {'icon': 'DefaultHardDisk.png'}
    if poster_url: art['poster'] = poster_url
    if fanart_url: art['fanart'] = fanart_url
    header.setArt(art)
    if series_overview:
        header.setInfo('video', {'plot': series_overview})
    xbmcplugin.addDirectoryItem(handle, builder(action='noop'), header, isFolder=False)

    # Seznam streamů
    total = 0
    for s_i, e_i, ep, st in _iter_saved_streams(data):
        total += 1
        ep_title = ep.get('name') or ep.get('title') or ep.get('source_name') or ''
        left = f"S{int(s_i):02d}E{int(e_i):02d}"
        orig_name = st.get('name') or 'Stream'
        label = f"{left} · {orig_name}"

        li = xbmcgui.ListItem(label=label)
        li.setProperty('IsPlayable', 'false')  # klik NEPŘEHRÁVÁ

        # Artwork
        art_st = {'icon': 'DefaultVideo.png'}
        if fanart_url: art_st['fanart'] = fanart_url
        if poster_url: art_st['poster'] = poster_url
        li.setArt(art_st)

        # Info pro skiny
        info = {
            'title': ep_title or f'Epizoda {e_i}',
            'season': int(s_i) if str(s_i).isdigit() else s_i,
            'episode': int(e_i) if str(e_i).isdigit() else e_i,
            'mediatype': 'video'
        }
        if isinstance(series_year, int):
            info['year'] = series_year
        if series_overview:
            info['plot'] = series_overview
        li.setInfo('video', info)
        
        # >>> SEM vlož blok krabiček <<<
        feats = st.get('features') or []
        vinfo, ainfo = _features_to_stream_info(feats, runtime_min=None)
        li.addStreamInfo('video', vinfo)
        li.addStreamInfo('audio', ainfo)

        # Kontext: rychlý start fronty
        cm = [
            ('Spustit frontu', f'RunPlugin({builder(action="queue_start_all")})'),
        ]
        li.addContextMenuItems(cm, replaceItems=False)

        # Klik => queue_add (doplníme size pro ETA v menu fronty)
        url = builder(
            action='queue_add',
            ident=st.get('ident'),
            name=orig_name,
            series_name=series_name,
            season=str(s_i),
            episode=str(e_i),
            size=str(st.get('size') or 0)   # kvůli zobrazení GB + ETA
        )
        xbmcplugin.addDirectoryItem(handle, url, li, isFolder=False)

    # Kategorie / obsah a korektní ukončení adresáře
    xbmcplugin.setPluginCategory(handle, f'Uložené streamy: {data.get("name") or series_name} ({total})')
    xbmcplugin.setContent(handle, 'files')  # nechceme „přehrát“, jen operovat se soubory
    xbmcplugin.endOfDirectory(handle)
