
# -*- coding: utf-8 -*-
import os
import json
import hashlib
import xbmcvfs
import xbmcaddon
import xbmcgui
import xbmc
from urllib.parse import quote_plus

ADDON_ID = 'plugin.video.mmirousek_v2'
ADDON = xbmcaddon.Addon(ADDON_ID)
PROFILE = xbmcvfs.translatePath(ADDON.getAddonInfo('profile'))
ADDON_PATH = xbmcvfs.translatePath(ADDON.getAddonInfo('path'))

# --- Kořeny knihoven v addon data -------------------------------------------
SERIES_DB = os.path.join(PROFILE, 'series_db')            # JSONy seriálů
LIB_SERIES_ROOT = os.path.join(PROFILE, 'KodiLibrary', 'Serialy')    # seriály (.strm)
LIB_MOVIES_ROOT = os.path.join(PROFILE, 'KodiLibrary', 'Filmy')  # filmy (.strm)

# --- Index pro M3U export (inkrementální) -----------------------------------
M3U_INDEX_JSON = os.path.join(PROFILE, 'playlist_index.json')


# ---------------------------------------------------------------------------
# Pomocné: notifikace, dirs, safe názvy
# ---------------------------------------------------------------------------
def _notify(msg: str, icon=xbmcgui.NOTIFICATION_INFO, ms: int = 3500):
    try:
        xbmcgui.Dialog().notification(ADDON.getAddonInfo('name'), msg, icon, ms)
    except Exception:
        pass


def _ensure_dir(path: str):
    if not xbmcvfs.exists(path):
        try:
            xbmcvfs.mkdirs(path)
        except Exception:
            try:
                os.makedirs(path, exist_ok=True)
            except Exception:
                pass


def _safe_name(s: str) -> str:
    if not s:
        return "unknown"
    safe = (s.replace('/', '-')
              .replace('\\', '-')
              .replace(':', '-')
              .replace('\n', '-')
              .replace('*', '-')
              .replace('?', '-')
              .replace('"', "'")
              .strip(" .\t\r\n"))
    return safe or "unknown"


def _sha1(s: str) -> str:
    return hashlib.sha1((s or '').encode('utf-8')).hexdigest()


# =============================================================================
# 0) Chytrý výběr cesty k playlistu M3U
# =============================================================================
def _get_playlist_path() -> str:
    """
    Vrátí existující cestu k playlistu M3U v pořadí:
      1) Nastavení addonu (playlist_path) — pouze, pokud soubor existuje.
      2) Fallback v addonu: resources/media/playlist.m3u — pokud existuje.
      3) Kodi profily: special://profile/playlist.m3u, special://home/userdata/playlist.m3u — první existující.
    Pokud nic nenajde, vrátí původní hodnotu ze Settings (kvůli diagnostice).
    V logu vypíše, kterou variantu použil.
    """
    # 1) Uživatelské nastavení
    user_path_raw = ADDON.getSetting('playlist_path') or ''
    if user_path_raw:
        try:
            user_path = xbmcvfs.translatePath(user_path_raw)
        except Exception:
            user_path = user_path_raw
        try:
            if xbmcvfs.exists(user_path):
                xbmc.log(f"[library_export] Using playlist from settings: {user_path}", xbmc.LOGINFO)
                return user_path
            else:
                xbmc.log(f"[library_export] Settings path not found, will fallback: {user_path}", xbmc.LOGWARNING)
        except Exception:
            xbmc.log("[library_export] Failed to probe settings path, will fallback", xbmc.LOGWARNING)

    # 2) Fallback uvnitř addonu
    try:
        fallback = xbmcvfs.translatePath(ADDON_PATH + "/resources/media/playlist.m3u")
        if xbmcvfs.exists(fallback):
            xbmc.log(f"[library_export] Using fallback playlist: {fallback}", xbmc.LOGINFO)
            return fallback
        else:
            xbmc.log(f"[library_export] Fallback file missing: {fallback}", xbmc.LOGWARNING)
    except Exception:
        xbmc.log("[library_export] Failed to resolve addon fallback path", xbmc.LOGWARNING)

    # 3) Běžné umístění v Kodi profilu
    candidates = [
        "special://profile/playlist.m3u",
        "special://home/userdata/playlist.m3u",
    ]
    for cand in candidates:
        try:
            p = xbmcvfs.translatePath(cand)
        except Exception:
            p = cand
        try:
            if xbmcvfs.exists(p):
                xbmc.log(f"[library_export] Using profile candidate: {p}", xbmc.LOGINFO)
                return p
        except Exception:
            pass

    xbmc.log("[library_export] No valid playlist found. Returning last known settings/raw path.", xbmc.LOGWARNING)
    return user_path_raw or ''


# =============================================================================
# 1) Export SERIÁLŮ → .strm (Varianta A: action=play&select=1)
# =============================================================================
def export_series_to_kodi_library():
    """
    Projde všechny JSONy v series_db a vytvoří .strm soubory (Seriály).
    Každý .strm: plugin://.../?action=play&select=1&series_name=...&season=...&episode=...
    """
    try:
        _ensure_dir(LIB_SERIES_ROOT)
        count_series = 0
        count_files = 0
        dirs, files = xbmcvfs.listdir(SERIES_DB)
        for filename in files:
            if not filename.endswith('.json'):
                continue
            fp = os.path.join(SERIES_DB, filename)
            with xbmcvfs.File(fp) as f:
                data = json.loads(f.read())
            series_name = data.get('name') or filename.replace('.json', '')
            series_folder = os.path.join(LIB_SERIES_ROOT, _safe_name(series_name))
            _ensure_dir(series_folder)

            seasons = data.get('seasons', {})
            for season_key, episodes in (seasons.items() if isinstance(seasons, dict) else []):
                season_folder = os.path.join(series_folder, f"Sezona {season_key}")
                _ensure_dir(season_folder)

                for ep_key, ep_data in (episodes.items() if isinstance(episodes, dict) else []):
                    ep_title = ep_data.get('name') or ep_data.get('title') or f"Epizoda {ep_key}"
                    strm_filename = f"S{int(season_key):02d}E{int(ep_key):02d} - {_safe_name(ep_title)}.strm"
                    strm_path = os.path.join(season_folder, strm_filename)

                    # URL pro přehrání přes plugin: play(select=1)
                    plugin_url = (
                        f"plugin://{ADDON_ID}/?action=play"
                        f"&select=1&series_name={quote_plus(series_name)}"
                        f"&season={quote_plus(str(season_key))}"
                        f"&episode={quote_plus(str(ep_key))}"
                    )
                    with xbmcvfs.File(strm_path, 'w') as sf:
                        sf.write(plugin_url)

                    count_files += 1
                count_series += 1

        _notify(f"Kodi knihovna (Seriály): {count_series} seriálů, {count_files} epizod.", xbmcgui.NOTIFICATION_INFO, 4000)
    except Exception as e:
        _notify(f"Kodi knihovna (Seriály) – chyba: {e}", xbmcgui.NOTIFICATION_ERROR, 5000)


# =============================================================================
# 2) Export FILMŮ z playlist.m3u → .strm (s podporou FORCE režimu)
# =============================================================================
def _load_m3u_index():
    try:
        if xbmcvfs.exists(M3U_INDEX_JSON):
            with xbmcvfs.File(M3U_INDEX_JSON, 'r') as f:
                return json.loads(f.read())
    except Exception:
        pass
    return {"entries": {}, "playlist_hash": "", "playlist_mtime": 0}


def _save_m3u_index(idx: dict):
    try:
        with xbmcvfs.File(M3U_INDEX_JSON, 'w') as f:
            f.write(json.dumps(idx, ensure_ascii=False, indent=2))
    except Exception:
        try:
            with open(M3U_INDEX_JSON, 'w', encoding='utf-8') as f:
                json.dump(idx, f, ensure_ascii=False, indent=2)
        except Exception:
            pass


def _read_text_file(path: str) -> str:
    try:
        with xbmcvfs.File(path, 'r') as f:
            return f.read()
    except Exception:
        try:
            with open(path, 'r', encoding='utf-8', errors='ignore') as f:
                return f.read()
        except Exception:
            return ""


def _parse_m3u_entries(text: str):
    import re
    items = []
    if not text:
        return items
    lines = [l.strip() for l in text.replace('\r', '').split('\n') if l.strip()]
    i = 0
    while i < len(lines):
        line = lines[i]
        if line.startswith('#EXTINF'):
            try:
                title_part = line.split(',', 1)[1].strip()
            except Exception:
                title_part = "Unknown"
            ym = re.search(r'\((19\d{2}|20\d{2})\)', title_part)
            year = int(ym.group(1)) if ym else None
            title = _safe_name(title_part)

            url = None
            j = i + 1
            while j < len(lines):
                if not lines[j].startswith('#'):
                    url = lines[j].strip()
                    break
                j += 1

            if url:
                items.append({"title": title, "year": year, "url": url})
            i = j if j > i else i + 1
        else:
            i += 1
    return items


def _movie_folder_name(title: str, year: int = None) -> str:
    if year and 1900 <= int(year) <= 2099:
        return f"{title} ({int(year)})"
    return title


def _strm_path_for_movie(title: str, year: int = None) -> str:
    folder = os.path.join(LIB_MOVIES_ROOT, _movie_folder_name(title, year))
    _ensure_dir(folder)
    return os.path.join(folder, f"{_safe_name(title)}.strm")


def _write_movie_strm(path: str, url: str) -> bool:
    plugin_url = (
        f"plugin://{ADDON_ID}/?action=play_m3u_stream"
        f"&m3u_url={quote_plus(url)}"
    )
    try:
        with xbmcvfs.File(path, 'w') as f:
            f.write(plugin_url)
        return True
    except Exception:
        try:
            with open(path, 'w', encoding='utf-8') as f:
                f.write(plugin_url)
            return True
        except Exception:
            return False


# --- FORCE přepínač ----------------------------------------------------------
def _is_force_m3u_export() -> bool:
    """
    Dočasný přepínač 'force' pro M3U export.
    - ADDON setting 'm3u_export_force' == 'true'  → zapne
    - NEBO existuje soubor PROFILE/force_m3u_export.flag → zapne
    """
    try:
        if (ADDON.getSetting('m3u_export_force') or '').lower() == 'true':
            return True
    except Exception:
        pass
    try:
        flag_path = os.path.join(PROFILE, 'force_m3u_export.flag')
        return xbmcvfs.exists(flag_path)
    except Exception:
        return False


def export_movies_from_m3u(force: bool = False):
    """
    Inkrementální export filmů z playlist.m3u do KodiMovies/Filmy (.strm).
    Pokud je 'force' True nebo je aktivní _is_force_m3u_export(),
    provede export vždy (ignoruje index/změnu souboru).
    """
    _ensure_dir(LIB_MOVIES_ROOT)

    pl_path = _get_playlist_path()
    if not xbmcvfs.exists(pl_path):
        _notify(f"Playlist nenalezen: {pl_path}", xbmcgui.NOTIFICATION_WARNING)
        return

    text = _read_text_file(pl_path)

    # mtime + hash (pro běžný režim)
    try:
        st = xbmcvfs.Stat(pl_path)
        mtime = int(getattr(st, "st_mtime", lambda: 0)() if callable(getattr(st, "st_mtime", None)) else 0)
    except Exception:
        try:
            mtime = int(os.path.getmtime(pl_path))
        except Exception:
            mtime = 0
    pl_hash = _sha1(text)

    # načti index
    idx = _load_m3u_index()

    # dočasný FORCE přepínač (runtime)
    force_runtime = bool(force) or _is_force_m3u_export()

    # je playlist změněný?
    changed = force_runtime or (idx.get("playlist_hash") != pl_hash or int(idx.get("playlist_mtime", 0)) != mtime)

    entries = _parse_m3u_entries(text)
    if not entries:
        _notify("Playlist je prázdný nebo nečitelný.", xbmcgui.NOTIFICATION_WARNING)
        return

    exported = 0
    idx_entries = idx.get("entries", {})

    for item in entries:
        url = item["url"]
        title = item["title"]
        year = item.get("year")
        key = _sha1(url)

        if not force_runtime:
            # běžný režim: když položka už v indexu je a playlist se nezměnil → přeskoč
            if key in idx_entries and not changed:
                continue
            # ochrana proti duplicitnímu přidání téže položky v rámci jednoho běhu
            if key in idx_entries:
                continue

        strm_path = _strm_path_for_movie(title, year)

        # zápis .strm vždy přepíše (xbmcvfs.File(..., 'w'))
        ok = _write_movie_strm(strm_path, url)
        if not ok:
            # fallback jméno s hash prefixem; pokus o zápis znovu
            strm_path = _strm_path_for_movie(f"{title} - {key[:6]}", year)
            _write_movie_strm(strm_path, url)

        # aktualizuj index (i v force režimu – ať máme poslední umístění .strm)
        idx_entries[key] = {
            "title": title,
            "year": year,
            "strm_path": strm_path,
            "url": url
        }
        exported += 1

    # ulož index + metadata playlistu
    idx["entries"] = idx_entries
    idx["playlist_mtime"] = mtime
    idx["playlist_hash"] = pl_hash
    _save_m3u_index(idx)

    if exported > 0:
        mode_txt = " (FORCE)" if force_runtime else ""
        _notify(f"M3U → knihovna: +{exported} filmů{mode_txt}", xbmcgui.NOTIFICATION_INFO)
    else:
        _notify("Playlist beze změny (žádné nové filmy).", xbmcgui.NOTIFICATION_INFO)
