# -*- coding: utf-8 -*-
"""
trakt_module.py — Trakt.tv integrace pro plugin.video.mmirousek
- TraktAPI: obálka nad Trakt API (GET/POST, token hlavičky, cache)
- FileCache: gzip JSON cache (namespace 'trakt')
- ProgressCacheManager: lokální JSON cache pokroku seriálů (cache-first + full refresh na vyžádání s progress barem)
- TraktMenu: renderery (Filmy/Seriály huby, Veřejné seznamy, Můj Trakt.tv, Watchlist, Kalendář,
  Pokrok sledování, Kolekce, Historie filmů/seriálů, People/Related/Next-Last)
- Prompt helpery: trakt_prompt_mark_season(), trakt_prompt_mark_episode()
- Utility: trakt_cache_invalidate_people(), trakt_cache_clear_all()
"""
import sys
import os
import json
import time
import gzip
import hashlib
from typing import Any, Dict, Optional, List
import requests
import xbmc
import xbmcvfs
import xbmcaddon
import xbmcgui
import xbmcplugin
from xbmcvfs import translatePath
from urllib.parse import urlencode
from datetime import datetime

from resources.lib.ui_utils import build_trakt_context

# TMDb obrázky (poster/profile base URL) — používáme pro postery i profilové fotky
from resources.lib.tmdb_module import IMAGE_BASE_URL_POSTER

ADDON = xbmcaddon.Addon('plugin.video.mmirousek')
_url = sys.argv[0]

# === KONFIGURACE TRAKT ===
TRAKT_CLIENT_ID = "4c2c45518d4accf713853b837c1251d787878434d76e71ca279db0aae40ae343"  # konstantní Client ID
TRAKT_CLIENT_SECRET = ADDON.getSetting('trakt_client_secret') or ""                    # musí být vyplněno v Settings
TRAKT_API_URL = "https://api.trakt.tv"

def get_url(**kwargs) -> str:
    """Bezpečné kódování parametrů do plugin URL."""
    return f'{_url}?{urlencode(kwargs, encoding="utf-8")}'

# === IKONY PODMENU (resources/media/icons/trakt/*.png) =======================
def _icon(name: str) -> str:
    """
    Vrátí plnou cestu k ikoně v resources/media/icons/trakt/<name>.png.
    """
    try:
        base = ADDON.getAddonInfo('path')
        p = os.path.join(base, 'resources', 'media', 'icons', 'trakt', f'{name}.png')
        return p
    except Exception:
        return 'DefaultFolder.png'

# === TRAKT HEADERS / TOKEN ===================================================
def _trakt_headers_base() -> Dict[str, str]:
    h = {
        'trakt-api-version': '2',
        'trakt-api-key': TRAKT_CLIENT_ID,
        'User-Agent': 'plugin.video.mmirousek/1.0 (+Kodi)'
    }
    try:
        lang = ADDON.getSetting('trakt_lang') or ''
        if lang:
            h['Accept-Language'] = lang
    except Exception:
        pass
    return h

def _auth_headers(for_post: bool = False) -> Optional[Dict[str, str]]:
    token = ADDON.getSetting('trakt_access_token')
    if not token:
        return None
    h = _trakt_headers_base()
    h['Authorization'] = f'Bearer {token}'
    if for_post:
        h['Content-Type'] = 'application/json'
    return h

def get_trakt_headers(for_post: bool = False):
    """
    Vrátí hlavičky pro volání Trakt API.
    - for_post=True → vyžaduje platný access token, jinak None
    - for_post=False → pokud není token, vrátí veřejné hlavičky (api-key, UA)
    """
    h_auth = _auth_headers(for_post=for_post)
    if h_auth:
        return h_auth
    if for_post:
        return None
    return _trakt_headers_base()

# === FILE CACHE (gzip) =======================================================
class FileCache:
    """Jednoduchý souborový cache layer (gzip JSON)."""
    def __init__(self, namespace: str = 'trakt'):
        profile = translatePath(ADDON.getAddonInfo('profile'))
        cache_root = os.path.join(profile, f'{namespace}_cache')
        os.makedirs(cache_root, exist_ok=True)
        self.root = cache_root

    @staticmethod
    def _hash_key(key: str) -> str:
        return hashlib.md5(key.encode('utf-8')).hexdigest()

    def _path(self, key: str) -> str:
        fname = self._hash_key(key) + '.json.gz'
        return os.path.join(self.root, fname)

    def get(self, key: str) -> Optional[Dict[str, Any]]:
        path = self._path(key)
        if not os.path.exists(path):
            return None
        try:
            with gzip.open(path, 'rt', encoding='utf-8') as f:
                obj = json.load(f)
            ts = obj.get('ts', 0)
            ttl = obj.get('ttl', 0)
            if ttl > 0 and (time.time() - ts) > ttl:
                try:
                    os.remove(path)
                except Exception:
                    pass
                return None
            return obj
        except Exception as e:
            xbmc.log(f"[TraktCache] Read fail: {e}", xbmc.LOGWARNING)
            return None

    def set(self, key: str, data: Any, hdr: Optional[Dict[str, Any]], ttl_sec: int):
        path = self._path(key)
        payload = {'ts': int(time.time()), 'ttl': int(ttl_sec), 'hdr': hdr or {}, 'data': data}
        try:
            with gzip.open(path, 'wt', encoding='utf-8') as f:
                json.dump(payload, f, ensure_ascii=False)
        except Exception as e:
            xbmc.log(f"[TraktCache] Write fail: {e}", xbmc.LOGWARNING)

    def clear_namespace(self):
        try:
            for fname in os.listdir(self.root):
                if fname.endswith('.gz'):
                    os.remove(os.path.join(self.root, fname))
        except Exception as e:
            xbmc.log(f"[TraktCache] Clear namespace error: {e}", xbmc.LOGERROR)

# TTL (sekundy)

# === TTL (sekundy) ============================================================

# === TTL (sekundy) ============================================================
TTL_MEDIUM = 10 * 60        # 10 min – dříve používané
TTL_LONG   = 24 * 60 * 60   # 24 h – people, genres
# ⏩ Nové: rychlé žebříčky / public lists (Trakt data)
TTL_FAST   = 60 * 60        # 60 min – trending/popular/anticipated, public lists
# ⏩ Nové: TMDb detaily (postery/plot) – chceme držet déle
TTL_TMDB_DETAILS = 14 * 24 * 60 * 60  # 7 dní
# ⏩ Prefetch TMDb na pozadí – limitujeme počet paralelních vláken
TMDB_PREFETCH_WORKERS = 4


# === TMDb details cache (poster/plot/year) ====================================
_TMDBCACHE = FileCache(namespace='tmdb')

def _tmdb_cache_key(media_type: str, tmdb_id: int) -> str:
    # Jednoduchý klíč: tmdb:<type>:<id>
    return f"tmdb:{('movie' if media_type=='movie' else 'tv')}:{int(tmdb_id)}"

def get_tmdb_details_from_cache_only(tmdb_id: int, media_type: str):
    """
    Vrátí cached TMDb detaily nebo None (bez síťového volání).
    """
    if not tmdb_id:
        return None
    key = _tmdb_cache_key(media_type, tmdb_id)
    obj = _TMDBCACHE.get(key)
    # FileCache.get vrací dict {'ts','ttl','hdr','data'} nebo None
    return (obj or {}).get('data') if obj else None

def store_tmdb_details(tmdb_id: int, media_type: str, details: dict, ttl_sec: int = TTL_TMDB_DETAILS):
    """
    Uloží TMDb detaily do cache (gzip JSON).
    """
    if not (tmdb_id and isinstance(details, dict) and details):
        return
    key = _tmdb_cache_key(media_type, tmdb_id)
    _TMDBCACHE.set(key, details, hdr=None, ttl_sec=ttl_sec)

# Pozadí: omezená paralelizace
import threading
_TMDB_SEM = threading.Semaphore(TMDB_PREFETCH_WORKERS)

def _prefetch_tmdb_detail_async(tmdb_id: int, media_type: str):
    """
    Na pozadí stáhne TMDb detaily (CZ) a uloží do cache, pokud tam nejsou.
    První návštěva listu tak bude rychlá; další už využije cached poster+plot.
    """
    if not tmdb_id:
        return

    # Pokud už je v cache, nic nedělej
    existing = get_tmdb_details_from_cache_only(tmdb_id, media_type)
    if existing:
        return

    def _task():
        with _TMDB_SEM:
            try:
                # Reuse existující fallback fetcher (v souboru je get_tmdb_details_fallback)
                det = get_tmdb_details_fallback(tmdb_id, 'movie' if media_type == 'movie' else 'tv')
                if det:
                    store_tmdb_details(tmdb_id, media_type, det, ttl_sec=TTL_TMDB_DETAILS)
                    xbmc.log(f"[TMDb Prefetch] cached {media_type} tmdb_id={tmdb_id}", xbmc.LOGINFO)
            except Exception as e:
                xbmc.log(f"[TMDb Prefetch] error tmdb_id={tmdb_id}: {e}", xbmc.LOGWARNING)

    try:
        threading.Thread(target=_task, name=f"tmdb_prefetch_{media_type}_{tmdb_id}", daemon=True).start()
    except Exception:
        # Fallback bez threadingu (neblokovat listing!)
        xbmc.log("[TMDb Prefetch] failed to start thread; skipping", xbmc.LOGWARNING)

def _build_cache_key(endpoint: str, params: Optional[Dict[str, Any]], auth: bool) -> str:
    p = params or {}
    try:
        p_ser = json.dumps(p, sort_keys=True, ensure_ascii=False)
    except Exception:
        p_ser = str(p)
    return f"{endpoint}\n{p_ser}\n{'A' if auth else 'N'}"

def _pick_pagination_headers(h: Dict[str, Any]) -> Dict[str, Any]:
    keep = {}
    for k, v in (h.items() if isinstance(h, dict) else []):
        if isinstance(k, str) and k.lower().startswith('x-pagination-'):
            keep[k] = v
    return keep

# === Pomocné formátování CZ data =============================================
def _format_cz_date(iso_date: str) -> str:
    """
    Převod 'YYYY-MM-DDTHH:MM:SSZ' -> 'dd. mm. yyyy' (bez závislosti na platformě).
    Bezpečný pro Windows i Linux.
    """
    try:
        import datetime
        d = (iso_date or '').strip()
        if not d:
            return ''
        date_part = d.split('T')[0]
        dt = datetime.datetime.strptime(date_part, '%Y-%m-%d')
        return dt.strftime('%d.%m.%Y')
    except Exception:
        return (iso_date or '').split('T')[0]

# === Trakt API obálka ========================================================
class TraktAPI:
    def __init__(self) -> None:
        self.base = TRAKT_API_URL
        self.last_headers: Dict[str, str] = {}
        self.cache = FileCache('trakt')

    # --- GET wrapper ---
    def get(self, endpoint: str, params: Optional[Dict[str, Any]] = None,
            auth: bool = True, timeout: int = 12, cache_ttl: int = 0,
            cache_key: Optional[str] = None) -> Optional[Any]:
        url = f"{self.base}/{endpoint.lstrip('/')}"
        headers = _auth_headers(False) if auth else _trakt_headers_base()
        if auth and not headers:
            return None
        key = cache_key or _build_cache_key(endpoint, params, auth)
        if cache_ttl > 0:
            cached = self.cache.get(key)
            if cached is not None:
                self.last_headers = cached.get('hdr', {}) or {}
                xbmc.log(f"[TraktCache] HIT {endpoint} {params}", xbmc.LOGINFO)
                return cached.get('data')
        try:
            r = requests.get(url, headers=headers, params=params, timeout=timeout)
            if r.status_code == 429:
                time.sleep(int(r.headers.get('Retry-After', '3')))
                r = requests.get(url, headers=headers, params=params, timeout=timeout)
            r.raise_for_status()
            self.last_headers = dict(r.headers) if hasattr(r, 'headers') else {}
            data = r.json()
            if cache_ttl > 0:
                self.cache.set(key, data, _pick_pagination_headers(self.last_headers), cache_ttl)
            return data
        except Exception as e:
            xbmc.log(f"TRAKT API ERROR: {e} [{endpoint}]", xbmc.LOGERROR)
            return None

    # --- Public endpoints / helpers ---
    def genres(self, media_type: str) -> List[Dict[str, Any]]:
        return self.get(f'genres/{media_type}', auth=False, cache_ttl=TTL_LONG) or []

    def movies(self, mode: str, page: int, limit: int,
               genres: Optional[str], years: Optional[str]):
        endpoint = f"movies/{mode}"
        params = {'extended': 'min', 'page': page, 'limit': limit}
        if genres: params['genres'] = genres
        if years:  params['years']  = years
        return self.get(endpoint, params=params, auth=False, cache_ttl=TTL_FAST)

    def shows(self, mode: str, page: int, limit: int,
              genres: Optional[str], years: Optional[str]):
        endpoint = f"shows/{mode}"
        params = {'extended': 'min', 'page': page, 'limit': limit}
        if genres: params['genres'] = genres
        if years:  params['years']  = years
        return self.get(endpoint, params=params, auth=False, cache_ttl=TTL_FAST)

    def list_popular_or_trending(self, mode: str, page: int, limit: int):
        endpoint = 'lists/popular' if mode == 'popular' else 'lists/trending'
        params = {'extended': 'min', 'page': page, 'limit': limit}
        return self.get(endpoint, params=params, auth=False, cache_ttl=TTL_FAST)

    def list_items(self, list_id: str, page: int, limit: int):
        endpoint = f'lists/{list_id}/items'
        params = {'extended': 'min', 'page': page, 'limit': limit}
        return self.get(endpoint, params=params, auth=True, cache_ttl=TTL_FAST)

    def people(self, media_type: str, trakt_id_or_slug: str, cache_ttl: int = TTL_LONG, auth: bool = False):
        endpoint = f'{media_type}/{trakt_id_or_slug}/people'
        xbmc.log(f"[TraktAPI.people] endpoint={endpoint} auth={'A' if auth else 'N'} ttl={cache_ttl}", xbmc.LOGINFO)
        return self.get(endpoint, params=None, auth=auth, cache_ttl=cache_ttl)

    def related(self, media_type: str, trakt_id_or_slug: str, page: int, limit: int):
        endpoint = f'{media_type}/{trakt_id_or_slug}/related'
        params = {'extended': 'full', 'page': page, 'limit': limit}
        return self.get(endpoint, params=params, auth=True, cache_ttl=TTL_MEDIUM)

    def users_watched_shows(self, page: int = 1, limit: int = 200,
                            extended: bool = False, extended_str: Optional[str] = 'min'):
        endpoint = 'users/me/watched/shows'
        params = {'page': page, 'limit': limit}
        params['extended'] = extended_str or ('full' if extended else 'min')
        return self.get(endpoint, params=params, auth=True, cache_ttl=TTL_MEDIUM) or []

    def users_watched_movies(self, page: int = 1, limit: int = 200,
                             extended: bool = True, extended_str: Optional[str] = 'full'):
        """
        Historie filmů uživatele (watched movies) — vrací i plays/last_watched_at.
        """
        endpoint = 'users/me/watched/movies'
        params = {'page': page, 'limit': limit}
        params['extended'] = extended_str or ('full' if extended else 'min')
        return self.get(endpoint, params=params, auth=True, cache_ttl=0) or []

    def users_watched_shows_history(self, page: int = 1, limit: int = 200,
                                    extended: bool = True, extended_str: Optional[str] = 'full'):
        """
        Historie seriálů uživatele (watched shows) — vrací i plays/last_watched_at.
        """
        endpoint = 'users/me/watched/shows'
        params = {'page': page, 'limit': limit}
        params['extended'] = extended_str or ('full' if extended else 'min')
        return self.get(endpoint, params=params, auth=True, cache_ttl=0) or []

    def users_history(self, type_: str = 'episodes',
                      start_at: Optional[str] = None, end_at: Optional[str] = None,
                      page: int = 1, limit: int = 200, extended: str = 'min',
                      cache_ttl: int = TTL_MEDIUM):
        endpoint = f'users/me/history/{type_}'
        params = {'page': page, 'limit': limit}
        if start_at: params['start_at'] = start_at
        if end_at:   params['end_at'] = end_at
        if extended: params['extended'] = extended
        return self.get(endpoint, params=params, auth=True, cache_ttl=cache_ttl) or []

    def shows_progress_watched(self, trakt_id_or_slug: str,
                               hidden: bool = False, specials: bool = False,
                               count_specials: bool = False, last_activity: bool = True,
                               cache_ttl: int = 0):
        if not trakt_id_or_slug:
            return None
        endpoint = f"shows/{trakt_id_or_slug}/progress/watched"
        params = {
            'hidden': 'true' if hidden else 'false',
            'specials': 'true' if specials else 'false',
            'count_specials': 'true' if count_specials else 'false',
            'last_activity': 'true' if last_activity else 'false'
        }
        return self.get(endpoint, params=params, auth=True, cache_ttl=cache_ttl)

    def last_activities(self) -> Optional[Dict[str, Any]]:
        return self.get('sync/last_activities', auth=True, cache_ttl=0)

    # NEW — show summary (public endpoint) — doplnění titulu/roku/ids/overview
    def show_summary(self, trakt_id_or_slug: str, extended: str = 'min') -> Dict[str, Any]:
        if not trakt_id_or_slug:
            return {}
        endpoint = f'shows/{trakt_id_or_slug}'
        params = {'extended': extended} if extended else None
        data = self.get(endpoint, params=params, auth=False, cache_ttl=TTL_LONG) or {}
        if isinstance(data, dict):
            return {
                'title': data.get('title') or '',
                'year': data.get('year'),
                'ids': data.get('ids') or {},
                'overview': data.get('overview') or ''
            }
        return {}

    # --- Collections (Movies/Shows) ---
    def collection_movies(self, sort_desc: bool = True):
        data = self.get('users/me/collection/movies', params={'extended': 'full'}, auth=True, cache_ttl=0) or []
        def _sort_key(row): return (row.get('listed_at') or row.get('collected_at') or '')
        data.sort(key=_sort_key, reverse=sort_desc)
        return data

    def collection_shows(self, sort_desc: bool = True):
        data = self.get('users/me/collection/shows', params={'extended': 'full'}, auth=True, cache_ttl=0) or []
        def _sort_key(row): return (row.get('listed_at') or row.get('collected_at') or '')
        data.sort(key=_sort_key, reverse=sort_desc)
        return data

    # --- Authorization (Device Code flow) ---
    def authorize_trakt(self) -> None:
        """
        Device Code flow – zobrazí kód, uživatel jej potvrdí na https://trakt.tv/activate,
        poté získá token a uloží do settings.
        """
        try:
            client_id = TRAKT_CLIENT_ID
            client_secret = TRAKT_CLIENT_SECRET
            if not client_id or not client_secret:
                raise ValueError("Chybí Trakt client_id / client_secret v nastavení.")
            # Step 1: device code
            r = requests.post(f"{self.base}/oauth/device/code",
                              json={'client_id': client_id}, timeout=10)
            r.raise_for_status()
            device = r.json()  # {'device_code','user_code','verification_url','expires_in','interval'}
            user_code = device.get('user_code')
            verify_url = device.get('verification_url')
            interval = int(device.get('interval', 5))
            expires_in = int(device.get('expires_in', 600))
            xbmcgui.Dialog().ok("Trakt přihlášení",
                                f"Na zařízení otevři: {verify_url}\n\nZadej kód: {user_code}\n\nA potvrď přístup.")
            # Step 2: poll for token
            token = None
            start = time.time()
            while (time.time() - start) < expires_in:
                time.sleep(interval)
                rr = requests.post(f"{self.base}/oauth/device/token",
                                   json={'client_id': client_id,
                                         'client_secret': client_secret,
                                         'code': device.get('device_code')},
                                   timeout=10)
                if rr.status_code == 200:
                    token_data = rr.json()
                    token = token_data.get('access_token')
                    refresh = token_data.get('refresh_token')
                    if token:
                        ADDON.setSetting('trakt_access_token', token)
                        if refresh:
                            ADDON.setSetting('trakt_refresh_token', refresh)
                        xbmcgui.Dialog().notification('Trakt', 'Přihlášení úspěšné.', xbmcgui.NOTIFICATION_INFO, 2500)
                        try:
                            update_trakt_status_setting()
                        except Exception:
                            pass
                        return
                elif rr.status_code in (400, 401, 403):
                    continue
            xbmcgui.Dialog().notification('Trakt', 'Přihlášení vypršelo.', xbmcgui.NOTIFICATION_WARNING, 3000)
        except Exception as e:
            xbmc.log(f"[TraktAuth] error: {e}", xbmc.LOGERROR)
            xbmcgui.Dialog().notification('Trakt', 'Chyba při autorizaci. Viz log.', xbmcgui.NOTIFICATION_ERROR, 3000)

# === Progress cache manager (cache-first) ====================================
class ProgressCacheManager:
    """
    Správa lokální cache pokroku seriálů:
    - progress_cache.json se strukturou:
    {
      "last_activity": ISO8601,
      "shows": {
        "<trakt_or_slug>": {
          "title": str,
          "year": int,
          "ids": {"trakt": int, "slug": str, "tmdb": int},
          "overview": str,
          "last_activity": ISO8601 "",
          "progress": {
            "aired": int,
            "completed": int,
            "next_episode": {"season": int, "number": int, "title": str} {}
          }
        }
      },
      "_last_full_fetch_ts": int (epoch) [informativně]
    }
    """
    def __init__(self, api: TraktAPI):
        self.api = api
        self.profile_root = translatePath(ADDON.getAddonInfo('profile'))
        os.makedirs(self.profile_root, exist_ok=True)
        self.cache_path = os.path.join(self.profile_root, 'progress_cache.json')

    def load_cache(self) -> Dict[str, Any]:
        if not os.path.exists(self.cache_path):
            return {"last_activity": "", "shows": {}}
        try:
            with open(self.cache_path, 'r', encoding='utf-8') as f:
                data = json.load(f)
            if not isinstance(data, dict):
                return {"last_activity": "", "shows": {}}
            data.setdefault("last_activity", "")
            data.setdefault("shows", {})
            return data
        except Exception as e:
            xbmc.log(f"[ProgressCache] read error: {e}", xbmc.LOGWARNING)
            return {"last_activity": "", "shows": {}}

    def save_cache(self, cache: Dict[str, Any]) -> None:
        try:
            with open(self.cache_path, 'w', encoding='utf-8') as f:
                json.dump(cache, f, ensure_ascii=False, indent=2)
            xbmc.log("[ProgressCache] cache saved", xbmc.LOGINFO)
        except Exception as e:
            xbmc.log(f"[ProgressCache] write error: {e}", xbmc.LOGERROR)

    def _compact_show_entry(self, show: Dict[str, Any], prog: Dict[str, Any]) -> Dict[str, Any]:
        ids = show.get('ids', {}) or {}
        title = show.get('title') or ''
        year = show.get('year')
        overview = show.get('overview') or ''
        p_air = (prog.get('aired') or 0) if isinstance(prog, dict) else 0
        p_comp = (prog.get('completed') or 0) if isinstance(prog, dict) else 0
        next_ep = (prog.get('next_episode') or {}) if isinstance(prog, dict) else {}
        last_act = (prog.get('last_activity') or '') if isinstance(prog, dict) else ''
        return {
            "title": title,
            "year": year,
            "ids": {"trakt": ids.get('trakt'), "slug": ids.get('slug'), "tmdb": ids.get('tmdb')},
            "overview": overview,
            "last_activity": last_act,
            "progress": {
                "aired": int(p_air),
                "completed": int(p_comp),
                "next_episode": {
                    "season": next_ep.get('season') if isinstance(next_ep.get('season'), int) else None,
                    "number": next_ep.get('number') if isinstance(next_ep.get('number'), int) else None,
                    "title": next_ep.get('title') or ''
                } if isinstance(next_ep, dict) else {}
            }
        }

    def _full_fetch(self) -> Dict[str, Any]:
        xbmc.log("[ProgressCache] full fetch start", xbmc.LOGINFO)
        cache = {"last_activity": "", "shows": {}}
        watched = self.api.users_watched_shows(page=1, limit=200, extended=False, extended_str='min') or []
        show_items: List[Dict[str, Any]] = []
        for row in watched:
            if isinstance(row, dict):
                sh = row.get('show') or row or {}
            elif isinstance(row, str):
                sh = {'title': '', 'year': None, 'ids': {'slug': row}, 'overview': ''}
            else:
                continue
            ids = sh.get('ids') or {}
            ident = ids.get('trakt') or ids.get('slug')
            # doplnit summary pokud chybí title/rok/tmdb
            if ident and (not sh.get('title') or sh.get('year') in (None, 0) or not ids.get('tmdb')):
                summary = self.api.show_summary(ident, extended='min') or {}
                if summary:
                    sh['title'] = summary.get('title') or sh.get('title') or ''
                    sh['year'] = summary.get('year') if summary.get('year') is not None else sh.get('year')
                    sh['overview'] = summary.get('overview') or sh.get('overview') or ''
                    sids = summary.get('ids') or {}
                    ids.setdefault('tmdb', sids.get('tmdb'))
                    ids.setdefault('slug', sids.get('slug'))
                    ids.setdefault('trakt', sids.get('trakt'))
                    sh['ids'] = ids
            if ident:
                show_items.append(sh)
        for idx, sh in enumerate(show_items):
            ids = sh.get('ids') or {}
            ident = ids.get('trakt') or ids.get('slug')
            if not ident:
                continue
            prog = self.api.shows_progress_watched(
                ident, hidden=False, specials=False, count_specials=False,
                last_activity=True, cache_ttl=0
            ) or {}
            entry = self._compact_show_entry(sh, prog)
            cache["shows"][str(ident)] = entry
            if (idx + 1) % 20 == 0:
                xbmc.log(f"[ProgressCache] fetched {idx + 1}/{len(show_items)}", xbmc.LOGINFO)
        act = self.api.last_activities() or {}
        episodes_ts = ((act.get('episodes') or {}).get('watched_at')) or ''
        cache["last_activity"] = episodes_ts or ''
        xbmc.log("[ProgressCache] full fetch done", xbmc.LOGINFO)
        return cache

    def ensure_cache(self) -> Dict[str, Any]:
        """
        Cache-first:
        - Pokud lokální cache neexistuje / je prázdná → full fetch + uložení.
        - Jinak vrátí lokální cache BEZ jakéhokoli kontaktu se serverem.
        """
        cache = self.load_cache()
        if not cache.get("shows"):
            xbmc.log("[ProgressCache] init: empty cache → full fetch", xbmc.LOGINFO)
            cache = self._full_fetch()
            self.save_cache(cache)
            return cache
        xbmc.log("[ProgressCache] using local cache (no server checks)", xbmc.LOGINFO)
        return cache

    def full_fetch_with_progress(self, show_cancel_button: bool = True, chunk_log_step: int = 20) -> bool:
        """
        Full fetch s progress barem a možností zrušit.
        - Stáhne watched shows → postupně progress/watched pro každou show.
        - Průběžně informuje dialog (2-argumentové update kvůli kompatibilitě Kodi API).
        - Po dokončení uloží cache a vrátí True; při zrušení False.
        """
        dlg = None
        try:
            dlg = xbmcgui.DialogProgress()
            dlg.create("Trakt • Aktualizace cache", "Získávám seznam rozkoukaných seriálů...")

            watched = self.api.users_watched_shows(page=1, limit=200, extended=False, extended_str='min') or []
            show_items: List[Dict[str, Any]] = []
            for row in watched:
                if isinstance(row, dict):
                    sh = row.get('show') or row or {}
                elif isinstance(row, str):
                    sh = {'title': '', 'year': None, 'ids': {'slug': row}, 'overview': ''}
                else:
                    continue
                ids = sh.get('ids') or {}
                ident = ids.get('trakt') or ids.get('slug')
                if ident:
                    show_items.append(sh)

            total = len(show_items)
            if total == 0:
                dlg.update(100, "Žádné rozkoukané seriály.")
                dlg.close()
                xbmcgui.Dialog().notification("Trakt", "Žádné rozkoukané seriály.", xbmcgui.NOTIFICATION_INFO, 2500)
                return True

            cache = {"last_activity": "", "shows": {}}
            for idx, sh in enumerate(show_items, start=1):
                if dlg.iscanceled():
                    dlg.close()
                    xbmc.log("[ProgressCache] full fetch canceled by user", xbmc.LOGINFO)
                    return False

                ident = (sh.get('ids') or {}).get('trakt') or (sh.get('ids') or {}).get('slug')
                if ident and (not sh.get('title') or sh.get('year') in (None, 0) or not (sh.get('ids') or {}).get('tmdb')):
                    summary = self.api.show_summary(str(ident), extended='min') or {}
                    sids = summary.get('ids') or {}
                    ids = sh.get('ids') or {}
                    ids.setdefault('tmdb', sids.get('tmdb'))
                    ids.setdefault('slug', sids.get('slug'))
                    ids.setdefault('trakt', sids.get('trakt'))
                    sh['ids'] = ids
                    sh['title'] = summary.get('title') or sh.get('title') or ''
                    sh['year'] = summary.get('year') if summary.get('year') is not None else sh.get('year')
                    sh['overview'] = summary.get('overview') or sh.get('overview') or ''

                prog = self.api.shows_progress_watched(
                    str(ident), hidden=False, specials=False, count_specials=False,
                    last_activity=True, cache_ttl=0
                ) or {}
                entry = self._compact_show_entry(sh, prog)
                cache["shows"][str(ident)] = entry

                percent = int((idx / float(total)) * 100.0)
                msg = f"Stahuji progres {idx}/{total}\n{sh.get('title') or str(ident)}"
                dlg.update(percent, msg)
                if idx % chunk_log_step == 0:
                    xbmc.log(f"[ProgressCache] fetched {idx}/{total}", xbmc.LOGINFO)

            act = self.api.last_activities() or {}
            episodes_ts = ((act.get('episodes') or {}).get('watched_at')) or ''
            cache["last_activity"] = episodes_ts or ''
            try:
                import time as _t
                cache["_last_full_fetch_ts"] = int(_t.time())
            except Exception:
                pass

            self.save_cache(cache)
            dlg.update(100, "Hotovo.")
            dlg.close()
            xbmcgui.Dialog().notification("Trakt", "Cache pokroku aktualizována.", xbmcgui.NOTIFICATION_INFO, 2500)
            return True

        except Exception as e:
            if dlg:
                try:
                    dlg.close()
                except Exception:
                    pass
            xbmc.log(f"[ProgressCache] full_fetch_with_progress error: {e}", xbmc.LOGERROR)
            xbmcgui.Dialog().notification("Trakt", "Chyba při aktualizaci cache.", xbmcgui.NOTIFICATION_ERROR, 3000)
            return False

    def drop_show_from_cache(self, ident: str) -> bool:
        try:
            cache = self.load_cache()
            shows = cache.get("shows", {}) or {}
            if ident in shows:
                del shows[ident]
                cache["shows"] = shows
                self.save_cache(cache)
                xbmc.log(f"[ProgressCache] dropped show {ident} from cache", xbmc.LOGINFO)
                return True
            return False
        except Exception as e:
            xbmc.log(f"[ProgressCache] drop_show_from_cache error: {e}", xbmc.LOGERROR)
            return False

    def drop_show_by_trakt_id(self, trakt_id: int) -> bool:
        ident_str = str(trakt_id)
        try:
            cache = self.load_cache()
            shows = cache.get("shows", {}) or {}
            if ident_str in shows:
                del shows[ident_str]
                cache["shows"] = shows
                self.save_cache(cache)
                xbmc.log(f"[ProgressCache] dropped show key={ident_str}", xbmc.LOGINFO)
                return True
            removed = False
            for k, v in list(shows.items()):
                ids = (v or {}).get("ids", {}) or {}
                if str(ids.get("trakt") or "") == ident_str:
                    del shows[k]
                    removed = True
            if removed:
                cache["shows"] = shows
                self.save_cache(cache)
                xbmc.log(f"[ProgressCache] dropped show by ids.trakt={ident_str}", xbmc.LOGINFO)
                return True
            return False
        except Exception as e:
            xbmc.log(f"[ProgressCache] drop_show_by_trakt_id error: {e}", xbmc.LOGERROR)
            return False

class WatchedMoviesCacheManager:
    """Správa lokální cache pro historii filmů."""
    def __init__(self):
        self.profile_root = translatePath(ADDON.getAddonInfo('profile'))
        os.makedirs(self.profile_root, exist_ok=True)
        self.cache_path = os.path.join(self.profile_root, 'watched_movies_cache.json')

    def load_cache(self) -> list:
        if not os.path.exists(self.cache_path):
            return []
        try:
            with open(self.cache_path, 'r', encoding='utf-8') as f:
                data = json.load(f)
                return data if isinstance(data, list) else []
        except Exception as e:
            xbmc.log(f"[MoviesCache] read error: {e}", xbmc.LOGWARNING)
            return []

    def save_cache(self, data: list) -> None:
        try:
            with open(self.cache_path, 'w', encoding='utf-8') as f:
                json.dump(data, f, ensure_ascii=False, indent=2)
            xbmc.log("[MoviesCache] cache saved", xbmc.LOGINFO)
        except Exception as e:
            xbmc.log(f"[MoviesCache] write error: {e}", xbmc.LOGERROR)

class WatchedShowsCacheManager:
    """Správa lokální cache pro historii seriálů."""
    def __init__(self):
        self.profile_root = translatePath(ADDON.getAddonInfo('profile'))
        os.makedirs(self.profile_root, exist_ok=True)
        self.cache_path = os.path.join(self.profile_root, 'watched_shows_cache.json')

    def load_cache(self) -> list:
        if not os.path.exists(self.cache_path):
            return []
        try:
            with open(self.cache_path, 'r', encoding='utf-8') as f:
                data = json.load(f)
                return data if isinstance(data, list) else []
        except Exception as e:
            xbmc.log(f"[ShowsCache] read error: {e}", xbmc.LOGWARNING)
            return []

    def save_cache(self, data: list) -> None:
        try:
            with open(self.cache_path, 'w', encoding='utf-8') as f:
                json.dump(data, f, ensure_ascii=False, indent=2)
            xbmc.log("[ShowsCache] cache saved", xbmc.LOGINFO)
        except Exception as e:
            xbmc.log(f"[ShowsCache] write error: {e}", xbmc.LOGERROR)


# === Trakt Menu ==============================================================
class TraktMenu:
    def __init__(self, handle: int, addon: xbmcaddon.Addon) -> None:
        self.api = TraktAPI()
        self._handle = handle
        self._addon = addon


    def update_trakt_cache_incremental(self):
        """
        Inkrementální aktualizace cache Trakt s progress barem:
        - načte timestamp poslední aktualizace,
        - stáhne jen rozdíly od posledního času (watched movies/shows + history),
        - spustí TMDb prefetch pro nové položky (na pozadí),
        - uloží nový timestamp,
        - bezpečně kontroluje typy (dict vs. str) a podporuje Cancel.
        """
        import os, json
        from datetime import datetime

        try:
            profile_root = translatePath(ADDON.getAddonInfo('profile'))
            last_update_file = os.path.join(profile_root, 'trakt_last_update.json')

            # 1) Načtení last_update (pokud existuje)
            last_ts = None
            if xbmcvfs.exists(last_update_file):
                try:
                    with xbmcvfs.File(last_update_file, 'r') as f:
                        content = f.read()
                        obj = json.loads(content) if content else {}
                        last_ts = obj.get('last_update')
                except Exception as e:
                    xbmc.log(f"[TraktCacheUpdate] read last_update failed: {e}", xbmc.LOGWARNING)

            # Progress dialog
            dlg = xbmcgui.DialogProgress()
            dlg.create("Trakt • Aktualizace cache", "Zjišťuji změny od poslední aktualizace…")

            # 2) Parametry s updated_since
            params = {'extended': 'min'}
            if last_ts:
                params['updated_since'] = last_ts

            new_items = []  # list of tuples: (media_type 'movie'|'tv', tmdb_id:int)

            def _update_progress(pct: int, line1: str = "", line2: str = ""):
                try:
                    dlg.update(max(0, min(100, int(pct))), line1 or "", line2 or "")
                except Exception:
                    pass

            # 3) Watched movies
            _update_progress(5, "Stahuji změny: filmy (watched)…")
            if dlg.iscanceled(): dlg.close(); return
            movies = self.api.get('sync/watched/movies', params=params, auth=True, cache_ttl=0) or []
            for m in movies:
                if not isinstance(m, dict):  # bezpečnost
                    continue
                movie_obj = m.get('movie')
                if not isinstance(movie_obj, dict):
                    continue
                ids = movie_obj.get('ids', {}) or {}
                tmdb_id = ids.get('tmdb')
                if isinstance(tmdb_id, int) and tmdb_id > 0:
                    new_items.append(('movie', tmdb_id))

            # 4) Watched shows
            _update_progress(25, "Stahuji změny: seriály (watched)…")
            if dlg.iscanceled(): dlg.close(); return
            shows = self.api.get('sync/watched/shows', params=params, auth=True, cache_ttl=0) or []
            for s in shows:
                if not isinstance(s, dict):
                    continue
                show_obj = s.get('show')
                if not isinstance(show_obj, dict):
                    continue
                ids = show_obj.get('ids', {}) or {}
                tmdb_id = ids.get('tmdb')
                if isinstance(tmdb_id, int) and tmdb_id > 0:
                    new_items.append(('tv', tmdb_id))

            # 5) Kombinovaná historie (movie/show/episode)
            _update_progress(45, "Stahuji změny: historie (sync/history)…")
            if dlg.iscanceled(): dlg.close(); return
            history = self.api.get('sync/history', params=params, auth=True, cache_ttl=0) or []
            for h in history:
                if not isinstance(h, dict):
                    continue
                media_type = h.get('type')  # 'movie' | 'show' | 'episode' | ...
                # Vyber správný podobjekt:
                obj = None
                if media_type == 'movie':
                    obj = h.get('movie')
                    target_mt = 'movie'
                elif media_type in ('show', 'episode'):
                    # Pro 'episode' použij ids show/episode → bereme to jako TV
                    obj = h.get(media_type)
                    target_mt = 'tv'
                else:
                    obj = None
                    target_mt = None

                if not isinstance(obj, dict) or target_mt is None:
                    continue

                ids = obj.get('ids', {}) or {}
                tmdb_id = ids.get('tmdb')
                if isinstance(tmdb_id, int) and tmdb_id > 0:
                    new_items.append((target_mt, tmdb_id))

            # 6) Unikátní TMDb ID
            uniq = []
            seen = set()
            for mt, tid in new_items:
                try:
                    tid_int = int(tid)
                except Exception:
                    continue
                key = (mt, tid_int)
                if key not in seen:
                    uniq.append(key)
                    seen.add(key)

            total = len(uniq)
            _update_progress(50, "Připravuji TMDb prefetch…", f"Nových položek: {total}")

            # 7) TMDb prefetch (na pozadí) s průběhem
            prefetched = 0
            if total == 0:
                _update_progress(95, "Žádné nové položky k doplnění.")
            else:
                step_base = 50.0
                step_span = 45.0  # 50 → 95 %
                per_item = step_span / float(total)
                for (media_type, tmdb_id) in uniq:
                    if dlg.iscanceled():
                        dlg.close()
                        xbmcgui.Dialog().notification('Trakt', f'Aktualizace zrušena ({prefetched}/{total})', xbmcgui.NOTIFICATION_INFO, 2500)
                        return
                    try:
                        _prefetch_tmdb_detail_async(tmdb_id, media_type)  # uloží do TMDb FileCache s TTL 14 dní
                        prefetched += 1
                    except Exception as e:
                        xbmc.log(f"[TraktCacheUpdate] prefetch error tmdb={tmdb_id}: {e}", xbmc.LOGWARNING)
                    pct = int(step_base + per_item * prefetched)
                    _update_progress(pct, "Přednačítám TMDb detaily…", f"{prefetched}/{total}")

            # 8) Ulož nový timestamp
            now_ts = datetime.utcnow().strftime('%Y-%m-%dT%H:%M:%S.000Z')
            try:
                with xbmcvfs.File(last_update_file, 'w') as f:
                    f.write(json.dumps({'last_update': now_ts}))
            except Exception as e:
                xbmc.log(f"[TraktCacheUpdate] write last_update failed: {e}", xbmc.LOGERROR)

            _update_progress(100, "Hotovo.", f"Doplněno: {prefetched} / {total}")
            dlg.close()
            xbmcgui.Dialog().notification('Trakt', f'Aktualizace dokončena ({prefetched}/{total})', xbmcgui.NOTIFICATION_INFO, 3000)

        except Exception as e:
            try:
                dlg.close()
            except Exception:
                pass
            xbmc.log(f"[TraktCacheUpdate] fatal: {e}", xbmc.LOGERROR)
            xbmcgui.Dialog().notification('Trakt', 'Aktualizace cache selhala (viz log).', xbmcgui.NOTIFICATION_ERROR, 3500)


    # -- Helpers pro paging (UI) --
    def _read_pagination(self) -> Dict[str, int]:
        h = getattr(self.api, "last_headers", {}) or {}
        def _to_int(v, default=0):
            try:
                return int(v)
            except Exception:
                return default
        return {
            "page": _to_int(h.get("X-Pagination-Page"), 1),
            "page_count": _to_int(h.get("X-Pagination-Page-Count"), 1),
            "item_count": _to_int(h.get("X-Pagination-Item-Count"), 0),
            "limit": _to_int(h.get("X-Pagination-Limit"), 0),
        }

    def _add_prev_page(self, action: str, **kwargs):
        try:
            current_page = int(kwargs.get('page', 1))
        except Exception:
            current_page = 1
        if current_page <= 1:
            return
        kwargs['page'] = current_page - 1
        li = xbmcgui.ListItem(label='<< Předchozí strana ...')
        li.setArt({'icon': _icon('trakt_lists_browse')})
        safe_kwargs = {k: str(v) for k, v in kwargs.items() if v is not None}
        url = get_url(action=action, **safe_kwargs)
        xbmcplugin.addDirectoryItem(self._handle, url, li, isFolder=True)

    def _add_next_page(self, action: str, only_if_more: bool = True, **kwargs):
        if only_if_more:
            pg = self._read_pagination()
            if pg.get("page") and pg.get("page_count") and pg["page"] >= pg["page_count"]:
                return
        try:
            current_page = int(kwargs.get('page', 1))
        except Exception:
            current_page = 1
        kwargs['page'] = current_page + 1
        li = xbmcgui.ListItem(label='>> Další strana ...')
        li.setArt({'icon': _icon('trakt_lists_browse')})
        safe_kwargs = {k: str(v) for k, v in kwargs.items() if v is not None}
        url = get_url(action=action, **safe_kwargs)
        xbmcplugin.addDirectoryItem(self._handle, url, li, isFolder=True)


    def list_movies_by_mode(self, mode: str, page: int = 1, limit: int = 30,
                            genres: Optional[str] = None, years: Optional[str] = None):
        data = self.api.movies(mode=mode, page=page, limit=limit, genres=genres, years=years)
        pg   = self._read_pagination()
        srv_page   = pg.get("page", page)
        page_count = pg.get("page_count", 1)

        xbmcplugin.setPluginCategory(self._handle, f"Trakt • Filmy • {mode} • strana {srv_page}/{page_count}")
        if not data:
            xbmcgui.Dialog().notification('Trakt', 'Nic k zobrazení.', xbmcgui.NOTIFICATION_WARNING)
            xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)
            return

        # Předchozí strana
        if srv_page > 1 or page > 1:
            self._add_prev_page('trakt_movies_list', mode=mode, page=srv_page, limit=limit,
                                genres=genres or '', years=years or '')

        for row in data:
            media = row.get('movie', row)
            if not isinstance(media, dict):
                continue

            title = media.get('title') or 'Bez názvu'
            year  = media.get('year')
            ids   = media.get('ids', {}) or {}
            imdb_id = ids.get('imdb')
            tmdb_id = ids.get('tmdb')
            trakt_id = ids.get('trakt')
            slug     = ids.get('slug')

            # === TMDb cache: pokus o načtení poster/plot ===
            poster_url = None
            plot_text  = ''
            if tmdb_id:
                det = get_tmdb_details_from_cache_only(tmdb_id, 'movie')
                if det:
                    if det.get('poster_path'):
                        poster_url = IMAGE_BASE_URL_POSTER + "/" + str(det['poster_path']).lstrip('/')
                    plot_text = det.get('plot') or ''
                else:
                    # není v cache → spustíme prefetch na pozadí
                    _prefetch_tmdb_detail_async(tmdb_id, 'movie')

            base_label = f"{title} ({year})" if year else title
            li = xbmcgui.ListItem(label=base_label)
            # artwork: pokud máme poster z cache, použij ho
            if poster_url:
                li.setArt({'icon': poster_url, 'thumb': poster_url, 'poster': poster_url})
            else:
                li.setArt({'icon': 'DefaultFolder.png'})
            li.setInfo('Video', {'title': title, 'plot': plot_text or '', 'year': year, 'mediatype': 'movie'})

            # Sjednocené context menu
            context = build_trakt_context(
                get_url_fn=get_url, title=title, tmdb_id=tmdb_id,
                trakt_id=trakt_id, slug=slug, media_type='movies',
                context_type='list_fast'
            )
            li.addContextMenuItems(context, replaceItems=False)

            # Klik → vyhledání na Webshare
            url = get_url(action='trakt_search', ident=imdb_id or '', title=title, year=year or '', is_movie='true')
            xbmcplugin.addDirectoryItem(self._handle, url, li, isFolder=True)

        # Další strana
        hdr_present = (pg.get("limit", 0) > 0) or (pg.get("item_count", 0) > 0) or (pg.get("page_count", 1) > 1)
        has_more = (srv_page < page_count) if hdr_present else (isinstance(data, list) and limit and (len(data) >= limit))
        if has_more:
            self._add_next_page('trakt_movies_list', only_if_more=False, mode=mode, page=srv_page, limit=limit,
                                genres=genres or '', years=years or '')

        xbmcplugin.setContent(self._handle, 'movies')
        xbmc.executebuiltin('Container.SetViewMode(50)')
        xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)


    def list_shows_by_mode(self, mode: str, page: int = 1, limit: int = 30,
                        genres: Optional[str] = None, years: Optional[str] = None):
        data = self.api.shows(mode=mode, page=page, limit=limit, genres=genres, years=years)
        pg   = self._read_pagination()
        srv_page   = pg.get("page", page)
        page_count = pg.get("page_count", 1)

        xbmcplugin.setPluginCategory(self._handle, f"Trakt • Seriály • {mode} • strana {srv_page}/{page_count}")
        if not data:
            xbmcgui.Dialog().notification('Trakt', 'Nic k zobrazení.', xbmcgui.NOTIFICATION_WARNING)
            xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)
            return

        if srv_page > 1 or page > 1:
            self._add_prev_page('trakt_shows_list', mode=mode, page=srv_page, limit=limit,
                                genres=genres or '', years=years or '')

        for row in data:
            media = row.get('show', row)
            if not isinstance(media, dict):
                continue

            ids   = media.get('ids', {}) or {}
            title = media.get('title') or (ids.get('slug') or str(ids.get('trakt') or 'Bez názvu'))
            year  = media.get('year')
            tmdb_id  = ids.get('tmdb')
            trakt_id = ids.get('trakt')
            slug     = ids.get('slug')

            # === TMDb cache: poster/plot pokud je k dispozici ===
            poster_url = None
            plot_text  = ''
            if tmdb_id:
                det = get_tmdb_details_from_cache_only(tmdb_id, 'tv')
                if det:
                    if det.get('poster_path'):
                        poster_url = IMAGE_BASE_URL_POSTER + "/" + str(det['poster_path']).lstrip('/')
                    plot_text = det.get('plot') or ''
                else:
                    _prefetch_tmdb_detail_async(tmdb_id, 'tv')

            base_label = f"{title} ({year})" if year else title
            li = xbmcgui.ListItem(label=base_label)
            if poster_url:
                li.setArt({'icon': poster_url, 'thumb': poster_url, 'poster': poster_url})
            else:
                li.setArt({'icon': 'DefaultFolder.png'})
            li.setInfo('Video', {'title': title, 'plot': plot_text or '', 'year': year, 'mediatype': 'tvshow'})

            context = build_trakt_context(
                get_url_fn=get_url, title=title, tmdb_id=tmdb_id,
                trakt_id=trakt_id, slug=slug, media_type='shows',
                context_type='list_fast'
            )
            li.addContextMenuItems(context, replaceItems=False)

            # Klik → TMDb sezóny
            url = get_url(action='tmdb_seasons', tmdb_id=tmdb_id or '', title=title)
            xbmcplugin.addDirectoryItem(self._handle, url, li, isFolder=True)

        hdr_present = (pg.get("limit", 0) > 0) or (pg.get("item_count", 0) > 0) or (pg.get("page_count", 1) > 1)
        has_more = (srv_page < page_count) if hdr_present else (isinstance(data, list) and limit and (len(data) >= limit))
        if has_more:
            self._add_next_page('trakt_shows_list', only_if_more=False, mode=mode, page=srv_page, limit=limit,
                                genres=genres or '', years=years or '')

        xbmcplugin.setContent(self._handle, 'tvshows')
        xbmc.executebuiltin('Container.SetViewMode(50)')
        xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)

    def movies_filters_wizard(self):
        """
        Jednoduchý wizard: zadáš žánry a rok/rozsah → přesměruje na listing filmů (trending) s filtry.
        Volá se z routeru akcí 'trakt_movies_filters'.
        """
        genres = xbmcgui.Dialog().input("Žánry (csv, např. action,comedy)", type=xbmcgui.INPUT_ALPHANUM)
        years  = xbmcgui.Dialog().input("Rok/rozsah (např. 2020-2025 nebo 2024)", type=xbmcgui.INPUT_ALPHANUM)
        url = get_url(
            action='trakt_movies_list',
            mode='trending',      # můžeš libovolně změnit na 'popular' apod.
            page=1,
            limit=30,
            genres=genres or '',
            years=years or ''
        )
        xbmc.executebuiltin(f'Container.Update({url})')


    def shows_filters_wizard(self):
        """
        Wizard pro seriály (pro akci 'trakt_shows_filters' v routeru).
        """
        genres = xbmcgui.Dialog().input("Žánry (csv)", type=xbmcgui.INPUT_ALPHANUM)
        years  = xbmcgui.Dialog().input("Rok/rozsah (např. 2015-2022 nebo 2024)", type=xbmcgui.INPUT_ALPHANUM)
        url = get_url(
            action='trakt_shows_list',
            mode='trending',
            page=1,
            limit=30,
            genres=genres or '',
            years=years or ''
        )
        xbmc.executebuiltin(f'Container.Update({url})')

    # -- Renderery položek --
    def _render_movie_item(self, media: Dict[str, Any], label_suffix: str = '', context_type: str = 'default'):
        title = media.get('title') or 'Bez názvu'
        year = media.get('year')
        ids = media.get('ids') or {}
        imdb_id = ids.get('imdb')
        tmdb_id = ids.get('tmdb')
        trakt_id = ids.get('trakt')
        slug = ids.get('slug')
        plot = media.get('overview') or ''

        poster_partial = None
        if tmdb_id:
            det = get_tmdb_details_fallback(tmdb_id, 'movie')
            if det:
                poster_partial = det.get('poster_path') or poster_partial
                title = det.get('title') or title
                plot = det.get('plot') or plot

        if context_type == 'watchlist':
            prefix = "[COLOR magenta][Film][/COLOR] "
            base_label = f"{prefix}{title} ({year})" if year else f"{prefix}{title}"
        else:
            base_label = f"{title} ({year})" if year else title
        li = xbmcgui.ListItem(label=(base_label + (label_suffix or '')))
        if poster_partial:
            poster_url = IMAGE_BASE_URL_POSTER + "/" + str(poster_partial).lstrip('/')
            li.setArt({'icon': poster_url, 'thumb': poster_url, 'poster': poster_url})
        li.setInfo('Video', {'title': title, 'plot': plot, 'year': year, 'mediatype': 'movie'})

        context = build_trakt_context(
            get_url_fn=get_url,
            title=title,
            tmdb_id=tmdb_id,
            trakt_id=trakt_id,
            slug=slug,
            media_type='movies',
            context_type=context_type  # běžné listy: trending/popular/anticipated/seznamy
        )
        li.addContextMenuItems(context, replaceItems=False)

        url = get_url(action='trakt_search', ident=imdb_id or '', title=title, year=year or '', is_movie='true')
        xbmcplugin.addDirectoryItem(self._handle, url, li, isFolder=True)

    def _render_show_item(self, media: Dict[str, Any], label_suffix: str = '', context_type: str = 'default'):
        ids = media.get('ids') or {}
        title = media.get('title') or (ids.get('slug') or str(ids.get('trakt') or 'Bez názvu'))
        year = media.get('year')
        tmdb_id = ids.get('tmdb')
        trakt_id = ids.get('trakt')
        slug = ids.get('slug')
        plot = media.get('overview') or ''

        poster_partial = None
        if tmdb_id:
            det = get_tmdb_details_fallback(tmdb_id, 'tv')
            if det:
                poster_partial = det.get('poster_path') or poster_partial
                title = det.get('title') or title
                plot = det.get('plot') or plot


        if context_type == 'watchlist':
            prefix = "[COLOR yellow][Seriál][/COLOR] "
            base_label = f"{prefix}{title} ({year})" if year else f"{prefix}{title}"
        else:
            base_label = f"{title} ({year})" if year else title

        li = xbmcgui.ListItem(label=(base_label + (label_suffix or '')))
        if poster_partial:
            poster_url = IMAGE_BASE_URL_POSTER + "/" + str(poster_partial).lstrip('/')
            li.setArt({'icon': poster_url, 'thumb': poster_url, 'poster': poster_url})
        li.setInfo('Video', {'title': title, 'plot': plot, 'year': year, 'mediatype': 'tvshow'})

        # Sjednocené kontext menu
        context = build_trakt_context(
            get_url_fn=get_url,
            title=title,
            tmdb_id=tmdb_id,
            trakt_id=trakt_id,
            slug=slug,
            media_type='shows',
            context_type=context_type
        )

        # Zachovej speciální akci: zobrazit Next/Last epizodu
        if slug or trakt_id:
            next_last = get_url(action='trakt_show_next_last', id=(slug or str(trakt_id)), title=title)
            context.append(('Zobrazit Next/Last epizodu', f'Container.Update({next_last})'))

        li.addContextMenuItems(context, replaceItems=False)

        url = get_url(action='tmdb_seasons', tmdb_id=tmdb_id or '', title=title)
        xbmcplugin.addDirectoryItem(self._handle, url, li, isFolder=True)

    def trakt_lists_hub(self):
        """
        Hub pro veřejné seznamy Traktu (Popular / Trending).
        Odpovídá akci 'trakt_lists_hub' z routeru.
        """
        # Info řádek (neklikací)
        info_li = xbmcgui.ListItem(label='(Public) Veřejné seznamy lze procházet i bez přihlášení k Traktu.')
        info_li.setArt({'icon': 'DefaultInfo.png'})
        info_li.setInfo('Video', {
            'title': 'Informace • Veřejné seznamy',
            'plot': ('Procházení veřejných seznamů uživatelů Trakt. '
                    'Najdete zde tematické kolekce filmů a seriálů, které právě trendují '
                    'nebo patří dlouhodobě mezi nejpopulárnější.')
        })
        xbmcplugin.addDirectoryItem(self._handle, get_url(action='noop'), info_li, False)

        # Popular lists
        li_pop = xbmcgui.ListItem(label='Seznamy • Popular')
        li_pop.setArt({'icon': 'DefaultFolder.png'})
        li_pop.setInfo('Video', {'title': 'Veřejné seznamy • Popular',
                                'plot': 'Nejčastěji sledované/odběry — dlouhodobě oblíbené seznamy.'})
        url_pop = get_url(action='trakt_lists_browse', mode='popular', page=1, limit=30)
        xbmcplugin.addDirectoryItem(self._handle, url_pop, li_pop, True)

        # Trending lists
        li_tr = xbmcgui.ListItem(label='Seznamy • Trending')
        li_tr.setArt({'icon': 'DefaultFolder.png'})
        li_tr.setInfo('Video', {'title': 'Veřejné seznamy • Trending',
                                'plot': 'Aktuálně nejvíce vyhledávané a sdílené seznamy.'})
        url_tr = get_url(action='trakt_lists_browse', mode='trending', page=1, limit=30)
        xbmcplugin.addDirectoryItem(self._handle, url_tr, li_tr, True)

        xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)


    # --- Hlavní menu Trakt (root) ---
    def show_main_menu(self):
        # Můj Trakt.tv (podmenu)
        li_my = xbmcgui.ListItem(label='Můj Trakt.tv')
        li_my.setArt({'icon': _icon('trakt_lists_hub')})
        li_my.setInfo('Video', {'title': 'Můj Trakt.tv',
                                'plot': 'Přehled vašeho Trakt účtu: Watchlist, Kalendář, pokrok, kolekce.'})
        xbmcplugin.addDirectoryItem(self._handle, get_url(action='trakt_my_hub'), li_my, True)

        # Info (public)
        info_li = xbmcgui.ListItem(label='[COLOR red]Veřejné žebříčky lze procházet i bez přihlášení k Traktu.[/COLOR]')
        info_li.setArt({'icon': _icon('trakt_lists_hub')})
        xbmcplugin.addDirectoryItem(self._handle, get_url(action='noop'), info_li, False)

        # Filmy hub
        li_movies = xbmcgui.ListItem(label='Filmy: Trending / Popular / Anticipated')
        li_movies.setArt({'icon': _icon('trakt_movies_hub')})
        xbmcplugin.addDirectoryItem(self._handle, get_url(action='trakt_movies_hub'), li_movies, True)

        # Seriály hub
        li_shows = xbmcgui.ListItem(label='Seriály: Trending / Popular / Anticipated')
        li_shows.setArt({'icon': _icon('trakt_shows_hub')})
        xbmcplugin.addDirectoryItem(self._handle, get_url(action='trakt_shows_hub'), li_shows, True)

        # Veřejné seznamy
        li_lists = xbmcgui.ListItem(label='Veřejné seznamy (Popular/Trending)')
        li_lists.setArt({'icon': _icon('trakt_lists_hub')})
        xbmcplugin.addDirectoryItem(self._handle, get_url(action='trakt_lists_hub'), li_lists, True)

        xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)

    # --- Watchlist ---
    def show_trakt_list(self, list_type: str):
        endpoint = f'users/me/{list_type}'.lstrip('/')
        data = self.api.get(endpoint, params={'extended': 'full'}, auth=True, cache_ttl=0)
        if not data:
            xbmcgui.Dialog().notification('Trakt', 'Seznam je prázdný.', xbmcgui.NOTIFICATION_WARNING)
            xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)
            return

        for item in data:
            if 'movie' in item:
                self._render_movie_item(item['movie'], context_type='watchlist')
            elif 'show' in item:
                self._render_show_item(item['show'], context_type='watchlist')
        

        xbmcplugin.setContent(self._handle, 'tvshows')
        xbmc.executebuiltin('Container.SetViewMode(50)')
        xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)

    # --- Kalendář (my shows) ---
    def list_calendar_shows(self, days: int = 30, start_date: Optional[str] = None):
        if not ADDON.getSetting('trakt_access_token'):
            xbmcgui.Dialog().ok('Trakt', 'Nejste přihlášeni k Trakt.')
            xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)
            return
        import datetime
        if not start_date:
            start_date = datetime.date.today().strftime('%Y-%m-%d')
        data = self.api.get(f'calendars/my/shows/{start_date}/{days}', auth=True, cache_ttl=0)
        if not data:
            xbmcgui.Dialog().ok('Trakt Kalendář', 'Žádné nadcházející epizody.')
            xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)
            return
        for item in data:
            episode = item.get('episode', {})
            show = item.get('show', {})
            air = (item.get('first_aired') or '').split('T')[0]
            s, e = episode.get('season'), episode.get('number')
            s_txt = f"S{int(s):02d}" if isinstance(s, int) else "S??"
            e_txt = f"E{int(e):02d}" if isinstance(e, int) else "E??"
            show_title = show.get('title') or 'Neznámý seriál'
            ep_title = episode.get('title') or 'Epizoda'
            date_colored = f"[COLOR magenta]{_format_cz_date(air)}[/COLOR]"
            se_colored = f"[COLOR yellow]{s_txt}{e_txt}[/COLOR]"
            label = f"{date_colored} {se_colored} - {show_title} ({ep_title})"
            li = xbmcgui.ListItem(label=label)
            li.setArt({'icon': _icon('trakt_calendar')})
            tmdb_id = show.get('ids', {}).get('tmdb')
            plot = episode.get('overview') or show.get('overview') or 'Popis chybí.'
            det = get_tmdb_details_fallback(tmdb_id, 'tv') if tmdb_id else None
            if det and det.get('poster_path'):
                poster = IMAGE_BASE_URL_POSTER + "/" + det['poster_path'].lstrip('/')
                li.setArt({'icon': poster, 'thumb': poster, 'poster': poster})
            li.setInfo('Video', {
                'title': ep_title, 'tvshowtitle': show_title,
                'season': s or 0, 'episode': e or 0,
                'plot': plot, 'mediatype': 'episode'
            })
            xbmcplugin.addDirectoryItem(self._handle, get_url(action='series_search', what=show_title), li, True)
        xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)

    # --- Huby (Movies/Shows) ---
    def trakt_movies_hub(self):
        li = xbmcgui.ListItem(label='Trending (filmy)')
        li.setArt({'icon': _icon('trakt_movies_hub')})
        xbmcplugin.addDirectoryItem(self._handle, get_url(action='trakt_movies_list', mode='trending', page=1, limit=30), li, True)

        li = xbmcgui.ListItem(label='Popular (filmy)')
        li.setArt({'icon': _icon('trakt_movies_hub')})
        xbmcplugin.addDirectoryItem(self._handle, get_url(action='trakt_movies_list', mode='popular', page=1, limit=30), li, True)

        li = xbmcgui.ListItem(label='Anticipated (filmy)')
        li.setArt({'icon': _icon('trakt_movies_hub')})
        xbmcplugin.addDirectoryItem(self._handle, get_url(action='trakt_movies_list', mode='anticipated', page=1, limit=30), li, True)

        li = xbmcgui.ListItem(label='Filtrovat (žánr/rok)...')
        li.setArt({'icon': _icon('trakt_movies_filters')})
        xbmcplugin.addDirectoryItem(self._handle, get_url(action='trakt_movies_filters'), li, True)

        xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)

    def trakt_shows_hub(self):
        li = xbmcgui.ListItem(label='Trending (seriály)')
        li.setArt({'icon': _icon('trakt_shows_hub')})
        xbmcplugin.addDirectoryItem(self._handle, get_url(action='trakt_shows_list', mode='trending', page=1, limit=30), li, True)

        li = xbmcgui.ListItem(label='Popular (seriály)')
        li.setArt({'icon': _icon('trakt_shows_hub')})
        xbmcplugin.addDirectoryItem(self._handle, get_url(action='trakt_shows_list', mode='popular', page=1, limit=30), li, True)

        li = xbmcgui.ListItem(label='Anticipated (seriály)')
        li.setArt({'icon': _icon('trakt_shows_hub')})
        xbmcplugin.addDirectoryItem(self._handle, get_url(action='trakt_shows_list', mode='anticipated', page=1, limit=30), li, True)

        li = xbmcgui.ListItem(label='Filtrovat (žánr/rok)...')
        li.setArt({'icon': _icon('trakt_shows_filters')})
        xbmcplugin.addDirectoryItem(self._handle, get_url(action='trakt_shows_filters'), li, True)

        xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)

    # --- Public lists ---


    def list_public_lists(self, mode: str, page: int = 1, limit: int = 30):
        data = self.api.list_popular_or_trending(mode=mode, page=page, limit=limit)
        pg   = self._read_pagination()
        srv_page   = pg.get("page", page)
        page_count = pg.get("page_count", 1)

        xbmcplugin.setPluginCategory(self._handle, f"Trakt • Seznamy • {mode} • strana {srv_page}/{page_count}")
        if not data:
            xbmcgui.Dialog().notification('Trakt', 'Žádné seznamy.', xbmcgui.NOTIFICATION_WARNING)
            xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)
            return

        if srv_page > 1 or page > 1:
            self._add_prev_page('trakt_lists_browse', mode=mode, page=srv_page, limit=limit)

        for row in data:
            lst = row.get('list', row)
            if not isinstance(lst, dict):
                continue
            name = lst.get('name') or 'Seznam'
            ids  = lst.get('ids', {}) or {}
            list_id = ids.get('slug') or (str(ids.get('trakt')) if ids.get('trakt') else None)
            if not list_id:
                continue

            li = xbmcgui.ListItem(label=name)
            li.setArt({'icon': 'DefaultFolder.png'})
            url = get_url(action='trakt_list_items', list_id=list_id, page=1, limit=50)
            xbmcplugin.addDirectoryItem(self._handle, url, li, isFolder=True)

        hdr_present = (pg.get("limit", 0) > 0) or (pg.get("page_count", 1) > 1)
        has_more = (srv_page < page_count) if hdr_present else (isinstance(data, list) and limit and (len(data) >= limit))
        if has_more:
            self._add_next_page('trakt_lists_browse', only_if_more=False, mode=mode, page=srv_page, limit=limit)

        xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)



    def list_items_of_list(self, list_id: str, page: int = 1, limit: int = 50):
        data = self.api.list_items(list_id=list_id, page=page, limit=limit)
        pg   = self._read_pagination()
        srv_page   = pg.get("page", page)
        page_count = pg.get("page_count", 1)

        xbmcplugin.setPluginCategory(self._handle, f"Trakt • Položky seznamu • {list_id} • strana {srv_page}/{page_count}")
        if not data:
            xbmcgui.Dialog().notification('Trakt', 'Seznam je prázdný.', xbmcgui.NOTIFICATION_WARNING)
            xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)
            return

        if srv_page > 1 or page > 1:
            self._add_prev_page('trakt_list_items', list_id=list_id, page=srv_page, limit=limit)

        for item in data:
            # --- Filmy ---
            if 'movie' in item and isinstance(item['movie'], dict):
                media = item['movie']
                title = media.get('title') or 'Bez názvu'
                year  = media.get('year')
                ids   = media.get('ids', {}) or {}
                imdb_id = ids.get('imdb')
                tmdb_id = ids.get('tmdb')
                trakt_id = ids.get('trakt')
                slug     = ids.get('slug')

                poster_url = None
                plot_text  = ''
                if tmdb_id:
                    det = get_tmdb_details_from_cache_only(tmdb_id, 'movie')
                    if det:
                        if det.get('poster_path'):
                            poster_url = IMAGE_BASE_URL_POSTER + "/" + str(det['poster_path']).lstrip('/')
                        plot_text = det.get('plot') or ''
                    else:
                        _prefetch_tmdb_detail_async(tmdb_id, 'movie')

                li = xbmcgui.ListItem(label=f"{title} ({year})" if year else title)
                if poster_url:
                    li.setArt({'icon': poster_url, 'thumb': poster_url, 'poster': poster_url})
                else:
                    li.setArt({'icon': 'DefaultFolder.png'})
                li.setInfo('Video', {'title': title, 'plot': plot_text or '', 'year': year, 'mediatype': 'movie'})

                cm = build_trakt_context(get_url_fn=get_url, title=title, tmdb_id=tmdb_id,
                                        trakt_id=trakt_id, slug=slug, media_type='movies',
                                        context_type='list_fast')
                li.addContextMenuItems(cm, replaceItems=False)

                url = get_url(action='trakt_search', ident=imdb_id or '', title=title, year=year or '', is_movie='true')
                xbmcplugin.addDirectoryItem(self._handle, url, li, isFolder=True)
                continue

            # --- Seriály ---
            if 'show' in item and isinstance(item['show'], dict):
                media = item['show']
                ids   = media.get('ids', {}) or {}
                title = media.get('title') or (ids.get('slug') or str(ids.get('trakt') or 'Bez názvu'))
                year  = media.get('year')
                tmdb_id  = ids.get('tmdb')
                trakt_id = ids.get('trakt')
                slug     = ids.get('slug')

                poster_url = None
                plot_text  = ''
                if tmdb_id:
                    det = get_tmdb_details_from_cache_only(tmdb_id, 'tv')
                    if det:
                        if det.get('poster_path'):
                            poster_url = IMAGE_BASE_URL_POSTER + "/" + str(det['poster_path']).lstrip('/')
                        plot_text = det.get('plot') or ''
                    else:
                        _prefetch_tmdb_detail_async(tmdb_id, 'tv')

                li = xbmcgui.ListItem(label=f"{title} ({year})" if year else title)
                if poster_url:
                    li.setArt({'icon': poster_url, 'thumb': poster_url, 'poster': poster_url})
                else:
                    li.setArt({'icon': 'DefaultFolder.png'})
                li.setInfo('Video', {'title': title, 'plot': plot_text or '', 'year': year, 'mediatype': 'tvshow'})

                cm = build_trakt_context(get_url_fn=get_url, title=title, tmdb_id=tmdb_id,
                                        trakt_id=trakt_id, slug=slug, media_type='shows',
                                        context_type='list_fast')
                li.addContextMenuItems(cm, replaceItems=False)

                url = get_url(action='tmdb_seasons', tmdb_id=tmdb_id or '', title=title)
                xbmcplugin.addDirectoryItem(self._handle, url, li, isFolder=True)
                continue

            # neznámá položka – přeskočit
            continue

        hdr_present = (pg.get("page_count", 1) > 1)
        has_more = (srv_page < page_count) if hdr_present else (isinstance(data, list) and limit and len(data) >= limit)
        if has_more:
            self._add_next_page('trakt_list_items', only_if_more=False, list_id=list_id, page=srv_page, limit=limit)

        xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)
    # --- People / Related / Next-Last ---
    def show_people(self, media_type: str, trakt_id_or_slug: str, title: str):
        """
        People (obsazení a štáb) s preferencí relevantních oddělení:
        - nejdříve CAST (herci),
        - poté jen vybraná crew oddělení: directing (Režie), writing (Scénář),
          sound (Hudba), production (Produkce).
        Klik na osobu: Container.Update(plugin://.../?action=tmdb_person_detail&tmdb_id=...)
        (varianta B – spolehlivě přepne aktuální kontejner na 'detail osoby').
        """
        xbmcplugin.setPluginCategory(self._handle, f"Trakt • Obsazení a štáb • {title}")
        xbmcplugin.setContent(self._handle, 'files')
        xbmc.log(f"[People] media_type={media_type} id={trakt_id_or_slug} title={title}", xbmc.LOGINFO)

        # 1) zkus public people, pak auth fallback
        data = self.api.people(media_type=media_type, trakt_id_or_slug=trakt_id_or_slug, cache_ttl=0, auth=False)
        if not data:
            xbmc.log("[TraktMenu.show_people] public empty, retry auth=True", xbmc.LOGWARNING)
            data = self.api.people(media_type=media_type, trakt_id_or_slug=trakt_id_or_slug, cache_ttl=0, auth=True)

        if not data or not isinstance(data, dict):
            xbmcgui.Dialog().notification('Trakt', 'Obsazení nedostupné.', xbmcgui.NOTIFICATION_WARNING)
            xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)
            return

        cast = data.get('cast') or []
        crew = data.get('crew') or {}

        # --- CAST (herci) ---
        MAX_CAST = 10
        for idx, p in enumerate(cast):
            if idx >= MAX_CAST:
                break
            person = p.get('person') or {}
            name = person.get('name') or 'Neznámé jméno'
            character = p.get('character') or ''
            tmdb_pid = (person.get('ids') or {}).get('tmdb')

            # Label: jméno — role (pokud je)
            label = f"{name} — {character}" if character else name
            li = xbmcgui.ListItem(label=label)
            li.setArt({'icon': _icon('trakt_people')})

            # TMDb fotka osoby (pokud je)
            thumb_url = get_tmdb_person_thumb(tmdb_pid) if tmdb_pid else None
            if thumb_url:
                li.setArt({'thumb': thumb_url, 'icon': thumb_url})

            # Klik → Container.Update na detail osoby
            if tmdb_pid:
                url = get_url(action='tmdb_person_detail', tmdb_id=int(tmdb_pid))
                xbmc.log(f"[People] cast person='{name}' tmdb_id={tmdb_pid} -> Container.Update", xbmc.LOGINFO)
                # Přidej do kontext menu možnost "Detail osoby" (necháváme i samotný click fungovat)
                li.addContextMenuItems([('Detail osoby (TMDb)', f'Container.Update({url})')], replaceItems=False)
                # Přidáváme položku jako "folder", aby click přirozeně vstoupil do dalšího listu.
                xbmcplugin.addDirectoryItem(self._handle, url, li, isFolder=True)
            else:
                xbmc.log(f"[People] cast person='{name}' tmdb_id=None", xbmc.LOGINFO)
                xbmcplugin.addDirectoryItem(self._handle, get_url(action='noop'), li, isFolder=False)

        # --- CREW (jen vybrané sekce) ---
        wanted_sections = {
            'directing': 'Režie',
            'writing': 'Scénář',
            'sound': 'Hudba',
            'production': 'Produkce',
        }
        for dept_key in ('directing', 'writing', 'sound', 'production'):
            members = crew.get(dept_key) or []
            if not members:
                continue

            header = xbmcgui.ListItem(label=f"{wanted_sections.get(dept_key, dept_key)} • {len(members)}")
            header.setArt({'icon': _icon('trakt_people')})
            xbmcplugin.addDirectoryItem(self._handle, get_url(action='noop'), header, isFolder=False)

            for m in members:
                person = m.get('person') or {}
                name = person.get('name') or 'Neznámé jméno'
                jobs = m.get('jobs') or m.get('job') or []  # list nebo string
                job_text = ''
                if isinstance(jobs, list) and jobs:
                    job_text = ', '.join([j for j in jobs if j])
                elif isinstance(jobs, str) and jobs.strip():
                    job_text = jobs.strip()

                label = f"{name}" + (f" — {job_text}" if job_text else "")
                li = xbmcgui.ListItem(label=label)
                li.setArt({'icon': _icon('trakt_people')})

                tmdb_pid = (person.get('ids') or {}).get('tmdb')
                thumb_url = get_tmdb_person_thumb(tmdb_pid) if tmdb_pid else None
                if thumb_url:
                    li.setArt({'thumb': thumb_url, 'icon': thumb_url})

                if tmdb_pid:
                    url = get_url(action='tmdb_person_detail', tmdb_id=int(tmdb_pid))
                    xbmc.log(f"[People] crew dept={dept_key} person='{name}' tmdb_id={tmdb_pid} -> Container.Update",
                             xbmc.LOGINFO)
                    li.addContextMenuItems([('Detail osoby (TMDb)', f'Container.Update({url})')], replaceItems=False)
                    xbmcplugin.addDirectoryItem(self._handle, url, li, isFolder=True)
                else:
                    xbmc.log(f"[People] crew dept={dept_key} person='{name}' tmdb_id=None", xbmc.LOGINFO)
                    xbmcplugin.addDirectoryItem(self._handle, get_url(action='noop'), li, isFolder=False)

        xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)

    def show_related(self, media_type: str, trakt_id_or_slug: str, page: int = 1, limit: int = 30):
        data = self.api.related(media_type=media_type, trakt_id_or_slug=trakt_id_or_slug, page=page, limit=limit)
        pg = self._read_pagination()
        srv_page = pg.get("page", page)
        page_count = pg.get("page_count", 1)
        xbmcplugin.setPluginCategory(self._handle, f"Trakt • Podobné ({media_type}) • strana {srv_page}/{page_count}")
        if not data:
            xbmcgui.Dialog().notification('Trakt', 'Žádné podobné položky.', xbmcgui.NOTIFICATION_WARNING)
            xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)
            return
        if srv_page > 1 or page > 1:
            self._add_prev_page('trakt_related', media_type=media_type, id=trakt_id_or_slug, page=srv_page, limit=limit)
        for media in data:
            if media_type == 'movies':
                self._render_movie_item(media)
            else:
                self._render_show_item(media)
        hdr_present = (pg.get("page_count", 0) > 1)
        has_more = (srv_page < page_count) if hdr_present else (isinstance(data, list) and limit and len(data) >= limit)
        if has_more:
            self._add_next_page('trakt_related', only_if_more=False, media_type=media_type, id=trakt_id_or_slug,
                                page=srv_page, limit=limit)
        xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)

    def show_next_last_episode(self, trakt_id_or_slug: str, show_title: str):
        next_ep = self.api.get(f'shows/{trakt_id_or_slug}/next_episode', params={'extended': 'full'},
                               auth=True, cache_ttl=0)
        last_ep = self.api.get(f'shows/{trakt_id_or_slug}/last_episode', params={'extended': 'full'},
                               auth=True, cache_ttl=0)
        def add_episode_li(ep, label_prefix):
            if not ep:
                return
            s = ep.get('season'); e = ep.get('number')
            s_txt = f"S{int(s):02d}" if isinstance(s, int) else "S??"
            e_txt = f"E{int(e):02d}" if isinstance(e, int) else "E??"
            ep_title = ep.get('title') or 'Epizoda'
            air_date_iso = (ep.get('first_aired') or '')
            air_date = _format_cz_date(air_date_iso) if air_date_iso else ''
            label = f"{label_prefix}: [{air_date}] {s_txt}{e_txt} - {show_title} ({ep_title})"
            li = xbmcgui.ListItem(label=label)
            li.setArt({'icon': _icon('trakt_next_last')})
            li.setInfo('Video', {'title': ep_title, 'tvshowtitle': show_title,
                                 'season': s if isinstance(s, int) else 0,
                                 'episode': e if isinstance(e, int) else 0,
                                 'plot': ep.get('overview') or ''})
            xbmcplugin.addDirectoryItem(self._handle, get_url(action='noop'), li, isFolder=False)
        add_episode_li(next_ep, "Další epizoda")
        add_episode_li(last_ep, "Poslední epizoda")
        xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)

    # --- Můj Trakt.tv hub + Pokrok + Kolekce + Historie ---
    def show_my_hub(self):
        token = ADDON.getSetting('trakt_access_token')

        # info_li = xbmcgui.ListItem(label='Můj Trakt.tv')
        # info_li.setArt({'icon': _icon('trakt_lists_hub')})
        # info_li.setInfo('Video', {'plot': 'Přehled vašeho Trakt účtu: Watchlist, Kalendář, pokrok, kolekce.'})
        # xbmcplugin.addDirectoryItem(self._handle, get_url(action='noop'), info_li, False)

        if not token:
            li = xbmcgui.ListItem(label='(Přihlásit se k Trakt)')
            li.setArt({'icon': _icon('trakt_lists_hub')})
            xbmcplugin.addDirectoryItem(self._handle, get_url(action='trakt_auth'), li, False)
            xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)
            return

        # Watchlist
        li_wl = xbmcgui.ListItem(label='Můj Watchlist (Filmy/Seriály)')
        li_wl.setArt({'icon': _icon('trakt_lists_hub')})
        xbmcplugin.addDirectoryItem(self._handle, get_url(action='trakt_list', list_type='watchlist'), li_wl, True)

        # Kalendář
        li_cal = xbmcgui.ListItem(label='Kalendář (Nové epizody)')
        li_cal.setArt({'icon': _icon('trakt_calendar')})
        xbmcplugin.addDirectoryItem(self._handle, get_url(action='trakt_calendar'), li_cal, True)

        # Pokrok (jen rozkoukané) — cache-first
        li_prog_incomplete = xbmcgui.ListItem(label='Pokrok sledování seriálů (jen rozkoukané)')
        li_prog_incomplete.setArt({'icon': _icon('trakt_shows_hub')})
        li_prog_incomplete.setInfo('Video', {'plot': 'Seznam pouze těch seriálů, které ještě nejsou dokoukané.'})
        xbmcplugin.addDirectoryItem(
            self._handle,
            get_url(action='trakt_shows_progress', page=1, limit=30, mode='cache', only_incomplete='true'),
            li_prog_incomplete, True
        )

        # Pokrok (posledních 90 dnů)
        li_prog = xbmcgui.ListItem(label='Pokrok sledování seriálů (posledních 90 dnů)')
        li_prog.setArt({'icon': _icon('trakt_shows_hub')})
        li_prog.setInfo('Video', {'plot': 'Relevantní seriály podle vaší historie za posledních 90 dnů.'})
        xbmcplugin.addDirectoryItem(
            self._handle,
            get_url(action='trakt_shows_progress', page=1, limit=30, mode='recent90'),
            li_prog, True
        )
        # Historie filmů
        li_watched_movies = xbmcgui.ListItem(label='Historie filmů (Watched Movies)')
        li_watched_movies.setArt({'icon': _icon('trakt_movies_hub')})
        li_watched_movies.setInfo('Video', {'plot': 'Seznam filmů, které byly zhlédnuty na Traktu.'})
        xbmcplugin.addDirectoryItem(
            self._handle,
            get_url(action='trakt_watched_movies', page=1, limit=50),
            li_watched_movies, True
        )

        # Historie seriálů
        li_watched_shows = xbmcgui.ListItem(label='Historie seriálů (Watched Shows)')
        li_watched_shows.setArt({'icon': _icon('trakt_shows_hub')})
        li_watched_shows.setInfo('Video', {'plot': 'Seznam seriálů, které byly zhlédnuty na Traktu.'})
        xbmcplugin.addDirectoryItem(
            self._handle,
            get_url(action='trakt_watched_shows', page=1, limit=50),
            li_watched_shows, True
        )

        li_inc = xbmcgui.ListItem(label='Aktualizace cache Trakt (inkrementálně)')
        xbmcplugin.addDirectoryItem(self._handle, get_url(action='trakt_cache_update_incremental'), li_inc, False)


        # # Full refresh cache pokroku (s progress barem)
        # li_refresh = xbmcgui.ListItem(label='Aktualizace cache Trakt (pokrok + historie)')
        # li_refresh.setArt({'icon': _icon('trakt_lists_hub')})
        # li_refresh.setInfo('Video', {'plot': 'Provede kompletní stažení pokroku z Trakt a uloží do lokální cache.'})
        # xbmcplugin.addDirectoryItem(
        #     self._handle,
        #     get_url(action='trakt_progress_cache_refresh_full_ui'),
        #     li_refresh, False
        # )
    # ... (předchozí kód v show_my_hub: položky Watchlist, Kalendář, Pokrok, Historie, Refresh) ...

        # --- VLOŽIT ZDE: Statistiky uživatele ---
        # neklikací statistiky (volitelné)
        def _fmt_minutes_to_days_hours(mins):
            try:
                m = int(mins) if mins else 0
            except Exception:
                m = 0
            days = m // (60 * 24)
            hours = (m // 60) % 24
            if days > 0:
                return f"{days}d {hours}h"
            if hours > 0:
                return f"{hours}h"
            return "0h"

        try:
            # 1. Zjistit username (pokud ho nemáme v settings, zkusíme fetch)
            # Pozn: Ideálně bys mohl username cachovat, ale GET users/me je rychlý
            user = self.api.get('users/me', auth=True, cache_ttl=3600) # cache na hodinu stačí
            username = (user or {}).get('username') if user else None
            
            stats = None
            if username:
                # 2. Načíst statistiky (cache 3 minuty)
                stats = self.api.get(f'users/{username}/stats', auth=True, cache_ttl=180)

            if stats:
                movies   = stats.get('movies', {}) or {}
                shows    = stats.get('shows', {}) or {}
                episodes = stats.get('episodes', {}) or {}

                movies_watched   = movies.get('watched', 0)
                movies_minutes   = movies.get('minutes', 0)
                shows_watched    = shows.get('watched', 0)
                episodes_minutes = episodes.get('minutes', 0)


                stats_label = (
                    f"[COLOR magenta]Filmy:[/COLOR] {int(movies_watched)} "
                    f"• [COLOR cyan]Čas:[/COLOR] {_fmt_minutes_to_days_hours(movies_minutes)} "
                    f"• [COLOR yellow]Seriály:[/COLOR] {int(shows_watched)} "
                    f"• [COLOR orange]Epizody čas:[/COLOR] {_fmt_minutes_to_days_hours(episodes_minutes)}"
                )

                li_stats = xbmcgui.ListItem(label=stats_label)
                li_stats.setArt({'icon': 'DefaultInfo.png'})
                li_stats.setInfo('Video', {'title': 'Trakt Statistiky', 'plot': 'Základní statistiky vašeho Trakt účtu.'})
                
                # isFolder=False a action='noop' zajistí, že to nic nedělá po kliknutí
                xbmcplugin.addDirectoryItem(self._handle, get_url(action='noop'), li_stats, False)

        except Exception as e:
            xbmc.log(f"[Trakt] users stats fetch error: {e}", xbmc.LOGWARNING)

        xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)


    # --- RYCHLÉ načítání pokroku ze CACHE + filtr jen rozkoukané + SORT last_activity DESC ---
    def list_shows_progress(self, page: int = 1, limit: int = 30, year: str = '',
                             mode: str = 'cache', start_at: str = '', end_at: str = '',
                             only_incomplete: str = ''):
        pcm = ProgressCacheManager(self.api)
        cache = pcm.ensure_cache()
        shows_map = cache.get("shows", {}) or {}
        shows_list = []
        for ident, entry in shows_map.items():
            ids = entry.get('ids') or {}
            title = entry.get('title') or (ids.get('slug') or str(ids.get('trakt') or 'Bez názvu'))
            entry['title'] = title
            if year and str(entry.get('year', '')) != str(year):
                continue
            prog = entry.get('progress') or {}
            aired = int(prog.get('aired') or 0)
            completed = int(prog.get('completed') or 0)
            next_ep = prog.get('next_episode') or {}
            if (only_incomplete or '').lower() == 'true' and aired > 0 and completed >= aired:
                xbmc.log(f"[FILTER] skipped complete show: {title} {completed}/{aired}", xbmc.LOGINFO)
                continue
            shows_list.append((ident, entry, aired, completed, next_ep))
        shows_list.sort(key=lambda it: it[1].get('last_activity') or '', reverse=True)
        total = len(shows_list)
        start_idx = max((page - 1), 0) * limit
        end_idx = start_idx + limit
        page_items = shows_list[start_idx:end_idx]
        total_pages = max(((total + limit - 1) // limit), 1)
        xbmcplugin.setPluginCategory(self._handle, f"Můj Trakt.tv • Pokrok • strana {page}/{total_pages}")
        if not page_items:
            xbmcgui.Dialog().notification('Trakt', 'V dané množině nejsou žádné seriály.', xbmcgui.NOTIFICATION_INFO)
            xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)
            return
        for ident, entry, aired, completed, next_ep in page_items:
            ids = entry.get('ids') or {}
            tmdb_id = ids.get('tmdb')
            trakt_id = ids.get('trakt')
            title = entry.get('title')
            year_val = entry.get('year')
            ratio_color = 'orange' if completed < aired else 'lime'
            ratio_txt = f"[COLOR {ratio_color}]{completed}/{aired}[/COLOR]"
            label = f"{title} ({year_val}) {ratio_txt}" if year_val else f"{title} {ratio_txt}"
            s = next_ep.get('season'); e = next_ep.get('number'); t = next_ep.get('title') or ''


            s = next_ep.get('season'); e = next_ep.get('number'); t = next_ep.get('title') or ''
            if isinstance(s, int) and isinstance(e, int):
                label += f" • [COLOR lightblue]Další:[/COLOR] S{s:02d}E{e:02d}" + (f" „{t}“" if t else '')
            li = xbmcgui.ListItem(label=label)
            li.setArt({'icon': _icon('trakt_shows_hub')})
            plot = entry.get('overview') or ''
            poster_partial = None
            if tmdb_id:
                det = get_tmdb_details_fallback(tmdb_id, 'tv')
                if det:
                    poster_partial = det.get('poster_path') or poster_partial
                    plot = det.get('plot') or plot
            if poster_partial:
                poster_url = IMAGE_BASE_URL_POSTER + "/" + str(poster_partial).lstrip('/')
                li.setArt({'icon': poster_url, 'thumb': poster_url, 'poster': poster_url})
            li.setInfo('Video', {'title': title, 'plot': plot, 'year': year_val, 'mediatype': 'tvshow'})

            cm = build_trakt_context(
                get_url_fn=get_url,
                title=title,
                tmdb_id=tmdb_id,
                trakt_id=trakt_id,
                slug=None,
                media_type='shows',
                context_type='progress',
                next_season_hint=(int(s) if isinstance(s, int) and s > 0 else None)
            )
            li.addContextMenuItems(cm, replaceItems=False)

            url = get_url(action='tmdb_seasons', tmdb_id=tmdb_id or '', title=title)
            xbmcplugin.addDirectoryItem(self._handle, url, li, True)
        if page > 1:
            self._add_prev_page('trakt_shows_progress', page=page, limit=limit,
                                mode=mode, start_at=start_at, end_at=end_at, only_incomplete=only_incomplete or '')
        if (page * limit) < total:
            self._add_next_page('trakt_shows_progress', only_if_more=False, page=page, limit=limit,
                                mode=mode, start_at=start_at, end_at=end_at, only_incomplete=only_incomplete or '')
        xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)

    # --- Historie: Filmy ---

    def list_watched_movies(self, page: int = 1, limit: int = 50):
        """Zobrazí historii zhlédnutých filmů z lokální cache."""
        xbmcplugin.setPluginCategory(self._handle, "Můj Trakt.tv • Historie filmů")
        cache = WatchedMoviesCacheManager().load_cache()
        if not cache:
            xbmcgui.Dialog().notification('Trakt', 'Cache historie filmů je prázdná. Spusťte aktualizaci.', xbmcgui.NOTIFICATION_INFO)
            xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)
            return

        # Stránkování
        total = len(cache)
        start_idx = (page - 1) * limit
        end_idx = start_idx + limit
        page_items = cache[start_idx:end_idx]
        total_pages = max(((total + limit - 1) // limit), 1)
        xbmcplugin.setPluginCategory(self._handle, f"Můj Trakt.tv • Historie filmů • strana {page}/{total_pages}")

        for item in page_items:
            movie = item.get('movie') or item
            if not isinstance(movie, dict):
                continue
            title = movie.get('title') or 'Bez názvu'
            year = movie.get('year')
            ids = movie.get('ids') or {}
            tmdb_id = ids.get('tmdb')
            plot = movie.get('overview') or ''
            poster_partial = None
            if tmdb_id:
                det = get_tmdb_details_fallback(tmdb_id, 'movie')
                if det:
                    poster_partial = det.get('poster_path') or poster_partial
                    title = det.get('title') or title
                    plot = det.get('plot') or plot
            plays = item.get('plays')
            last_watched_iso = item.get('last_watched_at') or ''
            last_date = _format_cz_date(last_watched_iso) if last_watched_iso else ''
            suffix_parts = []
            if last_date:
                suffix_parts.append(last_date)
            if isinstance(plays, int) and plays >= 0:
                suffix_parts.append(f"{plays}×")
            suffix = f" [COLOR blue]{' • '.join(suffix_parts)}[/COLOR]" if suffix_parts else ""
            self._render_movie_item(movie, label_suffix=suffix)

        # Navigace
        if page > 1:
            self._add_prev_page('trakt_watched_movies', page=page, limit=limit)
        if (page * limit) < total:
            self._add_next_page('trakt_watched_movies', only_if_more=False, page=page, limit=limit)


        xbmcplugin.setContent(self._handle, 'movies')
        xbmc.executebuiltin('Container.SetViewMode(50)')
        xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)

    # --- Historie: Seriály ---

    def list_watched_shows(self, page: int = 1, limit: int = 50):
        """Zobrazí historii zhlédnutých seriálů z lokální cache."""
        xbmcplugin.setPluginCategory(self._handle, "Můj Trakt.tv • Historie seriálů")
        cache = WatchedShowsCacheManager().load_cache()
        if not cache:
            xbmcgui.Dialog().notification('Trakt', 'Cache historie seriálů je prázdná. Spusťte aktualizaci.', xbmcgui.NOTIFICATION_INFO)
            xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)
            return

        total = len(cache)
        start_idx = (page - 1) * limit
        end_idx = start_idx + limit
        page_items = cache[start_idx:end_idx]
        total_pages = max(((total + limit - 1) // limit), 1)
        xbmcplugin.setPluginCategory(self._handle, f"Můj Trakt.tv • Historie seriálů • strana {page}/{total_pages}")

        for item in page_items:
            show = item.get('show') or item
            if not isinstance(show, dict):
                continue
            ids = show.get('ids') or {}
            title = show.get('title') or (ids.get('slug') or str(ids.get('trakt') or 'Bez názvu'))
            year = show.get('year')
            tmdb_id = ids.get('tmdb')
            plot = show.get('overview') or ''
            poster_partial = None
            if tmdb_id:
                det = get_tmdb_details_fallback(tmdb_id, 'tv')
                if det:
                    poster_partial = det.get('poster_path') or poster_partial
                    title = det.get('title') or title
                    plot = det.get('plot') or plot
            plays = item.get('plays')
            last_watched_iso = item.get('last_watched_at') or ''
            last_date = _format_cz_date(last_watched_iso) if last_watched_iso else ''
            suffix_parts = []
            if last_date:
                suffix_parts.append(last_date)
            if isinstance(plays, int) and plays >= 0:
                suffix_parts.append(f"{plays}×")
            suffix = f" [COLOR blue]{' • '.join(suffix_parts)}[/COLOR]" if suffix_parts else ""
            self._render_show_item(show, label_suffix=suffix)

        if page > 1:
            self._add_prev_page('trakt_watched_shows', page=page, limit=limit)
        if (page * limit) < total:
            self._add_next_page('trakt_watched_shows', only_if_more=False, page=page, limit=limit)
        xbmcplugin.endOfDirectory(self._handle, cacheToDisc=False)

# === TMDb Fallback ===========================================================
def get_tmdb_details_fallback(tmdb_id: Optional[int], media_type: str) -> Optional[Dict[str, Any]]:
    """
    Vrátí základní info z TMDb pro film/seriál (poster_path, title/name, plot, year).
    Nepoužívá žádné Trakt hlavičky, pouze TMDb API key z nastavení doplňku.
    """
    tmdb_api_key = ADDON.getSetting('tmdb_api_key')
    language_code = 'cs-CZ'
    if not (tmdb_id and tmdb_api_key):
        return None
    media_type_str = 'movie' if media_type == 'movie' else 'tv'
    endpoint = f"https://api.themoviedb.org/3/{media_type_str}/{int(tmdb_id)}"
    params = {'api_key': tmdb_api_key, 'language': language_code}
    try:
        r = requests.get(endpoint, params=params, timeout=7)
        r.raise_for_status()
        data = r.json()
        return {
            'poster_path': data.get('poster_path'),
            'title': data.get('title') if media_type_str == 'movie' else data.get('name'),
            'plot': data.get('overview'),
            'year': (data.get('release_date') or data.get('first_air_date') or '')[:4]
        }
    except Exception as e:
        xbmc.log(f"TMDb fallback error: {e}", xbmc.LOGERROR)
        return None

# === TMDb Osoba: fotka + detail =============================================
def get_tmdb_person_thumb(tmdb_person_id: Optional[int]) -> Optional[str]:
    """Vrátí URL profilové fotky osoby (TMDb 'profile_path')."""
    try:
        api_key = ADDON.getSetting('tmdb_api_key')
        if not (tmdb_person_id and api_key):
            return None
        url = f"https://api.themoviedb.org/3/person/{int(tmdb_person_id)}"
        params = {'api_key': api_key}
        r = requests.get(url, params=params, timeout=7)
        r.raise_for_status()
        data = r.json()
        profile_path = data.get('profile_path')
        if profile_path:
            return IMAGE_BASE_URL_POSTER + "/" + str(profile_path).lstrip('/')
    except Exception as e:
        xbmc.log(f"[TMDb Person] thumb error: {e}", xbmc.LOGWARNING)
    return None

def get_tmdb_person_detail(tmdb_person_id: Optional[int], language_code: str = 'cs-CZ') -> Optional[Dict[str, Any]]:
    """Vrátí detail osoby: jméno, biography (CZ, fallback EN), profile_path."""
    api_key = ADDON.getSetting('tmdb_api_key')
    if not (tmdb_person_id and api_key):
        xbmc.log("[TMDb Person] API key nebo ID chybí", xbmc.LOGWARNING)
        return None
    url = f"https://api.themoviedb.org/3/person/{int(tmdb_person_id)}"
    try:
        # 1) CZ pokus
        r = requests.get(url, params={'api_key': api_key, 'language': language_code}, timeout=7)
        r.raise_for_status()
        data = r.json()
        bio_cz = (data.get('biography') or '').strip()
        if not bio_cz:
            # 2) Fallback EN
            r_en = requests.get(url, params={'api_key': api_key, 'language': 'en-US'}, timeout=7)
            r_en.raise_for_status()
            data_en = r_en.json()
            bio_en = (data_en.get('biography') or '').strip()
            data['biography'] = bio_en  # přepíšeme prázdné CZ bio EN verzí
        return {
            'name': data.get('name'),
            'biography': data.get('biography') or '',
            'profile_path': data.get('profile_path')
        }
    except Exception as e:
        xbmc.log(f"[TMDb Person] detail error: {e}", xbmc.LOGERROR)
        return None

# === Detail osoby (TMDb) =====================================================

def get_tmdb_person_credits(tmdb_person_id: int, language_code: str = 'cs-CZ') -> list:
    """
    Vrátí combined_credits (cast) pro danou osobu z TMDb.
    Pokud CZ lokalizace neobsahuje názvy, ponechá je tak jak jsou; při chybě provede fallback na EN.
    """
    api_key = ADDON.getSetting('tmdb_api_key')
    if not (tmdb_person_id and api_key):
        xbmc.log("[TMDb Person] API key nebo ID chybí – credits skip", xbmc.LOGWARNING)
        return []
    url = f"https://api.themoviedb.org/3/person/{int(tmdb_person_id)}/combined_credits"
    try:
        # 1) Pokus s CZ
        r = requests.get(url, params={'api_key': api_key, 'language': language_code}, timeout=8)
        r.raise_for_status()
        data = r.json() or {}
        cast = data.get('cast') or []
        # Pokud by nebylo nic (nebo by se zjevně rozbil dotaz), zkus EN fallback
        if not cast:
            r_en = requests.get(url, params={'api_key': api_key, 'language': 'en-US'}, timeout=8)
            r_en.raise_for_status()
            data_en = r_en.json() or {}
            cast = data_en.get('cast') or []
        return cast
    except Exception as e:
        xbmc.log(f"[TMDb Person] combined_credits error: {e}", xbmc.LOGERROR)
        return []


def show_tmdb_person_detail(tmdb_id: int):
    """
    Zobrazí detail osoby (jméno, fotka, bio) a pod ním filmografii (combined_credits).
    - Filmy: klik otevře Webshare vyhledávání (trakt_search).
    - Seriály: klik otevře TMDb sezóny (tmdb_seasons).
    """
    handle = int(sys.argv[1])
    xbmcplugin.setPluginCategory(handle, "TMDb • Osoba")

    # 1) Detail osoby (bio CZ/EN, fotka)
    info = get_tmdb_person_detail(int(tmdb_id), language_code='cs-CZ') or {}
    name = info.get('name') or 'Neznámé jméno'
    bio = (info.get('biography') or '').strip()
    profile_path = info.get('profile_path')

    li_info = xbmcgui.ListItem(label=name)
    art = {'icon': _icon('trakt_people')}
    if profile_path:
        thumb = IMAGE_BASE_URL_POSTER + "/" + str(profile_path).lstrip('/')
        art.update({'thumb': thumb, 'icon': thumb})
    li_info.setArt(art)
    li_info.setInfo('Video', {'title': name, 'plot': bio or 'Biografie není k dispozici.'})
    xbmcplugin.addDirectoryItem(handle, get_url(action='noop'), li_info, isFolder=False)

    # 2) Filmografie (combined_credits) – seřadit podle popularity, limit
    credits = get_tmdb_person_credits(int(tmdb_id), language_code='cs-CZ') or []
    try:
        credits.sort(key=lambda c: float(c.get('popularity') or 0.0), reverse=True)
    except Exception:
        pass
    MAX_ITEMS = 40
    credits = credits[:MAX_ITEMS]

    def _year_from_str(s: Optional[str]) -> Optional[int]:
        try:
            t = (s or '').strip()
            return int(t[:4]) if len(t) >= 4 else None
        except Exception:
            return None

    for c in credits:
        media_type = c.get('media_type')  # 'movie' / 'tv'
        # Název a rok:
        title = c.get('title') if media_type == 'movie' else c.get('name')
        title = title or 'Bez názvu'
        year = _year_from_str(c.get('release_date') if media_type == 'movie' else c.get('first_air_date'))
        character = c.get('character') or ''
        poster_path = c.get('poster_path')
        tmdb_mid = c.get('id')  # TMDb title ID (film/seriál)

        label = (f"[Film] {title} ({year})" if media_type == 'movie'
                 else f"[Seriál] {title} ({year})") if year else \
                (f"[Film] {title}" if media_type == 'movie' else f"[Seriál] {title}")
        if character:
            label += f" — {character}"

        li = xbmcgui.ListItem(label=label)
        if poster_path:
            purl = IMAGE_BASE_URL_POSTER + "/" + str(poster_path).lstrip('/')
            li.setArt({'thumb': purl, 'icon': purl, 'poster': purl})

        # Navigace:
        if media_type == 'movie':
            # Otevřít Trakt/Webshare vyhledávání (stejný styl jako v _render_movie_item)
            url = get_url(action='trakt_search', ident='', title=title, year=str(year or ''), is_movie='true')
            xbmcplugin.addDirectoryItem(handle, url, li, isFolder=True)
        elif media_type == 'tv':
            # Otevřít TMDb sezóny (pokud máme tmdb_id seriálu)
            if tmdb_mid:
                url = get_url(action='tmdb_seasons', tmdb_id=int(tmdb_mid), title=title)
                xbmcplugin.addDirectoryItem(handle, url, li, isFolder=True)
            else:
                xbmcplugin.addDirectoryItem(handle, get_url(action='noop'), li, isFolder=False)
        else:
            # Neznámý typ – bez akce
            xbmcplugin.addDirectoryItem(handle, get_url(action='noop'), li, isFolder=False)

    xbmcplugin.endOfDirectory(handle, cacheToDisc=False)

# === Prompt helpery (module-level pro router) ================================

def trakt_prompt_mark_season(tmdb_id: int) -> None:
    """
    Prompt pro označení celé SEZÓNY na Traktu:
    - vyžádá číslo sezóny,
    - nabídne výběr data (NOW vs AIRED),
    - zavolá mark_season_watched(tmdb_id, season, watched_mode).
    """
    LOG = "mm.trakt_prompt"

    try:
        tmdb_id = int(tmdb_id or 0)
    except Exception:
        tmdb_id = 0

    if tmdb_id <= 0:
        xbmcgui.Dialog().notification('Trakt', 'Chybí TMDb ID seriálu.', xbmcgui.NOTIFICATION_WARNING, 2500)
        xbmc.log(f"[{LOG}] tmdb_id invalid: {tmdb_id}", xbmc.LOGERROR)
        return

    # 1) vyžádej číslo sezóny
    s_txt = xbmcgui.Dialog().input('Číslo sezóny', type=xbmcgui.INPUT_NUMERIC)
    if not s_txt or not s_txt.isdigit():
        xbmcgui.Dialog().notification('Trakt', 'Zadejte číslo sezóny (jen čísla).', xbmcgui.NOTIFICATION_WARNING, 2500)
        xbmc.log(f"[{LOG}] season input invalid: {s_txt}", xbmc.LOGWARNING)
        return
    season_num = int(s_txt)
    if season_num <= 0:
        xbmcgui.Dialog().notification('Trakt', 'Neplatné číslo sezóny.', xbmcgui.NOTIFICATION_WARNING, 2500)
        xbmc.log(f"[{LOG}] season <= 0", xbmc.LOGWARNING)
        return

    # 2) výběr data
    choice = xbmcgui.Dialog().select(
        f"Zapsat zhlédnutí sezóny S{season_num:02d}",
        ["Použít aktuální datum", "Použít datum vysílání (premiéra epizod)"]
    )
    if choice < 0:
        xbmc.log(f"[{LOG}] user canceled date prompt", xbmc.LOGINFO)
        return
    watched_mode = 'now' if choice == 0 else 'aired'
    xbmc.log(f"[{LOG}] SEASON tmdb={tmdb_id} S={season_num} mode={watched_mode}", xbmc.LOGINFO)

    # 3) zápis na Trakt
    try:
        from resources.lib.trakt_bulk import mark_season_watched
        mark_season_watched(tmdb_id, season_num, watched_mode=watched_mode)
        xbmcgui.Dialog().notification('Trakt', 'Sezóna zapsána.', xbmcgui.NOTIFICATION_INFO, 2000)
    except Exception as e:
        xbmc.log(f"[{LOG}] mark_season_watched error: {e}", xbmc.LOGERROR)
        xbmcgui.Dialog().notification('Trakt', 'Chyba při odeslání sezóny.', xbmcgui.NOTIFICATION_ERROR, 3000)


def trakt_prompt_mark_episode(tmdb_id: int) -> None:
    """
    Prompt pro označení EPIZODY na Traktu:
    - vyžádá S/E,
    - nabídne výběr data (NOW vs AIRED),
    - zavolá mark_episode_watched(tmdb_id, season, episode, watched_mode).
    """
    LOG = "mm.trakt_prompt"

    try:
        tmdb_id = int(tmdb_id or 0)
    except Exception:
        tmdb_id = 0

    if tmdb_id <= 0:
        xbmcgui.Dialog().notification('Trakt', 'Chybí TMDb ID seriálu.', xbmcgui.NOTIFICATION_WARNING, 2500)
        xbmc.log(f"[{LOG}] tmdb_id invalid: {tmdb_id}", xbmc.LOGERROR)
        return

    # 1) vyžádej S/E
    s_txt = xbmcgui.Dialog().input('Číslo sezóny', type=xbmcgui.INPUT_NUMERIC)
    if not s_txt or not s_txt.isdigit():
        xbmcgui.Dialog().notification('Trakt', 'Zadejte číslo sezóny (jen čísla).', xbmcgui.NOTIFICATION_WARNING, 2500)
        xbmc.log(f"[{LOG}] season input invalid: {s_txt}", xbmc.LOGWARNING)
        return
    e_txt = xbmcgui.Dialog().input('Číslo epizody', type=xbmcgui.INPUT_NUMERIC)
    if not e_txt or not e_txt.isdigit():
        xbmcgui.Dialog().notification('Trakt', 'Zadejte číslo epizody (jen čísla).', xbmcgui.NOTIFICATION_WARNING, 2500)
        xbmc.log(f"[{LOG}] episode input invalid: {e_txt}", xbmc.LOGWARNING)
        return
    season_num = int(s_txt)
    episode_num = int(e_txt)
    if season_num <= 0 or episode_num <= 0:
        xbmcgui.Dialog().notification('Trakt', 'Neplatné S/E.', xbmcgui.NOTIFICATION_WARNING, 2500)
        xbmc.log(f"[{LOG}] invalid S/E: S={season_num} E={episode_num}", xbmc.LOGWARNING)
        return

    # 2) výběr data
    choice = xbmcgui.Dialog().select(
        f"Zapsat zhlédnutí S{season_num:02d}E{episode_num:02d}",
        ["Použít aktuální datum", "Použít datum vysílání (premiéra epizody)"]
    )
    if choice < 0:
        xbmc.log(f"[{LOG}] user canceled date prompt", xbmc.LOGINFO)
        return
    watched_mode = 'now' if choice == 0 else 'aired'
    xbmc.log(f"[{LOG}] EP tmdb={tmdb_id} S={season_num} E={episode_num} mode={watched_mode}", xbmc.LOGINFO)

    # 3) zápis na Trakt
    try:
        from resources.lib.trakt_bulk import mark_episode_watched
        mark_episode_watched(tmdb_id, season_num, episode_num, watched_mode=watched_mode)
        xbmcgui.Dialog().notification('Trakt', 'Epizoda zapsána.', xbmcgui.NOTIFICATION_INFO, 2000)
    except Exception as e:
        xbmc.log(f"[{LOG}] mark_episode_watched error: {e}", xbmc.LOGERROR)
        xbmcgui.Dialog().notification('Trakt', 'Chyba při odeslání epizody.', xbmcgui.NOTIFICATION_ERROR, 3000)

# === Odstranění historie (remove) ============================================
def trakt_delete_history(kind: str, trakt_id: int, season: Optional[int] = None) -> None:
    """
    Odstranění záznamů z historie Traktu.
    Endpoint: POST /sync/history/remove
    Chování:
    - Na Trakt odešle remove payload (celý seriál nebo konkrétní sezónu).
    - Po úspěchu (200 OK) lokálně odstraní daný seriál z progress_cache.json (drop_show_by_trakt_id).
    """
    headers = _auth_headers(for_post=True)  # Authorization + Content-Type
    if not headers:
        xbmcgui.Dialog().notification('Trakt', 'Nejste přihlášeni.', xbmcgui.NOTIFICATION_ERROR)
        return
    if not trakt_id:
        xbmcgui.Dialog().notification('Trakt', 'Chybí trakt_id.', xbmcgui.NOTIFICATION_WARNING)
        return
    kind = (kind or '').lower()
    if kind == 'season':
        if season is None or not isinstance(season, int) or season <= 0:
            xbmcgui.Dialog().notification('Trakt', 'Neplatná sezóna.', xbmcgui.NOTIFICATION_WARNING)
            return
        payload = {
            "shows": [
                {
                    "ids": {"trakt": int(trakt_id)},
                    "seasons": [{"number": int(season)}]
                }
            ]
        }
    else:
        payload = {"shows": [{"ids": {"trakt": int(trakt_id)}}]}
    url = f"{TRAKT_API_URL}/sync/history/remove"
    try:
        r = requests.post(url, headers=headers, json=payload, timeout=12)
        xbmc.log(f"[TraktDelete] url={url} kind={kind} status={getattr(r,'status_code',None)} body={payload} "
                 f"resp={getattr(r, 'text', '')[:200]}", xbmc.LOGINFO)
        if r.status_code == 200:
            try:
                pcm = ProgressCacheManager(TraktAPI())
                pcm.drop_show_by_trakt_id(trakt_id)
            except Exception as e:
                xbmc.log(f"[TraktDelete] local cache drop error: {e}", xbmc.LOGWARNING)
            msg = 'Progress seriálu smazán.' if kind == 'show' else f'Sezóna {season} smazána.'
            xbmcgui.Dialog().notification('Trakt', msg, xbmcgui.NOTIFICATION_INFO)
        elif r.status_code == 401:
            xbmcgui.Dialog().notification('Trakt', 'Neplatný token (401) – přihlaste se znovu.', xbmcgui.NOTIFICATION_ERROR)
        else:
            xbmcgui.Dialog().notification('Trakt', f"Chyba {r.status_code}: {getattr(r, 'text', '')[:120]}",
                                          xbmcgui.NOTIFICATION_ERROR)
    except Exception as e:
        xbmc.log(f"[TraktDelete] error: {e}", xbmc.LOGERROR)
        xbmcgui.Dialog().notification('Trakt', 'Chyba při mazání progressu.', xbmcgui.NOTIFICATION_ERROR)

# === Servisní utility: stav v Settings =======================================
def update_trakt_status_setting() -> None:
    """
    Zapíše do settings pole 'trakt_status' informaci o stavu přihlášení.
    Pokud je token, pokusí se načíst /users/me a ukáže username; jinak 'Nepřihlášen'.
    """
    try:
        addon = xbmcaddon.Addon('plugin.video.mmirousek')
        token = addon.getSetting('trakt_access_token') or ''
        if not token:
            addon.setSetting('trakt_status', 'Nepřihlášen')
            return
        api = TraktAPI()
        me = api.get('users/me', params=None, auth=True, cache_ttl=0) or {}
        username = me.get('username') or (me.get('name') or '')
        if username:
            addon.setSetting('trakt_status', f'Přihlášen: {username}')
        else:
            addon.setSetting('trakt_status', 'Přihlášen (uživatel neidentifikován)')
    except Exception as e:
        xbmc.log(f"[Trakt] update_trakt_status_setting error: {e}", xbmc.LOGWARNING)

# === Servisní utility: cache people, clear all ===============================
def trakt_cache_invalidate_people(media_type: str, trakt_id_or_slug: str):
    api = TraktAPI()
    key_n = _build_cache_key(f'{media_type}/{trakt_id_or_slug}/people', None, auth=False)
    path_n = api.cache._path(key_n)
    key_a = _build_cache_key(f'{media_type}/{trakt_id_or_slug}/people', None, auth=True)
    path_a = api.cache._path(key_a)
    removed = []
    try:
        for p in (path_n, path_a):
            if os.path.exists(p):
                os.remove(p)
                removed.append(p)
        xbmc.log(f"[TraktCache] invalidated people keys: {removed}", xbmc.LOGINFO)
    except Exception as e:
        xbmc.log(f"[TraktCache] invalidate error: {e}", xbmc.LOGWARNING)

def trakt_cache_clear_all():
    api = TraktAPI()
    try:
        for fname in os.listdir(api.cache.root):
            if fname.endswith('.json.gz'):
                os.remove(os.path.join(api.cache.root, fname))
        xbmcgui.Dialog().notification('Trakt', 'Cache vyčištěna.', xbmcgui.NOTIFICATION_INFO)
        xbmc.log("[TraktCache] cleared all", xbmc.LOGINFO)
    except Exception as e:
        xbmc.log(f"[TraktCache] clear-all error: {e}", xbmc.LOGERROR)
        xbmcgui.Dialog().notification('Trakt', 'Chyba při čištění cache.', xbmcgui.NOTIFICATION_ERROR)

# --- Progress cache FULL refresh helper --------------------------------------

def trakt_progress_cache_refresh_full_ui():
    """
    Aktualizuje všechny Trakt cache: pokrok seriálů + historie filmů + historie seriálů.
    Zobrazuje progress dialog.
    """
    LOG = "mm.trakt_cache_refresh"
    try:
        api = TraktAPI()
        dlg = xbmcgui.DialogProgress()
        dlg.create("Trakt • Aktualizace cache", "Stahuji pokrok seriálů...")
        # 1) Pokrok seriálů
        pcm = ProgressCacheManager(api)
        ok = pcm.full_fetch_with_progress(show_cancel_button=True, chunk_log_step=20)
        if not ok:
            dlg.close()
            return

        # 2) Historie filmů
        dlg.update(50, "Stahuji historii filmů...")
        movies_data = api.users_watched_movies(page=1, limit=500, extended=True, extended_str='full') or []
        WatchedMoviesCacheManager().save_cache(movies_data)

        # 3) Historie seriálů
        dlg.update(80, "Stahuji historii seriálů...")
        shows_data = api.users_watched_shows_history(page=1, limit=500, extended=True, extended_str='full') or []
        WatchedShowsCacheManager().save_cache(shows_data)

        dlg.update(100, "Hotovo.")
        dlg.close()
        xbmcgui.Dialog().notification('Trakt', 'Cache pokroku + historie aktualizována.', xbmcgui.NOTIFICATION_INFO, 3000)
    except Exception as e:
        xbmc.log(f"[{LOG}] error: {e}", xbmc.LOGERROR)
        try:
            dlg.close()
        except Exception:
            pass
        xbmcgui.Dialog().notification('Trakt', 'Chyba při aktualizaci cache.', xbmcgui.NOTIFICATION_ERROR, 3000)
