# -*- coding: utf-8 -*-
import xbmc
import xbmcgui
import xbmcplugin
import xbmcaddon
import sys
import urllib.parse
from resources.lib.ui_utils import pick_icon as ui_pick_icon, add_menu_item as ui_add_item
from resources.lib.ui_utils import build_trakt_context
from resources.lib.tmdb_utils import get_english_title

# Importujeme TMDb klienta pro získání dat
from resources.lib.tmdb_module import TMDb
# Společné utilitky pro ikonky a položky s plot

# --- TMDb Konstanty pro obrázky ---
IMAGE_BASE_URL = 'https://image.tmdb.org/t/p/w500' 
FANART_BASE_URL = 'https://image.tmdb.org/t/p/original' 

# Standardní Kodi setup pro práci s doplňkem
addon_handle = int(sys.argv[1])
addon = xbmcaddon.Addon('plugin.video.mmirousek')
base_url = sys.argv[0]


# --- build_url: původní funkce ---
def build_url(query):
    """Pomocná funkce pro sestavení URL, kterou Kodi pošle zpět do routeru."""
    return base_url + '?' + urllib.parse.urlencode(query)

# --- NOVÝ WRAPPER ---
def build_url_kwargs(**kwargs):
    """Wrapper kolem build_url, který akceptuje **kwargs (dict-like)."""
    return build_url(kwargs)

# V souboru tmdb_kodi.py, po definici build_url

MOVIE_CATEGORIES = {
    'movie_popular': '🎬 Populární filmy',
    'movie_top_rated': '⭐ Nejlépe hodnocené filmy',
    'movie_upcoming': '🗓️ Připravované filmy',
}

# 🎯 NOVÁ KATEGORIE 'tv_airing_today' PŘIDÁNA ZDE
TV_CATEGORIES = {
    'tv_popular': 'Populární seriály',
    'tv_top_rated': 'Nejlépe hodnocené seriály',
    'tv_on_the_air': 'Právě vysílané (týden)',
    'tv_airing_today': 'Vysílané dnes',
}
# ----------------------------------------------------------------------
# 1. HLAVNÍ MENU FUNKCE
# ----------------------------------------------------------------------

# V souboru tmdb_kodi.py

def show_tmdb_menu():
    """Vytvoří základní menu: Trendy, Filmy a Seriály (sjednocený look)."""
    xbmc.log("TMDb INFO: Generování hlavního menu (Trendy/Filmy/Seriály).", xbmc.LOGINFO)

    # 1) Vyhledávání
    ui_add_item(
        handle=addon_handle,
        build_url_fn=build_url,
        label='Vyhledat na TMDb',
        plot='Zadej název filmu nebo seriálu a zobraz výsledky vyhledávání přímo z TMDb.',
        action='tmdb_search',
        art_icon=ui_pick_icon(addon, 'trending_day.png', 'DefaultAddonsSearch.png', 'tmdb'),
        is_folder=True
    )

    # 2) Trendy (den)
    ui_add_item(
        handle=addon_handle,
        build_url_fn=build_url,
        label='Trendy dnes (Filmy & Seriály)',
        plot='Kombinovaný žebříček toho, co je dnes na TMDb nejvíce trendy.',
        action='tmdb_list',
        art_icon=ui_pick_icon(addon, 'trending_day.png', 'DefaultFavourites.png', 'tmdb'),
        is_folder=True,
        category='trending_day'
    )

    # 3) Filmy
    ui_add_item(
        handle=addon_handle,
        build_url_fn=build_url,
        label='Filmy',
        plot='Procházení filmových kategorií: Populární, Nejlépe hodnocené, Připravované.',
        action='tmdb_movie_categories',
        art_icon=ui_pick_icon(addon, 'movie_popular.png', 'DefaultMovies.png', 'tmdb'),
        is_folder=True
    )

    # 4) Seriály
    ui_add_item(
        handle=addon_handle,
        build_url_fn=build_url,
        label='Seriály',
        plot='Procházení seriálových kategorií: Populární, Nejlépe hodnocené, Právě vysílané, Dnes v TV.',
        action='tmdb_tv_categories',
        art_icon=ui_pick_icon(addon, 'tv_popular.png', 'DefaultTVShows.png', 'tmdb'),
        is_folder=True
    )

    xbmcplugin.endOfDirectory(addon_handle)

def show_movie_categories():
    """Zobrazí seznam kategorií pro FILMY (s ikonou + popisem)."""
    items = [
        ('movie_popular',   'Populární filmy',        'Filmy, které jsou právě nejvíce populární.',      'movie_popular.png'),
        ('movie_top_rated', 'Nejlépe hodnocené filmy', 'Filmy s nejvyšším hodnocením na TMDb.',            'movie_top_rated.png'),
        ('movie_upcoming',  'Připravované filmy',     'Nadcházející premiéry a brzké uvedení do kin.',    'movie_upcoming.png'),
    ]
    for key, label, plot, icon in items:
        ui_add_item(
            handle=addon_handle,
            build_url_fn=build_url,
            label=label,
            plot=plot,
            action='tmdb_list',
            art_icon=ui_pick_icon(addon, icon, 'DefaultFolder.png', 'tmdb'),
            is_folder=True,
            category=key
        )
    xbmcplugin.endOfDirectory(addon_handle)

def show_tv_categories():
    """Zobrazí seznam kategorií pro SERIÁLY (s ikonou + popisem)."""
    items = [
        ('tv_popular',       'Populární seriály',          'Seriály, které jsou právě nejvíce populární.',            'tv_popular.png'),
        ('tv_top_rated',     'Nejlépe hodnocené seriály',  'Seriály s nejvyšším hodnocením na TMDb.',                 'tv_top_rated.png'),
        ('tv_on_the_air',    'Právě vysílané (týden)',     'Epizody, které se aktuálně vysílají v tomto týdnu.',      'tv_on_the_air.png'),
        ('tv_airing_today',  'Vysílané dnes',              'Seriály, které mají novou epizodu právě dnes.',           'tv_airing_today.png'),
    ]
    for key, label, plot, icon in items:
        ui_add_item(
            handle=addon_handle,
            build_url_fn=build_url,
            label=label,
            plot=plot,
            action='tmdb_list',
            art_icon=ui_pick_icon(addon, icon, 'DefaultFolder.png', 'tmdb'),
            is_folder=True,
            category=key
        )
    xbmcplugin.endOfDirectory(addon_handle)

# V souboru tmdb_kodi.py, přidejte novou funkci (např. před show_tmdb_list)

# V souboru tmdb_kodi.py (Musíme se ujistit, že můžeme volat ask() a build_url())

# Budeme předpokládat, že funkce 'ask' je dostupná z hlavního modulu (yawsp)
# Pokud je ask definována ve stejném souboru jako router, budeme ji muset importovat
# z yawsp, ale pro zjednodušení použijeme její logiku přímo zde:

def tmdb_search_menu():
    """Zobrazí dialog pro vyhledávání na TMDb a zobrazí výsledky pomocí ověřeného xbmc.Keyboard."""
    import xbmc
    
    # Použijeme logiku z Vaší funkční 'ask'
    kb = xbmc.Keyboard('', 'Vyhledat na TMDb') # Druhý parametr je název okna
    kb.doModal() 
    
    if kb.isConfirmed():
        search_query = kb.getText()
        if not search_query:
            # Uživatel potvrdil, ale nezadal nic
            xbmcplugin.endOfDirectory(addon_handle, succeeded=False)
            return

        xbmc.log(f"TMDb INFO: Vyhledávání dotazu (Keyboard): {search_query}", xbmc.LOGINFO)

        # 2. Sestavení URL a přesměrování (stejné jako dříve)
        url = build_url({
            'action': 'tmdb_list', 
            'category': 'search', 
            'q': search_query       # Předáváme dotaz jako parametr 'q'
        })

        # Nahrazení aktuálního okna výsledky vyhledávání
        xbmc.executebuiltin(f'Container.Update({url}, replace)')
        
    # V případě, že se dialog nezobrazil, nebo byl zrušen
    xbmcplugin.endOfDirectory(addon_handle)

# ----------------------------------------------------------------------
# 2. ZOBRAZENÍ SEZNAMU POLOŽEK (Filmy/Seriály)
# ----------------------------------------------------------------------



def show_tmdb_list(params):
    """
    Získá seznam filmů/seriálů z TMDb (nebo výsledky vyhledávání) a vykreslí je.
    Přidává sjednocené kontextové menu přes build_trakt_context (Trakt akce).
    """

    # --- Parametry z routeru ---
    category = params.get('category')  # např. 'movie_popular', 'tv_top_rated', 'trending_day'
    query = params.get('q')            # u vyhledávání obsahuje dotaz, jinak None

    # --- Ověření TMDb API klíče ---
    api_key = addon.getSetting('tmdb_api_key')
    if not api_key:
        xbmcgui.Dialog().notification('TMDb chyba', 'Chybí TMDb API klíč v nastavení doplňku.',
                                      xbmcgui.NOTIFICATION_WARNING)
        xbmcplugin.endOfDirectory(addon_handle, succeeded=False)
        return

    # --- Inicializace TMDb klienta ---
    tmdb = TMDb(api_key)

    # --- Načtení seznamu položek (max 3 stránky pro rychlost) ---
    try:
        items = tmdb.get_list(category, query=query, max_pages=3)
    except Exception as e:
        xbmc.log(f"TMDb ERROR: Chyba při získávání seznamu/vyhledávání: {e}", xbmc.LOGERROR)
        xbmcgui.Dialog().notification('TMDb chyba', f'Chyba při volání API: {e}', xbmcgui.NOTIFICATION_ERROR)
        xbmcplugin.endOfDirectory(addon_handle, succeeded=False)
        return

    # --- Pokud nic nenalezeno ---
    if not items:
        xbmc.log(f"TMDb INFO: Načteno 0 položek pro kategorii {category} s dotazem {query}", xbmc.LOGINFO)
        xbmcplugin.endOfDirectory(addon_handle)
        return

    # --- Vykreslování položek ---
# resources/lib/tmdb_kodi.py (uvnitř def show_tmdb_list(params): )

    for item in items:
        media_type = item['media_type']
        tmdb_id = item.get('id')
        
        # --- ZAJIŠTĚNÍ RESETU PRO KAŽDOU POLOŽKU ---
        secondary_label = None
        fallback_title = ""
        
        # Získání původního jazyka
        original_language = item.get('original_language')
        original_label = item.get('original_label') 
        
        # -----------------------------------------------------------------
        # --- KOREKCE CHYBNÉHO (NEPŘELOŽENÉHO) NÁZVU (PAZNAKY) ---
        # Vynutíme získání čitelného EN názvu.
        if original_language not in ['en', 'cs'] and tmdb_id:
            english_label = get_english_title(media_type, tmdb_id)
            
            if english_label:
                # Přepíšeme hlavní název, aby byl čitelný (např. 'Dear X')
                item['label'] = english_label
                
                # SCÉNÁŘ 1: ASIA (Nyní se primární název i fallback rovnají)
                # Fallback_title nastavíme, protože primární hledání se provede na chybnou transliteraci
                # a potřebujeme Fallback na čistý EN název.
                fallback_title = english_label
        # -----------------------------------------------------------------
        
        # --- LOGIKA PRO SEKUNDÁRNÍ NÁZEV A FALLBACK (POUZE CS/EN SERIÁLY) ---
        # Tato logika se spustí pouze tehdy, pokud SERIÁL NEBYL PŘEPSÁN v kroku výše
        if original_language == 'en' and original_label and original_label != item['label']:
            # item['label'] je CS název (např. 'Tváří v tvář')
            # original_label je EN název (např. 'Face to Face')
            secondary_label = original_label # pro zobrazení [Anglický Název]
            fallback_title = original_label # pro fallback vyhledávání (CS -> EN)

        # --- Logování do Kodi logu pro debug ---
        # xbmc.log(f"[TMDb DEBUG] Item: '{item['label']}' (EN/CS), Fallback Title to URL: '{fallback_title}'", xbmc.LOGINFO)

        # --- URL pro kliknutí na položku ---
        if media_type == 'tv':
            # Klik vede na seznam sezón
            url = build_url({
                'action': 'tmdb_seasons', 
                'tmdb_id': tmdb_id, 
                'title': item['label'],
                'fallback_title': fallback_title  # ⬅️ Nyní už je správně
            })
        else:
            # Klik vede na vyhledávání na Webshare
            url = build_url({'action': 'search', 'what': item['label']})
            
        # --- LOGIKA PRO SEKUNDÁRNÍ NÁZEV (KONTROLUJEME PŘEDĚLANÝ LABEL) ---
        secondary_label = None
        
        # Zkontrolujeme, zda původní název (předpokládáme EN) NEBYL PŘEPSÁN
        # Toto se stane, pokud TMDb vrátilo 'en' jako original_language.
        if original_language == 'en' and original_label and original_label != item['label']:
            # Použijeme ho jen, pokud TMDb vrátilo CS název (item['label']) a liší se od EN (original_label).
            secondary_label = original_label
            
        # --- Prefix podle typu (barevné označení) ---
        prefix = "[COLOR yellow][Seriál][/COLOR] " if media_type == 'tv' else "[COLOR magenta][Film][/COLOR] "
        
        # --- SESTAVENÍ HLAVNÍHO NÁZVU (item['label'] je nyní čitelný) ---
        display_label = prefix + item['label']

        # Přidáme sekundární název (nyní se přidá jen, pokud se liší)
        if secondary_label and secondary_label != item['label']:
            display_label += f" [{secondary_label}]" 

        # --- Doplnění roku a hodnocení ---
        if item.get('year'):
            display_label += f" ({item['year']})"
        if item.get('rating') and item['rating'] > 0:
            display_label += f" [★{item['rating']:.1f}]"

        # --- Vytvoření ListItem ---
        li = xbmcgui.ListItem(label=display_label)
        
        
        # --- Artwork (poster + fanart) ---
        if item.get('poster_path'):
            li.setArt({'icon': IMAGE_BASE_URL + item['poster_path'], 'thumb': IMAGE_BASE_URL + item['poster_path']})
        if item.get('backdrop_path'):
            li.setArt({'fanart': FANART_BASE_URL + item['backdrop_path']})

        # --- InfoLabels pro Kodi ---
        info_labels = {
            # V InfoLabels necháme jen CS název, aby to nepletlo knihovnu
            'title': item['label'],
            'plot': item.get('plot'),
            'year': item.get('year'),
            'rating': item.get('rating'),
            'mediatype': 'tvshow' if media_type == 'tv' else 'movie',
        }
        li.setInfo(type='Video', infoLabels=info_labels)

        # --- KONTEKSTOVÉ MENU přes build_trakt_context ---
        # Použijeme get_url z yawsp.py (podporuje **kwargs), jinak by build_url selhal
        cm = build_trakt_context(
            get_url_fn=build_url_kwargs,  # ✅ wrapper pro **kwargs
            title=item['label'],
            tmdb_id=tmdb_id,
            media_type='shows' if media_type == 'tv' else 'movies',
            context_type='default'
        )
        if cm:
            li.addContextMenuItems(cm, replaceItems=False)

        # --- Přidání položky do adresáře ---
        xbmcplugin.addDirectoryItem(handle=addon_handle, url=url, listitem=li, isFolder=True)

    # --- Ukončení adresáře ---
    xbmcplugin.endOfDirectory(addon_handle)
