
# -*- coding: utf-8 -*-
# Module: series_manager
# Author: user extension (+ curated streams/TMDb enrichment by mm)
# Updated: 2025-11-27 (refactor: clean_filename/path_exists/listdir_safe from ui_utils)
# License: AGPL v.3 https://www.gnu.org/licenses/agpl-3.0.html

import os
import io
import re
import json
import shutil
import tempfile
import xbmc
import xbmcaddon
import xbmcgui
import xml.etree.ElementTree as ET

from resources.lib.ui_utils import make_static_item
from resources.lib.ui_utils import clean_filename, path_exists, listdir_safe

# Helper: bezpečný builder URL, pokud nepředáš get_url z yawsp.py
def _default_build_url(**kwargs):
    """
    Fallback builder plugin URL, když create_*_menu není volána s build_url_fn.
    Použije sys.argv[0] jako base a urlencode s UTF-8.
    """
    try:
        import sys
        from urllib.parse import urlencode
        base = sys.argv[0]
        return "{}?{}".format(base, urlencode(kwargs, encoding="utf-8"))
    except Exception:
        # poslední možnost – minimální kompatibilita
        return "plugin://plugin.video.mmirousek/?" + "&".join(
            [f"{k}={v}" for k, v in kwargs.items()]
        )

# --- Atomický zápis JSON ------------------------------------------------------
def _atomic_write_json(file_path: str, data: dict) -> None:
    """
    Zapíše JSON atomicky: nejdřív do temp souboru, pak rename.
    Zajišťuje odolnost proti souběžnému zápisu.
    """
    try:
        dir_name = os.path.dirname(file_path)
        if dir_name and not os.path.exists(dir_name):
            os.makedirs(dir_name)
        fd, tmp_path = tempfile.mkstemp(dir=dir_name, prefix="tmp_", suffix=".json")
        with os.fdopen(fd, 'w', encoding='utf-8') as f:
            json.dump(data, f, ensure_ascii=False, indent=2)
        shutil.move(tmp_path, file_path)
    except Exception as e:
        xbmc.log(f'YaWSP SeriesManager: Chyba atomického zápisu: {e}', xbmc.LOGERROR)

# --- Py2/3 kompatibilní urlencode --------------------------------------------
try:
    from urllib.parse import urlencode
except ImportError:
    from urllib import urlencode  # Py2 fallback (Kodi Leia)

# --- Robustnější detekce S/E (CZ-friendly) -----------------------------------
RE_SxxEyy = re.compile(r'(?i)\bS(\d{1,2})\s*E(\d{1,3})\b')
RE_1x02   = re.compile(r'(?i)\b(\d{1,2})\s*[x×]\s*(\d{1,3})\b')
RE_CZ     = re.compile(r'(?i)\b(sezona|série|řada)\s*(\d{1,2}).*\b(díl|epizoda|ep)\s*(\d{1,3})\b')
RE_EPONLY = re.compile(r'(?i)\b(ep|epizoda)\s*(\d{1,3})\b')
SEASON_PACK_HINTS = re.compile(r'(?i)\b(pack|complete|season\s*pack|s\d{1,2}-s\d{1,2})\b')

PREF_LANG_ORDER = ("cs", "cz", "sk", "czsk", "en", "eng")

def _norm(s):
    return (s or "").strip()

def _light_clean(title):
    """Lehké očištění: ponechá SxxEyy/1x02, odstraní šum typu rozlišení/kodek apod."""
    if not title:
        return ""
    s = title.replace('.', ' ')
    s = re.sub(
        r'(?i)\b(480p|720p|1080p|2160p|4k|webrip|web[\- ]?dl|b[dr]rip|bluray|remux|hdtv|hdr|dv|x264|x265|hevc)\b',
        '', s
    )
    s = re.sub(r'\s{2,}', ' ', s).strip(" -.()[]")
    return s


class SeriesManager:
    def __init__(self, addon, profile):
        self.addon = addon
        self.profile = profile
        self.series_db_path = os.path.join(profile, 'series_db')
        self.ensure_db_exists()
        self.page_size = 100
        self.max_pages = 5  # dle dohody

    # --- NOVÉ/UPRAVENÉ METODY ROBUSTNÍHO NAČÍTÁNÍ ----------------------------

    def validate_series_streams(self, series_name, api_fn, token):
        """
        Validuje dostupnost všech streamů v seriálu přes Webshare API.
        Odstraní neplatné streamy z JSON, vrátí seznam epizod pro rescan.
        """
        import time
        import xbmcgui
        from xml.etree import ElementTree as ET

        data = self.load_series_data(series_name)
        if not data:
            return {"ok": 0, "invalid": 0, "errors": [], "rescan": []}

        ok_count, invalid_count = 0, 0
        errors = []
        rescan_eps = []

        dlg = xbmcgui.DialogProgress()
        dlg.create("Kontrola streamů", f"Seriál: {series_name}")
        total_streams = sum(len(ep.get("streams", [])) for s in data.get("seasons", {}).values() for ep in s.values())
        checked = 0

        for s_key, season in (data.get("seasons") or {}).items():
            for e_key, ep in (season or {}).items():
                streams = ep.get("streams", [])
                valid_streams = []
                for st in streams:
                    if dlg.iscanceled():
                        dlg.close()
                        return {"ok": ok_count, "invalid": invalid_count, "errors": errors, "rescan": rescan_eps}

                    ident = st.get("ident")
                    checked += 1
                    progress_value = int((checked / total_streams) * 100)
                    line1 = f"S{int(s_key):02d}E{int(e_key):02d}"
                    line2 = f"Zkontrolováno: {checked}/{total_streams}"

                    # Kodi 21 podporuje více řádků, fallback pro starší verze
                    try:
                        dlg.update(progress_value, line1, line2)
                    except TypeError:
                        dlg.update(progress_value)

                    try:
                        resp = api_fn("file_info", {"ident": ident, "wst": token})
                        xml = ET.fromstring(resp.content)
                        if xml.findtext("status") == "OK":
                            ok_count += 1
                            valid_streams.append(st)
                        else:
                            invalid_count += 1
                            tag = f"S{int(s_key):02d}E{int(e_key):02d}"
                            if tag not in errors:
                                errors.append(tag)
                    except Exception:
                        invalid_count += 1
                        tag = f"S{int(s_key):02d}E{int(e_key):02d}"
                        if tag not in errors:
                            errors.append(tag)

                    time.sleep(0.17)  # ~6 req/s

                if len(valid_streams) != len(streams):
                    ep["streams"] = valid_streams
                    rescan_eps.append((s_key, e_key))

        # Atomický zápis JSON
        safe_name = self._safe_filename(series_name, data.get("tmdb", {}).get("id"))
        file_path = os.path.join(self.series_db_path, f"{safe_name}.json")
        _atomic_write_json(file_path, data)

        dlg.close()
        return {"ok": ok_count, "invalid": invalid_count, "errors": errors, "rescan": rescan_eps}


    def _resolve_series_file_path(self, series_name_or_path: str) -> str:
        """
        Robustně určí cestu k JSON souboru seriálu.
        Přijme:
        - plnou cestu k .json (vrátí ji, pokud existuje),
        - název seriálu (CZ/EN),
        - 'tmdb_12345' (safe name z filename).
        Vrátí absolutní path nebo None.
        """
        if not isinstance(series_name_or_path, str) or not series_name_or_path.strip():
            return None
        key = series_name_or_path.strip()

        # 1) Je to přímo path k JSON souboru?
        if key.endswith('.json') and (os.path.sep in key):
            return key if os.path.exists(key) else None

        # 2) Je to safe name z filename? (např. 'tmdb_12345')
        if key.startswith('tmdb_'):
            try:
                tmdb_id = int(key[5:])
                fp = os.path.join(self.series_db_path, f"tmdb_{tmdb_id}.json")
                if os.path.exists(fp):
                    return fp
            except Exception:
                pass  # pokračuj dál

        # 3) Slug podle názvu seriálu (bez TMDb)
        slug = self._safe_filename(key)
        fp_slug = os.path.join(self.series_db_path, f"{slug}.json")
        if os.path.exists(fp_slug):
            return fp_slug

        # 4) Najdi odpovídající tmdb_<id>.json podle obsahu (name)
        try:
            for fn in os.listdir(self.series_db_path):
                if not fn.startswith('tmdb_') or not fn.endswith('.json'):
                    continue
                fp = os.path.join(self.series_db_path, fn)
                try:
                    with io.open(fp, 'r', encoding='utf8') as f:
                        d = json.load(f)
                    if not isinstance(d, dict):
                        continue
                    nm = (d.get('name') or '').strip().lower()
                    if nm and nm == key.strip().lower():
                        return fp
                except Exception:
                    continue
        except Exception:
            pass
        return None

    def load_series_data(self, series_name_or_path):
        """
        Načte data seriálu — název i path jsou podporované.
        Preferuje soubor tmdb_<id>.json, pokud existuje.
        """
        file_path = self._resolve_series_file_path(series_name_or_path)
        if not file_path or not os.path.exists(file_path):
            return None
        try:
            with io.open(file_path, 'r', encoding='utf8') as f:
                data = f.read()
            series_data = json.loads(data)  # Py3-safe
            return self._migrate_schema(series_data)
        except Exception as e:
            xbmc.log(f'YaWSP Series Manager: Error loading series data: {e}', level=xbmc.LOGERROR)
            return None

    def _get_tmdb_id(self, series_name: str) -> int:
        """
        Vrátí TMDb ID pro daný seriál — robustně.
        Zkusí načíst data přes _resolve_series_file_path a přečte tmdb.id.
        Pokud nenajde, vrátí None.
        """
        try:
            # 1) Zkus najít soubor podle názvu/safe name
            fp = self._resolve_series_file_path(series_name)
            if fp and os.path.exists(fp):
                with io.open(fp, 'r', encoding='utf8') as f:
                    d = json.load(f)
                if isinstance(d, dict):
                    tid = (d.get("tmdb") or {}).get("id")
                    if isinstance(tid, int) and tid > 0:
                        return tid

            # 2) Fallback: projdi tmdb_* soubory a porovnej name
            for fn in os.listdir(self.series_db_path):
                if not fn.startswith('tmdb_') or not fn.endswith('.json'):
                    continue
                try:
                    with io.open(os.path.join(self.series_db_path, fn), 'r', encoding='utf8') as f:
                        d = json.load(f)
                    if not isinstance(d, dict):
                        continue
                    nm = (d.get('name') or '').strip().lower()
                    if nm and nm == (series_name or '').strip().lower():
                        tid = (d.get("tmdb") or {}).get("id")
                        if isinstance(tid, int) and tid > 0:
                            return tid
                except Exception:
                    continue
            return None
        except Exception:
            return None

    def mark_episode_progress(self, series_name: str, season: int, episode: int, progress: float,
                              set_by: str = 'manual') -> None:
        """Označí jednu epizodu jako zhlédnutou/rozkoukanou nebo resetuje stav."""
        try:
            data = self.load_series_data(series_name)
            if not data:
                xbmc.log(f"[SeriesManager] mark_episode_progress: Data nenalezena pro {series_name}", xbmc.LOGWARNING)
                return

            s_key, e_key = str(season), str(episode)
            ep = data.get('seasons', {}).get(s_key, {}).get(e_key)
            if not ep:
                xbmc.log(f"[SeriesManager] mark_episode_progress: Epizoda {s_key}/{e_key} nenalezena", xbmc.LOGWARNING)
                return

            ep['progress'] = round(progress, 1)
            ep['watched'] = progress >= 90.0

            if ep['watched']:
                ep['play_count'] = ep.get('play_count', 0) + 1
                ep['last_watched_at'] = xbmc.getInfoLabel('System.Date') + 'T' + xbmc.getInfoLabel('System.Time')
                ep['set_by'] = set_by

            # Přepočítat stav sezóny
            self._recalc_season_state(data, s_key)

            # Uložit změny
            safe_name = self._safe_filename(series_name, data.get('tmdb', {}).get('id'))
            file_path = os.path.join(self.series_db_path, f"{safe_name}.json")
            _atomic_write_json(file_path, data)

            xbmc.log(f"[SeriesManager] mark_episode_progress: {series_name} S{s_key}E{e_key} -> {progress}%", xbmc.LOGINFO)
        except Exception as e:
            xbmc.log(f"[SeriesManager] CHYBA mark_episode_progress: {e}", xbmc.LOGERROR)

    def recalc_and_save_season_state(self, data: dict, season_key: str):
        """
        Přepočítá stav sezóny a uloží změny do souboru.
        """
        try:
            # Přepočítat stav sezóny
            self._recalc_season_state(data, season_key)

            # Uložit změny atomicky
            safe_name = self._safe_filename(data.get('name'), data.get('tmdb', {}).get('id'))
            file_path = os.path.join(self.series_db_path, f"{safe_name}.json")
            _atomic_write_json(file_path, data)

            xbmc.log(f"[SeriesManager] Season {season_key} state recalculated and saved", xbmc.LOGINFO)
        except Exception as e:
            xbmc.log(f"[SeriesManager] CHYBA recalc_and_save_season_state: {e}", xbmc.LOGERROR)

    # --- Přepočet stavu sezóny ------------------------------------------------
    def _recalc_season_state(self, data: dict, season_key: str) -> None:
        """
        Spočítá watched_count, total a watched flag pro sezónu.
        Uloží do data['season_state'][season_key].
        """
        try:
            season = data.get('seasons', {}).get(season_key, {})
            total = len(season)
            watched_count = sum(1 for ep in season.values() if ep.get('watched'))
            watched_flag = (watched_count == total and total > 0)

            season_state = data.setdefault('season_state', {})
            season_state[season_key] = {
                'watched': watched_flag,
                'watched_count': watched_count,
                'total': total,
                'last_updated': xbmc.getInfoLabel('System.Date')
            }
        except Exception as e:
            xbmc.log(f'YaWSP SeriesManager: _recalc_season_state CHYBA: {e}', xbmc.LOGERROR)

    def _label_suffix_for_episode(self, ep: dict, color: bool) -> str:
        """
        Vrátí suffix pro stav epizody: [OK], [75%], [New] s/bez barvy.
        """
        try:
            progress = float(ep.get('progress', 0.0) or 0.0)
            if ep.get('watched'):
                return '[COLOR lime]ok[/COLOR]' if color else '[ok]'
            elif progress > 5.0:
                p = int(round(progress))
                return f"[COLOR lime]{p}%[/COLOR]" if color else f"[{p}%]"
            else:
                return '[COLOR dimgray]New[/COLOR]' if color else '[New]'
        except Exception:
            return ''

    # --- FS zajištění ---------------------------------------------------------
    def ensure_db_exists(self):
        """Ensure that the series database directory exists"""
        try:
            if not os.path.exists(self.profile):
                os.makedirs(self.profile)
            if not os.path.exists(self.series_db_path):
                os.makedirs(self.series_db_path)
        except Exception as e:
            xbmc.log(f'YaWSP Series Manager: Error creating directories: {e}', level=xbmc.LOGERROR)

    # --- Vyhledávání + kurátorování + TMDb enrichment -------------------------
    def search_series(self, series_name, api_function, token, **kwargs):
        # --- režimové přepínače (kvůli quality režimům) ---
        skip_high_quality = bool(kwargs.get('skip_high_quality', False))
        force_hd_only     = bool(kwargs.get('force_hd_only', False))
        name_only         = bool(kwargs.get('name_only', False))
        only_uhd          = bool(kwargs.get('only_uhd', False))

        if skip_high_quality:
            xbmc.log("[SeriesManager] skip_high_quality=True → kurátorování bez 4K", xbmc.LOGINFO)
        if force_hd_only:
            xbmc.log("[SeriesManager] force_hd_only=True → pouze 1080p/720p", xbmc.LOGINFO)
        if name_only:
            xbmc.log("[SeriesManager] name_only=True → ignoruji kvalitu, zachovám pořadí API", xbmc.LOGINFO)
        if only_uhd:
            xbmc.log("[SeriesManager] only_uhd=True → filtruji striktně na 2160p/4K", xbmc.LOGINFO)

        # Init dat struktury
        series_data = self._init_series_data(series_name)

        # ⬅️ DŮLEŽITÉ: zkusit získat TMDb hned na začátku (aby filtr měl ID)
        try:
            self._enrich_with_tmdb(series_data, series_name)
        except Exception as e:
            xbmc.log(f'YaWSP Series Manager: TMDb enrich (early) skipped: {e}', xbmc.LOGWARNING)

        known_idents = set(series_data.get("known_idents", []))
        slug_series = re.sub(r'[^a-z0-9]', '', series_name.lower())

        # Hledací dotazy (jednoduchá kombinace)
        queries = [series_name, f"{series_name} s01", f"{series_name} episode", f"{series_name} season"]
        raw_items, seen_idents = [], set()

        for q in queries:
            for page in range(self.max_pages):
                results = self._perform_search(q, api_function, token,
                                               offset=page * self.page_size,
                                               limit=self.page_size)
                if not results:
                    break
                for r in results:
                    name = r.get('name') or ''
                    ident = r.get('ident') or ''
                    if not ident or ident in seen_idents:
                        continue
                    # Vyřazení season-packů
                    if SEASON_PACK_HINTS.search(name or ''):
                        continue
                    # Slug kontrola
                    slug_name = re.sub(r'[^a-z0-9]', '', name.lower())
                    if self._is_likely_episode(name, series_name) and slug_name.startswith(slug_series):
                        raw_items.append(r)
                    else:
                        xbmc.log(f"[SeriesManager] Vyřazeno (slug mismatch): {name}", xbmc.LOGDEBUG)
                    seen_idents.add(ident)

        xbmc.log(f"[SeriesManager] Po slug filtru zůstává {len(raw_items)} kandidátů", xbmc.LOGINFO)

        # Seskupení do bucketů (S,E)
        buckets = {}
        for item in raw_items:
            s, e = self._detect_episode_info(item.get('name', ''), series_name)
            if s is None or e is None:
                continue
            buckets.setdefault((s, e), []).append(item)

        # ⬅️ TMDb validace sezón/epizod — teď už máme tmdb.id → filtr nebude přeskakovat
        xbmc.log(f"[SeriesManager] Buckets před TMDb filtrem: {len(buckets)}", xbmc.LOGINFO)
        try:
            buckets = filter_invalid_seasons(series_data, buckets)
        except Exception as ex:
            xbmc.log(f"[SeriesManager] Filtr TMDb selhal: {ex}", xbmc.LOGWARNING)
        xbmc.log(f"[SeriesManager] Buckets po TMDb filtru: {len(buckets)}", xbmc.LOGINFO)

        # Uložení kurátorovaných streamů (bez řazení, zachovat pořadí API)
        for (s, e), cand_list in buckets.items():
            if not cand_list:
                continue
            curated_streams = self._curate_streams(
                cand_list,
                skip_high_quality=skip_high_quality,
                force_hd_only=force_hd_only,
                name_only=name_only,
                only_uhd=only_uhd
            )
            if not curated_streams:
                continue

            s_str, e_str = str(s), str(e)
            if s_str not in series_data['seasons']:
                series_data['seasons'][s_str] = {}
            entry = series_data['seasons'][s_str].get(e_str, {})
            entry.setdefault('source_name', curated_streams[0].get('name') or '')
            entry.setdefault('name', '')
            entry.setdefault('tmdb', {})
            entry['streams'] = curated_streams
            entry['curated'] = True
            entry['locked'] = entry.get('locked', False)
            entry['updated_at'] = xbmc.getInfoLabel('System.Date')
            series_data['seasons'][s_str][e_str] = entry

            for st in curated_streams:
                if st.get('ident'):
                    known_idents.add(st['ident'])
            series_data['known_idents'] = sorted(list(known_idents))

        # Doplň finální TMDb metadata (tituly epizod, stills, runtime, poster/backdrop)
        try:
            self._enrich_with_tmdb(series_data, series_name)
        except Exception as e:
            xbmc.log(f'YaWSP Series Manager: TMDb enrich skipped: {e}', xbmc.LOGWARNING)

        self._save_series_data(series_name, series_data)
        return self._migrate_schema(series_data)

    def _is_likely_episode(self, filename, series_name):
        """Check if a filename is likely to be an episode of the series"""
        if not filename:
            return False
        # povolit i varianty názvu (diakritika/oddělovače)
        if series_name and re.search(re.escape(series_name), filename, re.IGNORECASE) is None:
            if re.sub(r'\W+', '', series_name.lower()) not in re.sub(r'\W+', '', filename.lower()):
                return False
        t = _light_clean(filename)
        if RE_SxxEyy.search(t) or RE_1x02.search(t) or RE_CZ.search(t) or RE_EPONLY.search(t):
            return True
        return False

    def _perform_search(self, search_query, api_function, token, offset=0, limit=100):
        """Perform the actual search using the provided API function with paging"""
        results = []
        try:
            response = api_function('search', {
                'what': search_query,
                'category': 'video',
                'sort': 'recent',
                'limit': int(limit),
                'offset': int(offset),
                'wst': token,
                'maybe_removed': 'true'
            })
            try:
                xml = ET.fromstring(response.content)
            except Exception as ex:
                xbmc.log(f'YaWSP Series Manager: XML parse error: {ex}', xbmc.LOGWARNING)
                return []
            status = xml.find('status')
            if status is not None and (status.text or '').upper() == 'OK':
                for file in xml.iter('file'):
                    item = {}
                    for elem in file:
                        item[elem.tag] = elem.text
                    results.append(item)
            return results
        except Exception as e:
            xbmc.log(f'YaWSP Series Manager: search error: {e}', xbmc.LOGERROR)
            return []

    def _detect_episode_info(self, filename, series_name):
        """Detect S/E from filename (robust, CZ friendly)."""
        if not filename:
            return None, None
        t = _light_clean(filename)
        m = RE_SxxEyy.search(t)
        if m:
            return int(m.group(1)), int(m.group(2))
        m = RE_1x02.search(t)
        if m:
            return int(m.group(1)), int(m.group(2))
        m = RE_CZ.search(t)
        if m:
            # groups: (sezona|série|řada) (S) (díl|epizoda|ep) (E)
            try:
                return int(m.group(2)), int(m.group(4))
            except Exception:
                return None, None
        m = RE_EPONLY.search(t)
        if m:
            return 1, int(m.group(2))
        return None, None

    def _save_series_data(self, series_name, series_data):
        """Save series data to the database, filename based on TMDb ID"""
        try:
            tmdb_id = series_data.get("tmdb", {}).get("id")
            safe_name = self._safe_filename(series_name, tmdb_id)
            file_path = os.path.join(self.series_db_path, f"{safe_name}.json")
            _atomic_write_json(file_path, series_data)
            xbmc.log(f"YaWSP SeriesManager: uložen soubor {file_path}", xbmc.LOGINFO)
        except Exception as e:
            xbmc.log(f"YaWSP SeriesManager: Error saving series data: {e}", xbmc.LOGERROR)

    def get_all_series(self):
        """Get a list of all saved series"""
        series_list = []
        try:
            for filename in os.listdir(self.series_db_path):
                if filename.endswith('.json'):
                    safe_name = os.path.splitext(filename)[0]
                    # Zobrazíme uživatelské jméno dle obsahu, pokud jde načíst
                    display_name = safe_name.replace('_', ' ')
                    fp = os.path.join(self.series_db_path, filename)
                    try:
                        with io.open(fp, 'r', encoding='utf8') as f:
                            d = json.load(f)
                        nm = (d.get('name') or '').strip()
                        if nm:
                            display_name = nm
                    except Exception:
                        pass
                    series_list.append({
                        'name': display_name,
                        'filename': filename,
                        'safe_name': safe_name
                    })
        except Exception as e:
            xbmc.log(f'YaWSP Series Manager: Error listing series: {e}', level=xbmc.LOGERROR)
        return series_list

    def _safe_filename(self, series_name: str, tmdb_id: int = None) -> str:
        """
        Vrátí bezpečný název souboru pro seriál.
        Preferuje TMDb ID, jinak fallback na očištěný název.
        """
        if tmdb_id:
            return f"tmdb_{tmdb_id}"
        return re.sub(r'[^a-z0-9]+', '_', (series_name or '').lower()).strip('_')

    # --- schéma v3 + migrace --------------------------------------------------
    def _init_series_data(self, series_name):
        data = self.load_series_data(series_name) or {}
        if not data:
            data = {
                "schema_version": 3,
                "name": series_name,
                "aliases": [],
                "last_updated": xbmc.getInfoLabel('System.Date'),
                "source": "webshare",
                "known_idents": [],
                "tmdb": {},
                "extras": {"season_packs": []},
                "seasons": {}
            }
        else:
            data.setdefault("schema_version", 3)
            data.setdefault("aliases", [])
            data.setdefault("source", "webshare")
            data.setdefault("known_idents", [])
            data.setdefault("tmdb", {})
            data.setdefault("extras", {"season_packs": []})
            data.setdefault("seasons", {})
            data["last_updated"] = xbmc.getInfoLabel('System.Date')
        return data

    def _migrate_schema(self, data):
        if not isinstance(data, dict):
            return data
        data.setdefault("schema_version", 3)
        data.setdefault("aliases", [])
        data.setdefault("source", "webshare")
        data.setdefault("known_idents", [])
        data.setdefault("tmdb", {})
        data.setdefault("extras", {"season_packs": []})
        seasons = data.setdefault("seasons", {})

        # migrace starého "ident" -> streams[0]
        for s in list(seasons.keys()):
            for e in list(seasons[s].keys()):
                ep = seasons[s][e]
                if isinstance(ep, dict) and "streams" not in ep:
                    if "ident" in ep:
                        streams = [{
                            "ident": ep.get("ident"),
                            "name": ep.get("name", ""),
                            "quality": ep.get("quality", ""),
                            "lang": ep.get("lang", ""),
                            "size": int(ep.get("size", "0") or 0),
                            "ainfo": ep.get("ainfo", ""),
                            "score": 0,
                            "added_at": xbmc.getInfoLabel('System.Date'),
                            "source": "webshare"
                        }]
                        # doplň features/label
                        for st in streams:
                            st['features'] = self._extract_features(st)
                            st['label'] = ', '.join(st['features']) if st['features'] else (_norm(st['quality']) or _norm(st['name']) or 'Stream')
                        ep["streams"] = streams
                        ep.setdefault("source_name", ep.get("name", ""))
                        ep.setdefault("tmdb", {})
                        ep.setdefault("curated", False)
                        ep.setdefault("locked", False)
                        ep.setdefault("updated_at", xbmc.getInfoLabel('System.Date'))
                        # ep['name'] bude později přepsán TMDb názvem
        return data

    # --- skórování a kurátorování --------------------------------------------
    def _quality_rank(self, q):
        ql = (q or "").lower()
        if "2160" in ql or "4k" in ql:
            return 100
        if "1080" in ql:
            return 80
        if "720" in ql:
            return 60
        if "540" in ql or "sd" in ql:
            return 30
        return 40

    def _lang_rank(self, lang):
        l = (lang or "").lower()
        for idx, tag in enumerate(PREF_LANG_ORDER):
            if tag in l:
                return (len(PREF_LANG_ORDER) - idx) * 10
        return 0

    def _candidate_score(self, item):
        q = (item.get('quality') or '') + ' ' + (item.get('ainfo') or '')
        lang = (item.get('lang') or '')
        score = self._quality_rank(q) + self._lang_rank(lang)
        ql = q.lower()
        if 'hdr' in ql or 'dolby vision' in ql or 'dv' in ql:
            score += 10
        if 'x265' in ql or 'hevc' in ql:
            score += 5
        try:
            size = float(item.get('size', 0) or 0)  # bytes
            score += min(size / (1024*1024*200), 10)  # až +10 podle velikosti
        except Exception:
            pass
        if re.search(r'(?i)\b(cam|telesync|ts|scr|screener)\b', q):
            score -= 50
        return score

    def _curate_streams(self, candidates, skip_high_quality=False, force_hd_only=False,
                        name_only=False, only_uhd=False):
        """
        Kurátorování streamů:
        - Zachová pořadí z Webshare API (Most relevant).
        - Použije jen filtry kvality podle režimu.
        - Žádné řazení podle velikosti ani skórování.
        """
        norm = []
        seen = set()
        if not name_only:
            if only_uhd:
                candidates = [c for c in candidates if re.search(r'(?i)(2160|4k)', (c.get('quality') or '') + (c.get('name') or ''))]
            if force_hd_only:
                candidates = [c for c in candidates if re.search(r'(?i)(1080|720)', (c.get('quality') or '') + (c.get('name') or ''))]
            if skip_high_quality:
                candidates = [c for c in candidates if not re.search(r'(?i)(2160|4k)', (c.get('quality') or '') + (c.get('name') or ''))]

        for c in candidates:
            ident = c.get('ident') or ''
            if not ident or ident in seen:
                continue
            seen.add(ident)
            item = {
                'ident': ident,
                'name': c.get('name') or '',
                'quality': c.get('quality') or c.get('ainfo') or '',
                'lang': c.get('lang') or '',
                'size': int(c.get('size', '0') or 0),
                'ainfo': c.get('ainfo') or '',
                'score': 0,
                'added_at': xbmc.getInfoLabel('System.Date'),
                'source': 'webshare'
            }
            feats = self._extract_features(item)
            item['features'] = feats
            item['label'] = ', '.join(feats) if feats else (_norm(item['quality']) or _norm(item['name']) or 'Stream')
            norm.append(item)
        return norm[:6]

    def _enrich_with_tmdb(self, series_data: dict, series_name: str):
        """
        Doplní detail seriálu (poster/backdrop/overview + show_runtime) a metadata epizod:
        series_data['tmdb'] = {
          "id", "title", "original_title", "year", "poster", "backdrop",
          "poster_url", "backdrop_url", "overview", "show_runtime": int|None,
          "imdb_id": str|None
        }
        Epizodám ukládá: title, overview, still_url, air_date, vote, runtime (fallback na show_runtime).
        """
        try:
            from resources.lib import tmdb_utils as TM
        except Exception:
            TM = None

        # 1) Najdi/ověř show ID (přes search)
        if not series_data.get('tmdb') or not series_data['tmdb'].get('id'):
            tv_meta = None
            try:
                if TM and hasattr(TM, 'search_tmdb_tv'):
                    tv_meta = TM.search_tmdb_tv(series_name)
            except Exception:
                tv_meta = None
            if tv_meta:
                series_data.setdefault('tmdb', {})
                series_data['tmdb'].update({
                    "id": tv_meta.get("tmdb_id"),
                    "title": tv_meta.get("title"),
                    "original_title": tv_meta.get("original_title") or tv_meta.get("title"),
                    "year": tv_meta.get("year"),
                })
                series_data['name'] = tv_meta.get("title") or series_name

        tv_id = series_data.get('tmdb', {}).get('id')
        if not tv_id:
            return

        # 2) Detaily seriálu: get_tv_show_meta → show_runtime + poster/backdrop/overview
        show_meta = None
        if TM and hasattr(TM, 'get_tv_show_meta'):
            try:
                show_meta = TM.get_tv_show_meta(tv_id=int(tv_id), language='cs')
            except Exception:
                show_meta = None

        poster_path   = show_meta.get('poster_path')   if show_meta else None
        backdrop_path = show_meta.get('backdrop_path') if show_meta else None
        overview      = show_meta.get('overview')      if show_meta else None
        rt            = show_meta.get('episode_run_time') if show_meta else None

        if isinstance(rt, list) and rt:
            show_runtime = int(rt[0]) if isinstance(rt[0], (int, float)) else None
        elif isinstance(rt, int):
            show_runtime = rt
        else:
            show_runtime = None

        # 2a) Získání IMDb ID z externích identifikátorů
        imdb_id = None
        if TM and hasattr(TM, 'get_tv_external_ids'):
            try:
                ext_ids = TM.get_tv_external_ids(tv_id=int(tv_id))
                imdb_id = ext_ids.get('imdb_id')
            except Exception:
                imdb_id = None

        img_base_poster = "https://image.tmdb.org/t/p/w500"
        img_base_fanart = "https://image.tmdb.org/t/p/original"

        series_tm = series_data.setdefault('tmdb', {})
        series_tm.update({
            "poster": poster_path,
            "backdrop": backdrop_path,
            "poster_url": (img_base_poster + poster_path) if poster_path else None,
            "backdrop_url": (img_base_fanart + backdrop_path) if backdrop_path else None,
            "overview": overview,
            "show_runtime": show_runtime,
            "imdb_id": imdb_id
        })

        # 2b) Pokud nemáme 'year', zkus dopočítat z první epizody S01E01 (air_date[:4])
        if not series_tm.get('year'):
            try:
                s1 = series_data.get('seasons', {}).get('1', {})
                e1 = s1.get('1', {})
                air = (e1.get('tmdb') or {}).get('air_date') or (e1.get('air_date') if isinstance(e1, dict) else None)
                if air and len(air) >= 4:
                    series_tm['year'] = int(air[:4])
            except Exception:
                pass

        # 3) Epizody – title/overview/still_url/air_date/vote/runtime (+name)
        for s in list(series_data['seasons'].keys()):
            for e in list(series_data['seasons'][s].keys()):
                ep = series_data['seasons'][s][e]
                if ep.get('locked'):
                    continue
                ep.setdefault('tmdb', {})
                ep_meta = None
                if TM and hasattr(TM, 'get_tv_episode_meta'):
                    try:
                        ep_meta = TM.get_tv_episode_meta(tv_id=int(tv_id), season=int(s), episode=int(e), language='cs')
                    except Exception:
                        ep_meta = None
                if ep_meta:
                    ep_title   = ep_meta.get('title')
                    ep_overview= ep_meta.get('overview')
                    ep_still   = ep_meta.get('still_url') or None
                    ep_air     = ep_meta.get('air_date')
                    ep_vote    = ep_meta.get('vote')
                    ep_runtime = ep_meta.get('runtime')
                    if (ep_runtime is None or (isinstance(ep_runtime, int) and ep_runtime <= 0)) and isinstance(show_runtime, int) and show_runtime > 0:
                        ep_runtime = show_runtime
                    ep['tmdb'] = {
                        "episode_id": ep_meta.get("episode_id"),
                        "title": ep_title,
                        "overview": ep_overview,
                        "still_path": ep_meta.get('still_path'),
                        "still_url": ep_still,
                        "air_date": ep_air,
                        "vote": ep_vote,
                        "runtime": ep_runtime
                    }
                    if ep_title:
                        ep['title'] = ep_title
                        ep['name']  = ep_title
                    else:
                        if not ep.get('name'):
                            ep['name'] = ep.get('source_name') or ep.get('title') or ''
                else:
                    if not ep.get('name'):
                        ep['name'] = ep.get('title') or ep.get('source_name') or ''

    @staticmethod
    def _extract_features(st):
        """
        Vrátí seznam vlastností jako ['CZ-EN-Atmos-5.1', 'HDR10', 'HEVC', '2160p'].
        Vstupy bere z:
        - st['name'] (původní filename),
        - st['quality'] a st['ainfo'],
        - st['lang'] (pokud není v názvu).
        """
        name = (st.get('name') or '')
        q    = (st.get('quality') or '')
        ai   = (st.get('ainfo') or '')
        lang = (st.get('lang') or '')
        blob = ' '.join([name, q, ai]).lower()

        feats = []
        # 1) jazyk(y) + audio layout
        langs = []
        if re.search(r'\b(cz|cs)\b', blob) or 'cz' in lang or 'cs' in lang:
            langs.append('CZ')
        if re.search(r'\bsk\b', blob) or 'sk' in lang:
            if 'CZ' in langs:
                langs[-1] = 'CZ-SK'
            else:
                langs.append('SK')
        if re.search(r'\b(en|eng|english)\b', blob) or 'en' in lang:
            if langs and langs[-1] in ('CZ', 'CZ-SK', 'SK'):
                langs[-1] = f"{langs[-1]}-EN"
            else:
                langs.append('EN')

        audio = []
        if re.search(r'\batmos\b', blob):
            audio.append('Atmos')
        ch = re.search(r'\b(7\.1|5\.1|2\.0)\b', blob)
        if ch:
            audio.append(ch.group(1))

        lang_audio = '-'.join(x for x in [(' - '.join(langs) if langs else ''), *audio] if x)
        lang_audio = lang_audio.replace(' - ', '-') if lang_audio else ''
        if lang_audio:
            feats.append(lang_audio)

        # 2) HDR/DV
        if re.search(r'\bhdr10\b', blob):
            feats.append('HDR10')
        elif re.search(r'\bhdr\b', blob):
            feats.append('HDR')
        if re.search(r'\b(dv|dolby\s*vision)\b', blob):
            feats.append('DV')

        # 3) kodek
        if re.search(r'\b(hevc|x265)\b', blob):
            feats.append('HEVC')
        elif re.search(r'\b(x264|avc)\b', blob):
            feats.append('H.264')

        # 4) rozlišení
        if re.search(r'\b(2160p|4k)\b', blob):
            feats.append('2160p')
        elif re.search(r'\b1080p\b', blob):
            feats.append('1080p')
        elif re.search(r'\b720p\b', blob):
            feats.append('720p')

        # unikátní pořadí zachovat
        uniq = []
        for f in feats:
            if f not in uniq:
                uniq.append(f)
        return uniq

    # --- public helper: smazání seriálu --------------------------------------
    def remove_series(self, series_name):
        """
        Odstraní JSON soubor seriálu (tmdb_<id>.json i slug.json).
        Vrací True, pokud byl alespoň jeden soubor smazán.
        """
        try:
            safe_name = self._safe_filename(series_name)
            fp_slug = os.path.join(self.series_db_path, f"{safe_name}.json")
            # Pokus s tmdb_<id>
            tmdb_id = self._get_tmdb_id(series_name)
            fp_tmdb = os.path.join(self.series_db_path, f"tmdb_{tmdb_id}.json") if tmdb_id else None

            removed = False
            if fp_tmdb and os.path.exists(fp_tmdb):
                os.remove(fp_tmdb)
                removed = True
            if os.path.exists(fp_slug):
                os.remove(fp_slug)
                removed = True
            return removed
        except Exception as e:
            xbmc.log(f'YaWSP Series Manager: remove_series error: {e}', xbmc.LOGERROR)
            return False


# === Utility functions for the UI layer =======================================
import unicodedata
def _slugify(text: str) -> str:
    """Normalizuje text: lowercase, bez diakritiky, bez mezer a speciálních znaků."""
    if not text:
        return ''
    nfkd = unicodedata.normalize('NFKD', text)
    return ''.join(c for c in nfkd if c.isalnum()).lower()


def filter_invalid_seasons(series_data: dict, buckets: dict) -> dict:
    """
    Odfiltruje z buckets ty položky, které neodpovídají TMDb rozsahu sezón/epizod.
    - Zahodí sezóny > number_of_seasons.
    - Zahodí epizody, které nejsou v TMDb sezóně.
    - Zachová pořadí streamů (neřadí).
    """
    try:
        from resources.lib import tmdb_utils as TM
    except Exception:
        xbmc.log("[SeriesManager] TMDb modul nedostupný, filtr přeskočen", xbmc.LOGWARNING)
        return buckets  # ← vrátit původní buckets

    tmdb_id = (series_data.get('tmdb') or {}).get('id')
    if not tmdb_id:
        xbmc.log("[SeriesManager] Chybí TMDb ID, filtr přeskočen", xbmc.LOGWARNING)
        return buckets  # ← vrátit původní buckets

    show_meta = TM.get_tv_show_meta(int(tmdb_id), language='cs') or {}
    num_seasons = show_meta.get('number_of_seasons', 0)
    tmdb_year = show_meta.get('year')
    slug_title = re.sub(r'[^a-z0-9]', '', (show_meta.get('title') or '').lower())

    if not isinstance(num_seasons, int) or num_seasons <= 0:
        xbmc.log("[SeriesManager] TMDb nevrátil validní počet sezón, filtr přeskočen", xbmc.LOGWARNING)
        return buckets  # ← vrátit původní buckets

    xbmc.log(f"[SeriesManager] TMDb filtr start: TMDb ID={tmdb_id}, slug='{slug_title}', seasons={num_seasons}", xbmc.LOGINFO)

    valid_buckets = {}
    for (s, e), streams in buckets.items():
        try:
            if s > num_seasons:
                xbmc.log(f"[SeriesManager] ❌ Sezóna {s} mimo rozsah TMDb ({num_seasons})", xbmc.LOGDEBUG)
                continue

            season_meta = TM.get_tv_season_meta(int(tmdb_id), s, language='cs') or {}
            episodes_meta = season_meta.get('episodes') or []
            ep_numbers = {ep.get('episode_number') for ep in episodes_meta if isinstance(ep.get('episode_number'), int)}
            max_ep = max(ep_numbers) if ep_numbers else None

            if e not in ep_numbers:
                xbmc.log(f"[SeriesManager] ❌ Epizoda S{s:02d}E{e:02d} v TMDb sezóně neexistuje → skip", xbmc.LOGDEBUG)
                continue

            filtered_streams = []
            for st in streams:
                name = st.get('name') or ''
                name_slug = re.sub(r'[^a-z0-9]', '', name.lower())
                if slug_title and slug_title not in name_slug:
                    xbmc.log(f"[SeriesManager] ❌ Slug mismatch: {name}", xbmc.LOGDEBUG)
                    continue
                if max_ep and e > max_ep:
                    xbmc.log(f"[SeriesManager] ❌ Epizoda {e} > max {max_ep}: {name}", xbmc.LOGDEBUG)
                    continue
                year_match = re.search(r'\b(19|20)\d{2}\b', name)
                if year_match and tmdb_year and int(year_match.group(0)) != int(tmdb_year):
                    xbmc.log(f"[SeriesManager] ❌ Rok mismatch {year_match.group(0)} vs {tmdb_year}: {name}", xbmc.LOGDEBUG)
                    continue
                filtered_streams.append(st)

            if filtered_streams:
                valid_buckets[(s, e)] = filtered_streams
        except Exception as ex:
            xbmc.log(f"[SeriesManager] Filtr: chyba při validaci S{s}E{e}: {ex}", xbmc.LOGWARNING)
            # při chybě v jedné položce pokračuj dál

    xbmc.log(f"[SeriesManager] TMDb filtr dokončen: {len(valid_buckets)} validních položek z původních {len(buckets)}", xbmc.LOGINFO)
    # Pokud užitečné položky nezbyly, vrať prázdno; ale ne v "no TMDb" větvi.
    return valid_buckets


def get_url(**kwargs):
    """Create a URL for calling the plugin recursively"""
    from yawsp import _url
    return '{0}?{1}'.format(_url, urlencode(kwargs, encoding='utf-8'))


def create_series_menu(series_manager, handle):
    """
    Menu seriálů:
    - Label: Název seriálu (+ offline souhrn, pokud existují stažené epizody)
    - Artwork: poster + fanart (+ offline ikona disku)
    - Kontext: Aktualizovat, Odebrat, Stáhnout celý seriál, Trakt…
    """
    import xbmcplugin
    import xbmcvfs
    import xbmcaddon

    ADDON = xbmcaddon.Addon()

    # SPRÁVNÉ ZÍSKÁNÍ A PŘÍPRAVA DFOLDER (funguje na SMB/NFS/Android/CoreELEC)
    dfolder = ADDON.getSetting('dfolder') or ''
    base_real = None
    if dfolder:
        base_real = xbmcvfs.translatePath(dfolder)
        if base_real and not base_real.endswith(('/', '\\')):
            base_real += '/'          # ← důležité!

    def _cz_plural_episodes(n: int) -> str:
        if n == 1:
            return "1 epizoda"
        if 2 <= n <= 4:
            return f"{n} epizody"
        return f"{n} epizod"

    def _count_offline_for_series(series_name: str, series_data: dict) -> int:
        """
        Spočítá celkový počet offline epizod napříč všemi sezónami daného seriálu.
        Používá path_exists + listdir_safe → funguje i na síťových discích.
        """
        if not base_real or not series_data or not isinstance(series_data, dict):
            return 0

        total_found = 0
        seasons = series_data.get('seasons', {}) or {}

        for season_key, season_dict in seasons.items():
            # SPRÁVNÉ SLOŽENÍ CESTY BEZ os.path.join → funguje i pro smb://
            season_folder = f"{base_real}{clean_filename(series_name)}/Sezona {season_key}/"

            if not path_exists(season_folder):
                continue

            files = listdir_safe(season_folder)
            if not files:
                continue

            expected_bases = set()
            for ep_key, ep in (season_dict.items() if isinstance(season_dict, dict) else []):
                ep_title = ep.get('name') or ep.get('title') or ep.get('source_name') or ''
                if ep_title:
                    expected_bases.add(clean_filename(ep_title).lower())
                try:
                    expected_bases.add(f"s{int(season_key):02d}e{int(ep_key):02d}".lower())
                except Exception:
                    expected_bases.add(f"s{season_key}e{ep_key}".lower())

            found = set()
            for fname in files:
                base_noext = os.path.splitext(fname)[0].lower()
                for b in expected_bases:
                    if base_noext == b or base_noext.startswith(b):
                        found.add(b)
                        break
            total_found += len(found)

        return total_found

    # Výpis všech uložených seriálů
    series_list = series_manager.get_all_series()
    for s in series_list:
        series_name = s['name']
        data = series_manager.load_series_data(series_name) or {}
        tm = data.get('tmdb', {}) if isinstance(data, dict) else {}

        # Rok (TMDb nebo z první epizody)
        year = tm.get('year')
        if not year:
            try:
                seasons = (data.get('seasons') or {}) if isinstance(data, dict) else {}
                s1 = seasons.get('1', {})
                e1 = s1.get('1', {})
                air = (e1.get('tmdb') or {}).get('air_date') or (e1.get('air_date') if isinstance(e1, dict) else None)
                if air and len(air) >= 4:
                    year = int(air[:4])
            except Exception:
                year = None

        # Počty sezón a epizod
        seasons = (data.get('seasons') or {}) if isinstance(data, dict) else {}
        try:
            seasons_count = len(seasons)
            episodes_count = sum(len(v) for v in seasons.values() if isinstance(v, dict))
        except Exception:
            seasons_count = episodes_count = 0

        # OFFLINE souhrn přes všechny sezóny
        offline_count = _count_offline_for_series(series_name, data)
        offline_tag = f" [COLOR green]offline: {_cz_plural_episodes(offline_count)}[/COLOR]" if offline_count > 0 else ""
        offline_thumb = 'DefaultHardDisk.png' if offline_count > 0 else None

        # Finální label
        label = data.get('name') or series_name
        if year:
            label += f" ({year})"
        label += f" [COLOR blue]{seasons_count} sez. / {episodes_count} ep.[/COLOR]"
        label += offline_tag

        # Artwork + plot
        poster_url = tm.get('poster_url')
        fanart_url = tm.get('backdrop_url')
        plot = tm.get('overview') or ''

        li = xbmcgui.ListItem(label=label)
        art = {'icon': 'DefaultFolder.png'}
        if poster_url:
            art['poster'] = poster_url
        if fanart_url:
            art['fanart'] = fanart_url
        if offline_thumb:
            art['thumb'] = offline_thumb
        li.setArt(art)
        if plot:
            li.setInfo('video', {'plot': plot})

        # Kontextové menu
        cm = []
        cm.append(("Aktualizovat seriál", f"RunPlugin({get_url(action='series_refresh', series_name=series_name)})"))
        cm.append(("Aktualizovat (bez 4K)", f"RunPlugin({get_url(action='series_refresh', series_name=series_name, skip_high_quality='true')})"))
        cm.append(("Odebrat seriál", f"RunPlugin({get_url(action='series_remove', series_name=series_name)})"))
        cm.append(("ČSFD detail (rozšířený)", f"RunPlugin({get_url(action='csfd_lookup_enhanced', series_name=series_name)})"))
        cm.append(("Stáhnout všechny sezóny", f"RunPlugin({get_url(action='series_download_show', series_name=series_name)})"))
        if tm.get('id'):
            cm.append(("Odeslat celý seriál na Trakt…",
                       f"RunPlugin({get_url(action='series_trakt_mark_show_prompt', tmdb_id=int(tm.get('id')))})"))
        cm.append(("Přidat do oblíbených", "Action(AddToFavorites)"))
        li.addContextMenuItems(cm, replaceItems=False)

        url = get_url(action='series_detail', series_name=series_name)
        xbmcplugin.addDirectoryItem(handle, url, li, isFolder=True)

    # Konec adresáře
    xbmcplugin.setPluginCategory(handle, "Seriály")
    xbmcplugin.setContent(handle, "tvshows")
    xbmcplugin.endOfDirectory(handle)

def is_series_completed(series_data: dict) -> bool:
    """
    Vrátí True, pokud všechny sezóny v season_state mají watched==True
    a total == watched_count (pokud jsou hodnoty k dispozici).
    Pokud season_state chybí, považujeme seriál za NEdosledovaný.
    """
    try:
        if not isinstance(series_data, dict):
            return False
        seasons = series_data.get('seasons') or {}
        season_state = series_data.get('season_state') or {}
        # Pokud nejsou žádné sezóny, neoznačujeme jako dosledované
        if not seasons:
            return False
        for s_key in seasons.keys():
            st = season_state.get(str(s_key)) or {}
            watched = bool(st.get('watched', False))
            total = st.get('total')
            watched_count = st.get('watched_count')
            # musí být watched True
            if not watched:
                return False
            # pokud máme čísla, musí sedět
            if isinstance(total, int) and isinstance(watched_count, int):
                if watched_count != total:
                    return False
        return True
    except Exception:
        return False


def create_series_menu_filtered(series_manager, handle, status: str):
    """
    Filtruje seriály podle stavu sledování:
    - status='completed'    → plně dosledované
    - status='in_progress'  → rozkoukané (nedosledované)

    Vše ostatní (vzhled, offline tagy, kontextové menu) je stejné jako v create_series_menu.
    """
    import xbmcplugin
    import xbmcgui
    import xbmcaddon
    import xbmcvfs

    ADDON = xbmcaddon.Addon()

    # SPRÁVNÉ ZÍSKÁNÍ DFOLDER – funguje i na síťových discích
    dfolder = ADDON.getSetting('dfolder') or ''
    base_real = None
    if dfolder:
        base_real = xbmcvfs.translatePath(dfolder)
        if base_real and not base_real.endswith(('/', '\\')):
            base_real += '/'          # ← důležité pro bezpečné složení cesty

    # České skloňování epizod
    def _cz_plural_episodes(n: int) -> str:
        if n == 1:
            return "1 epizoda"
        if 2 <= n <= 4:
            return f"{n} epizody"
        return f"{n} epizod"

    # Počítání offline epizod napříč všemi sezónami (funguje na SMB/NFS)
    def _count_offline_for_series(series_name: str, series_data: dict) -> int:
        if not base_real or not series_data or not isinstance(series_data, dict):
            return 0

        total_found = 0
        seasons = series_data.get('seasons', {}) or {}

        for season_key, season_dict in seasons.items():
            # SPRÁVNÉ SLOŽENÍ CESTY – BEZ os.path.join → funguje i na smb://
            season_folder = f"{base_real}{clean_filename(series_name)}/Sezona {season_key}/"

            if not path_exists(season_folder):
                continue

            files = listdir_safe(season_folder)
            if not files:
                continue

            expected_bases = set()
            for ep_key, ep in (season_dict.items() if isinstance(season_dict, dict) else []):
                ep_title = ep.get('name') or ep.get('title') or ep.get('source_name') or ''
                if ep_title:
                    expected_bases.add(clean_filename(ep_title).lower())
                try:
                    expected_bases.add(f"s{int(season_key):02d}e{int(ep_key):02d}".lower())
                except Exception:
                    expected_bases.add(f"s{season_key}e{ep_key}".lower())

            found = set()
            for fname in files:
                base_noext = os.path.splitext(fname)[0].lower()
                for b in expected_bases:
                    if base_noext == b or base_noext.startswith(b):
                        found.add(b)
                        break
            total_found += len(found)

        return total_found

    # Filtrace seriálů podle stavu sledování
    series_list = series_manager.get_all_series()
    filtered_series = []
    for s in series_list:
        series_name = s['name']
        data = series_manager.load_series_data(series_name) or {}
        completed = is_series_completed(data)  # předpokládám, že tato funkce existuje

        if (status == 'completed' and completed) or (status == 'in_progress' and not completed):
            filtered_series.append((series_name, data))

    # Titulek kategorie
    title = "Seriály · Dosledované" if status == 'completed' else "Seriály · Rozkoukané"
    xbmcplugin.setPluginCategory(handle, title)

    # Vykreslení filtrovaných seriálů
    for series_name, data in filtered_series:
        tm = data.get('tmdb', {}) if isinstance(data, dict) else {}

        # Rok (TMDb nebo z první epizody)
        year = tm.get('year')
        if not year:
            try:
                seasons = (data.get('seasons') or {}) if isinstance(data, dict) else {}
                s1 = seasons.get('1', {})
                e1 = s1.get('1', {})
                air = (e1.get('tmdb') or {}).get('air_date') or (e1.get('air_date') if isinstance(e1, dict) else None)
                if air and len(air) >= 4:
                    year = int(air[:4])
            except Exception:
                year = None

        # Počty sezón a epizod
        seasons = (data.get('seasons') or {}) if isinstance(data, dict) else {}
        try:
            seasons_count = len(seasons)
            episodes_count = sum(len(v) for v in seasons.values() if isinstance(v, dict))
        except Exception:
            seasons_count = episodes_count = 0

        # Offline souhrn
        offline_count = _count_offline_for_series(series_name, data)
        offline_tag = f" [COLOR green]offline: {_cz_plural_episodes(offline_count)}[/COLOR]" if offline_count > 0 else ""
        offline_thumb = 'DefaultHardDisk.png' if offline_count > 0 else None

        # Finální label
        label = data.get('name') or series_name
        if year:
            label += f" ({year})"
        label += f" [COLOR blue]{seasons_count} sez. / {episodes_count} ep.[/COLOR]"
        label += offline_tag

        # Artwork + plot
        poster_url = tm.get('poster_url')
        fanart_url = tm.get('backdrop_url')
        plot = tm.get('overview') or ''

        li = xbmcgui.ListItem(label=label)
        art = {'icon': 'DefaultFolder.png'}
        if poster_url:
            art['poster'] = poster_url
        if fanart_url:
            art['fanart'] = fanart_url
        if offline_thumb:
            art['thumb'] = offline_thumb
        li.setArt(art)
        if plot:
            li.setInfo('video', {'plot': plot})

        # Kontextové menu – přidáváme parametr status, aby se po smazání vrátilo do stejného filtru
        cm = []
        cm.append(("Aktualizovat seriál", f"RunPlugin({get_url(action='series_refresh', series_name=series_name)})"))
        cm.append(("Odebrat seriál", f"RunPlugin({get_url(action='series_remove', series_name=series_name, status=status)})"))
        cm.append(("ČSFD detail (rozšířený)", f"RunPlugin({get_url(action='csfd_lookup_enhanced', series_name=series_name)})"))
        cm.append(("Stáhnout všechny sezóny", f"RunPlugin({get_url(action='series_download_show', series_name=series_name)})"))
        if tm.get('id'):
            cm.append(("Odeslat celý seriál na Trakt…",
                       f"RunPlugin({get_url(action='series_trakt_mark_show_prompt', tmdb_id=int(tm.get('id')))})"))
        cm.append(("Přidat do oblíbených", "Action(AddToFavorites)"))
        li.addContextMenuItems(cm, replaceItems=False)

        url = get_url(action='series_detail', series_name=series_name)
        xbmcplugin.addDirectoryItem(handle, url, li, isFolder=True)

    # Nastavení zobrazení
    xbmcplugin.setContent(handle, 'tvshows')
    xbmc.executebuiltin('Container.SetViewMode(50)')
    xbmcplugin.endOfDirectory(handle)



def create_seasons_menu(series_manager, handle, series_name, build_url_fn=None):
    """
    Menu sezón pro daný seriál:
    - Label: Sezona X [watched_count/total] (+ ok pokud vše zhlédnuto)
    - Offline indikace: [COLOR green]offline[/COLOR] + počet epizod + ikona disku
    - Artwork: poster + fanart
    - Kontext: Aktualizovat, Stáhnout sezónu, Trakt, atd.
    """
    import xbmcplugin
    import xbmcvfs
    import xbmcaddon
    import xbmcgui

    # Builder URL s fallbackem
    builder = build_url_fn or _default_build_url

    ADDON = xbmcaddon.Addon()
    color_tags = ADDON.getSetting('series_progress_color_tags') == 'true'

    # SPRÁVNÉ ZÍSKÁNÍ DFOLDER – funguje i na NASu
    dfolder = ADDON.getSetting('dfolder') or ''
    base_real = None
    if dfolder:
        base_real = xbmcvfs.translatePath(dfolder)
        if base_real and not base_real.endswith(('/', '\\')):
            base_real += '/'  # ← klíčové!

    def _cz_plural_episodes(n: int) -> str:
        if n == 1:
            return "1 epizoda"
        if 2 <= n <= 4:
            return f"{n} epizody"
        return f"{n} epizod"

    def _count_offline_eps_for_season(season_folder: str, season_dict: dict, season_num: int) -> int:
        """Spočítá offline epizody v dané sezóně – funguje na SMB/NFS."""
        if not season_folder or not path_exists(season_folder):
            return 0
        try:
            files = listdir_safe(season_folder)
        except Exception:
            return 0
        if not files:
            return 0

        expected_bases = set()
        for ep_key, ep in season_dict.items():
            ep_title = ep.get('name') or ep.get('title') or ep.get('source_name') or ''
            if ep_title:
                expected_bases.add(clean_filename(ep_title).lower())
            try:
                expected_bases.add(f"s{int(season_num):02d}e{int(ep_key):02d}".lower())
            except Exception:
                expected_bases.add(f"s{season_num}e{ep_key}".lower())

        found = set()
        for fname in files:
            base_noext = os.path.splitext(fname)[0].lower()
            for b in expected_bases:
                if base_noext == b or base_noext.startswith(b):
                    found.add(b)
                    break
        return len(found)

    # Načtení dat seriálu
    series_data = series_manager.load_series_data(series_name)
    if not series_data:
        xbmcgui.Dialog().notification('YaWSP', 'Data seriálu nenalezena', xbmcgui.NOTIFICATION_WARNING)
        xbmcplugin.endOfDirectory(handle, succeeded=False)
        return

    tm = series_data.get('tmdb', {}) if isinstance(series_data, dict) else {}
    poster_url = tm.get('poster_url')
    fanart_url = tm.get('backdrop_url')
    plot = tm.get('overview') or ''

    # Hlavička – název seriálu
    li_info = xbmcgui.ListItem(label=series_name)
    art_info = {'icon': 'DefaultFolder.png'}
    if poster_url:
        art_info['poster'] = poster_url
    if fanart_url:
        art_info['fanart'] = fanart_url
    li_info.setArt(art_info)
    if plot:
        li_info.setInfo('video', {'plot': plot})
    cm_info = [
        ("Aktualizovat seriál", f"RunPlugin({builder(action='series_refresh', series_name=series_name)})"),
        ("Odebrat seriál", f"RunPlugin({builder(action='series_remove', series_name=series_name)})"),
    ]
    li_info.addContextMenuItems(cm_info, replaceItems=False)
    xbmcplugin.addDirectoryItem(handle, builder(action='noop'), li_info, False)

    # Explicitní tlačítka
    for label, action, icon in [
        ("Aktualizovat seriál", 'series_refresh', 'DefaultAddonsSearch.png'),
        ("Aktualizovat s výběrem kvality", 'series_refresh_select', 'DefaultAddonsSearch.png'),
        ("Ověřit dostupnost streamů", 'series_validate_streams', 'DefaultAddonService.png'),
        ("Odebrat seriál", 'series_remove', 'DefaultIconError.png'),
    ]:
        li = xbmcgui.ListItem(label=label)
        li.setArt({'icon': icon, 'poster': poster_url or '', 'fanart': fanart_url or ''})
        xbmcplugin.addDirectoryItem(handle, builder(action=action, series_name=series_name),
                                    li, action != 'series_validate_streams')

    # Výpis sezón
    for season_key in sorted(series_data.get('seasons', {}).keys(), key=int):
        season_data = series_data['seasons'][season_key]
        season_state = series_data.get('season_state', {}).get(season_key, {})
        watched_count = season_state.get('watched_count', 0)
        total = season_state.get('total', len(season_data))
        watched_flag = season_state.get('watched', False)
        suffix = f"[{watched_count}/{total}]"
        if watched_flag:
            suffix += " " + ("[COLOR lime]ok[/COLOR]" if color_tags else "[ok]")

        # OFFLINE detekce – SPRÁVNĚ složená cesta!
        offline_tag = ""
        offline_thumb = None
        offline_count = 0
        if base_real:
            season_folder = f"{base_real}{clean_filename(series_name)}/Sezona {season_key}/"
            if path_exists(season_folder):
                offline_count = _count_offline_eps_for_season(season_folder, season_data, int(season_key))
                if offline_count > 0:
                    offline_tag = f" [COLOR green]offline[/COLOR] {_cz_plural_episodes(offline_count)}"
                    offline_thumb = 'DefaultHardDisk.png'

        label = f"Sezona {season_key} {suffix}{offline_tag}"
        li = xbmcgui.ListItem(label=label)
        art_season = {'icon': 'DefaultFolder.png'}
        if poster_url:
            art_season['poster'] = poster_url
        if fanart_url:
            art_season['fanart'] = fanart_url
        if offline_thumb:
            art_season['thumb'] = offline_thumb
        li.setArt(art_season)
        if plot:
            li.setInfo('video', {'plot': plot})

        # Kontextové menu sezóny
        cm_season = [
            ("Aktualizovat seriál", f"RunPlugin({builder(action='series_refresh', series_name=series_name)})"),
            ("Odebrat seriál", f"RunPlugin({builder(action='series_remove', series_name=series_name)})"),
            ("Přidat sezónu do fronty",
             f"RunPlugin({builder(action='series_download_season', series_name=series_name, season=season_key)})"),
            ("Spustit frontu (na pozadí)", f"RunPlugin({builder(action='queue_start_all')})"),
            ("Označit sezónu jako zhlédnutou",
             f"RunPlugin({builder(action='series_mark_season', series_name=series_name, tmdb_id=tm.get('id'), season=season_key, progress=100)})"),
            ("Vymazat stav sezóny",
             f"RunPlugin({builder(action='series_mark_season', series_name=series_name, season=season_key, progress=0)})"),
            ("Smazat sezónu",
             f"RunPlugin({builder(action='series_remove_season', series_name=series_name, season=season_key)})"),
        ]
        if tm.get('id'):
            try:
                cm_season.append(("Odeslat SEZÓNU na Trakt…",
                                  f"RunPlugin({builder(action='series_trakt_mark_season_prompt', tmdb_id=int(tm.get('id')), season=int(season_key))})"))
            except Exception:
                cm_season.append(("Odeslat SEZÓNU na Trakt…",
                                  f"RunPlugin({builder(action='series_trakt_mark_season_prompt', tmdb_id=tm.get('id'), season=season_key)})"))
        li.addContextMenuItems(cm_season, replaceItems=False)

        url = builder(action='series_season', series_name=series_name, season=season_key)
        xbmcplugin.addDirectoryItem(handle, url, li, isFolder=True)

    # Nastavení zobrazení
    xbmcplugin.setPluginCategory(handle, f"Sezóny: {series_name}")
    xbmcplugin.setContent(handle, "seasons")
    xbmc.executebuiltin("Container.SetViewMode(50)")
    xbmcplugin.endOfDirectory(handle)



def create_episodes_menu(series_manager, handle, series_name, season_num, build_url_fn=None):
    """
    Menu epizod pro danou sezónu:
    - Pokud je epizoda STAŽENÁ → přehrává se rovnou z disku (IsPlayable=true)
    - Pokud NENÍ stažená → otevře se klasický výběr streamu z Webshare
    """
    import xbmcplugin
    import xbmcvfs
    import xbmcgui
    import xbmcaddon

    # Builder URL s fallbackem
    builder = build_url_fn or _default_build_url

    ADDON = xbmcaddon.Addon()
    color_tags = ADDON.getSetting('series_progress_color_tags') == 'true'

    # Načtení dat seriálu
    series_data = series_manager.load_series_data(series_name)
    if not series_data or str(season_num) not in series_data.get('seasons', {}):
        xbmcgui.Dialog().notification('YaWSP', 'Data sezóny nenalezena', xbmcgui.NOTIFICATION_WARNING)
        xbmcplugin.endOfDirectory(handle, succeeded=False)
        return

    tm_series = series_data.get('tmdb', {}) if isinstance(series_data, dict) else {}
    series_poster = tm_series.get('poster_url')
    series_fanart = tm_series.get('backdrop_url')
    series_overview = tm_series.get('overview') or ''
    series_year = tm_series.get('year')
    show_runtime = tm_series.get('show_runtime')

    season_key = str(season_num)
    season = series_data['seasons'][season_key]

    # SPRÁVNÁ CESTA K OFFLINE SLOŽCE – funguje na SMB/NFS/Android
    dfolder = ADDON.getSetting('dfolder') or ''
    base_real = None
    if dfolder:
        base = xbmcvfs.translatePath(dfolder)
        if base and not base.endswith(('/', '\\')):
            base += '/'
        base_real = base

    season_folder = f"{base_real}{clean_filename(series_name)}/Sezona {season_key}/" if base_real else None

    # Procházení všech epizod
    for episode_num in sorted(season.keys(), key=int):
        ep = season[episode_num]
        tm_ep = ep.get('tmdb', {}) if isinstance(ep, dict) else {}
        ep_title = ep.get('name') or ep.get('title') or ep.get('source_name') or ''
        ep_plot = tm_ep.get('overview') or series_overview
        ep_still = tm_ep.get('still_url')
        ep_air = tm_ep.get('air_date')
        ep_runtime = tm_ep.get('runtime') or show_runtime

        # Stav sledování (OK / 75% / New)
        suffix = series_manager._label_suffix_for_episode(ep, color_tags)

        # OFFLINE DETEKCE + NALEZENÍ SKUTEČNÉHO SOUBORU
        local_file_path = None
        is_offline = False
        if season_folder and path_exists(season_folder):
            try:
                files = listdir_safe(season_folder)
                patterns = []
                if ep_title:
                    patterns.append(clean_filename(ep_title).lower())  # SPRÁVNĚ!
                patterns.append(f"s{int(season_num):02d}e{int(episode_num):02d}".lower())
                for f in files:
                    name_noext = os.path.splitext(f)[0].lower()
                    if any(name_noext == p or name_noext.startswith(p) for p in patterns):
                        local_file_path = season_folder + f
                        is_offline = True
                        break
            except Exception as e:
                xbmc.log(f"[EpisodesMenu] Offline detekce selhala: {e}", xbmc.LOGDEBUG)

        # Finální label s offline indikací
        offline_tag = " [COLOR limegreen]offline[/COLOR]" if is_offline else ""
        label = f"S{int(season_num):02d}E{int(episode_num):02d} - {ep_title} {suffix}{offline_tag}"
        li = xbmcgui.ListItem(label=label)

        # Artwork
        art = {}
        if ep_still:
            art.update({'thumb': ep_still, 'icon': ep_still})
        else:
            art['icon'] = 'DefaultVideo.png'
        if is_offline:
            art['thumb'] = 'DefaultHardDisk.png'
        if series_fanart:
            art['fanart'] = series_fanart
        if series_poster:
            art['poster'] = series_poster
        li.setArt(art)

        # InfoLabels
        info = {
            'title': ep_title or f"Epizoda {episode_num}",
            'plot': ep_plot,
            'aired': ep_air or '',
            'season': int(season_num),
            'episode': int(episode_num),
            'mediatype': 'episode'
        }
        if isinstance(ep_runtime, int) and ep_runtime > 0:
            info['duration'] = ep_runtime
        if isinstance(series_year, int):
            info['year'] = series_year
        li.setInfo('video', info)

        # Kontextové menu
        cm = [
            ("Označit jako zhlédnuté",
             f"RunPlugin({builder(action='series_mark_episode', series_name=series_name, season=season_num, episode=episode_num, progress=100)})"),
            ("Označit jako rozkoukané",
             f"RunPlugin({builder(action='series_mark_episode', series_name=series_name, season=season_num, episode=episode_num, progress=75)})"),
            ("Vymazat stav",
             f"RunPlugin({builder(action='series_mark_episode', series_name=series_name, season=season_num, episode=episode_num, progress=0)})"),
            ("Smazat epizodu",
             f"RunPlugin({builder(action='series_remove_episode', series_name=series_name, season=season_num, episode=episode_num)})"),
            ("Přidat do fronty",
             f"RunPlugin({builder(action='series_download_episode', series_name=series_name, season=season_num, episode=episode_num)});Container.Refresh"),
            ("Spustit frontu",
             f"RunPlugin({builder(action='queue_start_all')})"),
        ]
        if tm_series.get('id'):
            cm.append(
                ("Odeslat na Trakt…",
                 f"RunPlugin({builder(action='series_trakt_mark_episode_prompt', tmdb_id=int(tm_series.get('id')), season=int(season_num), episode=int(episode_num))})")
            )
        li.addContextMenuItems(cm, replaceItems=True)

        # HLAVNÍ LOGIKA: offline → přehrává rovnou, online → výběr streamu
        if is_offline and local_file_path:
            li.setProperty('IsPlayable', 'true')
            xbmcplugin.addDirectoryItem(handle, local_file_path, li, isFolder=False)
        else:
            li.setProperty('IsPlayable', 'false')
            url = builder(
                action='play',
                series_name=series_name,
                season=str(season_num),
                episode=str(episode_num),
                select='1'
            )
            xbmcplugin.addDirectoryItem(handle, url, li, isFolder=False)

    # Nastavení zobrazení
    xbmcplugin.setPluginCategory(handle, f"Epizody: Sezóna {season_num}")
    xbmcplugin.setContent(handle, "tvshows")
    xbmc.executebuiltin("Container.SetViewMode(50)")  # Wall view – nejlepší pro epizody
    xbmcplugin.endOfDirectory(handle)

# --- Nové pomocné API pro yawsp.py -------------------------------------------
def _format_stream_label(st):
    """Vytvoří label pro dialog: pouze podstatné vlastnosti (bez velikosti)."""
    feats = st.get('features')
    if isinstance(feats, list) and feats:
        return ', '.join(feats)
    # Fallback: spočítej jednoduše na místě
    try:
        name = st.get('name') or ''
        q    = st.get('quality') or ''
        ai   = st.get('ainfo') or ''
        lang = st.get('lang') or ''
        blob = ' '.join([name, q, ai, lang])
        tmp  = {'name': blob, 'quality': q, 'ainfo': ai, 'lang': lang}
        feats2 = SeriesManager._extract_features(tmp)
        return ', '.join(feats2) if feats2 else (_norm(st.get('label')) or _norm(st.get('quality')) or _norm(st.get('name')) or 'Stream')
    except Exception:
        return _norm(st.get('label')) or _norm(st.get('quality')) or _norm(st.get('name')) or 'Stream'


def get_episode_streams(series_manager_obj, series_name: str, season: str, episode: str):
    """Vrátí kurátorované streamy pro danou epizodu (seřazené)."""
    data = series_manager_obj.load_series_data(series_name)
    if not data:
        return []
    s = str(season); e = str(episode)
    ep = data.get('seasons', {}).get(s, {}).get(e)
    if not ep:
        return []
    return ep.get('streams', [])


def format_stream_two_lines(st: dict, runtime: int = None) -> (str, str):
    """
    Vrátí dvojici (line1, line2) pro dvouřádkový dialog výběru streamu.
    - line1: plný název streamu (původní filename z Webshare)
    - line2: klíčové vlastnosti + velikost GB + runtime v minutách (pokud je znám)
    """
    # 1) Line1 = celé jméno streamu (filename)
    line1 = (st.get('name') or '').strip()

    # 2) Line2 = slož z features; pokud chybí, dopočti on-the-fly
    feats = st.get('features')
    if not isinstance(feats, list) or not feats:
        try:
            tmp = {
                'name': st.get('name') or '',
                'quality': st.get('quality') or '',
                'ainfo': st.get('ainfo') or '',
                'lang': st.get('lang') or ''
            }
            feats = SeriesManager._extract_features(tmp)
        except Exception:
            feats = []

    # Velikost v GB (na konec řádku)
    size_gb = ''
    try:
        n = float(st.get('size') or 0.0)
        size_gb = f"{n/(1024*1024*1024):.1f} GB" if n > 0 else ''
    except Exception:
        size_gb = ''

    parts = []
    parts.extend(feats)
    if size_gb:
        parts.append(size_gb)
    if isinstance(runtime, int) and runtime > 0:
        parts.append(f"{runtime} min")

    line2 = ' • '.join(p.strip() for p in parts if p and p.strip())

    # Fallbacky
    if not line1:
        line1 = 'Stream'
    if not line2:
        line2 = (st.get('quality') or st.get('ident') or '').strip() or '—'

    return (line1, line2)
