# -*- coding: utf-8 -*-
"""
UI utilitky pro Kodi doplněk (plugin.video.mmirousek):
- icons_path: základní cesta k ikonám v doplňku (včetně podsložek).
- pick_icon: výběr ikony podle názvu souboru s fallbackem na default Kodi.
- pick_icon_path: totéž, ale zadává se přímo addon_path (pro moduly bez Addon instance).
- add_menu_item: jednotné vytváření položek (label + plot + art + URL).

Použitelné z yawsp.py, tmdb_kodi.py i tvprogram.py.
"""

from __future__ import annotations
import os

try:
    from xbmcvfs import translatePath
except Exception:
    from xbmc import translatePath

import xbmcgui
import xbmcplugin
import xbmcvfs
import xbmc
import sys
from urllib.parse import quote, parse_qs, unquote
from typing import Optional, Tuple





def icons_path(addon, *subdirs) -> str:
    """Vrátí absolutní cestu do resources/media/icons[/subdirs...]."""
    addon_path = translatePath(addon.getAddonInfo('path'))
    parts = [addon_path, 'resources', 'media', 'icons'] + list(subdirs)
    return os.path.join(*parts)


def pick_icon(addon, file_name: str | None, kodi_fallback: str = 'DefaultFolder.png', *subdirs) -> str:
    """
    Vrátí plnou cestu k ikoně resources/media/icons[/subdirs]/<file_name>, pokud existuje.
    Jinak vrátí defaultní Kodi ikonu (kodi_fallback).
    Použití tam, kde je k dispozici objekt Addon.
    """
    if file_name:
        base = icons_path(addon, *subdirs)
        full = os.path.join(base, file_name)
        if os.path.exists(full):
            return full
    return kodi_fallback


def pick_icon_path(addon_path: str, file_name: str | None, kodi_fallback: str = 'DefaultFolder.png', *subdirs) -> str:
    """
    Stejné jako pick_icon(), ale bere přímo addon_path.
    Praktické v modulech, které mají jen cesty (např. tvprogram.py).
    """
    if file_name:
        base = os.path.join(addon_path, 'resources', 'media', 'icons', *subdirs)
        full = os.path.join(base, file_name)
        if os.path.exists(full):
            return full
    return kodi_fallback


# resources/lib/ui_utils.py

import xbmcaddon  # ← doplň, pokud ještě není



def make_static_item(label: str,
                     plot: str,
                     addon: xbmcaddon.Addon = None,
                     icon_name: str = None,
                     default_icon: str = 'DefaultFolder.png',
                     extra_art: dict = None,
                     info_labels: dict = None) -> xbmcgui.ListItem:
    """
    Vytvoří sjednocený ListItem pro statické menu položky:
      - label: titulek položky
      - plot: víceřádkový popis (použij \n pro zalomení)
      - icon_name: název souboru ikony v resources/media/icons (např. 'search.png');
                   pokud None → použije se default_icon
      - default_icon: Kodi fallback (např. 'DefaultAddonsSearch.png')
      - extra_art: volitelně doplní 'poster', 'fanart', 'thumb' apod. (dict)
      - info_labels: volitelný dict s dalšími Video infoLabels (title/genre/year…)
    """
    addon = addon or xbmcaddon.Addon()
    li = xbmcgui.ListItem(label=label)

    icon_file = (icon_name or '').strip()
    icon_final = pick_icon(addon, icon_file, default_icon) if icon_file else default_icon

    # Artwork – přidáme fanart, aby Estuary nepoužil fallback
    art = {
        'icon': icon_final,
        'thumb': icon_final,
        'poster': icon_final,
        'fanart': icon_final  # klíčové pro Estuary
    }
    if isinstance(extra_art, dict):
        for k, v in extra_art.items():
            if v:
                art[k] = v
    li.setArt(art)

    info = {'title': label, 'plot': (plot or '')}
    if isinstance(info_labels, dict):
        info.update({k: v for k, v in info_labels.items() if v is not None})
    li.setInfo('video', info)

    return li



def add_menu_item(
    handle: int,
    build_url_fn,
    label: str,
    plot: str,
    action: str,
    art_icon: str | None = None,
    is_folder: bool = True,
    **params
) -> None:
    """
    Vytvoří ListItem se sjednoceným plotem a artworkem a přidá jej do adresáře.
    - build_url_fn: funkce pro sestavení URL. Podporuje 2 signatury:
        a) build_url_fn(dict)           # např. tmdb_kodi.build_url(query)
        b) build_url_fn(**kwargs)       # např. yawsp.get_url(**kwargs)
    - action: hodnota parametru 'action' v URL.
    """
    li = xbmcgui.ListItem(label=label)
    li.setInfo('video', {'title': label, 'plot': plot})
    if art_icon:
        li.setArt({'icon': art_icon, 'thumb': art_icon})

    # Vytvoř parametrický dict pro URL
    url_params = {'action': action, **params}

    # Podpora obou signatur build_url_fn
    try:
        # 1) build_url_fn(dict)
        url = build_url_fn(url_params)
    except TypeError:
        # 2) build_url_fn(**kwargs)
        url = build_url_fn(**url_params)

    xbmcplugin.addDirectoryItem(handle, url, li, isFolder=is_folder)



# --- UI helpery (společné) ----------------------------------------------------

def build_trakt_context(get_url_fn,
                        title: str,
                        tmdb_id: int = None,
                        trakt_id: int = None,
                        slug: str = None,
                        media_type: str = 'movies',
                        context_type: str = 'default',
                        next_season_hint: int = None) -> list:
    """
    Sjednocené kontextové menu pro Trakt akce.
    - get_url_fn: funkce pro stavbu plugin URL (např. trakt_module.get_url)
    - title, tmdb_id, trakt_id, slug: identifikace položky
    - media_type: 'movies' nebo 'shows'
    - context_type:
        'default'   → běžné položky (Obsazení, Related, Mark watched, Add to Watchlist)
        'watchlist' → stejné + 'Odebrat z Watchlist'
        'progress'  → akce pro mazání progressu + prompt označení S/E
    - next_season_hint: číslo sezóny pro „rychlé“ označení (volitelné, používá se v progress listingu)
    """
    context = []
    ident = slug or (str(trakt_id) if trakt_id else '')

    # # Obsazení a štáb
    # if ident:
    #     ppl_url = get_url_fn(action='trakt_people', media_type=media_type, id=ident, title=title)
    #     context.append(('mm Obsazení a štáb (Trakt)', f'Container.Update({ppl_url})'))

    # # Podobné tituly
    # if ident:
    #     rel_url = get_url_fn(action='trakt_related', media_type=media_type, id=ident, page=1, limit=30)
    #     context.append(('mm Podobné (Related)', f'Container.Update({rel_url})'))
    
    # Obsazení a štáb (Trakt)
    if ident or tmdb_id:
        ppl_url = get_url_fn(
            action='trakt_people',
            media_type=media_type,
            id=ident if ident else tmdb_id,  # fallback na tmdb_id
            title=title
        )
        context.append(('mm Obsazení a štáb (Trakt)', f'Container.Update({ppl_url})'))

    # Podobné tituly (Related)
    if ident or tmdb_id:
        rel_url = get_url_fn(
            action='trakt_related',
            media_type=media_type,
            id=ident if ident else tmdb_id,  # fallback na tmdb_id
            page=1,
            limit=30
        )
        context.append(('mm Podobné (Related)', f'Container.Update({rel_url})'))


    # Označit jako zhlédnuté
    if tmdb_id and context_type in ('default', 'watchlist'):
        mark_action = 'trakt_mark_movie' if media_type == 'movies' else 'trakt_mark_show'
        mark_url = get_url_fn(action=mark_action, tmdb_id=tmdb_id)
        context.append(('mm Označit zhlédnuté (Trakt)', f'RunPlugin({mark_url})'))


    # Přidat do Watchlistu (jen u běžných listingů, když máme tmdb_id)
    if tmdb_id and context_type == 'default':
        add_url = get_url_fn(action='trakt_watchlist_add', tmdb_id=tmdb_id)
        context.append(('mm Add do Trakt Watchlist', f'RunPlugin({add_url})'))

    # Odebrat z Watchlistu (jen ve watchlistu) – bez podmínky na tmdb_id

    if context_type == 'watchlist':
        remove_url = get_url_fn(action='trakt_watchlist_remove', tmdb_id=(tmdb_id or ''))
        context.append(('mm Odebrat z Trakt Watchlist', f'RunPlugin({remove_url})'))

    # Specifické pro progress listing (seriélní kontext)
    if context_type == 'progress':
        # Mazání progressu
        if trakt_id:
            context.append(('mm Smazat progress seriálu na Trakt',
                            f'RunPlugin({get_url_fn(action="trakt_delete_history", kind="show", trakt_id=trakt_id)})'))
            context.append(('mm Smazat progress sezóny na Trakt…',
                            f'RunPlugin({get_url_fn(action="trakt_delete_history", kind="season", trakt_id=trakt_id)})'))
        # Označení seriálu / sezóny / epizody (prompty)
        if tmdb_id:
            context.append(('mm Označit SERIÁL jako zhlédnutý…',
                            f'RunPlugin({get_url_fn(action="series_trakt_mark_show_prompt", tmdb_id=int(tmdb_id))})'))
            context.append(('mm Označit SEZÓNU jako zhlédnutou…',
                            f'RunPlugin({get_url_fn(action="series_trakt_mark_season_prompt", tmdb_id=int(tmdb_id))})'))
            context.append(('mm Označit EPIZODU jako zhlédnutou…',
                            f'RunPlugin({get_url_fn(action="series_trakt_mark_episode_prompt", tmdb_id=int(tmdb_id))})'))
            # Quick: podle next_season_hint přidej předvyplněnou sezónu
            if isinstance(next_season_hint, int) and next_season_hint > 0:
                quick = get_url_fn(action='series_trakt_mark_season_prompt',
                                   tmdb_id=int(tmdb_id), season=int(next_season_hint))
                context.append((f'mm Označit SEZÓNU S{next_season_hint:02d} jako zhlédnutou (vybrat datum)',
                                f'RunPlugin({quick})'))
    return context



# ---------------------------------------------------------------------------
# File & Path Utilities (nové)
# ---------------------------------------------------------------------------
import re

def clean_filename(name: str) -> str:
    """
    Očistí název souboru od zakázaných znaků a trailing teček/whitespace.
    Používá se pro stahování filmů/epizod i offline detekci.
    """
    if not name:
        return "download"
    safe = re.sub(r'[\\\/:*?"<>|\n]', '_', str(name))
    return safe.rstrip('. ').strip() or "download"

# ---------------------------------------------------------------------------
# Download Control Utilities
# ---------------------------------------------------------------------------
import xbmcgui

def reset_cancel_flags():
    """Resetuje všechny globální cancel flagy (mm_cancel_download, mm_cancel_download_all)."""
    win = xbmcgui.Window(10000)
    win.setProperty('mm_cancel_download', '0')
    win.setProperty('mm_cancel_download_all', '0')

def set_cancel_flags():
    """Nastaví globální cancel flagy na TRUE."""
    win = xbmcgui.Window(10000)
    win.setProperty('mm_cancel_download', '1')
    win.setProperty('mm_cancel_download_all', '1')

def is_cancel_requested() -> bool:
    """Vrátí True, pokud je aktivní cancel pro transfer nebo all."""
    win = xbmcgui.Window(10000)
    return (win.getProperty('mm_cancel_download') == '1' or win.getProperty('mm_cancel_download_all') == '1')


# ---------------------------------------------------------------------------
# File System Utilities (nové)
# ---------------------------------------------------------------------------

def path_exists(p: str) -> bool:
    """
    Bezpečně ověří existenci cesty: preferuje Kodi VFS (xbmcvfs), fallback na OS.
    Funguje pro soubory i složky, včetně 'special://' cest.
    """
    if not p:
        return False
    try:
        # xbmcvfs.exists vyžaduje trailing '/' pro složky; pro univerzalitu zkusíme obě varianty
        if xbmcvfs.exists(p):
            return True
        if p.endswith('/'):
            return xbmcvfs.exists(p[:-1])
        else:
            return xbmcvfs.exists(p + '/')
    except Exception:
        pass
    try:
        return os.path.exists(p) or os.path.isdir(p)
    except Exception:
        return False


def listdir_safe(p: str) -> list:
    """
    Bezpečné vypsání souborů v cestě (jen jména položek).
    Preferuje xbmcvfs.listdir, fallback na os.listdir.
    Při chybě vrací [].
    """
    if not p:
        return []
    try:
        dirs, files = xbmcvfs.listdir(p)
        # sjednotíme návrat: zajímají nás soubory/jména; Kodi vrací separátně dirs vs files
        return list(files) + list(dirs)
    except Exception:
        pass
    try:
        return os.listdir(p)
    except Exception:
        return []
# --- ADD: Folder Utilities ---

def ensure_folder(path: str) -> None:
    """
    Bezpečně vytvoří adresář (přes real path z xbmcvfs.translatePath).
    1:1 náhrada původního _ensure_folder v search/yawsp.
    """
    if not path:
        return
    real = xbmcvfs.translatePath(path)
    try:
        os.makedirs(real, exist_ok=True)
    except Exception as e:
        xbmc.log(f"[ui_utils.ensure_folder] cannot create folder {real}: {e}", xbmc.LOGWARNING)


# --- ADD: Target Path Builder ---
from .ui_utils import clean_filename  # pokud je soubor samostatný, import není nutný; nebo použij přímo local name

def make_target_path(addon, name: str) -> str:
    """
    Sestaví cílový target_path z nastavení dfolder + clean_filename(name) + zachování přípony.
    1:1 náhrada původního _make_target_path (bez závislosti na yawsp globálech).
    """
    try:
        dfolder = addon.getSetting('dfolder') or ''
    except Exception:
        dfolder = ''
    if not dfolder:
        # Bez dfolderu nebudeme cílovou cestu přidávat – queue_worker by skončil na error.
        return ''
    dest_real = xbmcvfs.translatePath(dfolder)
    ensure_folder(dest_real)
    base = clean_filename(name or '')
    _, ext = os.path.splitext(name or '')
    filename = base if not ext else (base + ext)
    return os.path.join(dest_real, filename)


def prompt_and_filter_list(action: str, query_key: str = 'title_filter', dialog_heading: str = 'Zadejte počáteční písmena') -> None:
    """
    Zobrazí klávesnici pro zadání textu a aktualizuje kontejner (list)
    s novým parametrem filtru.

    Args:
        action (str): Název akce pro volání v routeru (např. 'show_progress_watching').
        query_key (str): Klíč parametru, pod kterým bude filtr uložen (např. 'title_filter').
        dialog_heading (str): Text, který se zobrazí na klávesnici.
    """
    keyboard = xbmc.Keyboard('', dialog_heading)
    keyboard.doModal()

    if keyboard.isConfirmed():
        filter_text = keyboard.getText().strip()
        
        # Sestavíme URL pro aktualizaci kontejneru
        # sys.argv[0] je vždy cesta k addon.py, get_url() by mělo být dostupné
        # (pokud je get_url definováno v jiném modulu, je třeba ho sem importovat/definovat)
        
        # Použijeme přímou URL manipulaci, pokud get_url není dostupné
        
        base_url = sys.argv[0]
        params = f'?action={action}'

        if filter_text:
            # Přejdeme na hlavní menu s filtrem
            params += f'&{query_key}={quote(filter_text)}'
            
        xbmc.executebuiltin('Container.Update(%s%s)' % (base_url, params))


def get_filter_param(url: str, query_key: str = 'title_filter') -> Optional[str]:
    """
    Parsuje URL a vrací hodnotu filtru podle klíče.

    Args:
        url (str): Celá URL kontejneru, včetně parametrů.
        query_key (str): Klíč parametru filtru (např. 'title_filter').
        
    Returns:
        str | None: Hodnota filtru, nebo None.
    """
    try:
        # Zpracování URL parametrů
        query_params = parse_qs(url.split('?')[-1])
        filter_list = query_params.get(query_key, [None])
        
        if filter_list[0]:
            return unquote(filter_list[0])
        return None
    except Exception:
        return None
