# -*- coding: utf-8 -*-
# resources/lib/trakt_watchlist.py

import sys
import os
import json
import requests
import xbmc
import xbmcgui
import xbmcaddon

ADDON_ID = 'plugin.video.mmirousek'
ADDON = xbmcaddon.Addon(ADDON_ID)

# -----------------------------------------------------------------------------
# DYNAMICKÁ ÚPRAVA sys.path PRO SPRÁVNÉ IMPORTY PŘI RunScript
# -----------------------------------------------------------------------------
# Potřebujeme, aby šel importovat balíček "resources" (tj. kořen addonu je v sys.path)
# a případně i přímé moduly z resources/lib.
try:
    from xbmcvfs import translatePath  # moderní Kodi
except Exception:
    from xbmc import translatePath      # fallback pro starší verze

addon_path = translatePath(ADDON.getAddonInfo('path'))  # .../addons/plugin.video.mmirousek
lib_path = os.path.join(addon_path, 'resources', 'lib')

# Přidej kořen doplňku i lib do sys.path (na začátek pro jistotu)
if addon_path and addon_path not in sys.path:
    sys.path.insert(0, addon_path)
if lib_path and lib_path not in sys.path:
    sys.path.insert(0, lib_path)

# Teď už můžeme bezpečně importovat cokoliv z resources.lib.*
from resources.lib.trakt.trakt_service import get_trakt_headers

def _get_bool_setting(key, default=False):
    try:
        return ADDON.getSettingBool(key)
    except Exception:
        return (ADDON.getSetting(key) or '').lower() == 'true'

# Volitelný debug (pokud si ho někdy přidáš do settings.xml jako 'watchlist_debug')
SHOW_DEBUG = _get_bool_setting('watchlist_debug', False)


# -----------------------------------------------------------------------------
# Pomocné funkce
# -----------------------------------------------------------------------------
def _infer_media_type():
    """
    Určí 'movie' nebo 'show'.
    Preferuje ListItem.Mediatype (protože u custom položek DBType často chybí), fallback DBType.
    """
    mt = xbmc.getInfoLabel('ListItem.Mediatype')  # 'movie', 'tvshow', 'episode', ...
    if mt == 'movie':
        return 'movie'
    if mt in ('tvshow', 'episode', 'season'):
        return 'show'
    dbt = xbmc.getInfoLabel('ListItem.DBType')
    if dbt == 'movie':
        return 'movie'
    return 'show'


def find_trakt_ids(title, year=None):
    """
    Interaktivní vyhledání titulů na Traktu a výběr uživatelem.
    Vrací tuple (imdb, tmdb) nebo (None, None).
    """
    try:
        search_type = _infer_media_type()  # 'movie' nebo 'show'

        # Použij i OriginalTitle, bývá přesnější; padáme na 1–2 dotazy.
        original_title = xbmc.getInfoLabel('ListItem.OriginalTitle')
        if not original_title or original_title == title:
            original_title = None

        queries = []
        if original_title:
            queries.append(original_title)
        queries.append(title)

        # Pro /search stačí public hlavičky (api-key). Zkusíme si je půjčit z get_trakt_headers().
        headers_public = {'trakt-api-version': '2'}
        h = get_trakt_headers() or {}
        if 'trakt-api-key' in h:
            headers_public['trakt-api-key'] = h['trakt-api-key']
        else:
            xbmcgui.Dialog().ok("Trakt", "Chybí Client ID (trakt-api-key). Ověřte nastavení/konstantu.")
            return None, None

        for q in queries:
            params = {'query': q, 'limit': 5}
            if year and str(year).isdigit():
                params['year'] = int(year)

            url = 'https://api.trakt.tv/search/{}'.format(search_type)
            try:
                resp = requests.get(url, params=params, headers=headers_public, timeout=10)
            except Exception as e:
                xbmc.log(f"[WL] Trakt search network error: {e}", xbmc.LOGWARNING)
                continue

            if resp.status_code != 200:
                xbmc.log(f"[WL] Trakt search HTTP {resp.status_code}", xbmc.LOGWARNING)
                continue

            results = resp.json() or []
            if not results:
                continue

            options = []
            for result in results:
                item = result.get(search_type, {}) or {}
                t = item.get('title', '')
                y = item.get('year', '')
                ids = item.get('ids', {}) or {}
                imdb = ids.get('imdb', 'N/A')
                tmdb = ids.get('tmdb', 'N/A')
                options.append(f"{t} ({y}) — IMDb: {imdb} | TMDb: {tmdb}")

            sel = xbmcgui.Dialog().select("Vyberte správný titul:", options)
            if sel >= 0:
                chosen = results[sel].get(search_type, {}) or {}
                ids = chosen.get('ids', {}) or {}
                xbmcgui.Dialog().notification(
                    "Trakt",
                    f"Používám: {chosen.get('title', title)} ({chosen.get('year', year or '')})",
                    xbmcgui.NOTIFICATION_INFO, 2000
                )
                return ids.get('imdb'), ids.get('tmdb')

            xbmcgui.Dialog().ok("Trakt", "Výběr zrušen.")
            return None, None

        return None, None

    except Exception as e:
        xbmc.log(f"[WL] Trakt search fatal: {e}", xbmc.LOGERROR)
        return None, None


def get_item_data():
    """
    Poskládá položku pro payload /sync/watchlist (jen s IDs).
    - Pokud ID (IMDb/TMDb) existují v ListItem, použije je.
    - Jinak spustí find_trakt_ids() (interaktivní výběr).
    Vrací (item_dict, dbtype) nebo (None, None).
    """
    title = xbmc.getInfoLabel('ListItem.Title') or xbmc.getInfoLabel('ListItem.Label')
    year = xbmc.getInfoLabel('ListItem.Year')
    imdb = xbmc.getInfoLabel('ListItem.IMDBNumber')          # např. 'tt1234567'
    tmdb = xbmc.getInfoLabel('ListItem.UniqueID(tmdb)')      # číslo
    dbtype = xbmc.getInfoLabel('ListItem.DBType')

    if not title:
        return None, None

    ids = {}
    if imdb and imdb.startswith('tt'):
        ids['imdb'] = imdb
    if tmdb and tmdb.isdigit():
        ids['tmdb'] = int(tmdb)

    if ids:
        return {'ids': ids}, dbtype

    # Nemáme ID → interaktivní dohledání
    xbmcgui.Dialog().notification("Trakt", "Hledám přesně...", xbmcgui.NOTIFICATION_INFO, 1500)
    found_imdb, found_tmdb = find_trakt_ids(title, year if year and str(year).isdigit() else None)
    if found_imdb or found_tmdb:
        ids = {}
        if found_imdb:
            ids['imdb'] = found_imdb
        if found_tmdb:
            ids['tmdb'] = int(found_tmdb)
        return {'ids': ids}, dbtype

    xbmcgui.Dialog().ok("Trakt", f"Nelze najít:\n{title} ({year})")
    return None, None


# -----------------------------------------------------------------------------
# Hlavní akce
# -----------------------------------------------------------------------------
def send_to_trakt(endpoint, action_name):
    """
    endpoint: "" (add) | "remove"
    action_name: hezké hlášení do notifikace
    """
    # 1) Jednotná autorizační kontrola + hlavičky (stejné jako scrobble.py)
    headers = get_trakt_headers(for_post=True)
    if not headers:
        xbmcgui.Dialog().ok("Trakt", "Trakt není autorizován.\nOvěřte přihlášení přes trakt.tv/activate.")
        return

    # 2) Získání položky + IDs (nebo interaktivní výběr)
    res = get_item_data()
    if not res:
        return
    item, dbtype = res

    # 3) Urči typ a postav payload
    search_type = _infer_media_type()  # 'movie' | 'show'
    payload_key = 'movies' if search_type == 'movie' else 'shows'
    payload = {payload_key: [item]}

    if SHOW_DEBUG:
        try:
            xbmc.log(f"[WL] Payload: {json.dumps(payload, ensure_ascii=False)}", xbmc.LOGINFO)
            xbmcgui.Dialog().ok("DEBUG", json.dumps(payload, indent=2, ensure_ascii=False))
        except Exception:
            pass

    # 4) URL pro add/remove
    base = 'https://api.trakt.tv/sync/watchlist'
    url = base if endpoint == "" else f"{base}/{endpoint}"

    # 5) Odeslání
    try:
        r = requests.post(url, json=payload, headers=headers, timeout=12)
        if r.status_code in (200, 201):
            xbmcgui.Dialog().notification("Trakt", f"{action_name}!", xbmcgui.NOTIFICATION_INFO, 2500)
            xbmc.executebuiltin('Container.Refresh')
        elif r.status_code == 204:
            xbmcgui.Dialog().notification("Trakt", "Žádná změna (už bylo / už tam není).", xbmcgui.NOTIFICATION_INFO, 2500)
            xbmc.executebuiltin('Container.Refresh')
        elif r.status_code in (401, 403):
            xbmcgui.Dialog().ok("Trakt", "Neplatný/expir. token. Přihlašte se znovu na trakt.tv/activate.")
        else:
            body = ''
            try:
                body = (r.text or '')[:300]
            except Exception:
                pass
            xbmcgui.Dialog().ok("CHYBA Trakt", f"HTTP {r.status_code}\n{body}")
    except Exception as e:
        xbmcgui.Dialog().ok("CHYBA", str(e))


def add_to_watchlist():
    send_to_trakt("", "Přidáno do Watchlist")


def remove_from_watchlist():
    send_to_trakt("remove", "Odebráno z Watchlist")


# -----------------------------------------------------------------------------
# Entrypoint (RunScript)
# -----------------------------------------------------------------------------
if __name__ == '__main__':
    action = sys.argv[1] if len(sys.argv) > 1 else ""
    if action == "add":
        add_to_watchlist()
    elif action == "remove":
        remove_from_watchlist()