# -*- coding: utf-8 -*-
import requests
import json
import random
import xbmc
from requests.exceptions import HTTPError

# --- 1. KONFIGURACE ---
# DŮLEŽITÉ: Zde je jen placeholder, v Kodi se má používat klíč z nastavení
API_URL = 'https://api.themoviedb.org/3'
IMAGE_BASE_URL_POSTER = 'https://image.tmdb.org/t/p/w500' 
IMAGE_BASE_URL_FANART = 'https://image.tmdb.org/t/p/original' 

# ROZŠÍŘENÝ SEZNAM KATEGORIÍ (vychází z tvého snippetu)
TMDB_BASIC_LISTS = {
    'movie_popular': {'tmdb_type': 'movie', 'path': '/movie/popular'},
    'tv_top_rated': {'tmdb_type': 'tv', 'path': '/tv/top_rated'},
    'movie_upcoming': {'tmdb_type': 'movie', 'path': '/movie/upcoming'},
    'tv_on_the_air': {'tmdb_type': 'tv', 'path': '/tv/on_the_air'},
    'tv_popular': {'tmdb_type': 'tv', 'path': '/tv/popular'}, 
    'movie_top_rated': {'tmdb_type': 'movie', 'path': '/movie/top_rated'},
    
    # 🎯 NOVINKA 1: Seriály, které se vysílají dnes
    'tv_airing_today': {'tmdb_type': 'tv', 'path': '/tv/airing_today'},    

    # 🎯 NOVINKA 2: Trendy za dnešek (Mix Filmy/Seriály)
    'trending_day': {'tmdb_type': 'all', 'path': '/trending/all/day'},
}
# --- 2. TŘÍDY TMDb API Klienta ---


# V souboru tmdb_module.py, uvnitř class TMDb:

class TMDb:
    def __init__(self, api_key):
        self.api_key = api_key

    def _call_api(self, api_path, page=1, extra_params={}):
        """
        Hlavní metoda pro volání TMDb API. (Tato je již u Vás funkční)
        """
        
        # Sestavení plné URL. api_path (např. /movie/popular) již obsahuje lomítko.
        url = API_URL + api_path
        
        params = {
            'api_key': self.api_key,
            'language': 'cs-CZ',
            'page': page
        }
        params.update(extra_params)
        
        # Volání API
        r = requests.get(url, params=params)
        xbmc.log(f"TMDb DEBUG: Volaná URL: {r.url}", xbmc.LOGDEBUG)

        # Kontrola, zda volání nevrátilo chybu
        try:
            r.raise_for_status()
        except HTTPError as e:
            # Speciální logování pro kód 34 (Zdroj nenalezen) a 401 (Neautorizováno)
            if r.status_code == 404: 
                 xbmc.log(f"TMDb ERROR: Zdroj nebyl nalezen (status 34) - zkontroluj {api_path}", xbmc.LOGERROR)
            if r.status_code == 401:
                xbmc.log(f"TMDb ERROR: API Klíč není platný (status 401).", xbmc.LOGERROR)

            raise e
            
        return r # Vrací se celý response objekt pro další zpracování
        
    def call(self, category_key, query=None, page=1):
        """
        Nový vstupní bod, který rozhoduje, zda se jedná o seznam nebo vyhledávání.
        """
        extra_params = {}
        api_path = None
        
        # 🎯 NOVÁ LOGIKA PRO VYHLEDÁVÁNÍ
        if category_key == 'search':
            if not query:
                raise ValueError("Vyhledávací dotaz 'q' chybí pro TMDb vyhledávání.")
            
            # Vyhledávací endpoint pro filmy a seriály ('multi' search)
            api_path = "/search/multi"
            extra_params['query'] = query
            xbmc.log(f"TMDb INFO: Volání TMDb SEARCH pro dotaz: {query}", xbmc.LOGINFO)
            
        else:
            # Původní logika pro získání seznamu z TMDB_BASIC_LISTS
            list_info = TMDB_BASIC_LISTS.get(category_key)
            if not list_info:
                raise ValueError(f"Neznámá kategorie: {category_key}")
            
            api_path = list_info['path']
        
        # Předáváme volání do stávající funkční metody _call_api
        return self._call_api(api_path, page, extra_params)
      

# V souboru tmdb_module.py, uvnitř class TMDb:
# ...

    def _call_api(self, api_path, page, extra_params=None):
        """
        Sestaví plnou URL pro jakýkoli dotaz, přidá číslo stránky a extra parametry.
        """
        full_url = f"{API_URL}{api_path}"
        
        final_params = {'page': page}
        if extra_params:
            final_params.update(extra_params)
        
        # Voláme stávající metodu _call_api_raw, která se stará o requests a kontrolu chyby
        return self._call_api_raw(full_url, final_params)


    def call(self, category_key, query=None, page=1):
        """
        Získá API cestu a extra parametry na základě klíče kategorie nebo vyhledávání.
        Vrací response objekt.
        """
        extra_params = {}
        api_path = None
        
        if category_key == 'search':
            if not query:
                # Tohle by se nemělo stát, protože jsme to ošetřili v tmdb_kodi.py
                raise ValueError("Vyhledávací dotaz 'q' chybí pro TMDb vyhledávání.")
            
            api_path = "/search/multi"
            extra_params['query'] = query # Zde se přidává vyhledávací dotaz
            xbmc.log(f"TMDb INFO: Volání TMDb SEARCH pro dotaz: {query}", xbmc.LOGINFO)
            
        elif category_key in TMDB_BASIC_LISTS:
            api_path = TMDB_BASIC_LISTS[category_key]['path']
            
        else:
            raise ValueError(f"TMDb ERROR: Neznámý klíč kategorie: {category_key}")

        # Nyní voláme novou robustní metodu _call_api
        return self._call_api(api_path, page, extra_params)

# V souboru tmdb_module.py, uvnitř class TMDb:
# ...

    def get_list(self, list_key, query=None, max_pages=3): # 🚨 OPRAVA: Přidán parametr query
        """Získá seznam dat pro daný klíč nebo vyhledávání z více stránek."""
        
        # Kontrola základních chyb (pokud není klíč a není to ani vyhledávání)
        if list_key != 'search' and list_key not in TMDB_BASIC_LISTS:
            xbmc.log(f"TMDb ERROR: Neznámý klíč seznamu: {list_key}", xbmc.LOGERROR)
            return []
            
        all_results = []
        list_processor = ListBasic(tmdb_api=self)
        
        # Opakujeme volání pro každou stránku až do max_pages
        for page_num in range(1, max_pages + 1):
            try:
                # 1. Zavoláme novou flexibilní metodu call()
                response = self.call(list_key, query=query, page=page_num)
                data = response.json()
                
                # 2. Zpracujeme výsledky a přidáme je do seznamu
                page_results = list_processor.process_results(data)
                all_results.extend(page_results)
                
                # 3. Zkontrolujeme, zda TMDb hlásí další stránky
                if page_num >= data.get('total_pages', max_pages):
                    break
                    
            except Exception as e:
                xbmc.log(f"TMDb EXCEPTION: Nepodarilo se nacist stranku {page_num}: {e}", xbmc.LOGERROR)
                break 
                
        return all_results




    def _call_api_raw(self, full_url, extra_params={}):
            """
            Volá API s jakoukoli plnou URL (pro detaily, sezóny atd.).
            Používá se, když nemáme definovanou cestu v TMDB_BASIC_LISTS.
            """
            params = {
                'api_key': self.api_key,
                'language': 'cs-CZ',
            }
            params.update(extra_params)
            
            r = requests.get(full_url, params=params)
            xbmc.log(f"TMDb DEBUG RAW: Volaná URL: {r.url}", xbmc.LOGDEBUG)

            try:
                r.raise_for_status()
            except HTTPError as e:
                xbmc.log(f"TMDb ERROR RAW CALL: {e}", xbmc.LOGERROR)
                raise e
                
            return r # Vrací se celý response objekt, který se pak zpracuje


class ListBasic:
    """Třída pro zpracování a normalizaci výsledků z TMDb pro Kodi."""
    def __init__(self, tmdb_api):
        self.tmdb_api = tmdb_api

    def process_results(self, data):
        """Zpracuje JSON data z TMDb a vytvoří seznam slovníků pro Kodi."""
        
        results = []
        for item in data.get('results', []):
            media_type = item.get('media_type', 'movie' if 'title' in item else 'tv') # Detekce typu média
            
            # Normalizace dat pro Kodi ListItem
            result = {
                'id': item.get('id'),
                'media_type': media_type,
                'label': item.get('title') or item.get('name'),
                'plot': item.get('overview'),
                'rating': item.get('vote_average'),
                'year': item.get('release_date', '').split('-')[0] or item.get('first_air_date', '').split('-')[0],
                'poster_path': item.get('poster_path'),
                'backdrop_path': item.get('backdrop_path'),
            }
            # Filtrování položek bez názvu nebo posteru
            if result['label'] and result['poster_path']:
                results.append(result)
                
        return results

# ... (zbytek tmdb_module.py)