
# -*- coding: utf-8 -*-
import os
import traceback
import xbmc
import xbmcgui
import xbmcplugin
import unidecode

from resources.lib.ui_utils import make_static_item
import series_manager
from resources.lib.download_queue import load_queue

# ──────────────────────────────────────────────────────────────────────────────
# Helper pro název vyhledávání
# ──────────────────────────────────────────────────────────────────────────────
def _safe_search_name(data: dict, fallback: str) -> str:
    """
    Vrátí vhodný název pro vyhledávání:
    - Preferuje original_title, pokud po unidecode zůstane čitelné (latinka)
    - Jinak použije title nebo fallback
    """
    tmdb = data.get('tmdb', {}) if isinstance(data, dict) else {}
    orig = tmdb.get('original_title') or ''
    title = tmdb.get('title') or fallback
    if orig:
        ascii_version = unidecode.unidecode(orig).strip()
        if ascii_version and len(ascii_version) >= 3:
            return ascii_version
    return title or fallback

# ──────────────────────────────────────────────────────────────────────────────
# Funkce pro vyhledávání seriálu
# ──────────────────────────────────────────────────────────────────────────────
def series_search(params, addon, handle, profile, get_url, api, revalidate, popinfo):
    """Search for a TV series and organize it into seasons and episodes"""
    token = revalidate()
    series_name = params.get('what') or xbmcgui.Dialog().input("Zadejte název seriálu")
    if not series_name:
        xbmcplugin.endOfDirectory(handle, succeeded=False)
        return

    sm = series_manager.SeriesManager(addon, profile)
    data = sm.load_series_data(series_name) or {}
    search_name = _safe_search_name(data, series_name)
    xbmc.log(f"[SeriesSearch] Používám název pro vyhledávání: '{search_name}' (původní: '{series_name}')", xbmc.LOGINFO)

    progress = xbmcgui.DialogProgress()
    progress.create('StreamManager', f'Vyhledávám seriál {search_name}...')
    try:
        series_data = sm.search_series(search_name, api, token)
        if not series_data or not series_data['seasons']:
            progress.close()
            popinfo('Nenalezeny žádné epizody tohoto seriálu', icon=xbmcgui.NOTIFICATION_WARNING)
            xbmcplugin.endOfDirectory(handle, succeeded=False)
            return
        progress.close()
        dialog = xbmcgui.Dialog()
        message = (f"Seriál '{series_name}' byl úspěšně nalezen a uložen.\n\n"
                   f"Nalezeno {sum(len(season) for season in series_data['seasons'].values())} epizod "
                   f"v {len(series_data['seasons'])} sezónách.")
        dialog.ok(addon.getAddonInfo('name'), message)
        xbmc.executebuiltin(f'Container.Update({get_url(action="series_detail", series_name=series_name)})')
    except Exception as e:
        progress.close()
        traceback.print_exc()
        popinfo(f'Chyba: {str(e)}', icon=xbmcgui.NOTIFICATION_ERROR)

# ──────────────────────────────────────────────────────────────────────────────
# Hlavní menu Seriály
# ──────────────────────────────────────────────────────────────────────────────
def series_menu(params, addon, handle, profile, get_url, api, revalidate, popinfo):
    xbmcplugin.setPluginCategory(handle, addon.getAddonInfo('name') + " \\\\ Seriály")
    sm = series_manager.SeriesManager(addon, profile)

    # (1) Hledat nový seriál
    li_new = make_static_item(
        label="Hledat nový seriál",
        plot=("Vyhledá nový seriál na Webshare.\n"
              "Uloží data do Správce seriálů pro další práci."),
        addon=addon,
        icon_name='search.png',
        default_icon='DefaultAddonsSearch.png'
    )
    li_new.setLabel2("Vyhledá nový seriál • uloží do Správce")
    xbmcplugin.addDirectoryItem(handle, get_url(action='series_search'), li_new, False)

    # (1a) Zrušit stahování + stav fronty
    try:
        q = load_queue() or []
        downloading_count = sum(1 for i in q if (i or {}).get('status') == 'downloading')
        pending_count = sum(1 for i in q if (i or {}).get('status') == 'pending')
    except Exception:
        downloading_count = 0
        pending_count = 0

    status_tag = f"[COLOR lime]↓ {downloading_count}[/COLOR] • [COLOR gray]ve frontě: {pending_count}[/COLOR]"
    li_cancel = make_static_item(
        label=f"Zrušit stahování {status_tag}",
        plot=("Okamžitě zastaví všechny probíhající stahovací úlohy.\n"
              "Použijte pouze v případě potřeby."),
        addon=addon,
        icon_name='clear_queue.png',
        default_icon='DefaultScript.png'
    )
    li_cancel.setLabel2("Zastaví všechny stahovací úlohy")
    xbmcplugin.addDirectoryItem(handle, get_url(action='cancel_download'), li_cancel, False)

    # (2) Rozkoukané
    li_inp = make_static_item(
        label="Rozkoukané",
        plot=("Zobrazí uložené seriály, které ještě nejsou plně dosledované.\n"
              "Umožní rychlý vstup do sezón a epizod."),
        addon=addon,
        icon_name='progress.png',
        default_icon='DefaultRecentlyAddedEpisodes.png'
    )
    li_inp.setLabel2("Seznam nedosledovaných seriálů")
    xbmcplugin.addDirectoryItem(handle, get_url(action='series_filtered', status='in_progress'), li_inp, True)

    # (3) Dosledované
    li_comp = make_static_item(
        label="Dosledované",
        plot=("Zobrazí uložené seriály, které jsou označené jako plně zhlédnuté.\n"
              "Přehled hotových titulů."),
        addon=addon,
        icon_name='finish.png',
        default_icon='DefaultTvShows.png'
    )
    li_comp.setLabel2("Seznam plně dosledovaných seriálů")
    xbmcplugin.addDirectoryItem(handle, get_url(action='series_filtered', status='completed'), li_comp, True)

    try:
        xbmcplugin.setContent(handle, 'tvshows')
    except Exception:
        pass

    xbmcplugin.endOfDirectory(handle)

# ──────────────────────────────────────────────────────────────────────────────
# Detail seriálu (sezóny)
# ──────────────────────────────────────────────────────────────────────────────
def series_detail(params, addon, handle, profile, get_url, api, revalidate, popinfo):
    xbmcplugin.setPluginCategory(handle, addon.getAddonInfo('name') + " \\\\ " + params['series_name'])
    sm = series_manager.SeriesManager(addon, profile)
    series_manager.create_seasons_menu(sm, handle, params['series_name'], build_url_fn=get_url)

# ──────────────────────────────────────────────────────────────────────────────
# Detail sezóny (epizody)
# ──────────────────────────────────────────────────────────────────────────────
def series_season(params, addon, handle, profile, get_url, api, revalidate, popinfo):
    xbmcplugin.setPluginCategory(handle, addon.getAddonInfo('name') + " \\\\ " +
                                 params['series_name'] + f" \\\\ Sezona {params['season']}")
    sm = series_manager.SeriesManager(addon, profile)
    series_manager.create_episodes_menu(sm, handle, params['series_name'], params['season'], build_url_fn=get_url)

# ──────────────────────────────────────────────────────────────────────────────
# Aktualizace seriálu
# ──────────────────────────────────────────────────────────────────────────────
def series_refresh(params, addon, handle, profile, get_url, api, revalidate, popinfo):
    token = revalidate()
    series_name = params['series_name']
    sm = series_manager.SeriesManager(addon, profile)
    data = sm.load_series_data(series_name) or {}
    orig_title = data.get('tmdb', {}).get('original_title') or ''
    title = data.get('tmdb', {}).get('title') or series_name
    search_name = orig_title if orig_title.isascii() else title
    xbmc.log(f"[SeriesRefresh] 1. pokus hledání: '{search_name}'", xbmc.LOGINFO)

    quality_mode = params.get('quality_mode', 'default')
    skip_high_quality = False
    force_hd_only = False
    name_only = False
    if quality_mode == '4k':
        pass
    elif quality_mode == 'hd':
        skip_high_quality = True
        force_hd_only = True
    elif quality_mode == 'name':
        name_only = True

    progress = xbmcgui.DialogProgress()
    progress.create('StreamManager', f'Aktualizuji seriál {search_name}...')
    try:
        series_data = sm.search_series(search_name, api, token,
                                       skip_high_quality=skip_high_quality,
                                       force_hd_only=force_hd_only,
                                       name_only=name_only)
        if (not series_data or not series_data.get('seasons')) and search_name != title:
            xbmc.log(f"[SeriesRefresh] Nenalezeno podle '{search_name}', zkouším fallback '{title}'", xbmc.LOGINFO)
            series_data = sm.search_series(title, api, token,
                                           skip_high_quality=skip_high_quality,
                                           force_hd_only=force_hd_only,
                                           name_only=name_only)
        if not series_data or not series_data.get('seasons'):
            progress.close()
            popinfo('Nenalezeny žádné epizody tohoto seriálu', icon=xbmcgui.NOTIFICATION_WARNING)
            xbmc.executebuiltin(f'Container.Update({get_url(action="series")})')
            return
        progress.close()
        episode_count = sum(len(season) for season in series_data["seasons"].values())
        mode_txt = {"4k": "4K", "hd": "HD only", "name": "jen podle názvu"}.get(quality_mode, "standardně")
        popinfo(f"Nalezeno {episode_count} epizod v {len(series_data['seasons'])} sezónách ({mode_txt})")
        xbmc.executebuiltin(f'Container.Update({get_url(action="series_detail", series_name=series_name)})')
    except Exception as e:
        progress.close()
        traceback.print_exc()
        popinfo(f'Chyba: {str(e)}', icon=xbmcgui.NOTIFICATION_ERROR)
        xbmc.executebuiltin(f'Container.Update({get_url(action="series")})')

# ──────────────────────────────────────────────────────────────────────────────
# Výběr režimu aktualizace
# ──────────────────────────────────────────────────────────────────────────────
def series_refresh_select(params, addon, handle, profile, get_url, api, revalidate, popinfo):
    series_name = params['series_name']
    dialog = xbmcgui.Dialog()
    choice = dialog.select("Vyberte režim aktualizace", [
        "2160p / 4K / HDR",
        "1080p + 720p",
        "Pouze podle názvu (SxxExx)"
    ])
    if choice < 0:
        return
    quality_mode = {0: '4k', 1: 'hd', 2: 'name'}[choice]
    xbmc.log(f"[SeriesRefreshSelect] Vybrán režim: {quality_mode}", xbmc.LOGINFO)
    xbmc.executebuiltin(
        f'RunPlugin({get_url(action="series_refresh", series_name=series_name, quality_mode=quality_mode)})'
    )

# ──────────────────────────────────────────────────────────────────────────────
# Odstranění seriálu
# ──────────────────────────────────────────────────────────────────────────────
def series_remove(params, addon, handle, profile, get_url, api, revalidate, popinfo):
    series_name = params['series_name']
    status = (params.get('status') or '').strip().lower()
    dialog = xbmcgui.Dialog()
    win = xbmcgui.Window(10000)
    guard_key = 'mm_last_series_removed'
    last_removed = win.getProperty(guard_key)
    if not dialog.yesno('StreamManager', f'Opravdu chcete odebrat seriál "{series_name}"?'):
        return
    sm = series_manager.SeriesManager(addon, profile)
    ok = sm.remove_series(series_name)
    if ok:
        if last_removed != series_name:
            xbmcgui.Dialog().notification(addon.getAddonInfo('name'), 'Seriál odstraněn.', xbmcgui.NOTIFICATION_INFO, 2500)
            win.setProperty(guard_key, series_name)
    else:
        xbmcgui.Dialog().notification(addon.getAddonInfo('name'), 'Nelze odstranit seriál.', xbmcgui.NOTIFICATION_WARNING, 2500)
    try:
        if status in ('in_progress', 'completed'):
            xbmc.executebuiltin(f'Container.Update({get_url(action="series_filtered", status=status)})')
        else:
            xbmc.executebuiltin(f'Container.Update({get_url(action="series")})')
    except Exception:
        xbmc.executebuiltin(f'Container.Update({get_url(action="series")})')

# ──────────────────────────────────────────────────────────────────────────────
# Odstranění sezóny
# ──────────────────────────────────────────────────────────────────────────────
def series_remove_season(params, addon, handle, profile, get_url, api, revalidate, popinfo):
    series_name = params.get('series_name')
    season_key = params.get('season')
    if not series_name or not season_key:
        xbmcgui.Dialog().notification(addon.getAddonInfo('name'), "Chybí parametry", xbmcgui.NOTIFICATION_ERROR)
        return
    sm = series_manager.SeriesManager(addon, profile)
    data = sm.load_series_data(series_name)
    if not data:
        xbmcgui.Dialog().notification(addon.getAddonInfo('name'), "Data seriálu nenalezena", xbmcgui.NOTIFICATION_ERROR)
        return
    if not xbmcgui.Dialog().yesno("StreamManager", f"Opravdu smazat sezónu {season_key} ze seriálu '{series_name}'?"):
        return
    try:
        if season_key in data.get('seasons', {}):
            del data['seasons'][season_key]
        if season_key in data.get('season_state', {}):
            del data['season_state'][season_key]
        safe_name = sm._safe_filename(series_name, data.get('tmdb', {}).get('id'))
        file_path = os.path.join(sm.series_db_path, f"{safe_name}.json")
        series_manager._atomic_write_json(file_path, data)
        xbmcgui.Dialog().notification(addon.getAddonInfo('name'), f"Sezóna {season_key} odstraněna", xbmcgui.NOTIFICATION_INFO, 2500)
    except Exception as e:
        xbmc.log(f"[SeriesRemoveSeason] CHYBA: {e}", xbmc.LOGERROR)
        xbmcgui.Dialog().notification(addon.getAddonInfo('name'), "Chyba při mazání sezóny", xbmcgui.NOTIFICATION_ERROR)
    xbmc.executebuiltin(f'Container.Update({get_url(action="series_detail", series_name=series_name)})')

# ──────────────────────────────────────────────────────────────────────────────
# Odstranění epizody
# ──────────────────────────────────────────────────────────────────────────────
def series_remove_episode(params, addon, handle, profile, get_url, api, revalidate, popinfo):
    series_name = params.get('series_name')
    season_key = str(params.get('season'))
    episode_key = str(params.get('episode'))
    if not series_name or not season_key or not episode_key:
        xbmcgui.Dialog().notification(addon.getAddonInfo('name'), "Chybí parametry", xbmcgui.NOTIFICATION_ERROR)
        return
    sm = series_manager.SeriesManager(addon, profile)
    data = sm.load_series_data(series_name)
    if not data:
        xbmcgui.Dialog().notification(addon.getAddonInfo('name'), "Data seriálu nenalezena", xbmcgui.NOTIFICATION_ERROR)
        return
    if not xbmcgui.Dialog().yesno("StreamManager",
                                  f"Opravdu smazat epizodu S{season_key}E{episode_key} ze seriálu '{series_name}'?"):
        return
    try:
        if season_key in data.get('seasons', {}) and episode_key in data['seasons'][season_key]:
            del data['seasons'][season_key][episode_key]
        sm._recalc_season_state(data, season_key)
        safe_name = sm._safe_filename(series_name, data.get('tmdb', {}).get('id'))
        file_path = os.path.join(sm.series_db_path, f"{safe_name}.json")
        series_manager._atomic_write_json(file_path, data)
        xbmcgui.Dialog().notification(addon.getAddonInfo('name'),
                                      f"Epizoda S{season_key}E{episode_key} odstraněna",
                                      xbmcgui.NOTIFICATION_INFO, 2500)
    except Exception as e:
        xbmc.log(f"[SeriesRemoveEpisode] CHYBA: {e}", xbmc.LOGERROR)
        xbmcgui.Dialog().notification(addon.getAddonInfo('name'), "Chyba při mazání epizody", xbmcgui.NOTIFICATION_ERROR)
    xbmc.executebuiltin(
        f'Container.Update({get_url(action="series_season", series_name=series_name, season=season_key)})'
    )
