
# -*- coding: utf-8 -*-
# Module: offline_movie
# Purpose: Seznam stažených filmů z dfolder včetně TMDb metadat + přehrání/mazání
# Author: mmirousek (modularizace)
# License: AGPL v3

import os
import re
import xbmc
import xbmcgui
import xbmcplugin
import xbmcaddon
import xbmcvfs

from urllib.parse import urlencode
from resources.lib.tmdb_utils import search_tmdb_movie, get_movie_meta

ADDON_ID = 'plugin.video.mmirousek'
ADDON = xbmcaddon.Addon(ADDON_ID)

# Pevná základní URL pluginu – aby kontextové akce měly vždy platný cíl
PLUGIN_BASE = f"plugin://{ADDON_ID}"

# Přípustné filmové přípony
MOVIE_EXTS = {'.mkv', '.mp4', '.avi', '.mov', '.m4v', '.mpg', '.mpeg', '.ts', '.m2ts', '.webm'}

MIME_MAP = {
    '.mkv': 'video/x-matroska',
    '.mp4': 'video/mp4',
    '.avi': 'video/x-msvideo',
    '.m4v': 'video/mp4',
    '.ts': 'video/MP2T',
    '.mov': 'video/quicktime',
    '.webm': 'video/webm',
}

def _build_url(**kwargs) -> str:
    """
    Sestaví správné plugin URL: plugin://plugin.video.mmirousek?... (RunPlugin očekává platný PluginPath).
    """
    try:
        return f"{PLUGIN_BASE}?{urlencode(kwargs, encoding='utf-8')}"
    except Exception:
        # Fallback bez explicitního encodingu (starší buildy Pythonu v Kodi)
        return f"{PLUGIN_BASE}?{urlencode(kwargs)}"

def _listdir_files(root: str) -> list:
    """Vrátí absolutní cesty k souborům v kořenu složky – plně kompatibilní se SMB/NFS."""
    if not root or not xbmcvfs.exists(root):
        return []
    try:
        dirs, files = xbmcvfs.listdir(root)
        # Zajistíme správné složení cesty i pro smb:// a nfs://
        return [root.rstrip('/\\') + '/' + f for f in files]
    except Exception as e:
        xbmc.log(f"[OfflineMovies] _listdir_files error: {root} → {e}", xbmc.LOGERROR)
        return []
    
def _nice_title(meta: dict, fallback_name: str) -> str:
    """Title (Year), jinak fallback na filename bez přípony."""
    t = (meta or {}).get('title') or ''
    y = (meta or {}).get('year')
    if t and isinstance(y, int):
        return f"{t} ({y})"
    if t:
        return t
    return os.path.splitext(fallback_name)[0]

def _plot(meta: dict, fallback_name: str) -> str:
    p = (meta or {}).get('overview') or ''
    return p if p else f"Lokální soubor: {fallback_name}"

def _genres(meta: dict) -> str:
    g = (meta or {}).get('genres') or []
    try:
        return ", ".join([s for s in g if isinstance(s, str)]) if g else ""
    except Exception:
        return ""

def _poster(meta: dict) -> str:
    return (meta or {}).get('poster_url') or ''

def _fanart(meta: dict) -> str:
    return (meta or {}).get('backdrop_url') or ''


# ------------------------------------------------------------
# Veřejné funkce volané z routeru (yawsp.py → ACTIONS)
# ------------------------------------------------------------


def show_downloaded_movies(handle: int) -> None:
    """Zobrazí stažené filmy z nastavené dfolder – nyní funguje i na síťových discích."""
    xbmcplugin.setPluginCategory(handle, "Stažené filmy")
    dfolder = ADDON.getSetting('dfolder') or ''
    if not dfolder:
        xbmcgui.Dialog().notification("Stažené filmy", "Nenastavena složka pro stahování.", xbmcgui.NOTIFICATION_WARNING, 3000)
        xbmcplugin.endOfDirectory(handle, succeeded=False)
        return

    dest_real = xbmcvfs.translatePath(dfolder)
    if not dest_real:
        xbmcgui.Dialog().notification("Stažené filmy", "Chyba při získání cesty.", xbmcgui.NOTIFICATION_ERROR, 3000)
        xbmcplugin.endOfDirectory(handle, succeeded=False)
        return

    # Zajistíme trailing slash
    if not dest_real.endswith(('/', '\\')):
        dest_real += '/'

    # Kontrola existence přes xbmcvfs (funguje i na SMB/NFS)
    if not xbmcvfs.exists(dest_real):
        xbmcgui.Dialog().notification("Stažené filmy", "Cílová složka neexistuje nebo není dostupná.", xbmcgui.NOTIFICATION_ERROR, 4000)
        xbmcplugin.endOfDirectory(handle, succeeded=False)
        return

    files = [f for f in _listdir_files(dest_real) if os.path.splitext(f)[1].lower() in MOVIE_EXTS]
    if not files:
        xbmcgui.Dialog().notification("Stažené filmy", "Žádné stažené filmy.", xbmcgui.NOTIFICATION_INFO, 2500)
        xbmcplugin.endOfDirectory(handle, succeeded=True)
        return

    # Pokud je složka velká, přeskočíme TMDb lookup
    skip_tmdb = len(files) > 50

    for fpath in sorted(files, key=lambda p: os.path.basename(p).lower()):
        fname = os.path.basename(fpath)
        title = os.path.splitext(fname)[0]
        plot = f"Lokální soubor: {fname}"
        art = {}

        if not skip_tmdb:
            try:
                basic = search_tmdb_movie(fname)
                if basic:
                    title = basic.get('title') or title
                    plot = basic.get('overview') or plot
                    poster = basic.get('poster_path')
                    if poster:
                        art['poster'] = f"https://image.tmdb.org/t/p/w500{poster}"
                        art['thumb'] = art['poster']
                        art['icon'] = art['poster']
            except Exception as e:
                xbmc.log(f"[OfflineMovies] TMDb lookup error for {fname}: {e}", xbmc.LOGWARNING)

        li = xbmcgui.ListItem(label=title)
        li.setInfo('video', {'title': title, 'plot': plot, 'mediatype': 'movie'})
        if art:
            li.setArt(art)
        li.setProperty('IsPlayable', 'true')

        # Kontextové menu
        delete_url = _build_url(action='delete_downloaded_file', path=fpath)
        ctx = [
            ("Smazat soubor", f"RunPlugin({delete_url});Container.Refresh"),
            ("Otevřít složku", f"RunPlugin({_build_url(action='show_folder', path=os.path.dirname(fpath))})")
        ]
        li.addContextMenuItems(ctx)

        xbmcplugin.addDirectoryItem(handle, fpath, li, isFolder=False)

    xbmcplugin.endOfDirectory(handle)

def delete_downloaded_file(path: str) -> None:
    """Bezpečné mazání souboru – funguje i na SMB/NFS."""
    try:
        # path už přichází jako plná VFS cesta (smb:// nebo lokální)
        if xbmcvfs.exists(path):
            xbmcvfs.delete(path)
            xbmcgui.Dialog().notification("Stažené filmy", "Soubor smazán.", xbmcgui.NOTIFICATION_INFO, 2500)
        else:
            xbmcgui.Dialog().notification("Stažené filmy", "Soubor nenalezen.", xbmcgui.NOTIFICATION_WARNING, 2500)
    except Exception as e:
        xbmc.log(f"[OfflineMovies] Chyba při mazání {path}: {e}", xbmc.LOGERROR)
        xbmcgui.Dialog().notification("Stažené filmy", "Chyba při mazání souboru.", xbmcgui.NOTIFICATION_ERROR, 3000)
    finally:
        xbmc.executebuiltin("Container.Refresh")

def show_folder(handle: int, path: str) -> None:
    """Jednoduchý náhled obsahu složky (klik na kontext 'Otevřít složku')."""
    real = xbmcvfs.translatePath(path) or path
    if not real or not os.path.isdir(real):
        xbmcgui.Dialog().notification("Složka", "Nelze otevřít složku.", xbmcgui.NOTIFICATION_WARNING, 2500)
        return

    xbmcplugin.setPluginCategory(handle, os.path.basename(real) or "Složka")
    try:
        for fname in sorted(os.listdir(real)):
            full = os.path.join(real, fname)
            li = xbmcgui.ListItem(label=fname)
            if os.path.isfile(full):
                li.setProperty('IsPlayable', 'true')
                # Přímé přehrání souboru
                xbmcplugin.addDirectoryItem(handle, full, li, isFolder=False)
            else:
                # Zanořený listing – opět volat přes router tuto funkci
                xbmcplugin.addDirectoryItem(handle, _build_url(action='show_folder', path=full), li, isFolder=True)
    except Exception as e:
        xbmc.log(f"[OfflineMovies] show_folder error: {e}", xbmc.LOGERROR)
        xbmcgui.Dialog().notification("Složka", "Chyba při listování.", xbmcgui.NOTIFICATION_ERROR, 2500)

    xbmcplugin.endOfDirectory(handle)
