# resources/lib/local_history.py
import json
import xbmcaddon
import xbmcvfs
import os
import xbmc
import time
from resources.lib import tmdb_utils
import xbmcgui

try:
    from urllib.parse import urlencode
except ImportError:
    from urllib import urlencode  # Py2 fallback

# --- KONSTANTY ---
ADDON_ID = 'plugin.video.mmirousek'
LOG_TAG = '[YaWSP: Local History]'

# Cesta k profilu doplňku (e.g., .../userdata/addon_data/plugin.video.mmirousek/)
PROFILE_PATH = xbmcvfs.translatePath(xbmcaddon.Addon(ADDON_ID).getAddonInfo('profile'))
HISTORY_FILE = os.path.join(PROFILE_PATH, 'watch_history.json')


# ---------------------------------------------------------------------------
# FUNKCE PRO PRÁCI SE SOUBOREM (Load/Store)
# ---------------------------------------------------------------------------

def load_history():
    """Načte historii sledování ze souboru. Vrací slovník s klíči 'filmy' a 'serialy'."""
    try:
        if not os.path.exists(HISTORY_FILE):
            xbmc.log(f'{LOG_TAG} Soubor historie nenalezen v: {HISTORY_FILE}. Vracím prázdná data.', xbmc.LOGINFO)
            return {"filmy": [], "serialy": []}
        with open(HISTORY_FILE, 'r', encoding='utf-8') as f:
            data = json.load(f)
        filmy_count = len(data.get("filmy", []))
        serialy_count = len(data.get("serialy", []))
        xbmc.log(f'{LOG_TAG} Načteno z DB: Filmy: {filmy_count}, Seriály: {serialy_count}', xbmc.LOGINFO)
        # Zajištění, že se vždy vrátí struktura se seznamy
        return {
            "filmy": data.get("filmy", []),
            "serialy": data.get("serialy", [])
        }
    except Exception as e:
        xbmc.log(f'{LOG_TAG} FATÁLNÍ chyba při načítání historie: {e}', xbmc.LOGERROR)
        return {"filmy": [], "serialy": []}


def store_history(data):
    """Uloží aktuální data historie do souboru, zajistí existenci adresáře."""
    try:
        # 💡 ZAJISTÍ EXISTENCI ADRESÁŘE
        if not os.path.exists(PROFILE_PATH):
            os.makedirs(PROFILE_PATH)
            xbmc.log(f'{LOG_TAG} Adresář profilu vytvořen: {PROFILE_PATH}', xbmc.LOGINFO)
        # Uložení dat
        with open(HISTORY_FILE, 'w', encoding='utf-8') as f:
            json.dump(data, f, ensure_ascii=False, indent=4)
        xbmc.log(f'{LOG_TAG} Historie úspěšně uložena do: {HISTORY_FILE}', xbmc.LOGDEBUG)
    except Exception as e:
        xbmc.log(f'{LOG_TAG} FATÁLNÍ chyba při ukládání historie: {e}', xbmc.LOGERROR)


# ---------------------------------------------------------------------------
# HLAVNÍ FUNKCE (Business Logic)
# ---------------------------------------------------------------------------

def store_item(item_data):
    """
    Uloží nebo aktualizuje položku v lokální historii.
    item_data by ideálně obsahovala: tmdb_id, progress, media_type, name, stream_url.
    NOVĚ: pokud tmdb_id chybí, uložíme s tmdb_id='0' a deduplikujeme podle stream_url+name (filmy).
    """
    history = load_history()

    # 1. Normalizace klíčových dat
    tmdb_id = str(item_data.get('tmdb_id') or '').strip()
    if not tmdb_id:
        tmdb_id = '0'  # syntetické ID pro "bez identifikace"

    media_type = item_data.get('media_type', 'movie')  # Fallback na movie
    list_key = 'serialy' if media_type == 'episode' else 'filmy'
    item_list = history.get(list_key, [])

    # Zajištění přítomnosti klíčů
    item_data['watched_at'] = time.strftime('%Y-%m-%dT%H:%M:%S', time.localtime())
    item_data['stream_url'] = item_data.get('stream_url', '')
    # 💡 NOVÝ KLÍČ: Uložíme Trakt payload (pokud je k dispozici)
    item_data['trakt_payload'] = item_data.get('trakt_payload', {})

    # 2. Hledání duplicity a odstranění staré položky
    if list_key == 'filmy':
        # Filmy: primárně podle tmdb_id; když není (tmdb_id='0'), dedup podle (stream_url + name)
        if tmdb_id != '0':
            new_list = [item for item in item_list if str(item.get('tmdb_id')) != tmdb_id]
        else:
            su = (item_data.get('stream_url') or '').strip()
            nm = (item_data.get('name') or '').strip().lower()

            def is_same_movie(item):
                return ((item.get('stream_url') or '').strip() == su and
                        (item.get('name') or '').strip().lower() == nm)

            new_list = [item for item in item_list if not is_same_movie(item)]

    elif list_key == 'serialy':
        # Seriály: Hledáme podle tmdb_id, sezóny a epizody
        s_num = item_data.get('season_num')
        e_num = item_data.get('episode_num')

        def is_same_episode(item):
            return (str(item.get('tmdb_id')) == tmdb_id and
                    item.get('season_num') == s_num and
                    item.get('episode_num') == e_num)

        new_list = [item for item in item_list if not is_same_episode(item)]

    else:
        return

    # 3. Přidání nové (aktualizované) položky na začátek seznamu ("naposledy sledované")
    new_list.insert(0, item_data)

    # 4. Uložení upraveného seznamu
    history[list_key] = new_list
    store_history(history)


# 💡 NOVÁ FUNKCE pro načtení pro Kodi list
def get_list_items(media_type='movie'):
    """
    Vrátí seznam položek pro Kodi (tuple: (url, ListItem, isFolder)) podle typu (movie/episode).
    - Bez závislosti na yawsp.get_url() (žádný cirkulární import)
    - URL staví přes plugin base + urlencode
    - Přidá artwork (poster z TMDb)
    - Přidá stav [OK]/[75%]/[New] + Trakt status (přes add_status)
    - Kontextové menu: Smazat z historie (+ u filmů fix&scrobble)
    """
    items = []
    history = load_history()

    # Spoj filmy + seriály, seřaď podle watched_at (nové nahoře)
    all_items = (history.get('filmy', []) or []) + (history.get('serialy', []) or [])

    def _ts(i):
        return i.get('watched_at') or ''
    all_items.sort(key=_ts, reverse=True)

    PLUGIN_BASE = 'plugin://plugin.video.mmirousek/'

    for idx, item in enumerate(all_items):
        if item.get('media_type') != media_type:
            continue

        # Stav a label
        status = add_status(item)
        label = f"{item.get('name', 'Neznámý')} {status}"
        li = xbmcgui.ListItem(label=label)

        # Artwork
        poster = item.get('poster_path')
        if poster:
            poster_url = f"https://image.tmdb.org/t/p/w500{poster}"
            li.setArt({'icon': poster_url, 'thumb': poster_url, 'poster': poster_url})

        # InfoLabels (pozn.: setInfo je v Kodi 21+ deprecated, ale zatím tolerovatelné)
        li.setInfo('Video', {
            'title': item.get('name', ''),
            'plot': item.get('plot', 'Položka z lokální historie.'),
            'mediatype': 'movie' if media_type == 'movie' else 'episode'
        })

        # Kontextové menu (RunPlugin URL si postavíme ručně)
        cm = []
        # Smazat z historie
        delete_qs = urlencode({
            'action': 'delete_history_item',
            'tmdb_id': item.get('tmdb_id'),
            'media_type': item.get('media_type'),
        }, encoding='utf-8')
        cm.append(("Smazat z historie", f"RunPlugin({PLUGIN_BASE}?{delete_qs})"))

        # Fix & scrobble (jen pro filmy)
        if media_type == 'movie':
            fix_qs = urlencode({
                'action': 'fix_and_scrobble_item',
                'index': idx,
                'media_type': 'movie',
            }, encoding='utf-8')
            cm.append(("Opravit název a odeslat na Trakt", f"RunPlugin({PLUGIN_BASE}?{fix_qs})"))
        li.addContextMenuItems(cm, replaceItems=False)

        # URL pro přehrání z historie (ručně)
        play_qs = urlencode({
            'action': 'play_history_item',
            'url': item.get('stream_url', ''),
            'tmdb_id': item.get('tmdb_id'),
            'media_type': item.get('media_type'),
        }, encoding='utf-8')
        play_url = f"{PLUGIN_BASE}?{play_qs}"

        li.setProperty('IsPlayable', 'true')
        items.append((play_url, li, False))

    # ⬇️ return až po zpracování všech položek
    return items


# 💡 NOVÁ FUNKCE pro mazání
def remove_item(tmdb_id, media_type):
    """Odstraní položku/seriál z historie podle ID a typu."""
    history = load_history()
    tmdb_id = str(tmdb_id)
    list_key = 'serialy' if media_type == 'episode' else 'filmy'
    item_list = history.get(list_key, [])
    # Vytvoříme nový seznam, který neobsahuje položky s daným tmdb_id
    new_list = [item for item in item_list if str(item.get('tmdb_id')) != tmdb_id]
    # Uložíme upravený seznam
    history[list_key] = new_list
    store_history(history)
    xbmc.log(f'{LOG_TAG} Položka/Seriál {tmdb_id} smazána z historie.', xbmc.LOGINFO)
    return True


# resources/lib/local_history.py
# ... na konci souboru ...
def get_single_item(tmdb_id, media_type):
    """Získá jednu položku z historie podle ID a typu."""
    history = load_history()
    tmdb_id_str = str(tmdb_id)
    list_key = 'serialy' if media_type == 'episode' else 'filmy'
    item_list = history.get(list_key, [])
    for item in item_list:
        if str(item.get('tmdb_id')) == tmdb_id_str:
            # Pro seriály by se měla kontrolovat i sezóna/epizoda, ale stačí ID
            return item
    return None


# ===================================================================
# NOVÉ FUNKCE (pouze do local_history.py)
# ===================================================================
def add_status(item):
    """
    Vrátí status jako string: [ok], [75%], [Nový] + Trakt OK/✗
    Používá se v show_history_list() v yawsp.py
    """
    progress = item.get('progress', 0.0)
    payload = item.get('trakt_payload', {})

    # --- PROGRESS ---
    if progress >= 95.0:
        prog_status = u'[COLOR lime]ok[/COLOR]'
    elif progress > 5.0:
        prog_status = f'[{progress:.0f}%]'
    else:
        prog_status = '[Nový]'

    # --- TRAKT ---
    if payload:
        trakt_status = '[COLOR lime]T OK[/COLOR]'
    else:
        trakt_status = '[COLOR red]T ne[/COLOR]'

    return f"{prog_status} {trakt_status}"


def fix_and_scrobble_item(item_index, _media_type):
    """
    Opraví položku v lokální historii (doplní TMDb/TVDB ID, název, plakát, děj),
    uloží ji zpět, připraví správný Trakt *scrobble* payload a ihned spustí
    scrobble_start() (přes 'trakt_launch_history_item').
    :param item_index: index položky v seznamu (podle 'filmy' nebo 'serialy')
    :param _media_type: 'movie' nebo 'episode' (typ přepíná TMDb hledání)
    """
    history = load_history()
    list_key = 'serialy' if _media_type == 'episode' else 'filmy'
    items = history.get(list_key, [])
    if item_index < 0 or item_index >= len(items):
        xbmcgui.Dialog().ok("Chyba", "Položka nenalezena.")
        return

    item = items[item_index]
    current_name = item.get('name', 'Neznámý název')

    # Zjisti skutečný typ média z položky (fallback na parametr)
    media_type = item.get('media_type', _media_type or 'movie')
    if media_type not in ['movie', 'episode']:
        xbmc.log(f'{LOG_TAG} Neplatný media_type: {media_type}. Opravuji na "movie".', xbmc.LOGWARNING)
        media_type = 'movie'

    # --- Zadání správného názvu od uživatele ---
    kb = xbmc.Keyboard(current_name, 'Opravit název pro TMDb')
    kb.doModal()
    if not kb.isConfirmed():
        return
    search_name = (kb.getText() or '').strip()
    if not search_name:
        xbmcgui.Dialog().ok("Chyba", "Název nesmí být prázdný.")
        return

    # --- Hledání na TMDb ---
    xbmc.executebuiltin('ActivateWindow(busydialognocancel)')
    try:
        if media_type == 'episode':
            result = tmdb_utils.search_tmdb_tv(search_name)
        else:
            result = tmdb_utils.search_tmdb_movie(search_name)
    finally:
        xbmc.executebuiltin('Dialog.Close(busydialognocancel)')

    if not result:
        xbmcgui.Dialog().ok("Chyba", f"Nenašel jsem na TMDb:\n{search_name}")
        return

    # --- Sestavení nové (opravené) položky ---
    fixed_item = {
        'tmdb_id': str(result['tmdb_id']),
        'name': result.get('title', search_name),
        'media_type': media_type,
        'progress': item.get('progress', 0.0),
        'stream_url': item.get('stream_url', ''),  # pro přehrání z historie
        'poster_path': result.get('poster_path'),  # může být None (OK)
        'plot': result.get('overview', ''),        # může být prázdné
        'watched_at': item.get('watched_at', time.strftime('%Y-%m-%dT%H:%M:%S', time.localtime())),
    }

    # Seriálová metadata + SCROBBLE payload
    if media_type == 'episode':
        fixed_item['season_num'] = item.get('season_num')
        fixed_item['episode_num'] = item.get('episode_num')
        # Preferuj TVDB ID pro show, fallback na TMDb ID
        show_ids = {}
        if result.get('tvdb_id'):
            try:
                show_ids['tvdb'] = int(result['tvdb_id'])
            except Exception:
                pass
        if not show_ids:
            # fallback: TMDb ID
            try:
                show_ids['tmdb'] = int(result['tmdb_id'])
            except Exception:
                pass
        fixed_item['trakt_payload'] = {
            "show": {"ids": show_ids},
            "episode": {
                "season": fixed_item['season_num'],
                "number": fixed_item['episode_num']
            }
        }
    else:
        # Film – čisté scrobble schema
        try:
            movie_tmdb_id = int(result['tmdb_id'])
        except Exception:
            movie_tmdb_id = int(fixed_item['tmdb_id']) if fixed_item['tmdb_id'].isdigit() else 0
        fixed_item['trakt_payload'] = {
            "movie": {"ids": {"tmdb": movie_tmdb_id}}
        }

    # --- Přepsání položky v historii ---
    try:
        del items[item_index]
        history[list_key] = items
        store_history(history)
    except Exception as e:
        xbmc.log(f'{LOG_TAG} WARN: Nelze odstranit původní položku: {e}', xbmc.LOGWARNING)

    store_item(fixed_item)  # idempotentní: ve store_item jsou ošetřené duplicity

    # --- Předání do scrobble.py přes 'trakt_launch_history_item' ---
    try:
        addon = xbmcaddon.Addon(ADDON_ID)
        addon.setSetting('trakt_launch_history_item', json.dumps(fixed_item))
    except Exception as e:
        xbmc.log(f'{LOG_TAG} Chyba při ukládání trakt_launch_history_item: {e}', xbmc.LOGERROR)

    # --- Spuštění scrobble START (odeslání na Trakt) ---
    try:
        from resources.lib.scrobble import scrobble_start
        scrobble_start(None)  # čte si fixed_item ze Settings; nečekáme návratovou hodnotu
        xbmcgui.Dialog().notification("Trakt", "Odesláno!", xbmcgui.NOTIFICATION_INFO, 2500)
    except Exception as e:
        xbmc.log(f'{LOG_TAG} Scrobble start selhal: {e}', xbmc.LOGERROR)
        xbmcgui.Dialog().ok("Upozornění",
                            "ID bylo opraveno a uloženo,\n"
                            "ale odeslání na Trakt se nezdařilo.\n"
                            "Zkontroluj autorizaci v Nastavení.")

    # --- Obnovení UI ---
    xbmc.executebuiltin('Container.Refresh')